//+------------------------------------------------------------------+
//|                                                    VolumeProfile |
//|                                  Copyright 2024, Yashar Seyyedin |
//|                    https://www.mql5.com/en/users/yashar.seyyedin |
//+------------------------------------------------------------------+

class VolumeProfile
  {
private:
   long              volumes[];
   datetime          from;
   datetime          to;
   datetime          resolution_points;
   int               from_index;
   int               to_index;
   double            highest;
   double            lowest;
   int               levels_count;
   double            HV_price;
   long              HV_volume;
public:
   void              VolumeProfile(datetime _from, datetime _to, int _resolution_points);
                    ~VolumeProfile() {};
   double            GetHVPrice();
   void              Plot();
private:
   double            HighestPrice();
   double            LowestPrice();
  };

//+------------------------------------------------------------------+
//|     constructor                                                  |
//+------------------------------------------------------------------+
void VolumeProfile::VolumeProfile(datetime _from, datetime _to, int _resolution_points)
  {
   ResetLastError();
   from=_from;
   to=_to;
   resolution_points=_resolution_points;
   from_index=iBarShift(_Symbol, PERIOD_M1, from, false);
   to_index=iBarShift(_Symbol, PERIOD_M1, to, false);
   highest = HighestPrice();
   lowest = LowestPrice();
   levels_count=(int)((highest-lowest)/(resolution_points*_Point));
   ArrayResize(volumes, levels_count);
   ArrayInitialize(volumes, 0);

   for(int j=from_index;j>=to_index;j--)
     {
      double price=iClose(_Symbol, PERIOD_M1, j);
      long vol=iVolume(_Symbol, PERIOD_M1, j);
      int index = (int)((price-lowest)/(resolution_points*_Point));
      if(index>=levels_count || index<0)
        {
         continue;
        }
      volumes[index]=volumes[index]+vol;
     }

   int max_vol_index=ArrayMaximum(volumes, 0, WHOLE_ARRAY);
   HV_volume=volumes[max_vol_index];
   HV_price = lowest+max_vol_index*resolution_points*_Point;
  }

//+------------------------------------------------------------------+
//|     Get price related to highest volume in range                 |
//+------------------------------------------------------------------+
double VolumeProfile::GetHVPrice()
  {
   return HV_price;
  }

//+------------------------------------------------------------------+
//|     Get highest price in range                                   |
//+------------------------------------------------------------------+
double VolumeProfile::HighestPrice()
  {
   double high[];
   ArraySetAsSeries(high,true);
   CopyHigh(_Symbol, PERIOD_M1, to_index, from_index-to_index, high);
   return high[ArrayMaximum(high, 0, WHOLE_ARRAY)];
  }

//+------------------------------------------------------------------+
//|     Get lowest price in range                                    |
//+------------------------------------------------------------------+
double VolumeProfile::LowestPrice()
  {
   double low[];
   ArraySetAsSeries(low,true);
   CopyLow(_Symbol, PERIOD_M1, to_index, from_index-to_index, low);
   return low[ArrayMinimum(low, 0, WHOLE_ARRAY)];
  }

//+------------------------------------------------------------------+
//|     Plot the volumes                                             |
//+------------------------------------------------------------------+
void VolumeProfile::Plot()
  {
   string name="VP_"+TimeToString(from)+TimeToString(to);
   ObjectsDeleteAll(ChartID(),name);
   for(int i=0;i<levels_count;i++)
     {
      double price=lowest+i*resolution_points*_Point;
      long volume=volumes[i];
      double ratio=(double)volume/(double)HV_volume;
      double _to=from+ratio*(to-from);
      ObjectCreate(ChartID(), name+(string)i,OBJ_TREND, 0, from, price, _to, price);
      ObjectSetInteger(ChartID(), name+(string)i,OBJPROP_RAY_RIGHT, false);
      ObjectSetInteger(ChartID(), name+(string)i,OBJPROP_COLOR, clrLightGray);
     }
   //ObjectCreate(ChartID(), name+"HLINE",OBJ_HLINE, 0, TimeCurrent(), HV_price);
  }
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
