//+------------------------------------------------------------------+
//|                                           ConstFileUnlimited.mq5 |
//+------------------------------------------------------------------+
#property library
#property copyright "TheXpert"
#property link      "theforexpert@gmail.com"
#property version   "1.00"

#include <TheXpert/FileUnlimitedConstants.mqh>
#include <TheXpert/StringUtils.mqh>

#import "kernel32.dll"
   bool   CloseHandle(int hObject);
   int    CreateFileW(     string lpFileName,         uint dwDesiredAccess, 
                           uint dwShareMode,          uint lpSecurityAttributes,
                           uint dwCreationDisposition,uint dwFlagsAndAttributes, 
                           int hTemplateFile
                     );
   uint   GetFileAttributesW(string lpFileName);
   uint   GetFileSize(int hFile, uint& lpFileSizeHigh[]);
   bool   GetFileTime(int hFile, uint& lpCreationTime[], uint& lpLastAccessTime[], uint& lpLastWriteTime[]);
   bool   ReadFile(int hFile, ushort& lpBuffer[], uint nNumberOfBytesToRead, uint& lpNumberOfBytesRead[], int lpOverlapped);
   uint   SetFilePointer(int hFile, int lDistanceToMove, int& lpDistanceToMoveHigh[], uint dwMoveMethod);
   bool   FileTimeToSystemTime(const uint& lpFileTime[], MqlDateTime& lpSystemTime[]);
#import

//========================================================


class ConstFileImpl
   : public ConstFile
{
public:
   ConstFileImpl(const string& path);
   
   bool IsValid() const 
   {
      return m_hFile != INVALID_HANDLE;
   }
   
   ~ConstFileImpl()
   {
      if (m_hFile != INVALID_HANDLE)
      {
         CloseHandle(m_hFile);
      }
   }
   
   virtual ulong Size() const 
   {
      return m_Size;
   }
   
   virtual string Path() const
   {
      return m_Path;
   }

   virtual datetime CreationTime() const 
   {
      return m_CreationTime;
   }
   
   virtual datetime LastAccessTime() const 
   {
      return m_LastAccessTime;
   }
   
   virtual datetime LastModifyTime() const 
   {
      return m_LastModifyTime;
   }
   
   virtual bool Read(bool& value, string separators);
   virtual bool Read(long& value, string separators);
   virtual bool Read(double& value, string separators);
   virtual bool Read(string& value, string separators);
   virtual bool Read(datetime& value, string separators);
   virtual bool Read(color& value, string separators);

   virtual bool SetPos(long pos, MovePointerMethod from = MPM_BEGIN);
   virtual string ToString();
   
private:
   void Init();
   bool ReadBlock();
   
   bool ReadString(string& value, string separators);
   
   void FixEndOfLine();

private:
   int m_hFile;
   ulong m_Size;
   datetime m_CreationTime;
   datetime m_LastAccessTime;
   datetime m_LastModifyTime;
   
   string m_Buffer;
   ulong m_Pos;
   uint m_Attributes;
   string m_Path;
};

//========================================================

ConstFileImpl::ConstFileImpl(const string& path)
   : ConstFile(path)
   , m_hFile(INVALID_HANDLE)
   , m_Size(0)
   , m_CreationTime(0)
   , m_LastAccessTime(0)
   , m_LastModifyTime(0)
   , m_Buffer("")
   , m_Pos(0)
   , m_Attributes(INVALID_FILE_ATTRIBUTES)
   , m_Path(path)
{
   m_hFile = CreateFileW( path, FU_GENERIC_READ, FU_FILE_SHARE_READ, 0, FU_OPEN_EXISTING, 0, 0);
   
   if (m_hFile != INVALID_HANDLE)
   {
      Init();
   }
}

void ConstFileImpl::Init()
{
   // requesting size
   uint hi[1] = {0};
   uint low = GetFileSize(m_hFile, hi);
   
   m_Size = ulong(hi);
   m_Size = m_Size >> 32;
   m_Size += low;
   
   // requesting time properties
   uint creation[4], lastAccess[4], lastModify[4];
   if (GetFileTime(m_hFile, creation, lastAccess, lastModify))
   {
      // request succeeded, need to tranform time
      MqlDateTime time[1];
      FileTimeToSystemTime(creation, time);
      m_CreationTime = StructToTime(time[0]);

      FileTimeToSystemTime(lastAccess, time);
      m_LastAccessTime = StructToTime(time[0]);

      FileTimeToSystemTime(lastModify, time);
      m_LastModifyTime = StructToTime(time[0]);
   }
   
   m_Attributes = GetFileAttributesW(m_Path);
}

bool ConstFileImpl::Read(bool& value, string separators)
{
   string tmp;
   if (ReadString(tmp, separators))
   {
      value = bool(StringToInteger(tmp));
      return true;
   }
   return false;
}

bool ConstFileImpl::Read(long& value, string separators)
{
   string tmp;
   if (ReadString(tmp, separators))
   {
      value = StringToInteger(tmp);
      return true;
   }
   return false;
}

bool ConstFileImpl::Read(double& value, string separators)
{
   string tmp;
   if (ReadString(tmp, separators))
   {
      value = StringToDouble(tmp);
      return true;
   }
   return false;
}

bool ConstFileImpl::ReadString(string& value, string separators)
{
   if (m_Buffer == "")
   {
      if (!ReadBlock()) return false;
   }
   
   int pos = StringFindAnyOfChar(m_Buffer, separators);
   while (pos == 0)
   {
      m_Buffer = StringSubstr(m_Buffer, 1);
      m_Pos += 2;
      pos = StringFindAnyOfChar(m_Buffer, separators);
   }
   
   string strValue = "";
   
   bool needFixEOL = false;
   
   if (pos == -1)
   {
      strValue = m_Buffer;
      m_Pos += 2*StringLen(m_Buffer);
      m_Buffer = "";
      
      bool succeeded = ReadBlock();
      
      while (succeeded)
      {
         if (strValue == "")
         {
            pos = StringFindAnyOfChar(m_Buffer, separators);
            while (pos = 0)
            {
               m_Buffer = StringSubstr(m_Buffer, 1);
               m_Pos += 2;
            }
            
            if (pos == -1)
            {
               strValue = m_Buffer;
               m_Pos += 2*StringLen(m_Buffer);
               m_Buffer = "";
            }
            else
            {
               strValue = StringSubstr(m_Buffer, 0, pos);

               ushort chr = StringGetCharacter(m_Buffer, pos);
               if (chr == '\r') needFixEOL = true;

               m_Pos += 2*(StringLen(strValue) + 1);
               m_Buffer = StringSubstr(m_Buffer, pos + 1);
               break;
            }
         }
         else
         {
            pos = StringFindAnyOfChar(m_Buffer, separators);
            if (pos == 0)
            {
               ushort chr = StringGetCharacter(m_Buffer, pos);
               if (chr == '\r') needFixEOL = true;

               m_Buffer = StringSubstr(m_Buffer, 1);
               m_Pos += 2;
               break;
            }
            else if (pos == -1)
            {
               strValue = m_Buffer;
               m_Pos += 2*StringLen(m_Buffer);
               m_Buffer = "";
            }
            else
            {
               strValue = strValue + StringSubstr(m_Buffer, 0, pos);

               ushort chr = StringGetCharacter(m_Buffer, pos);
               if (chr == '\r') needFixEOL = true;

               m_Pos += 2*(StringLen(strValue) + 1);
               m_Buffer = StringSubstr(m_Buffer, pos + 1);
               break;
            }
         }
         succeeded = ReadBlock();
      }

      if (strValue == "") return false;
   }
   else
   {
      strValue = StringSubstr(m_Buffer, 0, pos);

      ushort chr = StringGetCharacter(m_Buffer, pos);
      if (chr == '\r') needFixEOL = true;

      m_Pos += 2*(StringLen(strValue) + 1);
      m_Buffer = StringSubstr(m_Buffer, pos + 1);
   }
   
   if (needFixEOL) FixEndOfLine();
   
   value = strValue;
   
   return true;
}

bool ConstFileImpl::Read(string& value, string separators)
{
   if (m_Buffer == "")
   {
      if (!ReadBlock()) return false;
   }
   
   bool needFixEOL = false;
   
   int pos = StringFindAnyOfChar(m_Buffer, separators);
   if (pos == 0)
   {
      ushort chr = StringGetCharacter(m_Buffer, pos);
      if (chr == '\r') needFixEOL = true;

      m_Buffer = StringSubstr(m_Buffer, 1);
      m_Pos += 2;
      value = "";
      
      if (needFixEOL) FixEndOfLine();
      
      return true;
   }
   
   string strValue = "";
   
   if (pos == -1)
   {
      strValue = m_Buffer;
      m_Pos += 2*StringLen(m_Buffer);
      m_Buffer = "";
      
      bool succeeded = ReadBlock();
      
      while (succeeded)
      {
         pos = StringFindAnyOfChar(m_Buffer, separators);
         if (pos == 0)
         {
            ushort chr = StringGetCharacter(m_Buffer, pos);
            if (chr == '\r') needFixEOL = true;

            m_Buffer = StringSubstr(m_Buffer, 1);
            m_Pos += 2;
            break;
         }
         
         if (pos == -1)
         {
            strValue = m_Buffer;
            m_Pos += 2*StringLen(m_Buffer);
            m_Buffer = "";
         }
         else
         {
            ushort chr = StringGetCharacter(m_Buffer, pos);
            if (chr == '\r') needFixEOL = true;

            strValue = StringSubstr(m_Buffer, 0, pos);
            m_Pos += 2*(StringLen(strValue) + 1);
            m_Buffer = StringSubstr(m_Buffer, pos + 1);
            break;
         }

         succeeded = ReadBlock();
      }
   }
   else
   {
      ushort chr = StringGetCharacter(m_Buffer, pos);
      if (chr == '\r') needFixEOL = true;

      strValue = StringSubstr(m_Buffer, 0, pos);
      m_Pos += 2*(StringLen(strValue) + 1);
      m_Buffer = StringSubstr(m_Buffer, pos + 1);
   }
   
   if (needFixEOL) FixEndOfLine();
   
   value = strValue;
   
   return true;
}

bool ConstFileImpl::Read(datetime& value, string separators)
{
   string tmp;
   if (ReadString(tmp, separators))
   {
      value = StringToTime(tmp);
      return true;
   }
   return false;
}

bool ConstFileImpl::Read(color& value, string separators)
{
   string tmp;
   if (ReadString(tmp, separators))
   {
      value = StringToColor(tmp);
      return true;
   }
   return false;
}

bool ConstFileImpl::SetPos(long pos, MovePointerMethod from)
{
   ulong newPos = 0;
   
   switch (from)
   {
      case MPM_BEGIN:
         if (pos < 0 || pos >= long(m_Size)) return false;
         newPos = pos;
         break;
         
      case MPM_CURRENT:
         if (long(pos + m_Pos) < 0 || ulong(pos + m_Pos) >= m_Size) return false;
         newPos = pos + m_Pos;
         break;
         
      case MPM_END:
         if (long(m_Size - 1 + pos) < 0 || pos > 0) return false;
         newPos = m_Size - 1 + pos;
         break;
         
      default:
         return false;
   }
   
   m_Pos = newPos;
   m_Buffer = "";
   
   return true;
}

bool ConstFileImpl::ReadBlock()
{
   int lo = int (m_Pos & 0xFFFFFFFF);
   int hi[1];
   hi[0] = int ((m_Pos >> 32) & 0xFFFFFFFF);
   
   ulong newPos = SetFilePointer(m_hFile, lo, hi, MPM_BEGIN);
   if (newPos == INVALID_SET_FILE_POINTER)
   {
      Print(__FUNCTION__, " SetFilePointer failure");
      return false;
   }
   newPos = newPos + (ulong(hi[0] << 32));

   if (newPos != m_Pos)
   {
      Print(__FUNCTION__, " Possible logical failure");
      return false;
   }
   
   if (m_Pos >= m_Size - 1)
   {
      m_Buffer = "";
      return false;
   }
   
   int readBytes[] = {0};
   uint toRead = 100;
   if (m_Pos + toRead > m_Size)
   {
      toRead = uint(m_Size - m_Pos);
   }

   ushort buffer[];
   ArrayResize(buffer, (toRead + 1)/2);
   
   if (ReadFile(m_hFile, buffer, toRead, readBytes, NULL))
   {
      uint newBufSize = (readBytes[0] + 1)/2;
      
      StringInit(m_Buffer, newBufSize, ' ');
      for (uint i = 0; i < newBufSize; i++)
      {
         StringSetCharacter(m_Buffer, i, buffer[i]);
      }
   }
   
   return true;
}

string ConstFileImpl::ToString()
{
   m_Buffer = "";
   
   int hi[] = {0};

   uint newPos = SetFilePointer(m_hFile, 0, hi, MPM_BEGIN);

   if (newPos == INVALID_SET_FILE_POINTER)
   {
      Print(__FUNCTION__, " SetFilePointer failure");
      return "";
   }
   
   int readBytes[] = {0};

   ushort buffer[];
   ArrayResize(buffer, int((m_Size + 1)/2));
   
   if (ReadFile(m_hFile, buffer, int(m_Size), readBytes, NULL))
   {
      uint newBufSize = (readBytes[0] + 1)/2;
      
      string result;
      StringInit(result, newBufSize, ' ');
      for (uint i = 0; i < newBufSize; i++)
      {
         StringSetCharacter(result, i, buffer[i]);
      }
      
      return result;
   }
   
   return "";
}

void ConstFileImpl::FixEndOfLine(void)
{
   if (StringLen(m_Buffer) == 0)
   {
      if (!ReadBlock()) return;
   }
   
   ushort chr = StringGetCharacter(m_Buffer, 0);
   if (chr == '\n')
   {
      m_Pos += 2;
      m_Buffer = StringSubstr(m_Buffer, 1);
   }
}

ConstFile* OpenConstFile(const string& path) export
{
   ConstFileImpl* result = new ConstFileImpl(path);
   
   if (!result.IsValid())
   {
      delete result;
      result = NULL;
   }
   
   return result;
}

