//+------------------------------------------------------------------+
//|                                                     Expert01.mqh |
//|                                      Copyright © 2017, Marketeer |
//|                           Portions of the code are public domain |
//|                          https://www.mql5.com/en/users/marketeer |
//|                                       Simple expert adviser stub |
//|                            https://www.mql5.com/ru/articles/3442 |
//+------------------------------------------------------------------+

#define ASSERT(A) if(!(A)){Print("Condition '" + #A + "' failed in " + __FUNCTION__ + "; will stop with custom error!");double _assert_=1/(A);}

class Double
{
  private:
    double v;
    double eps;
    void init()
    {
      eps = MathPow(10, -1 * _Digits);
    }
    
  public:
    Double(const double &d): v(d){init();}
    Double(double d): v(d){init();}
    Double(const Double &c): v(c.v){init();}
    
    double operator=(double x)
    {
      v = x;
      return v;
    }
    bool operator==(double x)
    {
      return MathAbs(v - x) < eps;
    }
    bool operator!=(double x)
    {
      return !(this == x);
    }
};

class Expert
{
  private:
    int magic;
    double lot;
    int deviation;
    string orderPrefix;
    datetime lastOrderTime;
    int lastOrderStopLoss[2]; // OP_BUY, OP_SELL used as indices
    string workSymbol;
    int defaultExpiration;
  
  public:
    Expert(int m)
    {
      magic = m;
      lot = 0;
      deviation = 10;
      workSymbol = Symbol();
    }

    Expert(int m, double l)
    {
      magic = m;
      lot = l;
      deviation = 10;
      workSymbol = Symbol();
    }
    
    Expert(int m, double l, string s)
    {
      magic = m;
      lot = l;
      deviation = 10;
      workSymbol = s;
    }

    Expert(int m, double l, int d)
    {
      magic = m;
      lot = l;
      deviation = d;
      workSymbol = Symbol();
    }
    
    void setDefaultExpiration(const int bars)
    {
      defaultExpiration = bars;
    }

    string getWorkSymbol() const
    {
      return workSymbol;
    }

    int getOrderCount()
    {
      return getOrderCount(-1);
    }

    int getMarketOrderCount()
    {
      return getOrderCount(mask(OP_BUY) | mask(OP_SELL));
    }

    int getStopOrderCount()
    {
      return getOrderCount(mask(OP_BUYSTOP) | mask(OP_SELLSTOP));
    }

    bool selectLastStopOrder()
    {
      return selectLastOrder(mask(OP_BUYSTOP) | mask(OP_SELLSTOP));
    }

    bool selectLastMarketOrder(const int mode = MODE_TRADES)
    {
      return selectLastOrder(mask(OP_BUY) | mask(OP_SELL), mode);
    }
    
    int getOrderCount(const int op)
    {
      return getOrderCount(op, "");
    }
    
    int mask(int op)
    {
      return 1 << op;
    }

    /**
     op - mask of order types
    */
    int getOrderCount(const int op, const string CurrentSymbol)
    {
      int count = 0;
      for(int i = OrdersTotal() - 1; i >= 0; i--)
      {
        if(OrderSelect(i, SELECT_BY_POS))
        {
          int mask = 1 << OrderType();
          if(OrderMagicNumber() == magic && (CurrentSymbol == OrderSymbol() || CurrentSymbol == "") && ((mask & op) != 0))
          {
            count++;
          }
        }
      }
      return count;
    }

    bool selectLastOrder(const int op, const int mode = MODE_TRADES)
    {
      int n = mode == MODE_TRADES ? OrdersTotal() : OrdersHistoryTotal();
      for(int i = n - 1; i >= 0; i--)
      {
        if(OrderSelect(i, SELECT_BY_POS, mode))
        {
          int mask = 1 << OrderType();
          if(OrderMagicNumber() == magic && (getWorkSymbol() == OrderSymbol()) && ((mask & op) != 0))
          {
            return true;
          }
        }
      }
      return false;
    }

    void deletePendingOrders()
    {
      deletePendingOrders((1 << OP_BUYLIMIT) | (1 << OP_SELLLIMIT) | (1 << OP_BUYSTOP) | (1 << OP_SELLSTOP));
    }

    void deletePendingOrders(const int op)
    {
      deletePendingOrders(op, getWorkSymbol());
    }
    
    void deletePendingOrders(const int op, const string CurrentSymbol)
    {
      for(int i = OrdersTotal() - 1; i >= 0; i--)
      {
        if(OrderSelect(i, SELECT_BY_POS))
        {
          int mask = 1 << OrderType();
          if(OrderMagicNumber() == magic && (CurrentSymbol == OrderSymbol() || CurrentSymbol == "") && ((mask & op) != 0))
          {
            if(!OrderDelete(OrderTicket()))
            {
              Print("OrderDelete failed for #", OrderTicket(), " ", OrderSymbol(), " with error ", GetLastError());
            }
          }
        }
      }
    }


    int closeMarketOrders()
    {
      return closeMarketOrders((1 << OP_BUY) | (1 << OP_SELL));
    }

    int closeMarketOrders(const int op)
    {
      return closeMarketOrders(op, getWorkSymbol());
    }

    int closeMarketOrders(const int op, const string CurrentSymbol)
    {
      int count = 0;
      for(int i = OrdersTotal() - 1; i >= 0; i--)
      {
        if(OrderSelect(i, SELECT_BY_POS))
        {
          int mask = 1 << OrderType();
          if(OrderMagicNumber() == magic && (CurrentSymbol == OrderSymbol() || CurrentSymbol == "") && ((mask & op) != 0))
          {
            double price = MarketInfo(OrderSymbol(), IsOfBuyType(OrderType())?MODE_BID:MODE_ASK);
            
            if(!OrderClose(OrderTicket(), OrderLots(), price, deviation))
            {
              Print("OrderClose failed for #", OrderTicket(), " ", OrderSymbol(), " with error ", GetLastError());
            }
            else
            {
              count++;
            }
          }
        }
      }
      return count;
    }

    void trailStops(const string CurrentSymbol, const int slbuy, const int slsell)
    {
      for(int i = OrdersTotal() - 1; i >= 0; i--)
      {
        if(OrderSelect(i, SELECT_BY_POS))
        {
          if(OrderMagicNumber() == magic && (CurrentSymbol == OrderSymbol() || CurrentSymbol == "") && OrderType() < 2)
          {
            bool isBuy = IsOfBuyType(OrderType());
            if((isBuy && slbuy == 0) || (!isBuy && slsell == 0)) continue;
            double point = MarketInfo(OrderSymbol(), MODE_POINT);
            double price = MarketInfo(OrderSymbol(), isBuy?MODE_BID:MODE_ASK);
            double level = price + (isBuy?-1:+1) * (isBuy?slbuy:slsell) * point;
            
            if(OrderStopLoss() == 0 || (isBuy && OrderStopLoss() < level) || (!isBuy && OrderStopLoss() > level))
            {
              if(MathAbs(OrderStopLoss() - level) / point >= deviation)
              {
                if(!OrderModify(OrderTicket(), OrderOpenPrice(), level, OrderTakeProfit(), OrderExpiration()))
                {
                  Print("OrderModify failed for #", OrderTicket(), " ", OrderSymbol(), " with error ", GetLastError());
                  Print(level, " ", price, " ", slbuy, " ", slsell);
                }
              }
            }
          }
        }
      }
    }
    
    void trailStops()
    {
      if(lastOrderStopLoss[OP_BUY] != 0 || lastOrderStopLoss[OP_SELL] != 0)
      {
        trailStops(getWorkSymbol(), lastOrderStopLoss[OP_BUY], lastOrderStopLoss[OP_SELL]);
      }
    }

    int placePendingOrder(const int type, const int points, const int sl, const int tp)
    {
      return placePendingOrder(type, getWorkSymbol(), points, sl, tp);
    }

    int placeMarketOrder(const int type, const int sl, const int tp)
    {
      return placeMarketOrder(type, getWorkSymbol(), sl, tp);
    }

    int placeMarketOrder(const int type)
    {
      return placeMarketOrder(type, getWorkSymbol(), 0, 0);
    }
    
    double volumeCalculatePercent(const string symbol, const double percent, double customMargin = 0)
    {
      double NewLots = 0;
      double OneLot = MarketInfo(symbol, MODE_MARGINREQUIRED);
      if(OneLot == 0)
      {
        Print("Margin is zero ", symbol);
        return(0);
      }
      double MinLot = MarketInfo(symbol, MODE_MINLOT);
      if(MinLot == 0)
      {
        Print("MinLot is zero ", symbol);
        return(0);
      }
      double Step = MarketInfo(symbol, MODE_LOTSTEP);
      if(Step == 0)
      {
        Print("LotStep is zero");
        return(0);
      }
    
      double Money;
      if(customMargin == 0)
      {
        Money = AccountFreeMargin();
      }
      else
      {
        Money = customMargin;
      }
    
      NewLots = Money * MathMin(percent, 100) / 100 / OneLot;
      
      NewLots = MathFloor(NewLots / Step) * Step;
    
      if(NewLots < MinLot)
      {
        return 0;
      }
      
      double MaxLot = MarketInfo(symbol, MODE_MAXLOT);
      if(NewLots > MaxLot) NewLots = MaxLot;
      
      return(NewLots);
    }
    
    int getPointsForLotAndRisk(const int pointsOrRisk, double lotSize = 0)
    {
      if(pointsOrRisk < 0 && lotSize != 0)
      {
        double pointValue = _Point * MarketInfo(_Symbol, MODE_TICKVALUE) / MarketInfo(_Symbol, MODE_TICKSIZE);
        if(lotSize < 0) lotSize = volumeCalculatePercent(getWorkSymbol(), -lotSize);
        return (int)(-pointsOrRisk / 100.0 * AccountFreeMargin() / lotSize / pointValue);
      }
      
      ASSERT(pointsOrRisk > 0);
    
      return pointsOrRisk;
    }
    
    
    int placeMarketOrder(const int type, const string CurrentSymbol, const int sl, const int tp)
    {
      return placeMarketOrder(type, CurrentSymbol, lot, sl, tp);
    }

    int placeMarketOrder(const int type, const double customLot)
    {
      return placeMarketOrder(type, customLot, 0.0, 0.0);
    }
    
    int placeMarketOrder(const int type, const string CurrentSymbol, const double customLot, const int sl, const int tp)
    {
      ASSERT(type == OP_BUY || type == OP_SELL);
      
      double point = MarketInfo(CurrentSymbol, MODE_POINT);
      double l = customLot;
      if(l == 0) l = MarketInfo(CurrentSymbol, MODE_MINLOT);
      if(l < 0) l = volumeCalculatePercent(CurrentSymbol, -l);
      
      double price, stoploss, takeprofit;
      if(IsOfBuyType(type))
      {
        price = MarketInfo(CurrentSymbol, MODE_ASK);
        stoploss = sl > 0 ? price - sl * point : 0;
        takeprofit = tp > 0 ? price + tp * point : 0;
      }
      else
      {
        price = MarketInfo(CurrentSymbol, MODE_BID);
        stoploss = sl > 0 ? price + sl * point : 0;
        takeprofit = tp > 0 ? price - tp * point : 0;
      }
      
      int ticket = OrderSend(CurrentSymbol, type, l, price, deviation, stoploss, takeprofit, orderPrefix, magic);
      if(ticket == -1)
      {
        Print("OrderSend failed ", CurrentSymbol, " ", type, " ", (string)l, " ", price, " ", stoploss, " ", takeprofit, " with error ", GetLastError());
      }
      else
      {
        lastOrderTime = iTime(CurrentSymbol, 0, 0);
        lastOrderStopLoss[type] = sl;
      }
      return ticket;
    }

    int placeMarketOrder(const int type, const double customLot, const double stoploss, const double takeprofit)
    {
      ASSERT(type == OP_BUY || type == OP_SELL);

      double l = customLot;
      if(l == 0) l = MarketInfo(getWorkSymbol(), MODE_MINLOT);
      if(l < 0) l = volumeCalculatePercent(getWorkSymbol(), -l);

      double price = MarketInfo(getWorkSymbol(), IsOfBuyType(type)?MODE_ASK:MODE_BID);
      
      int ticket = OrderSend(getWorkSymbol(), type, l, price, deviation, stoploss, takeprofit, orderPrefix, magic);
      if(ticket == -1)
      {
        Print("OrderSend failed ", getWorkSymbol(), " ", type, " ", (string)l, " ", price, " ", stoploss, " ", takeprofit, " with error ", GetLastError());
      }
      else
      {
        lastOrderTime = iTime(getWorkSymbol(), 0, 0);
        if(stoploss != 0) lastOrderStopLoss[type] = (int)(MathAbs(stoploss - price) / MarketInfo(getWorkSymbol(), MODE_POINT));
        else lastOrderStopLoss[type] = 0;
      }
      return ticket;
    }
    
    int getLastOrderBar()
    {
      return iBarShift(getWorkSymbol(), 0, lastOrderTime, true);
    }
    
    int placePendingOrder(const int type, const string CurrentSymbol, const int points, const int sl, const int tp)
    {
      ASSERT(type >= 2);

      double point = MarketInfo(CurrentSymbol, MODE_POINT);
      double l = lot;
      if(l == 0) l = MarketInfo(CurrentSymbol, MODE_MINLOT);
      if(l < 0) l = volumeCalculatePercent(CurrentSymbol, -l);
      
      double price, stoploss, takeprofit;
      bool isStop = IsOfStopType(type);
      
      if(IsOfBuyType(type))
      {
        price = MarketInfo(CurrentSymbol, MODE_ASK);
        if(isStop)
        {
          price += point * points;
        }
        else
        {
          price -= point * points;
        }
        stoploss = sl > 0 ? price - sl * point : 0;
        takeprofit = tp > 0 ? price + tp * point : 0;
      }
      else
      {
        price = MarketInfo(CurrentSymbol, MODE_BID);
        if(isStop)
        {
          price -= point * points;
        }
        else
        {
          price += point * points;
        }
        stoploss = sl > 0 ? price + sl * point : 0;
        takeprofit = tp > 0 ? price - tp * point : 0;
      }
      
      datetime expiration = defaultExpiration > 0 ? iTime(CurrentSymbol, 0, 0) + PeriodSeconds() * defaultExpiration : 0;
      int ticket = OrderSend(CurrentSymbol, type, l, price, deviation, stoploss, takeprofit, orderPrefix, magic, expiration);
      if(ticket == -1)
      {
        Print("OrderSend failed ", CurrentSymbol, " ", type, " ", DoubleToString(l, 2), " ", price, " ", stoploss, " ", takeprofit, " ", (string)expiration, " with error ", GetLastError());
        Print("Ask ", MarketInfo(CurrentSymbol, MODE_ASK), " Bid ", MarketInfo(CurrentSymbol, MODE_BID));
      }
      else
      {
        if(OrderSelect(ticket, SELECT_BY_TICKET))
        {
          OrderPrint();
        }
        lastOrderStopLoss[type % 2] = sl;
      }
      return ticket;
    }
    
    int placePendingOrder(const int type, const double customLot, const double price, const double stoploss, const double takeprofit, const int customExpiration = 0)
    {
      ASSERT(type >= 2);

      double l = customLot;
      if(l == 0) l = MarketInfo(getWorkSymbol(), MODE_MINLOT);
      if(l < 0) l = volumeCalculatePercent(getWorkSymbol(), -l);

      datetime expiration = customExpiration > 0 ? iTime(getWorkSymbol(), 0, 0) + PeriodSeconds() * customExpiration : 0;
      int ticket = OrderSend(getWorkSymbol(), type, l, price, deviation, stoploss, takeprofit, orderPrefix, magic, expiration);
      if(ticket == -1)
      {
        Print("OrderSend failed ", getWorkSymbol(), " ", type, " ", DoubleToString(l, 2), " ", price, " ", stoploss, " ", takeprofit, " ", (string)expiration, " with error ", GetLastError());
        Print("Ask ", MarketInfo(getWorkSymbol(), MODE_ASK), " Bid ", MarketInfo(getWorkSymbol(), MODE_BID));
      }
      else
      {
        if(OrderSelect(ticket, SELECT_BY_TICKET))
        {
          OrderPrint();
        }
        if(stoploss != 0) lastOrderStopLoss[type % 2] = (int)(MathAbs(stoploss - price) / MarketInfo(getWorkSymbol(), MODE_POINT));
        else lastOrderStopLoss[type % 2] = 0;
      }
      return ticket;
    }
    
    double findBreakEvenLevel()
    {
      double price[2] = {0, 0}, lots[2] = {0, 0};
      int count = 0;
      for(int i = OrdersTotal() - 1; i >= 0; i--)
      {
        if(OrderSelect(i, SELECT_BY_POS))
        {
          if(OrderMagicNumber() == magic && OrderSymbol() == getWorkSymbol() && OrderType() < 2)
          {
            price[OrderType()] += OrderLots() * OrderOpenPrice();
            lots[OrderType()] += OrderLots();
            count++;
          }
        }
      }
      
      if(count < 2) return 0;
      
      double result = 0;
      int digits = (int)MarketInfo(getWorkSymbol(), MODE_DIGITS);
      if(lots[OP_BUY] > lots[OP_SELL])
      {
        result = (price[OP_BUY] - price[OP_SELL]) / (lots[OP_BUY] - lots[OP_SELL]);
        result = NormalizeDouble(result, digits);
      }
      else
      if(lots[OP_BUY] < lots[OP_SELL])
      {
        result = (price[OP_SELL] - price[OP_BUY]) / (lots[OP_BUY] - lots[OP_SELL]);
        result = NormalizeDouble(result, digits);
      }
      // else buy == sell, no netto position
      
      return result;
    }
    
    
    bool closeByAll(const string customSymbol = "")
    {
      bool result = false;
      
      string s = customSymbol == "" ? getWorkSymbol() : customSymbol;
      
      if(MarketInfo(s, MODE_CLOSEBY_ALLOWED) == 0.0)
      {
        Print("OrderCloseBy is not allowed. Please close orders manually");
        return false;
      }
      
      int buyTicket = -1;
      int sellTicket = -1;
       
      double buyLot = 0;
      double sellLot = 0;
       
      for(int i = OrdersTotal() - 1; i >= 0; i--)
      {
        if(OrderSelect(i, SELECT_BY_POS))
        {
          if(OrderMagicNumber() == magic && s == OrderSymbol())
          {
            if(OrderType() == OP_SELL)
            {
              sellTicket = OrderTicket();
              sellLot = OrderLots();
              if(buyTicket != -1) break;
            }
              
            if(OrderType() == OP_BUY)
            {
              buyTicket = OrderTicket();
              buyLot = OrderLots();
              if(sellTicket != -1) break;
            }
          }
        }
      }
       
      if(buyTicket == -1 || sellTicket == -1) return(false);

      if(sellLot > buyLot)
      {
        result = OrderCloseBy(sellTicket, buyTicket);
      }
      else
      {
        result = OrderCloseBy(buyTicket, sellTicket);
      }
      if(!result)
      {
        Print("OrderCloseBy failed with error ", GetLastError());
      }
      
      return(result);
    }
    
    void shrink(const string customSymbol = "")
    {
      while(closeByAll(customSymbol));
    }
    
    bool IsOfBuyType(int op)
    {
      return(!(op % 2));
    }

    bool IsOfLimitType(int op)
    {
      return(op == 2 || op == 3);
    }

    bool IsOfStopType(int op)
    {
      return(op == 4 || op == 5);
    }
    
    
};
