//+------------------------------------------------------------------+
//|                                           TradeObjects.[mq4|mq5] |
//|                                      Copyright © 2017, Marketeer |
//|                Trading by graphical objects on MetaTrader charts |
//|                          https://www.mql5.com/en/users/marketeer |
//|                            https://www.mql5.com/ru/articles/3442 |
//+------------------------------------------------------------------+

#property copyright "Copyright © 2017, Marketeer"
#property link      "https://www.mql5.com/en/users/marketeer"
#property version   "1.0"
#property description "Semi-automatic expert adviser for trading based on graphic markup using trendlines, hlines, vlines, channels, and other objects."

#property strict

#ifdef __MQL5__
  #include <mt4market.mqh>
  #include <ind4to5.mqh>
  #include <mt4orders.mqh>
#endif

input int Magic = 0;
input double Lot = 0.01 /*default lot*/;
input int Deviation = 10 /*tolerance to price changes during order execution*/;

input int DefaultTakeProfit = 0 /*points | -percent of risk*/;
input int DefaultStopLoss = 0 /*points | -percent of risk*/;
input int DefaultExpiration = 0 /*bars*/;

input string CommonPrefit = "exp" /*empty to handle all compatible objects*/;

input color BuyColor = clrBlue /*market and pending buy orders - open, close, sl, tp*/;
input color SellColor = clrRed /*market and pending sell orders - open, close, sl, tp*/;
input color ActivationColor = clrGray /*activation of pending orders placement, alert*/;

// All available styles: STYLE_SOLID, STYLE_DASH, STYLE_DOT, STYLE_DASHDOT, STYLE_DASHDOTDOT
input ENUM_LINE_STYLE InstantType = STYLE_SOLID /*opens market trades, alerts*/;
input ENUM_LINE_STYLE PendingType = STYLE_DASH /*defines probable pending orders (requires activation)*/;
input ENUM_LINE_STYLE CloseStopLossTakeProfitType = STYLE_DOT /*applied to open positions*/;

input int EventHotSpot = 10 /*points*/;
input int EventTimeSpan = 10 /*seconds*/;
input int EventInterval = 10 /*bars*/;
input bool CancelCounterparts = false /*opposite event disables line*/;
input bool AllowOrderCloseBy = false /*partial close enabled, counter orders will collapse*/;
input bool ShowBreakEven = false;


color DimmedBuyColor, DimmedSellColor, DimmedActivationColor;

#include <Expert01.mqh>
#include <Set.mqh>

Set<ENUM_OBJECT> _ln(OBJ_HLINE, OBJ_VLINE, OBJ_TREND, OBJ_CHANNEL, OBJ_FIBO);
Set<ENUM_LINE_STYLE> _st(InstantType, PendingType, CloseStopLossTakeProfitType);
Set<color> _cc(BuyColor, SellColor, ActivationColor);


class TradeObjects
{
  private:
    Expert *e;
    struct LineObject
    {
      string name;
      int status;
      void operator=(const LineObject &o)
      {
        name = o.name;
        status = o.status;
      }
    };
    
    LineObject objects[];
    double PreviousBid;
    
    double EMA(double previous, double current, int smoothing)
    {
      double k = 2.0 / (smoothing + 1);
      if(previous == 0) return current;
      return(current * k + previous * (1 - k));
    }


  protected:
    
    void showMessage(const string message)
    {
      static datetime lastMessage = 0;
      static double averageInterval = 0;
      double averageIntervalOld = averageInterval;
      averageInterval = EMA(averageInterval, TimeCurrent() - lastMessage, 10);
      const int annoyanceThreshold = 5; // seconds
      if(averageInterval < annoyanceThreshold)
      {
        if(averageIntervalOld >= annoyanceThreshold)
        {
          Alert("Too many messages per second. Bulk messages directed silently to log");
        }
        Print(message);
      }
      else
      {
        Alert(message);
      }
      lastMessage = TimeCurrent();
    }
    
    void startRefresh()
    {
      for(int i = 0; i < ArraySize(objects); i++)
      {
        objects[i].status = 0;
      }
    }
    
    int findObject(const string name) const
    {
      int n = ArraySize(objects);
      for(int i = 0; i < n; i++)
      {
        if(objects[i].name == name)
        {
          return i;
        }
      }
      return -1;
    }
    
    bool attachObject(const string name)
    {
      int i = findObject(name);
      
      if(i != -1)
      {
        objects[i].status = 1;
      }
      else // not found
      {
        int n = ArraySize(objects);
        ArrayResize(objects, n + 1);
        objects[n].name = name;
        objects[n].status = 1;
        return true;
      }
      
      return false;
    }
    
    bool stopRefresh()
    {
      int n = ArraySize(objects);
      int squeezed = 0;
      for(int i = n - 1; i >= 0; i--)
      {
        if(objects[i].status == 0)
        {
          // ArrayCopy(objects, objects, i, i + 1);
          for(int j = i; j < n - 1; j++)
          {
            objects[j] = objects[j + 1];
          }
          squeezed++;
        }
      }
      if(squeezed > 0)
      {
        ArrayResize(objects, n - squeezed);
        Print("Total lines: ", ArraySize(objects));
        return true;
      }
      return false;
    }
    
    bool removeObject(const string name)
    {
      bool found = false;
      int n = ArraySize(objects);
      int squeezed = 0;
      for(int i = n - 1; i >= 0; i--)
      {
        if(objects[i].name == name)
        {
          // ArrayCopy(objects, objects, i, i + 1);
          for(int j = i; j < n - 1; j++)
          {
            objects[j] = objects[j + 1];
          }
          squeezed++;
          found = true;
        }
      }
      if(squeezed > 0) ArrayResize(objects, n - squeezed);
      return found;
    }
    
    bool checkObjectCompliance(const string obj)
    {
      if(CommonPrefit == "" || StringFind(obj, CommonPrefit) == 0)
      {
        if(_ln[ObjectGetInteger(0, obj, OBJPROP_TYPE)]
        && _st[ObjectGetInteger(0, obj, OBJPROP_STYLE)]
        && _cc[(color)ObjectGetInteger(0, obj, OBJPROP_COLOR)])
        {
          if(ObjectGetInteger(0, obj, OBJPROP_TYPE) == OBJ_TREND)
          {
            if(!ObjectGetInteger(0, obj, OBJPROP_RAY))
            {
              return false;
            }
          }
          
          if(ObjectGetInteger(0, obj, OBJPROP_TYPE) != OBJ_VLINE)
          {
            for(int i = 0; i < 3; i++)
            {
              if(ObjectGetInteger(0, obj, OBJPROP_TIME, i) > TimeCurrent())
              {
                return false;
              }
            }
          }
          return true;
        }
      }
      return false;
    }
    
    bool detectLines()
    {
      startRefresh();
      int n = ObjectsTotal(ChartID(), 0);
      int count = 0;
      for(int i = 0; i < n; i++)
      {
        string obj = ObjectName(ChartID(), i, 0);
        if(checkObjectCompliance(obj))
        {
          if(attachObject(obj))
          {
            describe(obj);
            count++;
          }
        }
      }
      if(count > 0) Print("New lines: ", count);
      bool changes = stopRefresh() || (count > 0);
      
      if(changes)
      {
        display();
      }
      
      return changes;
    }
    
    void updateBELine(const double level)
    {
      const string ID = "TL_BEL";
      if(level != 0)
      {
        if(ObjectCreate(0, ID, OBJ_HLINE, 0, Time[0], MathAbs(level)))
        {
          ObjectSetInteger(0, ID, OBJPROP_COLOR, clrOrange);
          ObjectSetInteger(0, ID, OBJPROP_STYLE, STYLE_DASHDOTDOT);
          ObjectSetString(0, ID, OBJPROP_TEXT, level > 0 ? "Break Even Up" : "Break Even Down");
        }
        else
        {
          ObjectSetDouble(0, ID, OBJPROP_PRICE, 0, MathAbs(level));
        }
      }
      else
      {
        ObjectDelete(0, ID);
      }
    }
    
    template<typename T>
    void swap(T &t1, T &t2)
    {
      T t = t1;
      t1 = t2;
      t2 = t;
    }
    
    double getCurrentPrice(const int index, double &auxiliary, double &auxiliaryFibo)
    {
      string name = objects[index].name;
      int type = (int)ObjectGetInteger(0, name, OBJPROP_TYPE);
      if(type == OBJ_TREND)
      {
        datetime dt1 = (datetime)ObjectGetInteger(0, name, OBJPROP_TIME, 0);
        datetime dt2 = (datetime)ObjectGetInteger(0, name, OBJPROP_TIME, 1);
        int i1 = iBarShift(NULL, 0, dt1, true);
        int i2 = iBarShift(NULL, 0, dt2, true);
        if(i1 <= i2 || i1 == -1 || i2 == -1)
        {
          if(objects[index].status != -1)
          {
            showMessage("Incorrect line: " + name);
            objects[index].status = -1;
          }
          return 0;
        }
        else if(objects[index].status == -1)
        {
          objects[index].status = 1;
          display();
        }
        double p1 = ObjectGetDouble(0, name, OBJPROP_PRICE, 0);
        double p2 = ObjectGetDouble(0, name, OBJPROP_PRICE, 1);
        
        double k = -(p1 - p2)/(i2 - i1);
        double b = -(i1 * p2 - i2 * p1)/(i2 - i1);
        
        return b;
      }
      else if(type == OBJ_HLINE)
      {
        return ObjectGetDouble(0, name, OBJPROP_PRICE, 0);
      }
      else if(type == OBJ_VLINE)
      {
        return EMPTY_VALUE; // should not be a null, not used otherwise
      }
      else if(type == OBJ_CHANNEL)
      {
        datetime dt1 = (datetime)ObjectGetInteger(0, name, OBJPROP_TIME, 0);
        datetime dt2 = (datetime)ObjectGetInteger(0, name, OBJPROP_TIME, 1);
        datetime dt3 = (datetime)ObjectGetInteger(0, name, OBJPROP_TIME, 2);
        int i1 = iBarShift(NULL, 0, dt1, true);
        int i2 = iBarShift(NULL, 0, dt2, true);
        int i3 = iBarShift(NULL, 0, dt3, true);
        if(i1 <= i2 || i1 == -1 || i2 == -1 || i3 == -1)
        {
          if(objects[index].status != -1)
          {
            showMessage("Incorrect channel: " + name);
            objects[index].status = -1;
          }
          return 0;
        }
        else if(objects[index].status == -1)
        {
          objects[index].status = 1;
          display();
        }
        double p1 = ObjectGetDouble(0, name, OBJPROP_PRICE, 0);
        double p2 = ObjectGetDouble(0, name, OBJPROP_PRICE, 1);
        double p3 = ObjectGetDouble(0, name, OBJPROP_PRICE, 2);
        
        double k = -(p1 - p2)/(i2 - i1);
        double b = -(i1 * p2 - i2 * p1)/(i2 - i1);
        
        double dy = i3 * k + b - p3;
        
        auxiliary = p3 - i3 * k; // b - dy;
        
        return b;
      }
      else if(type == OBJ_FIBO)
      {
        // level 61.8 is enter point at retracement (buy/sell limit),
        // 38.2 and 161.8 as stoploss/takeprofit
        // set auxiliary and auxiliaryFibo to 38 and 161 levels, return 61.8
        
        double p1 = ObjectGetDouble(0, name, OBJPROP_PRICE, 0);
        double p2 = ObjectGetDouble(0, name, OBJPROP_PRICE, 1);
        datetime dt1 = (datetime)ObjectGetInteger(0, name, OBJPROP_TIME, 0);
        datetime dt2 = (datetime)ObjectGetInteger(0, name, OBJPROP_TIME, 1);
        
        if(dt2 < dt1)
        {
          swap(p1, p2);
        }
        
        double price = (p2 - p1) * ObjectGetDouble(0, name, OBJPROP_LEVELVALUE, 4) + p1;
        auxiliary = (p2 - p1) * ObjectGetDouble(0, name, OBJPROP_LEVELVALUE, 2) + p1;
        auxiliaryFibo = (p2 - p1) * ObjectGetDouble(0, name, OBJPROP_LEVELVALUE, 6) + p1;
        return price;
      }
      return 0;
    }
    
    bool checkActivation(const double price)
    {
      if(Bid >= price - EventHotSpot * _Point && Bid <= price + EventHotSpot * _Point)
      {
        return true;
      }
      
      if((PreviousBid < price && Bid >= price)
      || (PreviousBid > price && Bid <= price))
      {
        return true;
      }
      return false;
    }
    
    int checkMarket(const double price, const datetime last = 0) // returns direction of price movement
    {
      if(last != 0 && (TimeCurrent() - last) / PeriodSeconds() < EventInterval)
      {
        return 0;
      }
    
      if(PreviousBid >= price - EventHotSpot * _Point && PreviousBid <= price + EventHotSpot * _Point)
      {
        if(Bid > price + EventHotSpot * _Point)
        {
          return +1; // up
        }
        else if(Bid < price - EventHotSpot * _Point)
        {
          return -1; // down
        }
      }
    
      if(PreviousBid < price && Bid >= price && MathAbs(Bid - PreviousBid) >= EventHotSpot * _Point)
      {
        return +1;
      }
      else if(PreviousBid > price && Bid <= price && MathAbs(Bid - PreviousBid) >= EventHotSpot * _Point)
      {
        return -1;
      }
      
      return 0;
    }
    
    bool checkTime(const string name)
    {
      return (ObjectGetInteger(0, name, OBJPROP_TYPE) == OBJ_VLINE
        && (datetime)ObjectGetInteger(0, name, OBJPROP_TIME, 0) == Time[0]);
    }

    void disableLine(const string name)
    {
      int width = (int)ObjectGetInteger(0, name, OBJPROP_WIDTH);
      if(width > 1)
      {
        ObjectSetInteger(0, name, OBJPROP_WIDTH, width - 1);
        ObjectSetInteger(0, name, OBJPROP_ZORDER, TimeCurrent());
      }
      else
      {
        ObjectSetInteger(0, name, OBJPROP_BACK, true);
        ObjectSetInteger(0, name, OBJPROP_COLOR, darken((color)ObjectGetInteger(0, name, OBJPROP_COLOR)));
      }
      display();
    }
    
    void disableLine(const int i)
    {
      disableLine(objects[i].name);
    }
    
    bool processPending(const string text)
    {
      int index = findObject(text);
      if(index == -1)
      {
        Print("Unknown object: ", text);
        return false;
      }
      
      double aux2 = 0, aux3 = 0;
      double pending = getCurrentPrice(index, aux2, aux3);
      if(pending == 0)
      {
        Print("Incorrect object: ", text);
        display();
        return false;
      }
    
      string properties[]; // lot[%] [expiration in bars]
      int np = StringSplit(ObjectGetString(0, text, OBJPROP_TEXT), ' ', properties);
      int expiration = np > 1 ? (int)StringToInteger(properties[1]) : DefaultExpiration;
      color clrp = (color)ObjectGetInteger(0, text, OBJPROP_COLOR);
      int ticket = -1;
      
      double lot = np > 0 ? StringToDouble(properties[0]) : Lot;
      
      double sl = 0.0, tp = 0.0;
      if(aux2 != 0 && aux3 != 0)
      {
        if(clrp == BuyColor)
        {
          if(aux2 > pending)
          {
            tp = aux2;
            sl = aux3;
          }
          else
          {
            sl = aux2;
            tp = aux3;
          }
        }
        else
        {
          if(aux2 > pending)
          {
            sl = aux2;
            tp = aux3;
          }
          else
          {
            tp = aux2;
            sl = aux3;
          }
        }
      }
      else
      if(aux2 != 0)
      {
        if(clrp == BuyColor)
        {
          if(aux2 > pending)
          {
            tp = aux2;
            if(DefaultStopLoss != 0) sl = pending - e.getPointsForLotAndRisk(DefaultStopLoss, lot) * _Point;
          }
          else
          {
            sl = aux2;
            if(DefaultTakeProfit != 0) tp = pending + e.getPointsForLotAndRisk(DefaultTakeProfit, lot) * _Point;
          }
        }
        else // sell
        {
          if(aux2 > pending)
          {
            sl = aux2;
            if(DefaultTakeProfit != 0) tp = pending - e.getPointsForLotAndRisk(DefaultTakeProfit, lot) * _Point;
          }
          else
          {
            tp = aux2;
            if(DefaultStopLoss != 0) sl = pending + e.getPointsForLotAndRisk(DefaultStopLoss, lot) * _Point;
          }
        }
      }
      else
      {
        if(clrp == BuyColor)
        {
          if(DefaultStopLoss != 0) sl = pending - e.getPointsForLotAndRisk(DefaultStopLoss, lot) * _Point;
          if(DefaultTakeProfit != 0) tp = pending + e.getPointsForLotAndRisk(DefaultTakeProfit, lot) * _Point;
        }
        else // sell
        {
          if(DefaultStopLoss != 0) sl = pending + e.getPointsForLotAndRisk(DefaultStopLoss, lot) * _Point;
          if(DefaultTakeProfit != 0) tp = pending - e.getPointsForLotAndRisk(DefaultTakeProfit, lot) * _Point;
        }
      }
      
      pending = NormalizeDouble(pending, _Digits);
      sl = NormalizeDouble(sl, _Digits);
      tp = NormalizeDouble(tp, _Digits);
      
      if(pending > Bid) // buy stop or sell limit
      {
        ticket = e.placePendingOrder(clrp == BuyColor ? OP_BUYSTOP : OP_SELLLIMIT, lot, pending, sl, tp, expiration);
      }
      else // sell stop or buy limit
      {
        ticket = e.placePendingOrder(clrp == SellColor ? OP_SELLSTOP : OP_BUYLIMIT, lot, pending, sl, tp, expiration);
      }
      
      if(ticket != -1)
      {
        disableLine(text);
      }
      
      return ticket != -1;
    }
    
    int collectPendingLines(string &pendings[])
    {
      int n = ArraySize(objects);
      for(int i = 0; i < n; i++)
      {
        if(ObjectGetInteger(0, objects[i].name, OBJPROP_STYLE) == PendingType
        && (ObjectGetInteger(0, objects[i].name, OBJPROP_COLOR) == BuyColor
        ||  ObjectGetInteger(0, objects[i].name, OBJPROP_COLOR) == SellColor))
        {
          int m = ArraySize(pendings);
          ArrayResize(pendings, m + 1);
          pendings[m] = objects[i].name;
        }
      }
      return ArraySize(pendings);
    }
    
    void processLines()
    {
      int n = ArraySize(objects);
      for(int i = 0; i < n; i++)
      {
        string name = objects[i].name;
        if(ObjectGetInteger(ChartID(), name, OBJPROP_BACK)) continue;
        
        int style = (int)ObjectGetInteger(0, name, OBJPROP_STYLE);
        color clr = (color)ObjectGetInteger(0, name, OBJPROP_COLOR);
        string text = ObjectGetString(0, name, OBJPROP_TEXT);
        datetime last = (datetime)ObjectGetInteger(0, name, OBJPROP_ZORDER);
    
        double aux = 0, auxf = 0;
        double price = getCurrentPrice(i, aux, auxf);
        if(price == 0)
        {
          display();
          continue; // some error
        }
        
        if(clr == ActivationColor)
        {
          if(style == InstantType)
          {
            if(checkActivation(price) || checkTime(name))
            {
              disableLine(i);
              if(StringFind(text, "Alert:") == 0) Alert(StringSubstr(text, 6));
              else if(StringFind(text, "Push:") == 0) SendNotification(StringSubstr(text, 5));
              else if(StringFind(text, "Mail:") == 0) SendMail("TradeObjects", StringSubstr(text, 5));
              else Print(text);
            }
          }
          else
          if(style == PendingType)
          {
            if(checkActivation(price) || checkTime(name))
            {
              Print("Activated: ", name);

              int succeeded = 0;
              string pendings[];
              string failed = "";
              int m = StringSplit(text, '/', pendings);
              if(m == 0)
              {
                m = collectPendingLines(pendings);
              }
              for(int j = 0; j < m; j++)
              {
                if(processPending(pendings[j])) succeeded++;
                else failed += (j > 0 ? "/" : "") + pendings[j];
              }
              
              if(succeeded == m) // success
              {
                if(m == 0)
                {
                  showMessage("Incorrect description for pending orders in '" + name + "'");
                }
                disableLine(i);
              }
              else if(succeeded > 0) // partial retry
              {
                ObjectSetString(0, name, OBJPROP_TEXT, failed);
                showMessage("Incomplete state with pending orders ('" + text + "') in '" + name + "'");
              }
              else // otherwise full retry
              {
                showMessage("Pending orders failed with '" + name + "'");
              }
            }
          }
          else if(style == CloseStopLossTakeProfitType) // close all positions
          {
            int dir = checkMarket(price) || checkTime(name);
            if(dir != 0 && e.getMarketOrderCount() > 0)
            {
              Print("Activated: ", name);
              if(e.closeMarketOrders() > 0)
              {
                disableLine(i);
              }
              else
              {
                showMessage("Total close failed with '" + name + "'");
              }
            }
          }
        }
        else if(clr == BuyColor)
        {
          if(style == InstantType)
          {
            int dir = checkMarket(price, last);
            if((dir == 0) && checkTime(name))
            {
              if(clr == BuyColor) dir = +1;
              else if(clr == SellColor) dir = -1;
            }
            if(dir > 0)
            {
              Print("Activated: ", name);
              double lot = StringToDouble(ObjectGetString(0, name, OBJPROP_TEXT)); // lot[%]
              if(lot == 0) lot = Lot;
    
              double sl = 0.0, tp = 0.0;
              if(aux != 0 && auxf != 0)
              {
                if(aux > Ask)
                {
                  tp = aux;
                  sl = auxf;
                }
                else
                {
                  sl = aux;
                  tp = auxf;
                }
              }
              else if(aux != 0)
              {
                if(aux > Ask)
                {
                  tp = aux;
                  if(DefaultStopLoss != 0) sl = Bid - e.getPointsForLotAndRisk(DefaultStopLoss, lot) * _Point;
                }
                else
                {
                  sl = aux;
                  if(DefaultTakeProfit != 0) tp = Bid + e.getPointsForLotAndRisk(DefaultTakeProfit, lot) * _Point;
                }
              }
              else
              {
                if(DefaultStopLoss != 0) sl = Bid - e.getPointsForLotAndRisk(DefaultStopLoss, lot) * _Point;
                if(DefaultTakeProfit != 0) tp = Bid + e.getPointsForLotAndRisk(DefaultTakeProfit, lot) * _Point;
              }
              
              sl = NormalizeDouble(sl, _Digits);
              tp = NormalizeDouble(tp, _Digits);
            
              int ticket = e.placeMarketOrder(OP_BUY, lot, sl, tp);
              if(ticket != -1) // success
              {
                disableLine(i);
              }
              else
              {
                showMessage("Market buy failed with '" + name + "'");
              }
            }
            else if((dir < 0) && CancelCounterparts) // for example, breakthrough down is happened instead of rebound up
            {
              Print("Buy line " + name + " breakdown: ",
                DoubleToString(PreviousBid, _Digits), "->", DoubleToString(Bid, _Digits), "/", DoubleToString(price, _Digits));
              disableLine(i);
            }
          }
          else if(style == CloseStopLossTakeProfitType) // close sell position, stoploss for sell, takeprofit for sell
          {
            int dir = checkMarket(price) || checkTime(name);
            if(dir != 0 && e.getOrderCount(e.mask(OP_SELL)) > 0)
            {
              Print("Activated: ", name);
              double lot = StringToDouble(ObjectGetString(0, name, OBJPROP_TEXT)); // lot
              if(lot > 0)
              {
                if(e.placeMarketOrder(OP_BUY, lot) != -1) // will trigger OrderCloseBy();
                {
                  disableLine(i);
                }
                else
                {
                  showMessage("Partial sell close failed with '" + name + "'");
                }
              }
              else
              {
                if(e.closeMarketOrders(e.mask(OP_SELL)) > 0)
                {
                  disableLine(i);
                }
                else
                {
                  showMessage("Complete sell close failed with '" + name + "'");
                }
              }
            }
          }
        }
        else if(clr == SellColor)
        {
          if(style == InstantType)
          {
            int dir = checkMarket(price, last);
            if((dir == 0) && checkTime(name))
            {
              if(clr == BuyColor) dir = +1;
              else if(clr == SellColor) dir = -1;
            }
            if(dir < 0)
            {
              Print("Activated: ", name);
              double lot = StringToDouble(ObjectGetString(0, name, OBJPROP_TEXT)); // lot[%]
              if(lot == 0) lot = Lot;
    
              double sl = 0.0, tp = 0.0;
              if(aux != 0 && auxf != 0)
              {
                if(aux > Bid)
                {
                  sl = aux;
                  tp = auxf;
                }
                else
                {
                  tp = aux;
                  sl = auxf;
                }
              }
              else if(aux != 0)
              {
                if(aux > Bid)
                {
                  sl = aux;
                  if(DefaultTakeProfit != 0) tp = Ask - e.getPointsForLotAndRisk(DefaultTakeProfit, lot) * _Point;
                }
                else
                {
                  tp = aux;
                  if(DefaultStopLoss != 0) sl = Ask + e.getPointsForLotAndRisk(DefaultStopLoss, lot) * _Point;
                }
              }
              else
              {
                if(DefaultStopLoss != 0) sl = Ask + e.getPointsForLotAndRisk(DefaultStopLoss, lot) * _Point;
                if(DefaultTakeProfit != 0) tp = Ask - e.getPointsForLotAndRisk(DefaultTakeProfit, lot) * _Point;
              }
    
              sl = NormalizeDouble(sl, _Digits);
              tp = NormalizeDouble(tp, _Digits);
    
              int ticket = e.placeMarketOrder(OP_SELL, lot, sl, tp);
              if(ticket != -1) // success
              {
                disableLine(i);
              }
              else
              {
                showMessage("Market sell failed with '" + name + "'");
              }
            }
            else if((dir > 0) && CancelCounterparts) // for example, breakthrough up is happened instead of rebound down
            {
              Print("Sell line " + name + " breakup: ",
                DoubleToString(PreviousBid, _Digits), "->", DoubleToString(Bid, _Digits), "/", DoubleToString(price, _Digits));
              disableLine(i);
            }
          }
          else if(style == CloseStopLossTakeProfitType) // close buy position, stoploss for buy, takeprofit for buy
          {
            int dir = checkMarket(price) || checkTime(name);
            if(dir != 0 && e.getOrderCount(e.mask(OP_BUY)) > 0)
            {
              Print("Activated: ", name);
              double lot = StringToDouble(ObjectGetString(0, name, OBJPROP_TEXT)); // lot
              if(lot > 0)
              {
                if(e.placeMarketOrder(OP_SELL, lot) != -1) // will trigger OrderCloseBy();
                {
                  disableLine(i);
                }
                else
                {
                  showMessage("Partial buy close failed with '" + name + "'");
                }
              }
              else
              {
                if(e.closeMarketOrders(e.mask(OP_BUY)) > 0)
                {
                  disableLine(i);
                }
                else
                {
                  showMessage("Complete buy close failed with '" + name + "'");
                }
              }
            }
          }
        }
      }
    }
    
    color darken(color c)
    {
      return (color)(
        ((c & 0xFF) >> 1) +
        (((c & 0xFF00) >> 9) << 8) +
        (((c & 0xFF0000) >> 17) << 16)
      );
    }
    
    void display()
    {
      int n = ArraySize(objects);
      string comment = "TradeObjects: " + (string)n + " objects\n";
      for(int i = 0; i < n; i++)
      {
        comment += getObjectLegend(objects[i].name) + (objects[i].status == -1?" invalid":"") + "\n";
      }
      Comment(comment);
      
      if(ShowBreakEven)
      {
        updateBELine(e.findBreakEvenLevel());
      }
    }
    
    string getObjectLegend(const string sparam)
    {
      int style = (int)ObjectGetInteger(0, sparam, OBJPROP_STYLE);
      color clr = (color)ObjectGetInteger(0, sparam, OBJPROP_COLOR);
    
      string description = EnumToString((ENUM_OBJECT)ObjectGetInteger(0, sparam, OBJPROP_TYPE));
      string text = ObjectGetString(0, sparam, OBJPROP_TEXT);
      
      string opposite = "";
      if(clr == BuyColor || clr == DimmedBuyColor)
      {
        description += " buy";
        opposite = "sell";
        if(text == "") text = (style == CloseStopLossTakeProfitType ? "all" : DoubleToString(Lot, 2));
      }
      else if(clr == SellColor || clr == DimmedSellColor)
      {
        description += " sell";
        opposite = "buy";
        if(text == "") text = (style == CloseStopLossTakeProfitType ? "all" : DoubleToString(Lot, 2));
      }
      else if(clr == ActivationColor || clr == DimmedActivationColor)
      {
        description += " activation";
        opposite = "all";
      }
      else description += " n/a";
      
      if(text != "") description += " " + text;
    
      if(style == InstantType) { /* description += " market"; */ }
      else if(style == PendingType) description += " pending";
      else if(style == CloseStopLossTakeProfitType) description += ": close/sl/tp for " + opposite;
      else description += "?";
      
      int width = (int)ObjectGetInteger(0, sparam, OBJPROP_WIDTH);
      if(width > 1)
      {
        description += " [" + (string)width + "]";
      }
      
      if(ObjectGetInteger(ChartID(), sparam, OBJPROP_BACK))
      {
        description += " [disabled/done]";
      }
      
      return description;
    }
    
    void describe(string sparam)
    {
      Print("New line added: '", sparam, "' ", getObjectLegend(sparam));
    }
    
  public:
    void handleInit()
    {
      if(DefaultStopLoss < 0) // informational message about risk -> amount
      {
        double _PointValue = _Point * MarketInfo(_Symbol, MODE_TICKVALUE) / MarketInfo(_Symbol, MODE_TICKSIZE);
        double Loss = -DefaultStopLoss / 100.0 * AccountFreeMargin();
        Print("For lot ", DoubleToString(Lot, 2), " and Risk ", -DefaultStopLoss,
          "% of ", DoubleToString(AccountFreeMargin(), 2), "$ -> loss=",
          DoubleToString(Loss, 2), "$ SL=", (int)(Loss / Lot / _PointValue), "pts");
      }
      
      DimmedBuyColor = darken(BuyColor);
      DimmedSellColor = darken(SellColor);
      DimmedActivationColor = darken(ActivationColor);
      
      _cc << DimmedBuyColor << DimmedSellColor << DimmedActivationColor;
      
      detectLines();
    }
    
    void handleTick()
    {
      static datetime lastBar;
    
      #ifdef __MQL4__  
      if(MQLInfoInteger(MQL_TESTER))
      {
        static datetime lastTick = 0;
        if(TimeCurrent() != lastTick)
        {
          handleTimer();
          lastTick = TimeCurrent();
        }
      }
      #endif
    
      if(lastBar != Time[0])
      {
        lastBar = Time[0];
        detectLines();
      }
      
      e.trailStops();
    }
    
    void handleTimer()
    {
      static int counter = 0;
      
      // detectLines();
      
      counter++;
      if(counter == EventTimeSpan) // wait until we have history record of bid for EventTimeSpan
      {
        counter = 0;
        if(PreviousBid > 0) processLines();
        if(PreviousBid != Bid) PreviousBid = Bid;
      }

      if(AllowOrderCloseBy) e.shrink();
    }
    
    void handleChart(const int id, const long &lparam, const double &dparam, const string &sparam)
    {
      if(id == CHARTEVENT_OBJECT_CREATE || id == CHARTEVENT_OBJECT_CHANGE)
      {
        if(checkObjectCompliance(sparam))
        {
          if(attachObject(sparam))
          {
            display();
            describe(sparam);
          }
        }
        else
        {
          detectLines();
        }
      }
      else if(id == CHARTEVENT_OBJECT_DELETE)
      {
        if(removeObject(sparam))
        {
          display();
          Print("Line deleted: ", sparam);
        }
      }
      else if(id == CHARTEVENT_CHART_CHANGE)
      {
        detectLines();
      }
    }
    
    TradeObjects()
    {
      e = new Expert(Magic, Lot, Deviation);
    }
    
    ~TradeObjects()
    {
      delete e;
    }
};

TradeObjects to;

//+------------------------------------------------------------------+
//| expert initialization function                                   |
//+------------------------------------------------------------------+
void OnInit()
{
  ChartSetInteger(0, CHART_EVENT_OBJECT_DELETE, true);
  EventSetTimer(1);
  to.handleInit();
}

//+------------------------------------------------------------------+
//| expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
}

//+------------------------------------------------------------------+
//| chart events handler                                             |
//+------------------------------------------------------------------+
void OnChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam)
{
  to.handleChart(id, lparam, dparam, sparam);
}
                 
//+------------------------------------------------------------------+
//| timer handler                                                    |
//+------------------------------------------------------------------+
void OnTimer()
{
  to.handleTimer();
}

//+------------------------------------------------------------------+
//| expert start function                                            |
//+------------------------------------------------------------------+
void OnTick()
{
  to.handleTick();
}
//+------------------------------------------------------------------+
