#include "ExecutionAlgorithm.mqh"

//+------------------------------------------------------------------+
//| Iceberg Order Implementation                                      |
//+------------------------------------------------------------------+
class CIcebergOrder : public CExecutionAlgorithm
{
private:
   double            m_visibleVolume;       // Visible portion of the order
   double            m_minVisibleVolume;    // Minimum visible volume
   double            m_maxVisibleVolume;    // Maximum visible volume
   bool              m_useRandomVisibleVolume; // Whether to randomize visible volume
   int               m_orderPlacementDelay; // Delay between order placements (ms)
   bool              m_avoidRoundNumbers;   // Whether to avoid round numbers in price
   double            m_limitPrice;          // Limit price for the orders
   ulong             m_currentOrderTicket;  // Current active order ticket
   ENUM_ORDER_TYPE   m_orderType;           // Order type (buy or sell)
   bool              m_orderActive;         // Flag indicating if an order is currently active
   int               m_priceDeviation;      // Price deviation to avoid round numbers (in points)
   
public:
   // Constructor
   CIcebergOrder(string symbol, double volume, double limitPrice, ENUM_ORDER_TYPE orderType,
                 double visibleVolume, double minVisibleVolume = 0.0, double maxVisibleVolume = 0.0,
                 bool useRandomVisibleVolume = true, int orderPlacementDelay = 1000, 
                 bool avoidRoundNumbers = true, int priceDeviation = 2, int slippage = 3);
   
   // Destructor
   ~CIcebergOrder();
   
   // Implementation of virtual methods
   virtual bool      Initialize() override;
   virtual bool      Execute() override;
   virtual bool      Update() override;
   virtual bool      Terminate() override;
   
   // Iceberg specific methods
   double            GetRandomVisibleVolume();
   double            AdjustPriceToAvoidRoundNumbers(double price);
   bool              CheckAndReplaceOrder();
   bool              IsOrderFilled(ulong ticket);
   bool              IsOrderPartiallyFilled(ulong ticket, double &filledVolume);
};

//+------------------------------------------------------------------+
//| Constructor                                                       |
//+------------------------------------------------------------------+
CIcebergOrder::CIcebergOrder(string symbol, double volume, double limitPrice, ENUM_ORDER_TYPE orderType,
                           double visibleVolume, double minVisibleVolume, double maxVisibleVolume,
                           bool useRandomVisibleVolume, int orderPlacementDelay, 
                           bool avoidRoundNumbers, int priceDeviation, int slippage)
   : CExecutionAlgorithm(symbol, volume, 0, 0, slippage) // Iceberg doesn't use time constraints
{
   m_visibleVolume = visibleVolume;
   m_minVisibleVolume = minVisibleVolume;
   m_maxVisibleVolume = maxVisibleVolume;
   m_useRandomVisibleVolume = useRandomVisibleVolume;
   m_orderPlacementDelay = orderPlacementDelay;
   m_avoidRoundNumbers = avoidRoundNumbers;
   m_limitPrice = limitPrice;
   m_orderType = orderType;
   m_currentOrderTicket = 0;
   m_orderActive = false;
   m_priceDeviation = priceDeviation;
   
   // If min/max not specified, set defaults based on visible volume
   if(m_minVisibleVolume <= 0.0)
      m_minVisibleVolume = m_visibleVolume * 0.5;
      
   if(m_maxVisibleVolume <= 0.0)
      m_maxVisibleVolume = m_visibleVolume * 1.5;
}

//+------------------------------------------------------------------+
//| Destructor                                                        |
//+------------------------------------------------------------------+
CIcebergOrder::~CIcebergOrder()
{
   // Clean up resources if needed
   if(m_orderActive && m_currentOrderTicket > 0)
   {
      // Try to cancel the current order
      CancelOrder(m_currentOrderTicket);
   }
}

//+------------------------------------------------------------------+
//| Initialize the Iceberg Order                                      |
//+------------------------------------------------------------------+
bool CIcebergOrder::Initialize()
{
   if(!CExecutionAlgorithm::Initialize())
      return false;
      
   // Validate the visible volume
   double minVolume = SymbolInfoDouble(m_symbol, SYMBOL_VOLUME_MIN);
   double maxVolume = SymbolInfoDouble(m_symbol, SYMBOL_VOLUME_MAX);
   double stepVolume = SymbolInfoDouble(m_symbol, SYMBOL_VOLUME_STEP);
   
   // Ensure visible volume is within valid bounds
   m_visibleVolume = MathMax(minVolume, MathMin(maxVolume, m_visibleVolume));
   m_visibleVolume = MathFloor(m_visibleVolume / stepVolume) * stepVolume;
   
   // Ensure min/max visible volumes are within valid bounds
   m_minVisibleVolume = MathMax(minVolume, MathMin(m_visibleVolume, m_minVisibleVolume));
   m_minVisibleVolume = MathFloor(m_minVisibleVolume / stepVolume) * stepVolume;
   
   m_maxVisibleVolume = MathMax(m_visibleVolume, MathMin(maxVolume, m_maxVisibleVolume));
   m_maxVisibleVolume = MathFloor(m_maxVisibleVolume / stepVolume) * stepVolume;
   
   // Adjust the limit price if needed
   if(m_avoidRoundNumbers)
      m_limitPrice = AdjustPriceToAvoidRoundNumbers(m_limitPrice);
      
   m_isActive = true;
   m_orderActive = false;
   
   Print("Iceberg Order initialized for ", m_symbol, 
         ". Total volume: ", DoubleToString(m_totalVolume, 2), 
         ", Visible volume: ", DoubleToString(m_visibleVolume, 2),
         ", Limit price: ", DoubleToString(m_limitPrice, _Digits));
   
   return true;
}

//+------------------------------------------------------------------+
//| Execute the Iceberg Order                                         |
//+------------------------------------------------------------------+
bool CIcebergOrder::Execute()
{
   if(!m_isActive)
      return false;
      
   // If an order is already active, check its status
   if(m_orderActive)
   {
      return CheckAndReplaceOrder();
   }
   
   // Calculate the volume for this execution
   double volumeToExecute = m_useRandomVisibleVolume ? 
                           GetRandomVisibleVolume() : 
                           m_visibleVolume;
                           
   // Ensure we don't exceed the remaining volume
   if(volumeToExecute > m_remainingVolume)
      volumeToExecute = m_remainingVolume;
      
   // Place the order
   if(!PlaceOrder(m_orderType, volumeToExecute, m_limitPrice))
   {
      Print("Iceberg: Failed to place order");
      return false;
   }
   
   // Store the order ticket
   m_currentOrderTicket = OrderGetTicket(0);
   m_orderActive = true;
   
   Print("Iceberg: Placed order for ", DoubleToString(volumeToExecute, 2), 
         " at price ", DoubleToString(m_limitPrice, _Digits), 
         ". Remaining: ", DoubleToString(m_remainingVolume, 2),
         ", Ticket: ", m_currentOrderTicket);
         
   return true;
}

//+------------------------------------------------------------------+
//| Update the Iceberg Order state                                    |
//+------------------------------------------------------------------+
bool CIcebergOrder::Update()
{
   if(!m_isActive)
      return false;
      
   // Check if all volume has been executed
   if(m_remainingVolume <= 0)
   {
      Print("Iceberg: All volume executed. Terminating algorithm.");
      return Terminate();
   }
   
   // If an order is active, check its status
   if(m_orderActive)
   {
      return CheckAndReplaceOrder();
   }
   else
   {
      // If no order is active, execute a new one
      return Execute();
   }
}

//+------------------------------------------------------------------+
//| Terminate the Iceberg Order                                       |
//+------------------------------------------------------------------+
bool CIcebergOrder::Terminate()
{
   if(!m_isActive)
      return false;
      
   // If there's an active order, cancel it
   if(m_orderActive && m_currentOrderTicket > 0)
   {
      if(!CancelOrder(m_currentOrderTicket))
      {
         Print("Iceberg: Failed to cancel order ", m_currentOrderTicket);
      }
      else
      {
         Print("Iceberg: Cancelled order ", m_currentOrderTicket);
         m_orderActive = false;
      }
   }
   
   // If there's remaining volume and we want to execute it all at once
   if(m_remainingVolume > 0)
   {
      Print("Iceberg: Terminating with remaining volume: ", DoubleToString(m_remainingVolume, 2));
      
      // Place the final order for remaining volume
      if(!PlaceOrder(m_orderType, m_remainingVolume, m_limitPrice))
      {
         Print("Iceberg: Failed to place final order");
      }
   }
   
   m_isActive = false;
   
   Print("Iceberg Order terminated. Total executed volume: ", 
         DoubleToString(m_executedVolume, 2));
         
   return true;
}

//+------------------------------------------------------------------+
//| Get a randomized visible volume                                   |
//+------------------------------------------------------------------+
double CIcebergOrder::GetRandomVisibleVolume()
{
   // Generate a random volume between min and max visible volume
   double range = m_maxVisibleVolume - m_minVisibleVolume;
   double randomVolume = m_minVisibleVolume + (MathRand() / (double)32767) * range;
   
   // Ensure the volume is within valid bounds
   double minVolume = SymbolInfoDouble(m_symbol, SYMBOL_VOLUME_MIN);
   double maxVolume = SymbolInfoDouble(m_symbol, SYMBOL_VOLUME_MAX);
   double stepVolume = SymbolInfoDouble(m_symbol, SYMBOL_VOLUME_STEP);
   
   // Normalize to volume step
   randomVolume = MathFloor(randomVolume / stepVolume) * stepVolume;
   
   // Ensure within min/max bounds
   randomVolume = MathMax(minVolume, MathMin(maxVolume, randomVolume));
   
   return randomVolume;
}

//+------------------------------------------------------------------+
//| Adjust price to avoid round numbers                               |
//+------------------------------------------------------------------+
double CIcebergOrder::AdjustPriceToAvoidRoundNumbers(double price)
{
   // This function adjusts the price to avoid round numbers
   // which can make iceberg orders more detectable
   
   // Get the tick size
   double tickSize = SymbolInfoDouble(m_symbol, SYMBOL_TRADE_TICK_SIZE);
   
   // Check if the price is a round number (ending in .00 or .50)
   string priceStr = DoubleToString(price, _Digits);
   int len = StringLen(priceStr);
   
   bool isRoundNumber = false;
   
   if(len >= 3)
   {
      string lastTwoChars = StringSubstr(priceStr, len - 2, 2);
      if(lastTwoChars == "00" || lastTwoChars == "50" || 
         lastTwoChars == "25" || lastTwoChars == "75")
         isRoundNumber = true;
   }
   
   // If it's a round number, adjust it
   if(isRoundNumber)
   {
      // Add or subtract a small random deviation
      int points = m_priceDeviation + (int)(MathRand() % 3); // 2-4 points
      double adjustment = points * SymbolInfoDouble(m_symbol, SYMBOL_POINT);
      
      // For buy orders, subtract from the price
      // For sell orders, add to the price
      // This gives a slightly better price
      if(m_orderType == ORDER_TYPE_BUY)
         price -= adjustment;
      else
         price += adjustment;
         
      Print("Iceberg: Adjusted price from round number. New price: ", DoubleToString(price, _Digits));
   }
   
   return price;
}

//+------------------------------------------------------------------+
//| Check order status and replace if needed                          |
//+------------------------------------------------------------------+
bool CIcebergOrder::CheckAndReplaceOrder()
{
   if(!m_orderActive || m_currentOrderTicket == 0)
      return false;
      
   // Check if the order is filled
   if(IsOrderFilled(m_currentOrderTicket))
   {
      // Order is filled, update statistics
      double orderVolume = 0.0;
      
      // Get the order volume
      if(OrderSelect(m_currentOrderTicket))
         orderVolume = OrderGetDouble(ORDER_VOLUME_INITIAL);
      
      Print("Iceberg: Order filled. Ticket: ", m_currentOrderTicket, 
            ", Volume: ", DoubleToString(orderVolume, 2), 
            ", Remaining: ", DoubleToString(m_remainingVolume, 2));
            
      // Reset order tracking
      m_orderActive = false;
      m_currentOrderTicket = 0;
            
      // If there's still volume to execute, place a new order after a delay
      if(m_remainingVolume > 0)
      {
         // Add a delay before placing the next order
         Sleep(m_orderPlacementDelay);
         return Execute();
      }
      
      return true;
   }
   
   // Check if the order is partially filled
   double filledVolume = 0.0;
   if(IsOrderPartiallyFilled(m_currentOrderTicket, filledVolume))
   {
      // Order is partially filled
      Print("Iceberg: Order partially filled. Ticket: ", m_currentOrderTicket,
            ", Volume: ", DoubleToString(filledVolume, 2), 
            ", Remaining: ", DoubleToString(m_remainingVolume, 2));
            
      // Cancel the current order
      if(!CancelOrder(m_currentOrderTicket))
      {
         Print("Iceberg: Failed to cancel partially filled order ", m_currentOrderTicket);
         return false;
      }
      
      // Reset order tracking
      m_orderActive = false;
      m_currentOrderTicket = 0;
      
      // Place a new order for the remaining volume after a delay
      Sleep(m_orderPlacementDelay);
      return Execute();
   }
   
   return true;
}

//+------------------------------------------------------------------+
//| Check if an order is completely filled                            |
//+------------------------------------------------------------------+
bool CIcebergOrder::IsOrderFilled(ulong ticket)
{
   // This function checks if an order is completely filled
   
   if(!OrderSelect(ticket))
      return false;
      
   // Check the order state
   ENUM_ORDER_STATE orderState = (ENUM_ORDER_STATE)OrderGetInteger(ORDER_STATE);
   
   return (orderState == ORDER_STATE_FILLED);
}

//+------------------------------------------------------------------+
//| Check if an order is partially filled                             |
//+------------------------------------------------------------------+
bool CIcebergOrder::IsOrderPartiallyFilled(ulong ticket, double &filledVolume)
{
   // This function checks if an order is partially filled
   // and returns the filled volume
   
   filledVolume = 0.0;
   
   if(!OrderSelect(ticket))
      return false;
      
   // Check the order state
   ENUM_ORDER_STATE orderState = (ENUM_ORDER_STATE)OrderGetInteger(ORDER_STATE);
   
   // If the order is already filled, it's not partially filled
   if(orderState == ORDER_STATE_FILLED)
      return false;
      
   // Get the initial and current volume
   double initialVolume = OrderGetDouble(ORDER_VOLUME_INITIAL);
   double currentVolume = OrderGetDouble(ORDER_VOLUME_CURRENT);
   
   // Calculate the filled volume
   filledVolume = initialVolume - currentVolume;
   
   // If some volume is filled but not all, it's partially filled
   return (filledVolume > 0 && filledVolume < initialVolume);
}
//+------------------------------------------------------------------+
