#include "ExecutionAlgorithm.mqh"
#include "TWAP.mqh"
#include "VWAP.mqh"
#include "IcebergOrder.mqh"
#include "PerformanceAnalyzer.mqh"

// Market condition enumeration
enum ENUM_MARKET_CONDITION
{
   MARKET_CONDITION_NORMAL,     // Normal market conditions
   MARKET_CONDITION_VOLATILE,   // Volatile market conditions
   MARKET_CONDITION_TRENDING,   // Trending market conditions
   MARKET_CONDITION_ILLIQUID    // Illiquid market conditions
};

// Signal type enumeration
enum ENUM_SIGNAL_TYPE
{
   SIGNAL_TYPE_BUY,             // Buy signal
   SIGNAL_TYPE_SELL,            // Sell signal
   SIGNAL_TYPE_CLOSE_BUY,       // Close buy signal
   SIGNAL_TYPE_CLOSE_SELL       // Close sell signal
};

//+------------------------------------------------------------------+
//| Framework for integrating execution algorithms with strategies    |
//+------------------------------------------------------------------+
class CExecutionManager
{
private:
   CExecutionAlgorithm *m_algorithms[];  // Array of execution algorithms
   int               m_algorithmCount;   // Number of algorithms
   bool              m_isActive;         // Is manager active
   string            m_symbol;           // Trading symbol
   int               m_slippage;         // Default slippage in points
   bool              m_autoSelectAlgorithm; // Whether to auto-select algorithm based on market conditions
   
public:
   // Constructor/Destructor
   CExecutionManager(string symbol, int slippage = 3, bool autoSelectAlgorithm = true);
   ~CExecutionManager();
   
   // Management methods
   bool              AddAlgorithm(CExecutionAlgorithm *algorithm);
   bool              RemoveAlgorithm(int index);
   bool              StartExecution();
   bool              StopExecution();
   bool              UpdateAlgorithms();
   
   // Integration with strategies
   bool              ExecuteSignal(ENUM_SIGNAL_TYPE signalType, double volume, double price = 0.0);
   
   // Adaptive algorithm selection
   CExecutionAlgorithm* SelectBestAlgorithm(double volume, ENUM_MARKET_CONDITION marketCondition);
   
   // Market condition analysis
   ENUM_MARKET_CONDITION AnalyzeMarketCondition();
   
   // Utility methods
   int               GetAlgorithmCount() { return m_algorithmCount; }
   CExecutionAlgorithm* GetAlgorithm(int index);
   bool              IsActive() { return m_isActive; }
   
   // Create specific algorithm instances
   CTWAP*            CreateTWAP(double volume, datetime startTime, datetime endTime, 
                              int intervals, ENUM_ORDER_TYPE orderType, 
                              bool useRandomization = false);
                              
   CVWAP*            CreateVWAP(double volume, datetime startTime, datetime endTime, 
                              int intervals, ENUM_ORDER_TYPE orderType, 
                              bool adaptiveMode = true);
                              
   CIcebergOrder*    CreateIcebergOrder(double volume, double limitPrice, ENUM_ORDER_TYPE orderType,
                                      double visibleVolume, bool useRandomVisibleVolume = true);
};

//+------------------------------------------------------------------+
//| Constructor                                                       |
//+------------------------------------------------------------------+
CExecutionManager::CExecutionManager(string symbol, int slippage, bool autoSelectAlgorithm)
{
   m_symbol = symbol;
   m_slippage = slippage;
   m_autoSelectAlgorithm = autoSelectAlgorithm;
   m_algorithmCount = 0;
   m_isActive = false;
   
   // Initialize the algorithm array
   ArrayResize(m_algorithms, 0);
}

//+------------------------------------------------------------------+
//| Destructor                                                        |
//+------------------------------------------------------------------+
CExecutionManager::~CExecutionManager()
{
   // Clean up all algorithms
   for(int i = 0; i < m_algorithmCount; i++)
   {
      if(m_algorithms[i] != NULL)
      {
         delete m_algorithms[i];
         m_algorithms[i] = NULL;
      }
   }
   
   ArrayFree(m_algorithms);
}

//+------------------------------------------------------------------+
//| Add an algorithm to the manager                                   |
//+------------------------------------------------------------------+
bool CExecutionManager::AddAlgorithm(CExecutionAlgorithm *algorithm)
{
   if(algorithm == NULL)
      return false;
      
   // Resize the array
   int newSize = m_algorithmCount + 1;
   if(ArrayResize(m_algorithms, newSize) != newSize)
   {
      Print("Failed to resize algorithm array");
      return false;
   }
   
   // Add the algorithm
   m_algorithms[m_algorithmCount] = algorithm;
   m_algorithmCount++;
   
   Print("Algorithm added to manager. Total count: ", m_algorithmCount);
   
   return true;
}

//+------------------------------------------------------------------+
//| Remove an algorithm from the manager                              |
//+------------------------------------------------------------------+
bool CExecutionManager::RemoveAlgorithm(int index)
{
   if(index < 0 || index >= m_algorithmCount)
      return false;
      
   // Delete the algorithm
   if(m_algorithms[index] != NULL)
   {
      delete m_algorithms[index];
      m_algorithms[index] = NULL;
   }
   
   // Shift the array
   for(int i = index; i < m_algorithmCount - 1; i++)
      m_algorithms[i] = m_algorithms[i + 1];
      
   // Resize the array
   m_algorithmCount--;
   ArrayResize(m_algorithms, m_algorithmCount);
   
   Print("Algorithm removed from manager. Total count: ", m_algorithmCount);
   
   return true;
}

//+------------------------------------------------------------------+
//| Start execution of all algorithms                                 |
//+------------------------------------------------------------------+
bool CExecutionManager::StartExecution()
{
   if(m_isActive)
      return true;
      
   // Initialize all algorithms
   for(int i = 0; i < m_algorithmCount; i++)
   {
      if(m_algorithms[i] != NULL)
      {
         if(!m_algorithms[i].Initialize())
         {
            Print("Failed to initialize algorithm ", i);
            return false;
         }
      }
   }
   
   m_isActive = true;
   
   Print("Execution manager started");
   
   return true;
}

//+------------------------------------------------------------------+
//| Stop execution of all algorithms                                  |
//+------------------------------------------------------------------+
bool CExecutionManager::StopExecution()
{
   if(!m_isActive)
      return true;
      
   // Terminate all algorithms
   for(int i = 0; i < m_algorithmCount; i++)
   {
      if(m_algorithms[i] != NULL)
      {
         if(m_algorithms[i].IsActive())
         {
            if(!m_algorithms[i].Terminate())
            {
               Print("Failed to terminate algorithm ", i);
            }
         }
      }
   }
   
   m_isActive = false;
   
   Print("Execution manager stopped");
   
   return true;
}

//+------------------------------------------------------------------+
//| Update all algorithms                                             |
//+------------------------------------------------------------------+
bool CExecutionManager::UpdateAlgorithms()
{
   if(!m_isActive)
      return false;
      
   bool allInactive = true;
   
   // Update all algorithms
   for(int i = 0; i < m_algorithmCount; i++)
   {
      if(m_algorithms[i] != NULL)
      {
         if(m_algorithms[i].IsActive())
         {
            allInactive = false;
            
            // Execute and update the algorithm
            m_algorithms[i].Execute();
            m_algorithms[i].Update();
         }
      }
   }
   
   // If all algorithms are inactive, stop the manager
   if(allInactive && m_algorithmCount > 0)
   {
      Print("All algorithms are inactive. Stopping execution manager.");
      m_isActive = false;
   }
   
   return true;
}

//+------------------------------------------------------------------+
//| Execute a trading signal                                          |
//+------------------------------------------------------------------+
bool CExecutionManager::ExecuteSignal(ENUM_SIGNAL_TYPE signalType, double volume, double price)
{
   if(volume <= 0)
      return false;
      
   // Determine the order type
   ENUM_ORDER_TYPE orderType;
   switch(signalType)
   {
      case SIGNAL_TYPE_BUY:
         orderType = ORDER_TYPE_BUY;
         break;
      case SIGNAL_TYPE_SELL:
         orderType = ORDER_TYPE_SELL;
         break;
      case SIGNAL_TYPE_CLOSE_BUY:
         orderType = ORDER_TYPE_SELL; // To close a buy position, we sell
         break;
      case SIGNAL_TYPE_CLOSE_SELL:
         orderType = ORDER_TYPE_BUY;  // To close a sell position, we buy
         break;
      default:
         Print("Invalid signal type");
         return false;
   }
   
   // If auto-select is enabled, choose the best algorithm based on market conditions
   if(m_autoSelectAlgorithm)
   {
      ENUM_MARKET_CONDITION marketCondition = AnalyzeMarketCondition();
      CExecutionAlgorithm *algorithm = SelectBestAlgorithm(volume, marketCondition);
      
      if(algorithm != NULL)
      {
         // Add the algorithm to the manager
         AddAlgorithm(algorithm);
         
         // Start execution
         if(!m_isActive)
            StartExecution();
            
         return true;
      }
      else
      {
         Print("Failed to select an appropriate algorithm");
         return false;
      }
   }
   else
   {
      // If no auto-select, use the first available algorithm or create a default one
      if(m_algorithmCount > 0)
      {
         // Use the first algorithm
         // Note: In a real implementation, you might want to check if the algorithm is suitable
         return true;
      }
      else
      {
         // Create a default algorithm (TWAP)
         datetime startTime = TimeCurrent();
         datetime endTime = startTime + 3600; // 1 hour
         
         CTWAP *twap = CreateTWAP(volume, startTime, endTime, 6, orderType, true);
         
         if(twap != NULL)
         {
            // Start execution
            if(!m_isActive)
               StartExecution();
               
            return true;
         }
         else
         {
            Print("Failed to create default algorithm");
            return false;
         }
      }
   }
   
   return false;
}

//+------------------------------------------------------------------+
//| Select the best algorithm based on market conditions              |
//+------------------------------------------------------------------+
CExecutionAlgorithm* CExecutionManager::SelectBestAlgorithm(double volume, ENUM_MARKET_CONDITION marketCondition)
{
   // This function selects the most appropriate algorithm based on market conditions
   
   datetime currentTime = TimeCurrent();
   
   switch(marketCondition)
   {
      case MARKET_CONDITION_NORMAL:
         // In normal market conditions, TWAP is a good choice
         return CreateTWAP(volume, currentTime, currentTime + 3600, 6, ORDER_TYPE_BUY, true);
         
      case MARKET_CONDITION_VOLATILE:
         // In volatile markets, VWAP with adaptive mode is better
         return CreateVWAP(volume, currentTime, currentTime + 3600, 12, ORDER_TYPE_BUY, true);
         
      case MARKET_CONDITION_TRENDING:
         // In trending markets, TWAP with fewer intervals might be better
         return CreateTWAP(volume, currentTime, currentTime + 1800, 3, ORDER_TYPE_BUY, false);
         
      case MARKET_CONDITION_ILLIQUID:
      {
         // In illiquid markets, Iceberg orders are often the best choice
         double currentPrice = SymbolInfoDouble(m_symbol, SYMBOL_BID);
         return CreateIcebergOrder(volume, currentPrice, ORDER_TYPE_BUY, volume * 0.1, true);
      }
      
      default:
         // Default to TWAP
         return CreateTWAP(volume, currentTime, currentTime + 3600, 6, ORDER_TYPE_BUY, true);
   }
   
   return NULL;
}

//+------------------------------------------------------------------+
//| Analyze current market conditions                                 |
//+------------------------------------------------------------------+
ENUM_MARKET_CONDITION CExecutionManager::AnalyzeMarketCondition()
{
   // This function analyzes the current market conditions
   // and returns the appropriate market condition enum
   
   // For this example, we'll use a simple volatility-based approach
   
   // Get recent price data
   MqlRates rates[];
   int copied = CopyRates(m_symbol, PERIOD_M5, 0, 20, rates);
   
   if(copied <= 0)
   {
      Print("Failed to copy rates for market condition analysis");
      return MARKET_CONDITION_NORMAL; // Default to normal
   }
   
   // Calculate average true range (ATR) as a volatility measure
   double atr = 0.0;
   for(int i = 1; i < copied; i++)
   {
      double trueRange = MathMax(rates[i].high - rates[i].low,
                               MathMax(MathAbs(rates[i].high - rates[i-1].close),
                                      MathAbs(rates[i].low - rates[i-1].close)));
      atr += trueRange;
   }
   
   atr /= (copied - 1);
   
   // Calculate average volume
   double avgVolume = 0.0;
   for(int i = 0; i < copied; i++)
      avgVolume += rates[i].tick_volume;
      
   avgVolume /= copied;
   
   // Calculate price direction (trend)
   double priceChange = rates[copied-1].close - rates[0].close;
   
   // Get symbol info for normalization
   double tickSize = SymbolInfoDouble(m_symbol, SYMBOL_TRADE_TICK_SIZE);
   double avgPrice = (rates[0].close + rates[copied-1].close) / 2.0;
   
   // Normalize ATR as percentage of price
   double normalizedATR = atr / avgPrice * 100.0;
   
   // Determine market condition
   if(normalizedATR > 0.5) // High volatility threshold
      return MARKET_CONDITION_VOLATILE;
      
   if(MathAbs(priceChange) / avgPrice * 100.0 > 0.3) // Strong trend threshold
      return MARKET_CONDITION_TRENDING;
      
   if(avgVolume < 100) // Low volume threshold (adjust based on symbol)
      return MARKET_CONDITION_ILLIQUID;
      
   // Default case
   return MARKET_CONDITION_NORMAL;
}

//+------------------------------------------------------------------+
//| Get an algorithm by index                                         |
//+------------------------------------------------------------------+
CExecutionAlgorithm* CExecutionManager::GetAlgorithm(int index)
{
   if(index < 0 || index >= m_algorithmCount)
      return NULL;
      
   return m_algorithms[index];
}

//+------------------------------------------------------------------+
//| Create a TWAP algorithm instance                                  |
//+------------------------------------------------------------------+
CTWAP* CExecutionManager::CreateTWAP(double volume, datetime startTime, datetime endTime, 
                                   int intervals, ENUM_ORDER_TYPE orderType, 
                                   bool useRandomization)
{
   // Create a new TWAP algorithm
   CTWAP *twap = new CTWAP(m_symbol, volume, startTime, endTime, intervals, 
                          orderType, useRandomization, 0.2, m_slippage);
                          
   if(twap == NULL)
   {
      Print("Failed to create TWAP algorithm");
      return NULL;
   }
   
   // Add it to the manager
   if(!AddAlgorithm(twap))
   {
      delete twap;
      return NULL;
   }
   
   return twap;
}

//+------------------------------------------------------------------+
//| Create a VWAP algorithm instance                                  |
//+------------------------------------------------------------------+
CVWAP* CExecutionManager::CreateVWAP(double volume, datetime startTime, datetime endTime, 
                                   int intervals, ENUM_ORDER_TYPE orderType, 
                                   bool adaptiveMode)
{
   // Create a new VWAP algorithm
   CVWAP *vwap = new CVWAP(m_symbol, volume, startTime, endTime, intervals, 
                          orderType, 5, adaptiveMode, m_slippage);
                          
   if(vwap == NULL)
   {
      Print("Failed to create VWAP algorithm");
      return NULL;
   }
   
   // Add it to the manager
   if(!AddAlgorithm(vwap))
   {
      delete vwap;
      return NULL;
   }
   
   return vwap;
}

//+------------------------------------------------------------------+
//| Create an Iceberg Order algorithm instance                        |
//+------------------------------------------------------------------+
CIcebergOrder* CExecutionManager::CreateIcebergOrder(double volume, double limitPrice, 
                                                  ENUM_ORDER_TYPE orderType,
                                                  double visibleVolume, 
                                                  bool useRandomVisibleVolume)
{
   // Create a new Iceberg Order algorithm
   CIcebergOrder *iceberg = new CIcebergOrder(m_symbol, volume, limitPrice, orderType,
                                            visibleVolume, 0.0, 0.0, 
                                            useRandomVisibleVolume, 1000, 
                                            true, 2, m_slippage);
                                            
   if(iceberg == NULL)
   {
      Print("Failed to create Iceberg Order algorithm");
      return NULL;
   }
   
   // Add it to the manager
   if(!AddAlgorithm(iceberg))
   {
      delete iceberg;
      return NULL;
   }
   
   return iceberg;
}
//+------------------------------------------------------------------+
