//+------------------------------------------------------------------+
//|                                               asyncwebsocket.mqh |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include<asyncwinhttp.mqh>


//+------------------------------------------------------------------+
//|Class CWebsocket                                                  |
//| Purpose: class for websocket client                              |
//+------------------------------------------------------------------+
class CWebsocket
  {
private:
   ENUM_WEBSOCKET_STATE clientState;            //websocket state
   HINTERNET            hWebSocket;            //websocket handle
   string               serveraddress;          //full server address
   string               serverName;             //server domain name
   INTERNET_PORT        serverPort;             //port number
   string               serverPath;             //server path
   bool                 initialized;            //boolean flag that denotes the state of underlying winhttp infrastruture required for client
   BYTE                 rxbuffer[];             //internal buffer for reading from the socket
   ulong                rxsize;                 //rxbuffer arraysize
   // private methods
   void              reset(void);
   bool              clientsend(BYTE &txbuffer[],WINHTTP_WEB_SOCKET_BUFFER_TYPE buffertype);
   void              clientread(BYTE &rxbuffer[],WINHTTP_WEB_SOCKET_BUFFER_TYPE &buffertype,ulong &bytes);
   string            GetErrorDescription(const DWORD error_code);

public:
                     CWebsocket(void):hWebSocket(NULL),
                     serveraddress(NULL),
                     serverName(NULL),
                     serverPort(0),
                     serverPath(NULL),
                     initialized(false),
                     rxsize(65535)
     {
      ArrayResize(rxbuffer,(int)rxsize);
      ArrayFill(rxbuffer,0,rxsize,0);

     }

                    ~CWebsocket(void)
     {
      ArrayFree(rxbuffer);
      client_reset(hWebSocket);
     }
   //public methods

   bool              Connect(const string _serveraddress, const INTERNET_PORT port=443, bool secure = true);
   void              Close(void);
   bool              SendString(const string msg);
   bool              Send(BYTE &buffer[]);
   ulong             ReadString(string &response);
   ulong             Read(BYTE &buffer[],WINHTTP_WEB_SOCKET_BUFFER_TYPE &buffertype);
   void              Abort(void);
   string            LastErrorMessage(void);
   DWORD             LastError(void);
   //public getter methods
   ENUM_WEBSOCKET_STATE ClientState(void);
   HINTERNET         WebSocketHandle(void);
   ulong             CallBackResult(void);
   ulong             ReadAvailable(void);
   ulong             Poll(void);
   string            ServerAddress(void) {  return(serveraddress);  }
   string            DomainName(void)                {  return(serverName);  }
   INTERNET_PORT     Port(void)               {  return(serverPort);  }
   string            ServerPath(void)                {  return(serverPath);  }
  };

//+----------------------------------------------------------------------------+
//|the helper method releases all http resources (handles) no longer in use    |
//+----------------------------------------------------------------------------+
void CWebsocket::reset(void)
  {
   initialized = false;
   hWebSocket = NULL;
   serveraddress=serverName=serverPath=NULL;
   serverPort=0;
  }
//+------------------------------------------------------------------------------------------------------+
//|Connect method used to set server parameters and establish client connection                          |
//+------------------------------------------------------------------------------------------------------+
bool CWebsocket::Connect(const string _serveraddress,const INTERNET_PORT port=443, bool secure = true)
  {
   if(initialized)
     {
      if(StringCompare(_serveraddress,serveraddress,false))
         Close();
      else
         return(true);
     }

   serveraddress = _serveraddress;

   int dot=StringFind(serveraddress,".");

   int ss=(dot>0)?StringFind(serveraddress,"/",dot):-1;

   serverPath=(ss>0)?StringSubstr(serveraddress,ss+1):"/";

   int sss=StringFind(serveraddress,"://");

   if(sss<0)
      sss=-3;

   serverName=StringSubstr(serveraddress,sss+3,ss-(sss+3));
   serverPort=port;


   DWORD connect_error = client_connect(serveraddress,port,ulong(secure),hWebSocket);

   if(hWebSocket<=0)
     {
      Print(__FUNCTION__," Connection to ", serveraddress, " failed. \n", GetErrorDescription(connect_error));
      return(false);
     }
   else
      initialized = true;

   return(true);
  }

//+------------------------------------------------------------------+
//| helper method for sending data to the server                     |
//+------------------------------------------------------------------+
bool CWebsocket::clientsend(BYTE &txbuffer[],WINHTTP_WEB_SOCKET_BUFFER_TYPE buffertype)
  {
   DWORD len=DWORD(ArraySize(txbuffer));

   if(len<=0)
     {
      Print(__FUNCTION__," Send buffer is empty ");
      return(false);
     }

   DWORD send = client_send(hWebSocket,buffertype,txbuffer,len);

   return(true);

  }

//+------------------------------------------------------------------+
//|public method for sending raw string messages                     |
//+------------------------------------------------------------------+
bool CWebsocket::SendString(const string msg)
  {
   if(!initialized || hWebSocket == NULL)
     {
      Print(__FUNCTION__, " No websocket connection ");
      return(false);
     }

   if(StringLen(msg)<=0)
     {
      Print(__FUNCTION__, " Message buffer is empty ");
      return(false);
     }

   BYTE msg_array[];

   StringToCharArray(msg,msg_array,0,WHOLE_ARRAY);

   ArrayRemove(msg_array,ArraySize(msg_array)-1,1);

   DWORD len=(ArraySize(msg_array));

   return(clientsend(msg_array,WINHTTP_WEB_SOCKET_BINARY_MESSAGE_BUFFER_TYPE));
  }

//+------------------------------------------------------------------+
//|Public method for sending data prepackaged in an array            |
//+------------------------------------------------------------------+
bool CWebsocket::Send(BYTE &buffer[])
  {
   if(!initialized || hWebSocket == NULL)
     {
      Print(__FUNCTION__, " No websocket connection ");
      return(false);
     }

   return(clientsend(buffer,WINHTTP_WEB_SOCKET_BINARY_MESSAGE_BUFFER_TYPE));
  }

//+------------------------------------------------------------------+
//|helper method for reading received messages from the server       |
//+------------------------------------------------------------------+
void CWebsocket::clientread(BYTE &rbuffer[],WINHTTP_WEB_SOCKET_BUFFER_TYPE &buffertype,ulong &bytes)
  {

   buffertype=-1;
   bytes=0;
   if(rbuffer.Size()==0)
     {
      Print(__FUNCTION__, " Receiving buffer of insufficient size ");
      return;
     }
   ulong get=client_read(hWebSocket,rbuffer,ulong(rbuffer.Size()),buffertype);

   if(get)
     {
      Print(__FUNCTION__, " failed to read operation");
      return;
     }

   bytes = (ulong)rxbuffer.Size();

   return;

  }

//+------------------------------------------------------------------+
//|public method for reading data sent from the server               |
//+------------------------------------------------------------------+
ulong CWebsocket::Read(BYTE &buffer[],WINHTTP_WEB_SOCKET_BUFFER_TYPE &buffertype)
  {
   if(!initialized || hWebSocket == NULL)
     {
      Print(__FUNCTION__, " No websocket connection ");
      return(false);
     }

   ulong bytes_read_from_socket=0;

   clientread(buffer,buffertype,bytes_read_from_socket);

   return(bytes_read_from_socket);

  }
//+------------------------------------------------------------------+
//|public method for reading data sent from the server               |
//+------------------------------------------------------------------+
ulong CWebsocket::ReadString(string &_response)
  {
   if(!initialized || hWebSocket == NULL)
     {
      Print(__FUNCTION__, " No websocket connection ");
      return(false);
     }

   ulong bytes_read_from_socket=0;

   ZeroMemory(rxbuffer);

   WINHTTP_WEB_SOCKET_BUFFER_TYPE rbuffertype;

   clientread(rxbuffer,rbuffertype,bytes_read_from_socket);

   _response=(bytes_read_from_socket)?CharArrayToString(rxbuffer):"";

   return(bytes_read_from_socket);
  }

//+------------------------------------------------------------------+
//| Closes a websocket client connection                             |
//+------------------------------------------------------------------+
void CWebsocket::Close(void)
  {
   if(!initialized || hWebSocket == NULL)
      return;
   else
      client_disconnect(hWebSocket);
  }
//+--------------------------------------------------------------------------+
//|method for abandoning a client connection. All previous server connection |
//|   parameters are reset to their default state                            |
//+--------------------------------------------------------------------------+
void CWebsocket::Abort(void)
  {
   client_reset(hWebSocket);
   reset();
  }

//+------------------------------------------------------------------+
//|  get string error description                                    |
//+------------------------------------------------------------------+
string CWebsocket::GetErrorDescription(const DWORD error_code)
  {
   string errormsg=NULL;

   switch(int(error_code))
     {
      case  ERROR_WINHTTP_TIMEOUT                  :
         errormsg="The request has timed out";
         break;
      case  ERROR_WINHTTP_INTERNAL_ERROR           :
         errormsg="An internal error has occurred";
         break;
      case  ERROR_WINHTTP_INVALID_URL              :
         errormsg="The URL is not valid";
         break;
      case  ERROR_WINHTTP_UNRECOGNIZED_SCHEME      :
         errormsg="The URL specified an unfamiliar scheme";
         break;
      case  ERROR_WINHTTP_NAME_NOT_RESOLVED        :
         errormsg="The server name cannot be resolved";
         break;
      case  ERROR_WINHTTP_INVALID_OPTION           :
         errormsg="A request to WinHttpSetOption specified an invalid option value";
         break;
      case  ERROR_WINHTTP_OPTION_NOT_SETTABLE      :
         errormsg="The requested option cannot be set, only queried";
         break;
      case  ERROR_WINHTTP_SHUTDOWN                 :
         errormsg="The WinHTTP function support is being shut down or unloaded";
         break;
      case  ERROR_WINHTTP_LOGIN_FAILURE            :
         errormsg="The login attempt failed";
         break;
      case  ERROR_WINHTTP_OPERATION_CANCELLED      :
         errormsg="The operation was canceled";
         break;
      case  ERROR_WINHTTP_INCORRECT_HANDLE_TYPE    :
         errormsg="The type of handle supplied is incorrect for this operation";
         break;
      case  ERROR_WINHTTP_INCORRECT_HANDLE_STATE   :
         errormsg="The handle supplied is not in the correct state";
         break;
      case  ERROR_WINHTTP_CANNOT_CONNECT           :
         errormsg="Connection to the server failed.";
         break;
      case  ERROR_WINHTTP_CONNECTION_ERROR         :
         errormsg="The connection with the server has been reset or terminated, or an incompatible SSL protocol was encountered";
         break;
      case  ERROR_WINHTTP_RESEND_REQUEST           :
         errormsg="The WinHTTP function failed. The desired function can be retried on the same request handle";
         break;
      case  ERROR_WINHTTP_CLIENT_AUTH_CERT_NEEDED  :
         errormsg="the server requests client authentication";
         break;
      case  ERROR_WINHTTP_CANNOT_CALL_BEFORE_OPEN  :
         errormsg="requested operation cannot be performed before calling the Open method";
         break;
      case  ERROR_WINHTTP_CANNOT_CALL_BEFORE_SEND  :
         errormsg="requested operation cannot be performed before calling the Send method";
         break;
      case  ERROR_WINHTTP_CANNOT_CALL_AFTER_SEND   :
         errormsg="requested operation cannot be performed after calling the Send method";
         break;
      case  ERROR_WINHTTP_CANNOT_CALL_AFTER_OPEN   :
         errormsg="option cannot be requested after the Open method has been called";
         break;
      case  ERROR_WINHTTP_HEADER_NOT_FOUND             :
         errormsg="The requested header cannot be located";
         break;
      case  ERROR_WINHTTP_INVALID_SERVER_RESPONSE      :
         errormsg="The server response cannot be parsed";
         break;
      case  ERROR_WINHTTP_REDIRECT_FAILED              :
         errormsg="The redirection failed ";
         break;
      case  ERROR_WINHTTP_AUTO_PROXY_SERVICE_ERROR  :
         errormsg="proxy for the specified URL cannot be located";
         break;
      case  ERROR_WINHTTP_SECURE_FAILURE           :
         errormsg="errors were found in the Secure Sockets Layer (SSL) certificate sent by the server";
         break;
      case  ERROR_WINHTTP_SECURE_CERT_DATE_INVALID    :
         errormsg="A required certificate is not within its validity period ";
         break;
      case  ERROR_WINHTTP_SECURE_CERT_CN_INVALID      :
         errormsg="Certificate CN name does not match the passed value ";
         break;
      case  ERROR_WINHTTP_SECURE_INVALID_CA           :
         errormsg="root certificate is not trusted by the trust provider ";
         break;
      case  ERROR_WINHTTP_SECURE_CERT_REV_FAILED      :
         errormsg="revocation cannot be checked because the revocation server was offline ";
         break;
      case  ERROR_WINHTTP_SECURE_CHANNEL_ERROR        :
         errormsg="error occurred having to do with a secure channel ";
         break;
      case  ERROR_WINHTTP_SECURE_INVALID_CERT         :
         errormsg="certificate is invalid";
         break;
      case  ERROR_WINHTTP_SECURE_CERT_REVOKED         :
         errormsg="certificate has been revoked ";
         break;
      case  ERROR_WINHTTP_SECURE_CERT_WRONG_USAGE     :
         errormsg="certificate is not valid for the requested usage ";
         break;
      case  ERROR_WINHTTP_HEADER_COUNT_EXCEEDED                 :
         errormsg="larger number of headers present in response than WinHTTP could receive";
         break;
      case  ERROR_WINHTTP_HEADER_SIZE_OVERFLOW                  :
         errormsg=" size of headers received exceeds the limit for the request handle";
         break;
      case  ERROR_WINHTTP_CHUNKED_ENCODING_HEADER_SIZE_OVERFLOW :
         errormsg="overflow condition  encountered in the course of parsing chunked encoding";
         break;
      case  ERROR_WINHTTP_RESPONSE_DRAIN_OVERFLOW               :
         errormsg="response exceeds an internal WinHTTP size limit";
         break;
      case  ERROR_WINHTTP_CLIENT_CERT_NO_PRIVATE_KEY            :
         errormsg="The context for the SSL client certificate does not have a private key associated with it";
         break;
      case  ERROR_WINHTTP_CLIENT_CERT_NO_ACCESS_PRIVATE_KEY     :
         errormsg="The application does not have the required privileges to access the private key associated with the client certificate";
         break;
      case WEBSOCKET_ERROR_FAILED_OPERATION                     :
         errormsg="Invalid operation";
         break;
      case WEBSOCKET_ERROR_INVALID_PARAMETER                    :
         errormsg="Invalid parameter passed to function";
         break;
      case WEBSOCKET_ERROR_INVALID_HANDLE                       :
         errormsg="Websocket client has not been initialized";
         break;
      case WEBSOCKET_ERROR_EMPTY_SEND_BUFFER                    :
         errormsg="Send buffer is empty";
         break;
      case WEBSOCKET_ERROR_NOT_CONNECTED                        :
         errormsg="Websocket client is not connected to a server";
         break;
      default:
         errormsg="Unknown error "+IntegerToString(error_code);
         break;
     }

   return errormsg;
  }
//+------------------------------------------------------------------+
//|Get the last error emanating from operations in the dll           |
//+------------------------------------------------------------------+
string CWebsocket::LastErrorMessage(void)
  {

   DWORD error = client_lasterror(hWebSocket);

   return GetErrorDescription(error);
  }
//+------------------------------------------------------------------+
//|  get the last error                                              |
//+------------------------------------------------------------------+
DWORD CWebsocket::LastError(void)
  {
   if(!initialized || hWebSocket == NULL)
     {
      Print(__FUNCTION__, " No websocket connection ");
      return 0;
     }

   return client_lasterror(hWebSocket);
  }

//+------------------------------------------------------------------+
//| websocket state                                                  |
//+------------------------------------------------------------------+
ENUM_WEBSOCKET_STATE CWebsocket::ClientState(void)
  {
   if(hWebSocket!=NULL)
      return client_status(hWebSocket);
   else
      return CLOSED;
  }
//+------------------------------------------------------------------+
//|   websocket handle                                               |
//+------------------------------------------------------------------+
HINTERNET CWebsocket::WebSocketHandle(void)
  {
   if(hWebSocket!=NULL)
      return client_websocket_handle(hWebSocket);
   else
      return NULL;
  }
//+------------------------------------------------------------------+
//| call back notification                                           |
//+------------------------------------------------------------------+
ulong CWebsocket::CallBackResult(void)
  {
   if(hWebSocket!=NULL)
      return client_lastcallback_notification(hWebSocket);
   else
      return WINHTTP_CALLBACK_STATUS_DEFAULT;
  }
//+------------------------------------------------------------------+
//|  check if any data has read from the server                      |
//+------------------------------------------------------------------+
ulong CWebsocket::ReadAvailable(void)
  {
   if(hWebSocket!=NULL)
      return(client_readable(hWebSocket));
   else
      return 0;
  }
//+------------------------------------------------------------------+
//|  asynchronous read operation (polls for server response)         |
//+------------------------------------------------------------------+
ulong CWebsocket::Poll(void)
  {
   if(hWebSocket!=NULL)
      return client_poll(hWebSocket);
   else
      return WEBSOCKET_ERROR_INVALID_HANDLE;
  }

//+------------------------------------------------------------------+
