//+------------------------------------------------------------------+
//| Trade signal class that will be pooled                           |
//+------------------------------------------------------------------+
class CTradeSignal
{
public:
   datetime time;
   double price;
   ENUM_ORDER_TYPE type;
   double volume;
   bool isValid;
   
   // Reset the object for reuse
   void Reset()
   {
      time = 0;
      price = 0.0;
      type = ORDER_TYPE_BUY;
      volume = 0.0;
      isValid = false;
   }
};

//+------------------------------------------------------------------+
//| Object pool for CTradeSignal instances                           |
//+------------------------------------------------------------------+
class CTradeSignalPool
{
private:
   CTradeSignal* m_pool[];
   int m_poolSize;
   int m_nextAvailable;
   
public:
   // Constructor
   CTradeSignalPool(int initialSize = 100)
   {
      m_poolSize = initialSize;
      ArrayResize(m_pool, m_poolSize);
      m_nextAvailable = 0;
      
      // Pre-allocate objects
      for(int i = 0; i < m_poolSize; i++)
      {
         m_pool[i] = new CTradeSignal();
      }
      
      Print("Trade signal pool initialized with ", m_poolSize, " objects");
   }
   
   // Get an object from the pool
   CTradeSignal* Acquire()
   {
      // If we've used all objects, expand the pool
      if(m_nextAvailable >= m_poolSize)
      {
         int oldSize = m_poolSize;
         m_poolSize *= 2;  // Double the pool size
         ArrayResize(m_pool, m_poolSize);
         
         // Allocate new objects
         for(int i = oldSize; i < m_poolSize; i++)
         {
            m_pool[i] = new CTradeSignal();
         }
         
         Print("Trade signal pool expanded to ", m_poolSize, " objects");
      }
      
      // Get the next available object
      CTradeSignal* signal = m_pool[m_nextAvailable++];
      signal.Reset();  // Ensure it's in a clean state
      return signal;
   }
   
   // Return an object to the pool
   void Release(CTradeSignal* &signal)
   {
      if(signal == NULL)
         return;
         
      // In a more sophisticated implementation, we would
      // actually track which objects are in use and reuse them.
      // For simplicity, we're just decrementing the counter.
      if(m_nextAvailable > 0)
         m_nextAvailable--;
         
      signal = NULL;  // Clear the reference
   }
   
   // Destructor
   ~CTradeSignalPool()
   {
      // Clean up all allocated objects
      for(int i = 0; i < m_poolSize; i++)
      {
         delete m_pool[i];
      }
      
      Print("Trade signal pool destroyed");
   }
};


// Global pool instance
CTradeSignalPool* g_signalPool = NULL;

void OnInit()
{
   // Initialize the pool
   g_signalPool = new CTradeSignalPool(100);
}

void OnTick()
{
   // Acquire a signal from the pool
   CTradeSignal* signal = g_signalPool.Acquire();
   
   // Set signal properties
   signal.time = TimeCurrent();
   signal.price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   signal.type = ORDER_TYPE_BUY;
   signal.volume = 0.1;
   signal.isValid = true;
   
   // Process the signal...
   
   // Return the signal to the pool when done
   g_signalPool.Release(signal);
}

void OnDeinit(const int reason)
{
   // Clean up the pool
   delete g_signalPool;
   g_signalPool = NULL;
}