//+------------------------------------------------------------------+
//| Circular buffer for price data                                   |
//+------------------------------------------------------------------+
class CPriceBuffer
{
private:
   double m_prices[];
   int m_capacity;
   int m_head;
   int m_size;
   
public:
   // Constructor
   CPriceBuffer(int capacity = 1000)
   {
      m_capacity = capacity;
      ArrayResize(m_prices, m_capacity);
      m_head = 0;
      m_size = 0;
   }
   
   // Add a price to the buffer
   void Add(double price)
   {
      m_prices[m_head] = price;
      m_head = (m_head + 1) % m_capacity;
      
      if(m_size < m_capacity)
         m_size++;
   }
   
   // Get a price at a specific index (0 is the most recent)
   double Get(int index)
   {
      if(index < 0 || index >= m_size)
         return 0.0;
         
      int actualIndex = (m_head - 1 - index + m_capacity) % m_capacity;
      return m_prices[actualIndex];
   }
   
   // Get the current size
   int Size()
   {
      return m_size;
   }
   
   // Get the capacity
   int Capacity()
   {
      return m_capacity;
   }
   
   // Clear the buffer
   void Clear()
   {
      m_head = 0;
      m_size = 0;
   }
   
   // Calculate simple moving average
   double SMA(int period)
   {
      if(period <= 0 || period > m_size)
         return 0.0;
         
      double sum = 0.0;
      for(int i = 0; i < period; i++)
      {
         sum += Get(i);
      }
      
      return sum / period;
   }
};