//+------------------------------------------------------------------+
//|                                                BaseFunctions.mqh |
//|                                    Copyright 2025, Andy Thompson |
//|                                        mailto:andydoc1@gmail.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, Andy Thompson"
#property link      "mailto:andydoc1@gmail.com"

//+------------------------------------------------------------------+
// Examples of use of code in this file;                             |
//    All examples use the inputs and global variables in the code   |
//                                                                   |
//    1) ScoreRF = RF:                                               |
//       double ScoreRF = ReLU(DeviationRF);                         |
//       * there is an issue in that the target value, eg 5, will    |
//       actually return 0 and only values >5 will return non-zero   |
//                                                                   |
//    2) ScoreRF converges on 0 as RF approaches -infinity, remains  |
//    near zero until ~MinRF-3, then converges on RF:                |
//       double ScoreRF = Softplus(DeviationRF);                     |
//                                                                   |
//    3) ScoreRF is returned from a Sigmoid curve between 0 and 1    |
//    returning <.01 with RF below a value of MinRF; and             |
//    > 0.99 above a value of 14.22:                                 |
//       double ScoreRF = CorrectedSigmoid(DeviationRF);             |
//                                                                   |
//    4) ScoreTrades returns 1 when Trades == TargetTrades with      |
//    returns tending to 0 on either side:                           |
//       double ScoreTrades = Deriv4Sigmoid(DeviationTrades);        |
//                                                                   |
//    5) ScoreRF is returned from a similar curve although           |
//    not identical to the Sigmoid curve between 0 and 1             |
//    returning <.01 with RF below a value of MinRF; and             |
//    > 0.99 above a value of 9.6:                                   |
//       double ScoreRF = CorrectedRSTanh(DeviationRF);              |
//                                                                   |
//    6) ScoreTrades returns 1 when Trades == TargetTrades with      |
//    returns tending to 0 on either side on a similar curve         |
//    although not identical to the Sigmoid curve between 0 and 1:   |
//       double ScoreTrades = DerivTanh(DeviationTrades);            |
//                                                                   |
//    For fuller explanation see /en/articles/17429/180903/          |
//+------------------------------------------------------------------+
input double MinRF = 5; // Minimum Recovery Factor
input double TargetTrades = 100; // Target number of trades
sinput double SigCorrection = 4.6; // 4.6 Should give results above 0.01 above target
sinput double TanhCorrection = 2.5; // Anything between 2 and 3 would be reasonable

//+------------------------------------------------------------------+
//|                                                 Global Variables |
//+------------------------------------------------------------------+
double RF = TesterStatistics(STAT_RECOVERY_FACTOR);
double Trades = TesterStatistics(STAT_TRADES);
double DeviationRF = RF - MinRF;
double DeviationTrades = Trades - TargetTrades;
//+------------------------------------------------------------------+
//|                 ReLU Activation Function (Rectified Linear Unit) |
//+------------------------------------------------------------------+
double ReLU(double x) {
   return(MathMax(0, x));
}
//+------------------------------------------------------------------+
//|                                     Softplus Activation Function |
//+------------------------------------------------------------------+
double Softplus(double x) {
   return(MathLog(1 + MathExp(x)));
}
//+------------------------------------------------------------------+
//|                          Sigmoid Activation Function (Corrected) |
//+------------------------------------------------------------------+
double CorrectedSigmoid(double x) {
   return(1 / (1 + MathExp(-(x - SigCorrection))));
}
//+------------------------------------------------------------------+
//|                          UNCORRECTED Sigmoid Activation Function |
//+------------------------------------------------------------------+
double Sigmoid(double x) {
   return(1 / (1 + MathExp(-x)));
}
//+------------------------------------------------------------------+
//|                        Derivative (rescaled) of Sigmoid Function |
//+------------------------------------------------------------------+
double Deriv4Sigmoid(double x) {
   return(4 * (Sigmoid(x) * (1 - Sigmoid(x))));
}
//+------------------------------------------------------------------+
//|                   Tanh Activation Function (Rescaled &Corrected) |
//+------------------------------------------------------------------+
double CorrectedRSTanh(double x) {
   return((MathTanh(x  + TanhCorrection) + 1) / 2);  // rescaled and corrected
}
//+------------------------------------------------------------------+
//|                                      Derivative of Tanh Function |
//+------------------------------------------------------------------+
double DerivTanh(double x) {
   return(1 - MathPow(x, 2));
}

//+------------------------------------------------------------------+
