import numpy as np
import pandas as pd
import MetaTrader5 as mt5
from datetime import datetime
from sklearn.preprocessing import MinMaxScaler
import plotly.graph_objects as go
from plotly.subplots import make_subplots
import warnings
warnings.filterwarnings('ignore')

class TrendForceIndicator:
    def __init__(self, symbol: str, timeframe: int = mt5.TIMEFRAME_H1, lookback: int = 10000):
        """
        Indicator initialization
        
        Args:
            symbol (str): Trading stmbol (for example, "EURUSD")
            timeframe (int): Timeframe (default - H1)
            lookback (int): Number of bars for analysis
        """
        self.symbol = symbol
        self.timeframe = timeframe
        self.lookback = lookback
        self.scaler = MinMaxScaler(feature_range=(3, 9))

    def analyze_market(self) -> pd.DataFrame:
        """Main market analysis method"""
        # Get data from MT5
        rates = mt5.copy_rates_from_pos(self.symbol, self.timeframe, 0, self.lookback)
        if rates is None:
            raise ValueError(f"Failed to get data for {self.symbol}")
            
        df = pd.DataFrame(rates)
        df['time'] = pd.to_datetime(df['time'], unit='s')
        df.set_index('time', inplace=True)
        
        # Calculate components
        df = self._calculate_components(df)
        return df.tail(500)  # Return the last 500 bars

    def _calculate_components(self, df: pd.DataFrame) -> pd.DataFrame:
        """Calculate all indicator components"""
        # Basic components
        df['volatility'] = df['close'].pct_change().rolling(20).std()
        df['momentum'] = df['close'].pct_change(5)
        
        # Volumetric profile
        df['volume_ma'] = df['tick_volume'].rolling(20).mean()
        df['volume_trend'] = df['tick_volume'] / df['volume_ma']
        
        # Calculate trend strength
        df['trend_force'] = df['volatility'] * df['volume_trend'] * abs(df['momentum'])
        df['trend_force_norm'] = self.scaler.fit_transform(
            df['trend_force'].values.reshape(-1, 1)
        ).flatten()
        
        # Define direction
        df['trend_direction'] = np.sign(df['momentum'])
        
        # Calculate trading sessions
        df['session_coef'] = 1.0
        hour = df.index.hour
        
        # Ratios for different sessions
        asian_mask = (hour >= 0) & (hour < 8)
        european_mask = (hour >= 8) & (hour < 16)
        american_mask = (hour >= 16) & (hour < 24)
        
        df.loc[asian_mask, 'session_coef'] = 0.7    # Asian session
        df.loc[european_mask, 'session_coef'] = 1.0  # European session
        df.loc[american_mask, 'session_coef'] = 0.9  # American session
        
        # Total trend strength parameter
        df['trend_force_adjusted'] = df['trend_force_norm'] * df['session_coef']
        
        # Components for defining reversals
        df['direction_strength'] = df['trend_direction'] * df['trend_force_adjusted']
        
        # Reversal patterns
        df['reversal_pattern'] = np.where(
            (df['trend_force_adjusted'] > 6) & 
            (df['direction_strength'].diff().rolling(3).std() > 1.5),
            1, 0
        )
        
        # Confirm by volume
        df['volume_confirmation'] = np.where(
            (df['reversal_pattern'] == 1) &
            (df['volume_trend'] > df['volume_trend'].rolling(20).mean() * 1.5),
            'Strong',
            'Weak'
        )
        
        return df

    def create_visualization(self, df: pd.DataFrame = None):
        """Create interactive visualization"""
        if df is None:
            df = self.analyze_market()
        
        df = df.reset_index()
        
        # Create a chart with three subwindows
        fig = make_subplots(rows=3, cols=1, 
                           shared_xaxes=True,
                           subplot_titles=('Price', 'Trend Force', 'Trend Direction'),
                           row_heights=[0.5, 0.25, 0.25],
                           vertical_spacing=0.05)

        # Price chart
        fig.add_trace(
            go.Candlestick(
                x=df['time'],
                open=df['open'],
                high=df['high'],
                low=df['low'],
                close=df['close'],
                name='OHLC',
                hoverlabel=dict(
                    bgcolor='white',
                    font=dict(size=12)
                )
            ),
            row=1, col=1
        )

        # Trend strength chart
        fig.add_trace(
            go.Scatter(
                x=df['time'],
                y=df['trend_force_adjusted'],
                mode='lines',
                line=dict(color='blue', width=2),
                name='Trend Force',
                hovertemplate="<br>".join([
                    "Time: %{x}",
                    "Force: %{y:.2f}",
                    "<extra></extra>"
                ])
            ),
            row=2, col=1
        )

        # Reference levels
        fig.add_hline(y=3, line_dash="dash", line_color="yellow", row=2, col=1)
        fig.add_hline(y=6, line_dash="dash", line_color="green", row=2, col=1)

        # Trend direction graph
        fig.add_trace(
            go.Bar(
                x=df['time'],
                y=df['trend_direction'] * df['trend_force_adjusted'],
                name='Trend Direction',
                marker_color=np.where(df['trend_direction'] > 0, 'green', 'red'),
                hovertemplate="<br>".join([
                    "Time: %{x}",
                    "Direction: %{y:.2f}",
                    "Reversal: %{text}",
                    "<extra></extra>"
                ]),
                text=df['volume_confirmation']
            ),
            row=3, col=1
        )

        # Set up appearance
        fig.update_layout(
            height=1000,
            title_text=f"Trend Force Indicator - {self.symbol}",
            showlegend=True,
            xaxis3_title="Time",
            yaxis_title="Price",
            yaxis2_title="Force",
            yaxis3_title="Direction"
        )

        return fig

def main():
    # MT5 initialization
    if not mt5.initialize():
        print("Failed to initialize MetaTrader 5")
        return
        
    try:
        # Create and test indicator
        indicator = TrendForceIndicator('AUDUSD')
        df = indicator.analyze_market()
        fig = indicator.create_visualization(df)
        fig.show()
        
    finally:
        mt5.shutdown()

if __name__ == "__main__":
    main()
