//+------------------------------------------------------------------+
//|                                                      EasyXml.mqh |
//|                                  Copyright 2013, Paul van Hemmen |
//|                                          http://www.vanhemmen.de |
//+------------------------------------------------------------------+
#property copyright "Copyright 2013, Paul van Hemmen"
#property link      "http://www.vanhemmen.de"
#property version   "1.00"

//+------------------------------------------------------------------+
//| Imported functions description                                   |
//+------------------------------------------------------------------+
#import "wininet.dll"
int InternetAttemptConnect(int x);
int InternetOpenW(string sAgent,int lAccessType,string sProxyName="",string sProxyBypass="",int lFlags=0);
int InternetOpenUrlW(int hInternetSession,string sUrl,string sHeaders="",int lHeadersLength=0,int lFlags=0,int lContext=0);
int InternetReadFile(int hFile,uchar &sBuffer[],int lNumBytesToRead,int &lNumberOfBytesRead[]);
int HttpQueryInfoW(int hRequest,int dwInfoLevel,uchar &lpvBuffer[],int &lpdwBufferLength,int &lpdwIndex);
int InternetCloseHandle(int hInet);
#import

//+------------------------------------------------------------------+
//| macros                                                           |
//+------------------------------------------------------------------+
#define EASYXML_START_OPEN          "<"
#define EASYXML_START_CLOSE         ">"
#define EASYXML_SELFCLOSE           "/>"
#define EASYXML_CLOSE_OPEN          "</"
#define EASYXML_CLOSE_CLOSE         ">"
#define EASYXML_WHITESPACE          " \r\n\t"
#define EASYXML_LATIN               "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
#define EASYXML_CDATA_OPEN          "<![CDATA["
#define EASYXML_CDATA_CLOSE         "]]>"
#define EASYXML_COMMENT_OPEN        "<!--"
#define EASYXML_COMMENT_CLOSE       "-->"
#define EASYXML_PROLOG_OPEN         "<?xml"
#define EASYXML_PROLOG_CLOSE        "?>"
#define EASYXML_STYLESHEET_OPEN     "<?xml-stylesheet"
#define EASYXML_STYLESHEET_CLOSE    "?>"
#define EASYXML_DOCTYPE_OPEN        "<!DOCTYPE"
#define EASYXML_DOCTYPE_CLOSE        ">"
#define EASYXML_ATTRIBUTE_SEPARATOR "="
#define EASYXML_ATTRIBUTE_COLON     "\"'"
#define EASYXML_XMLFILE_ENDING      ".xml"
#define HTTP_QUERY_CONTENT_LENGTH   5

//+------------------------------------------------------------------+
//| includes                                                         |
//+------------------------------------------------------------------+
#include "easyxmlnode.mqh"
//#include "easyxmlattribute.mqh"
#include "easyxmlerrordescription.mqh"
//+------------------------------------------------------------------+
//| class declaration                                                |
//+------------------------------------------------------------------+
class CEasyXml
  {
private:
   // Properties
   CEasyXmlNode     *DocumentRoot;
   bool              blDebug;
   bool              blSaveToCache;
   string            sText;
   string            sFilename;
   int               Err;
   string            ErrMsg;

   // Methods :: parsing
   bool              parseRecursive(CEasyXmlNode *pActualNode,string &pText,int &pPos,int pLevel=0);
   bool              parseAttributes(CEasyXmlNode *pActualNode,string pAttributes,string pDebugSpace);
   void              skipWhitespace(string &pText,int &pPos);
   bool              skipWhitespaceAndComments(string &pText,int &pPos,string pDebugSpace);
   bool              hasSiblings(string &pText,int &pPos);
   bool              endOfXml(string &pText,int &pPos);
   bool              skipProlog(string &pText,int &pPos);

   // Methods :: helpers
   bool              writeStreamToCacheFile(string pStream);
   bool              Error(int pPos=-1,bool pClear=true);

public:
   // Methods :: parse by source
   bool              loadXmlFromFile(string pFilename);
   bool              loadXmlFromUrl(string pUrl);
   bool              loadXmlFromUrlWebReq(string pUrl);
   bool              loadXmlFromString(string pText);

   // Methods :: setters
   void              setDebugging(bool pDebug);
   bool              setUrlCacheFile(string pFilename);

   // Methods :: getters
   CEasyXmlNode     *getDocumentRoot(void);
   string            getText(void);
   string            GetErrorMsg(void){   return(ErrMsg);}
   void              Clear(void);

                     CEasyXml();
                    ~CEasyXml();
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CEasyXml::CEasyXml()
  {
   blDebug       = false;
   blSaveToCache = false;
   DocumentRoot  = new CEasyXmlNode;
  }
//+------------------------------------------------------------------+
//| Desctructor                                                      |
//+------------------------------------------------------------------+
CEasyXml::~CEasyXml()
  {
   if(CheckPointer(DocumentRoot)==POINTER_DYNAMIC) delete DocumentRoot;
  }
//+------------------------------------------------------------------+
//| enable debug output                                              |
//+------------------------------------------------------------------+
void CEasyXml::setDebugging(bool pDebug)
  {
   blDebug=pDebug;
  }
//+------------------------------------------------------------------+
//| enable url caching                                               |
//+------------------------------------------------------------------+
bool CEasyXml::setUrlCacheFile(string pFilename)
  {
   if(StringSubstr(pFilename,StringLen(pFilename)-StringLen(EASYXML_XMLFILE_ENDING),StringLen(EASYXML_XMLFILE_ENDING))==EASYXML_XMLFILE_ENDING)
     {
      sFilename     = pFilename;
      blSaveToCache = true;
        } else {
      Err=EASYXML_INVALID_FILENAME;
      return(Error(-1,false));
     }

   return(true);
  }
//+------------------------------------------------------------------+
//| write stream to cache file                                       |
//+------------------------------------------------------------------+
bool CEasyXml::writeStreamToCacheFile(string pStream)
  {
   int hFile=FileOpen(sFilename,FILE_BIN|FILE_WRITE);

   if(hFile==INVALID_HANDLE)
     {
      Err=EASYXML_ERR_CONNECTION_FILEOPEN;
      return(false);
     }

   FileWriteString(hFile,pStream);
   FileClose(hFile);

   return(true);
  }
//+------------------------------------------------------------------+
//| load xml by given url                                            |
//+------------------------------------------------------------------+
bool CEasyXml::loadXmlFromUrl(string pUrl)
  {
   int rv= InternetAttemptConnect(0);
   if(rv!=0)
     {
      Err=EASYXML_ERR_CONNECTION_ATTEMPT;
      return(Error());
     }

   int hInternetSession= InternetOpenW("Microsoft Internet Explorer", 0, "","", 0);
   if(hInternetSession<=0)
     {
      Err=EASYXML_ERR_CONNECTION_OPEN;
      return(Error());
     }

   int hURL= InternetOpenUrlW(hInternetSession, pUrl, "", 0, 0, 0);
   if(hURL<=0)
     {
      InternetCloseHandle(hInternetSession);
      Err=EASYXML_ERR_CONNECTION_URL;
      return(Error());
     }

//get url file size
   int iBufferLength  = 2048;
   int iIndex         = 0;
   int iContentLength = 0;

   int iRes;
   string sCount;
   uchar ucBuffer[2048];

   iRes=HttpQueryInfoW(hURL,HTTP_QUERY_CONTENT_LENGTH,ucBuffer,iBufferLength,iIndex);
   if(iRes==1)
     {
      for(int k=0; k<iBufferLength; k++)
        {
         sCount=sCount+CharToString(ucBuffer[k]);
        }
      if(StringLen(sCount)>0) iContentLength=(int)StringToInteger(sCount);
     }

   if(blDebug && iContentLength==0) Print("*** Content length of ",pUrl," is unknown ***");

// read url
   string sStream   = "";
   int iBytesCount  = 0;

   int dwBytesRead[1];
   uchar buffer[1024];

   while(!IsStopped())
     {
      bool bResult = InternetReadFile(hURL, buffer, 1024, dwBytesRead);
      iBytesCount += dwBytesRead[0];

      sStream+=CharArrayToString(buffer,0,dwBytesRead[0],CP_UTF8);

      if(dwBytesRead[0]==0) break;
     }

   if(iBytesCount==0)
     {
      InternetCloseHandle(hInternetSession);
      Err=EASYXML_ERR_CONNECTION_EMPTYSTREAM;
      return(Error());
     }

   if(blSaveToCache)
     {
      bool bResult=writeStreamToCacheFile(sStream);
      // In this case: just warn, don't return
      if(!bResult) Error(-1,false);
     }

   InternetCloseHandle(hInternetSession);

   return(loadXmlFromString(sStream));
  }
//+------------------------------------------------------------------+
//| load xml by given url using MQL5 webrequest function             |
//+------------------------------------------------------------------+
bool CEasyXml::loadXmlFromUrlWebReq(string pUrl)
  {
//---
   string cookie=NULL,headers;
   char post[],result[];
   int res;
//---
   string _url=pUrl;
   string sStream;
   res=WebRequest("GET",_url,cookie,NULL,5000,post,0,result,headers);
//--- check error
   if(res==-1)
     {
      Err=EASYXML_ERR_WEBREQUEST_URL;
      return(Error());
     }
//---success downloading file
   sStream=CharArrayToString(result,0,-1,CP_UTF8);
//---set up cach file
   if(blSaveToCache)
     {
      bool bResult=writeStreamToCacheFile(sStream);
      if(!bResult) Error(-1,false);
     }
//---
   return(loadXmlFromString(sStream));
  }
//+------------------------------------------------------------------+
//| Load XML by given file                                           |
//+------------------------------------------------------------------+
bool CEasyXml::loadXmlFromFile(string pFilename)
  {
   string sStream;
   int    iStringSize;

   int hFile=FileOpen(pFilename,FILE_TXT|FILE_READ|FILE_UNICODE);
   if(hFile==INVALID_HANDLE)
     {
      Err=EASYXML_ERR_CONNECTION_FILEOPEN;
      return(Error());
     }

   while(!FileIsEnding(hFile))
     {
      iStringSize = FileReadInteger(hFile, INT_VALUE);
      sStream    += FileReadString(hFile, iStringSize);
     }

   FileClose(hFile);

   return(loadXmlFromString(sStream));
  }
//+------------------------------------------------------------------+
//| Load XML by given string                                         |
//+------------------------------------------------------------------+
bool CEasyXml::loadXmlFromString(string pText)
  {
   bool blSuccess = false;
   int  iPos      = 0;
   sText          = pText;

   StringTrimLeft(pText);
   StringTrimRight(pText);

// Skip xml prolog
   blSuccess=skipProlog(pText,iPos);
   if(!blSuccess) return(Error(iPos));

// Parse
   blSuccess=parseRecursive(DocumentRoot,pText,iPos);
   if(!blSuccess) return(Error(iPos));

   return(true);
  }
//+------------------------------------------------------------------+
//| XML recursive parser logic                                       |
//+------------------------------------------------------------------+
bool CEasyXml::parseRecursive(CEasyXmlNode *pActualNode,string &pText,int &pPos,int pLevel=0)
  {
   bool blSuccess;
   bool blSibling=false;

//---
// At least parse one child element.
// Continue for as long as there are any siblings.
//---

   do
     {

      // Debugging output vars
      string sDebugSpace;
      string sDebugOutput;

      // Indent debug output for better readability
      StringInit(sDebugSpace,pLevel*4,StringGetCharacter(" ",0));

      string sTagName;
      string sCloseTagName;
      string sTagContent;
      string sAttributes;

      // Create sibling node
      if(blSibling)
        {
         pActualNode=pActualNode.createSibling(new CEasyXmlNode);
        }

      // Skip comments
      if(!skipWhitespaceAndComments(pText,pPos,sDebugSpace)) return(false);

      //---
      // Get start tag. If it contains attributes, parse them seperately.
      // If tag is self-closing, continue with siblings loop
      //---

      // Open tag
      if(StringFind(EASYXML_START_OPEN,StringSubstr(pText,pPos,1))==0)
        {
         pPos++;
           } else {
         Err=EASYXML_INVALID_OPENTAG_START;
         return(false);
        }

      // Tag name
      while(StringFind(EASYXML_WHITESPACE,StringSubstr(pText,pPos,1))==-1 && 
            StringCompare(EASYXML_START_CLOSE,StringSubstr(pText,pPos,1))!=0 && 
            !endOfXml(pText,pPos))
        {
         sTagName+=StringSubstr(pText,pPos,1);
         pPos++;
        }

      pActualNode.setName(sTagName);

      // Debugging
      if(blDebug) Print(sDebugSpace,"<"+sTagName+"> D:",IntegerToString(pLevel)," | P:",pPos);

      skipWhitespace(pText,pPos);

      // Attributes
      if(StringFind(EASYXML_LATIN,StringSubstr(pText,pPos,1))!=-1)
        {
         while(StringCompare(EASYXML_START_CLOSE,StringSubstr(pText,pPos, 1)) != 0 &&
               StringCompare(EASYXML_SELFCLOSE, StringSubstr(pText, pPos, 2)) != 0 &&
               !endOfXml(pText,pPos))
           {
            sAttributes+=StringSubstr(pText,pPos,1);
            pPos++;
           }
         blSuccess=parseAttributes(pActualNode,sAttributes,sDebugSpace);
         if(!blSuccess) return(false);
        }

      // Self closing tag
      if(StringCompare(EASYXML_SELFCLOSE,StringSubstr(pText,pPos,2))==0)
        {
         pPos+=2;
         skipWhitespace(pText,pPos);

         // Detect if next sibling exists
         blSibling=hasSiblings(pText,pPos);

         continue;
        }

      // Start tag close
      if(StringCompare(EASYXML_START_CLOSE,StringSubstr(pText,pPos,1))==0)
        {
         pPos++;
           } else {
         Err=EASYXML_INVALID_OPENTAG_CLOSE;
         return(false);
        }

      // Skip comments
      if(!skipWhitespaceAndComments(pText,pPos,sDebugSpace)) return(false);

      //---
      // Parse next lower level tag and/or read text content
      //---

      // Next level tag
      if(StringCompare(EASYXML_START_OPEN,StringSubstr(pText,pPos,1))==0 && 
         StringCompare(EASYXML_CLOSE_OPEN,StringSubstr(pText,pPos,2))!=0 && 
         StringCompare(EASYXML_CDATA_OPEN,StringSubstr(pText,pPos,StringLen(EASYXML_CDATA_OPEN)))!=0)
        {
         // Delve deeper
         pActualNode=pActualNode.createChild(new CEasyXmlNode);

         blSuccess=parseRecursive(pActualNode,pText,pPos,pLevel+1);
         if(!blSuccess) return(false);

         pActualNode=pActualNode.Parent();
        }

      // Read text content, even if it follows a closing tag
      if(StringCompare(EASYXML_CDATA_OPEN,StringSubstr(pText,pPos,StringLen(EASYXML_CDATA_OPEN)))!=0)
        {
         // Tags in between text won't get parsed as XML nodes
         while(StringCompare(EASYXML_CLOSE_OPEN+sTagName,StringSubstr(pText,pPos,StringLen(EASYXML_CLOSE_OPEN+sTagName)))!=0 && 
               !endOfXml(pText,pPos))
           {
            sTagContent+=StringSubstr(pText,pPos,1);
            pPos++;
           }
         pActualNode.setValue(sTagContent);
        }
      // Else read CDATA content, if there is any
      else
        {
         int iClose=StringFind(pText,EASYXML_CDATA_CLOSE,pPos+StringLen(EASYXML_CDATA_OPEN));

         if(iClose>0)
           {
            sTagContent = StringSubstr(pText, pPos + StringLen(EASYXML_CDATA_OPEN), (iClose - pPos - StringLen(EASYXML_CDATA_OPEN)));
            pPos        = iClose + StringLen(EASYXML_CDATA_CLOSE);
              } else {
            Err=EASYXML_INVALID_CDATA;
            return(false);
           }
         pActualNode.setValue(sTagContent);
        }

      // Debugging
      if(blDebug && StringLen(sTagContent)!=0)
        {
         sDebugOutput=sTagContent;
         StringTrimLeft(sDebugOutput);
         sDebugOutput=(StringLen(sDebugOutput)>=50) ? StringSubstr(sDebugOutput,0,50)+"..." : sDebugOutput;
         Print(sDebugSpace,"  ### Content ###    "+sDebugOutput);
        }

      skipWhitespace(pText,pPos);

      //---
      // Get end tag and compare it to start tag. return to upper level if valid
      //---

      if(StringFind(EASYXML_CLOSE_OPEN,StringSubstr(pText,pPos,2))==0)
        {
         pPos+=2;
           } else {
         Err=EASYXML_NO_CLOSETAG;
         return(false);
        }

      //read end tag name
      while(StringFind(EASYXML_CLOSE_CLOSE,StringSubstr(pText,pPos,1))==-1 && !endOfXml(pText,pPos))
        {
         sCloseTagName+=StringSubstr(pText,pPos,1);
         pPos++;
        }

      if(blDebug) Print(sDebugSpace,"</",sCloseTagName,"> D:",IntegerToString(pLevel)," | P:",pPos);

      //compare start and end tag names
      if(StringCompare(sCloseTagName,sTagName,false)==0)
        {
         pPos++;
           } else {
         Err=EASYXML_INVALID_CLOSETAG;
         return(false);
        }

      // Skip comments
      if(!skipWhitespaceAndComments(pText,pPos,sDebugSpace)) return(false);

      // Detect if next sibling exists
      blSibling=hasSiblings(pText,pPos);

     }
   while(blSibling==true);

//return to upper level
   return(true);
  }
//+------------------------------------------------------------------+
//| pares attributes                                                 |
//+------------------------------------------------------------------+
bool CEasyXml::parseAttributes(CEasyXmlNode *pActualNode,string pAttributes,string pDebugSpace)
  {
   int iAttrPos        = 0;
   int iValidAttrStart = 0;
   int iValidAttrEnd   = 0;

   string sDebugOutput;

   while(!endOfXml(pAttributes,iAttrPos))
     {
      string sAttributeName;
      string sAttributeValue;
      string sAttributeValueColon;

      // Some wellformed validity test
      if(StringFind(EASYXML_LATIN,StringSubstr(pAttributes,iAttrPos,1))!=-1)
        {
         iValidAttrStart++;
        }

      // Read Attributename
      while(StringCompare(EASYXML_ATTRIBUTE_SEPARATOR,StringSubstr(pAttributes,iAttrPos,1))!=0)
        {
         sAttributeName+=StringSubstr(pAttributes,iAttrPos,1);
         //if(iAttrPos>=StringLen(pAttributes))break;
         iAttrPos++;
        }

      // Skip attribute separator
      if(StringCompare(EASYXML_ATTRIBUTE_SEPARATOR,StringSubstr(pAttributes,iAttrPos,1))==0)
        {
         iAttrPos++;
        }

      // Read attribute value. Store Open Colon and use for further comparison
      if(StringFind(EASYXML_ATTRIBUTE_COLON,StringSubstr(pAttributes,iAttrPos,1))!=-1)
        {
         sAttributeValueColon=StringSubstr(pAttributes,iAttrPos,1);
         iAttrPos++;
        }

      while(StringFind(sAttributeValueColon,StringSubstr(pAttributes,iAttrPos,1))==-1)
        {
         sAttributeValue+=StringSubstr(pAttributes,iAttrPos,1);
         //if(iAttrPos>=StringLen(pAttributes))break;
         iAttrPos++;
        }

      if(StringFind(sAttributeValueColon,StringSubstr(pAttributes,iAttrPos,1))!=-1)
        {
         iAttrPos++;
         iValidAttrEnd++;
        }

      // If attribute is wellformed, set attribute to node
      if(iValidAttrStart==iValidAttrEnd)
        {
         pActualNode.setAttribute(sAttributeName,sAttributeValue);
           } else {
         Err=EASYXML_INVALID_ATTRIBUTE;
         return(false);
        }

      // Debugging
      if(blDebug)
        {
         sDebugOutput += (StringLen(sDebugOutput) != 0) ? " | " : "";
         sDebugOutput += IntegerToString(iValidAttrEnd) + ": " + sAttributeName + " -> " + sAttributeValue;
        }

      skipWhitespace(pAttributes,iAttrPos);
     }

// Debugging
   if(blDebug) Print(pDebugSpace,"  ### Attributes ###    ",sDebugOutput);

   return(true);
  }
//+------------------------------------------------------------------+
//| skip whitespace                                                  |
//+------------------------------------------------------------------+
void CEasyXml::skipWhitespace(string &pText,int &pPos)
  {
   while(StringFind(EASYXML_WHITESPACE,StringSubstr(pText,pPos,1))!=-1)
     {
      pPos++;
     }
  }
//+------------------------------------------------------------------+
//| check if node has siblings                                       |
//+------------------------------------------------------------------+
bool CEasyXml::hasSiblings(string &pText,int &pPos)
  {
   if(StringFind(EASYXML_START_OPEN,StringSubstr(pText,pPos,1))==0 &&
      StringFind(EASYXML_CLOSE_OPEN,StringSubstr(pText,pPos,2))==-1)
     {
      return true;
        } else {
      return false;
     }
  }
//+------------------------------------------------------------------+
//| check for end of xml                                             |
//+------------------------------------------------------------------+
bool CEasyXml::endOfXml(string &pText,int &pPos)
  {
   return !(pPos<StringLen(pText));
  }
//+------------------------------------------------------------------+
//| skip xml prolog                                                  |
//+------------------------------------------------------------------+
bool CEasyXml::skipProlog(string &pText,int &pPos)
  {
// Skip xml declaration
   if(StringCompare(EASYXML_PROLOG_OPEN,StringSubstr(pText,pPos,StringLen(EASYXML_PROLOG_OPEN)))==0)
     {
      int iClose=StringFind(pText,EASYXML_PROLOG_CLOSE,pPos+StringLen(EASYXML_PROLOG_OPEN));

      if(blDebug) Print("### Prolog ###    ",StringSubstr(pText,pPos,(iClose-pPos)+StringLen(EASYXML_PROLOG_CLOSE)));

      if(iClose>0)
        {
         pPos=iClose+StringLen(EASYXML_PROLOG_CLOSE);
           } else {
         Err=EASYXML_INVALID_PROLOG;
         return(false);
        }
     }
// skip stylesheet declarations
   if(StringCompare(EASYXML_STYLESHEET_OPEN,StringSubstr(pText,pPos,StringLen(EASYXML_STYLESHEET_OPEN)))==0)
     {
      int iClose=StringFind(pText,EASYXML_STYLESHEET_CLOSE,pPos+StringLen(EASYXML_STYLESHEET_OPEN));

      if(blDebug) Print("### Prolog ###    ",StringSubstr(pText,pPos,(iClose-pPos)+StringLen(EASYXML_STYLESHEET_CLOSE)));

      if(iClose>0)
        {
         pPos=iClose+StringLen(EASYXML_STYLESHEET_CLOSE);
           } else {
         Err=EASYXML_INVALID_PROLOG;
         return(false);
        }
     }
// Skip comments
   if(!skipWhitespaceAndComments(pText,pPos,"")) return(false);

// Skip doctype
   if(StringCompare(EASYXML_DOCTYPE_OPEN,StringSubstr(pText,pPos,StringLen(EASYXML_DOCTYPE_OPEN)))==0)
     {
      int iClose=StringFind(pText,EASYXML_DOCTYPE_CLOSE,pPos+StringLen(EASYXML_DOCTYPE_OPEN));

      if(blDebug) Print("### DOCTYPE ###    ",StringSubstr(pText,pPos,(iClose-pPos)+StringLen(EASYXML_DOCTYPE_CLOSE)));

      if(iClose>0)
        {
         pPos=iClose+StringLen(EASYXML_DOCTYPE_CLOSE);
           } else {
         Err=EASYXML_INVALID_DOCTYPE;
         return(false);
        }
     }

// Skip comments
   if(!skipWhitespaceAndComments(pText,pPos,"")) return(false);

   return(true);
  }
//+------------------------------------------------------------------+
//| skip xml comments                                                |
//+------------------------------------------------------------------+
bool CEasyXml::skipWhitespaceAndComments(string &pText,int &pPos,string pDebugSpace)
  {
   bool blNextComment=false;

// Do while there are consecutive comments
   do
     {
      skipWhitespace(pText,pPos);

      if(StringCompare(EASYXML_COMMENT_OPEN,StringSubstr(pText,pPos,StringLen(EASYXML_COMMENT_OPEN)))==0)
        {
         int iClose=StringFind(pText,EASYXML_COMMENT_CLOSE,pPos+StringLen(EASYXML_COMMENT_OPEN));

         if(blDebug) Print(pDebugSpace,"  ### Comment ###    ",StringSubstr(pText,pPos,(iClose-pPos)+StringLen(EASYXML_COMMENT_CLOSE)));

         if(iClose>0)
           {
            pPos=iClose+StringLen(EASYXML_COMMENT_CLOSE);
              } else {
            Err=EASYXML_INVALID_COMMENT;
            return(false);
           }
        }

      skipWhitespace(pText,pPos);

      if(StringCompare(EASYXML_COMMENT_OPEN,StringSubstr(pText,pPos,StringLen(EASYXML_COMMENT_OPEN)))==0)
        {
         blNextComment=true;
           } else {
         blNextComment=false;
        }

     }
   while(blNextComment==true);

   return(true);
  }
//+------------------------------------------------------------------+
//| error handling                                                   |
//+------------------------------------------------------------------+
bool CEasyXml::Error(int pPos=-1,bool pClear=true)
  {
   if(pClear) Clear();
   if(!ErrMsg==NULL)ErrMsg=NULL;
   ErrMsg=EasyXmlError(Err,pPos);
   Print(ErrMsg);
   return(false);
  }
//+------------------------------------------------------------------+
//| get paresd document root                                         |
//+------------------------------------------------------------------+
CEasyXmlNode *CEasyXml::getDocumentRoot(void)
  {
   return DocumentRoot;
  }
//+------------------------------------------------------------------+
//| return unparsed text                                             |
//+------------------------------------------------------------------+
string CEasyXml::getText(void)
  {
   return sText;
  }
//+------------------------------------------------------------------+
//| Clear the doc tree                                               |
//+------------------------------------------------------------------+
void CEasyXml::Clear(void)
  {
   StringInit(sText);
   if(CheckPointer(DocumentRoot)==POINTER_DYNAMIC) delete DocumentRoot;

   DocumentRoot=new CEasyXmlNode;
  }
//+------------------------------------------------------------------+
