//+------------------------------------------------------------------+
//|                                              SignalEmbedding.mqh |
//|                   Copyright 2009-2017, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertSignal.mqh>
#include <My\C_u_rbm.mqh>
//+------------------------------------------------------------------+
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals based on Price-Embedding with UnOrthodox RBM.      |
//| Type=SignalAdvanced                                              |
//| Name=Embedding                                                   |
//| ShortName=Embedding                                              |
//| Class=CSignalEmbedding                                           |
//| Page=signal_embedding                                            |
//| Parameter=TrainSet,int,65,Training Set Size                      |
//| Parameter=Epochs,int,5,Epochs                                    |
//| Parameter=LearningRate,double,0.0005,Learning Rate               |
//| Parameter=InitialWeight,double,0.005,Default Weight              |
//| Parameter=InitialBias,double,0.0,Default Bias                    |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalEmbedding.                                          |
//| Purpose: Class of Price-Embedding with UnOrthodox RBM.           |
//|            Derives from class CExpertSignal.                     |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
#define  __RBM_VISIBLE 8
#define  __RBM_HIDDEN 4
int      __RBM_ARCH[2] // RBM arch

=
{  __RBM_VISIBLE, __RBM_HIDDEN
};
#define  __MLP_INPUTS 4
#define  __MLP_HIDDEN 4
#define  __MLP_OUTPUTS 1
int      __MLP_ARCH[5] // MLP arch

=
{  __MLP_INPUTS, __MLP_HIDDEN, __MLP_HIDDEN, __MLP_HIDDEN, __MLP_OUTPUTS
};
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CSignalEmbedding   : public CExpertSignal
{
protected:
   int                           m_train_set;                       //
   int                           m_epochs;                              // Epochs
   double                        m_learning_rate;                       // Epsilon

   double                        m_initial_weight;
   double                        m_initial_bias;

   Slearning                     m_learning;
   Smlp                          m_mlp,m_u_rbm;

   double                        m_train_deposit;
   double                        m_train_criteria;

   
public:
                     CSignalEmbedding(void);
                    ~CSignalEmbedding(void);

   //--- methods of setting adjustable parameters
   void                          TrainSet(int value)
   {  m_train_set = value;
   }
   void                          Epochs(int value)
   {  m_epochs = value;
   }
   void                          LearningRate(double value)
   {  m_learning_rate = value;
   }


   void                          InitialWeight(double value)
   {  m_initial_weight = value;
   }
   void                          InitialBias(double value)
   {  m_initial_bias = value;
   }

   //--- method of verification of arch
   virtual bool      ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int       LongCondition(void);
   virtual int       ShortCondition(void);

protected:
   void                          Embedder(vector &Extraction);
   vector                        Extractor(vector &Embedding, vector &Labels, bool Train = true);

   C_u_rbm                       *U_RBM;
   Cmlp                          *MLP;
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CSignalEmbedding::CSignalEmbedding(void) :    m_train_set(65),
   m_epochs(5),
   m_learning_rate(0.0005),
   m_initial_weight(0.005),
   m_initial_bias(0.0)

{
//--- initialization of protected data
   m_used_series = USE_SERIES_OPEN + USE_SERIES_HIGH + USE_SERIES_LOW + USE_SERIES_CLOSE + USE_SERIES_SPREAD + USE_SERIES_TIME;
//
   ArrayResize(m_mlp.arch, ArraySize(__MLP_ARCH));
   for(int i = 0; i < ArraySize(__MLP_ARCH); i++)
   {  m_mlp.arch[i] = __MLP_ARCH[i];
   }
   m_mlp.initial_weight = m_initial_weight;
   m_mlp.initial_bias = m_initial_bias;
   m_mlp.loss = LOSS_HUBER;
   m_mlp.activation = AF_SOFTSIGN;
//
   MLP = new Cmlp(m_mlp);
//
   ArrayResize(m_u_rbm.arch, ArraySize(__RBM_ARCH));
   for(int i = 0; i < ArraySize(__RBM_ARCH); i++)
   {  m_u_rbm.arch[i] = __RBM_ARCH[i];
   }
   m_u_rbm.initial_weight = m_initial_weight;
   m_u_rbm.initial_bias = m_initial_bias;
   m_u_rbm.loss = LOSS_CCE;
   m_u_rbm.activation = AF_SOFTMAX;
//
   U_RBM = new C_u_rbm(m_u_rbm);
//
   m_learning.epochs = m_epochs;
   m_learning.initial_rate = m_learning_rate;
   m_learning.rate = m_learning_rate;
   m_learning.type = LEARNING_ADAPTIVE_GRADIENT;
   m_train_deposit = AccountInfoDouble(ACCOUNT_BALANCE);
   m_train_criteria = 0.0;
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
void CSignalEmbedding::~CSignalEmbedding(void)
{  delete MLP;
   delete U_RBM;
}
//+------------------------------------------------------------------+
//| Validation arch protected data.                              |
//+------------------------------------------------------------------+
bool CSignalEmbedding::ValidationSettings(void)
{  if(!CExpertSignal::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_epochs < 1)
   {  printf(__FUNCSIG__ + " epochs size should at least be 1! ");
      return(false);
   }
   if(m_learning_rate < 0.0)
   {  printf(__FUNCSIG__ + " learning rate should be greater than 0.0! ");
      return(false);
   }
   if(!MLP.validated)
   {  printf(__FUNCSIG__ + " invalid network arch. ");
      printf(__FUNCSIG__ +
             " invalid network arch! Settings size is: %i, Max layer size is: %i, Min layer size is: %i, and activation is %s ",
             ArraySize(m_mlp.arch), m_mlp.arch[ArrayMaximum(m_mlp.arch)], m_mlp.arch[ArrayMinimum(m_mlp.arch)], EnumToString(m_mlp.activation)
            );
      return(false);
   }
//read best weights
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalEmbedding::InitIndicators(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalEmbedding::LongCondition(void)
{  int result = 0;
   vector _o;
   Embedder(_o);
   //Print(__FUNCSIG__+" extract: ",_o);
   m_close.Refresh(-1);
   if(_o[0] > 0.0)
   {  result = int(round(100.0*fabs(_o[0])/(fabs(_o[0])+fabs(m_close.GetData(StartIndex())-m_close.GetData(StartIndex()+1)))));
   }
   return(result);
}
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalEmbedding::ShortCondition(void)
{  int result = 0;
   vector _o;
   Embedder(_o);
   m_close.Refresh(-1);
   if(_o[0] < 0.0)
   {  result = int(round(100.0*fabs(_o[0])/(fabs(_o[0])+fabs(m_close.GetData(StartIndex())-m_close.GetData(StartIndex()+1)))));
   }
   return(result);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSignalEmbedding::Embedder(vector &Extraction)
{  m_learning.rate = m_learning_rate;
   for(int i = 1; i <= m_epochs; i++)
   {  U_RBM.LearningType(m_learning, i);
      for(int ii = m_train_set; ii >= 0; ii--)
      {  vector _in, _in_new, _in_old, _out, _out_new, _out_old;
         if
         (
         _in_new.Init(__RBM_VISIBLE) && 
         _in_new.CopyRates(m_symbol.Name(), m_period, 8, ii + __MLP_OUTPUTS, __RBM_VISIBLE) && 
         _in_new.Size() == __RBM_VISIBLE &&
         _in_old.Init(__RBM_VISIBLE) && 
         _in_old.CopyRates(m_symbol.Name(), m_period, 8, ii + __MLP_OUTPUTS + 1, __RBM_VISIBLE) && 
         _in_old.Size() == __RBM_VISIBLE
         &&
         _out_new.Init(__MLP_OUTPUTS) &&
         _out_new.CopyRates(m_symbol.Name(), m_period, 8, ii, __MLP_OUTPUTS) &&
         _out_new.Size() == __MLP_OUTPUTS &&
         _out_old.Init(__MLP_OUTPUTS) &&
         _out_old.CopyRates(m_symbol.Name(), m_period, 8, ii + __MLP_OUTPUTS, __MLP_OUTPUTS) &&
         _out_old.Size() == __MLP_OUTPUTS
         )
         {  _in = _in_new - _in_old;
            _out = _out_new - _out_old;
            U_RBM.Set(_in);
            U_RBM.GetPositive();
            U_RBM.GetNegative();
            U_RBM.BackPropagate(m_learning.rate);
            Extraction = Extractor(U_RBM.output, _out, ii > 0);
         }
      }
   }
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector CSignalEmbedding::Extractor(vector &Embedding, vector &Labels, bool Train = true)
{  vector _extraction;
   _extraction.Init(MLP.output.Size());
   m_learning.rate = m_learning_rate;
   for(int i = 1; i <= m_epochs; i++)
   {  MLP.LearningType(m_learning, i);
      MLP.Set(Embedding);
      MLP.Forward();
      _extraction = MLP.output;
      if(Train)
      {  MLP.Get(Labels);
         MLP.Backward(m_learning, i);
      }
   }
   return(_extraction);
}
//+------------------------------------------------------------------+
