//+------------------------------------------------------------------+
//|                                            PatchTST Predictor    |
//|                                                   Copyright 2024 |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024"
#property link      "https://www.mql5.com"
#property version   "1.00"

#resource "\\PatchTST\\patchtst_model.onnx" as uchar PatchTSTModel[]

#define SEQ_LENGTH 168
#define PRED_LENGTH 24
#define INPUT_FEATURES 4

long ModelHandle = INVALID_HANDLE;
datetime ExtNextBar = 0;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
   // Load the ONNX model
   ModelHandle = OnnxCreateFromBuffer(PatchTSTModel, ONNX_DEFAULT);
   if (ModelHandle == INVALID_HANDLE)
   {
      Print("Error creating ONNX model: ", GetLastError());
      return(INIT_FAILED);
   }

   // Set input shape
   const long input_shape[] = {1, SEQ_LENGTH, INPUT_FEATURES};
   if (!OnnxSetInputShape(ModelHandle, ONNX_DEFAULT, input_shape))
   {
      Print("Error setting input shape: ", GetLastError());
      return(INIT_FAILED);
   }

   // Set output shape
   const long output_shape[] = {1, PRED_LENGTH, INPUT_FEATURES};
   if (!OnnxSetOutputShape(ModelHandle, 0, output_shape))
   {
      Print("Error setting output shape: ", GetLastError());
      return(INIT_FAILED);
   }

   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   if (ModelHandle != INVALID_HANDLE)
   {
      OnnxRelease(ModelHandle);
      ModelHandle = INVALID_HANDLE;
   }
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
   if (TimeCurrent() < ExtNextBar)
      return;

   ExtNextBar = TimeCurrent();
   ExtNextBar -= ExtNextBar % PeriodSeconds();
   ExtNextBar += PeriodSeconds();

   // Prepare input data
   float input_data[];
   if (!PrepareInputData(input_data))
   {
      Print("Error preparing input data");
      return;
   }

   // Make prediction
   float predictions[];
   if (!MakePrediction(input_data, predictions))
   {
      Print("Error making prediction");
      return;
   }

   // Draw hypothetical future bars
   DrawFutureBars(predictions);
}

//+------------------------------------------------------------------+
//| Prepare input data for the model                                 |
//+------------------------------------------------------------------+
bool PrepareInputData(float &input_data[])
{
   MqlRates rates[];
   ArraySetAsSeries(rates, true);
   int copied = CopyRates(_Symbol, PERIOD_H1, 0, SEQ_LENGTH, rates);
   
   if (copied != SEQ_LENGTH)
   {
      Print("Failed to copy rates data. Copied: ", copied);
      return false;
   }

   ArrayResize(input_data, SEQ_LENGTH * INPUT_FEATURES);
   for (int i = 0; i < SEQ_LENGTH; i++)
   {
      input_data[i * INPUT_FEATURES + 0] = (float)rates[SEQ_LENGTH - 1 - i].open;
      input_data[i * INPUT_FEATURES + 1] = (float)rates[SEQ_LENGTH - 1 - i].high;
      input_data[i * INPUT_FEATURES + 2] = (float)rates[SEQ_LENGTH - 1 - i].low;
      input_data[i * INPUT_FEATURES + 3] = (float)rates[SEQ_LENGTH - 1 - i].close;
   }

   return true;
}

//+------------------------------------------------------------------+
//| Make prediction using the ONNX model                             |
//+------------------------------------------------------------------+
bool MakePrediction(const float &input_data[], float &output_data[])
{
   ArrayResize(output_data, PRED_LENGTH * INPUT_FEATURES);

   if (!OnnxRun(ModelHandle, ONNX_NO_CONVERSION, input_data, output_data))
   {
      Print("Error running ONNX model: ", GetLastError());
      return false;
   }

   return true;
}

//+------------------------------------------------------------------+
//| Draw hypothetical future bars                                    |
//+------------------------------------------------------------------+
void DrawFutureBars(const float &predictions[])
{
   datetime current_time = TimeCurrent();
   for (int i = 0; i < PRED_LENGTH; i++)
   {
      datetime bar_time = current_time + PeriodSeconds(PERIOD_H1) * (i + 1);
      double open = predictions[i * INPUT_FEATURES + 0];
      double high = predictions[i * INPUT_FEATURES + 1];
      double low = predictions[i * INPUT_FEATURES + 2];
      double close = predictions[i * INPUT_FEATURES + 3];

      string obj_name = "FutureBar_" + IntegerToString(i);
      ObjectCreate(0, obj_name, OBJ_RECTANGLE, 0, bar_time, low, bar_time + PeriodSeconds(PERIOD_H1), high);
      ObjectSetInteger(0, obj_name, OBJPROP_COLOR, close > open ? clrGreen : clrRed);
      ObjectSetInteger(0, obj_name, OBJPROP_FILL, true);
      ObjectSetInteger(0, obj_name, OBJPROP_BACK, true);
   }

   ChartRedraw();
}