import torch
import torch.nn as nn
import numpy as np
import pandas as pd
import MetaTrader5 as mt5
import matplotlib.pyplot as plt

# Define LSTM model class (same as during training)
class LSTMModel(nn.Module):
    def __init__(self, input_size, hidden_layer_size, output_size):
        super(LSTMModel, self).__init__()
        self.hidden_layer_size = hidden_layer_size
        self.lstm = nn.LSTM(input_size, hidden_layer_size)
        self.linear = nn.Linear(hidden_layer_size, output_size)
        self.hidden_cell = (torch.zeros(1, 1, self.hidden_layer_size),
                            torch.zeros(1, 1, self.hidden_layer_size))

    def forward(self, input_seq):
        lstm_out, self.hidden_cell = self.lstm(input_seq.view(len(input_seq), 1, -1), self.hidden_cell)
        predictions = self.linear(lstm_out.view(len(input_seq), -1))
        return predictions[-1]

# Normalize prices on a rolling basis resetting at the start of each day
def normalize_daily_rolling(data):
    data['date'] = data.index.date
    data['rolling_high'] = data.groupby('date')['high'].transform(lambda x: x.expanding(min_periods=1).max())
    data['rolling_low'] = data.groupby('date')['low'].transform(lambda x: x.expanding(min_periods=1).min())

    data['norm_open'] = (data['open'] - data['rolling_low']) / (data['rolling_high'] - data['rolling_low'])
    data['norm_high'] = (data['high'] - data['rolling_low']) / (data['rolling_high'] - data['rolling_low'])
    data['norm_low'] = (data['low'] - data['rolling_low']) / (data['rolling_high'] - data['rolling_low'])
    data['norm_close'] = (data['close'] - data['rolling_low']) / (data['rolling_high'] - data['rolling_low'])

    # Replace NaNs with zeros
    data.fillna(0, inplace=True)
    return data[['norm_open', 'norm_high', 'norm_low', 'norm_close']]

# Load the saved model
input_size = 5  # time_token, norm_open, norm_high, norm_low, norm_close
hidden_layer_size = 100
output_size = 1

model = LSTMModel(input_size, hidden_layer_size, output_size)
model.load_state_dict(torch.load('lstm_model.pth'))
model.eval()

# Connect to MetaTrader 5
if not mt5.initialize():
    print("Initialize failed")
    mt5.shutdown()

# Load the latest 160 bars of market data
symbol = "EURUSD"
timeframe = mt5.TIMEFRAME_M15
bars = 160  # 60 for sequence length + 100 for evaluation steps
rates = mt5.copy_rates_from_pos(symbol, timeframe, 0, bars)
mt5.shutdown()

# Convert to DataFrame
data = pd.DataFrame(rates)
data['time'] = pd.to_datetime(data['time'], unit='s')
data.set_index('time', inplace=True)

# Normalize the new data
data[['norm_open', 'norm_high', 'norm_low', 'norm_close']] = normalize_daily_rolling(data)

# Tokenize time
data['time_token'] = (data.index.hour * 3600 + data.index.minute * 60 + data.index.second) / 86400

# Drop unnecessary columns
data = data[['time_token', 'norm_open', 'norm_high', 'norm_low', 'norm_close']]

# Fetch the last 100 sequences for evaluation
seq_length = 60
evaluation_steps = 100

# Initialize lists for storing evaluation results
all_true_prices = []
all_predicted_prices = []

model.eval()

for step in range(evaluation_steps, 0, -1):
    # Get the sequence ending at 'step'
    seq = data.values[-step-seq_length:-step]
    seq = torch.tensor(seq, dtype=torch.float32)

    # Make prediction
    with torch.no_grad():
        model.hidden_cell = (torch.zeros(1, 1, model.hidden_layer_size),
                             torch.zeros(1, 1, model.hidden_layer_size))
        prediction = model(seq).item()
    
    all_true_prices.append(data['norm_close'].values[-step])
    all_predicted_prices.append(prediction)

# Calculate percent changes and convert to percentages
true_pct_change = (np.diff(all_true_prices) / np.array(all_true_prices[:-1])) * 100
predicted_pct_change = (np.diff(all_predicted_prices) / np.array(all_predicted_prices[:-1])) * 100

# Make next prediction
next_seq = data.values[-seq_length:]
next_seq = torch.tensor(next_seq, dtype=torch.float32)

with torch.no_grad():
    model.hidden_cell = (torch.zeros(1, 1, model.hidden_layer_size),
                         torch.zeros(1, 1, model.hidden_layer_size))
    next_prediction = model(next_seq).item()

# Calculate percent change for the next prediction
next_true_price = data['norm_close'].values[-1]
next_price_pct_change = ((next_prediction - all_predicted_prices[-1]) / all_predicted_prices[-1]) * 100

print(f"Next predicted close price (normalized): {next_prediction}")
print(f"Percent change for the next prediction based on normalized price: {next_price_pct_change:.5f}%")
print("All Predicted Prices: ", all_predicted_prices)

# Plot the evaluation results with capped y-axis
plt.figure(figsize=(12, 8))

plt.subplot(2, 1, 1)
plt.plot(all_true_prices, label='True Prices (Normalized)')
plt.plot(all_predicted_prices, label='Predicted Prices (Normalized)')
plt.scatter(len(all_true_prices), next_prediction, color='red', label='Next Prediction')
plt.legend()
plt.title('True vs Predicted Prices (Normalized, Last 100 Steps)')
plt.ylim(min(min(all_true_prices), min(all_predicted_prices))-0.1, max(max(all_true_prices), max(all_predicted_prices))+0.1)

plt.subplot(2, 1, 2)
plt.plot(true_pct_change, label='True Percent Change')
plt.plot(predicted_pct_change, label='Predicted Percent Change')
plt.scatter(len(true_pct_change), next_price_pct_change, color='red', label='Next Prediction')
plt.legend()
plt.title('True vs Predicted Price Percent Change (Last 100 Steps)')
plt.ylabel('Percent Change (%)')
plt.ylim(-100, 100)  # Cap the y-axis at -100% to 100%

plt.tight_layout()
plt.show()

