//+------------------------------------------------------------------+
//|                                             SignalPerceptron.mqh |
//|                   Copyright 2009-2017, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Math\Alglib\dataanalysis.mqh>
#include <Cgini.mqh>
#include <Math\Stat\stat.mqh>
#include <Expert\ExpertSignal.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals based on Classifier Perceptron                     |
//| Type=SignalAdvanced                                              |
//| Name=Perceptron                                                  |
//| ShortName=Perceptron                                             |
//| Class=CSignalPerceptron                                          |
//| Page=signal_perceptron                                           |
//| Parameter=Features,int,8,Features                                |
//| Parameter=Hidden,int,1,Hidden Layers                             |
//| Parameter=Hidden_1_Size,int,5,Hidden Layer 1 Size                |
//| Parameter=Hidden_2_Size,int,3,Hidden Layer 2 Size                |
//| Parameter=Training_Points,int,4,Training Points                  |
//| Parameter=Training_Restarts,int,2,Training Restarts              |
//| Parameter=ActivationType,int,1,Activation Type (0 - 3)           |
//| Parameter=Hidden_1_Bias,double,0.05,Hidden Layer 1 Bias          |
//| Parameter=Hidden_2_Bias,double,0.05,Hidden Layer 2 Bias          |
//| Parameter=Output_Bias,double,0.05,Output Layer Bias              |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalPerceptron.                                         |
//| Purpose: Class of of trade signals based on Classifier Perceptron.|
//|            Derives from class CExpertSignal.                     |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CSignalPerceptron          : public CExpertSignal
  {
protected:
   
   int                           m_hidden;                              // 
   int                           m_features;                            // 
   int                           m_hidden_1_size;                       //
   int                           m_hidden_2_size;                       //  
   int                           m_training_points;                     //
   int                           m_training_restarts;                   //
   int                           m_activation_type;                     // 
   double                        m_hidden_1_bias;                       //
   double                        m_hidden_2_bias;                       //
   double                        m_output_bias;                         //

public:
                                 CSignalPerceptron(void);
                                 ~CSignalPerceptron(void);
   
   //--- method of verification of settings
   virtual bool                  ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool                  InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int                   LongCondition(void);
   virtual int                   ShortCondition(void);
   //---
   void                          Features(int value)           { m_features=value;           }
   void                          Hidden(int value)             { m_hidden=value;             }
   void                          Hidden_1_Size(int value)      { m_hidden_1_size=value;      }
   void                          Hidden_2_Size(int value)      { m_hidden_2_size=value;      }
   void                          Training_Points(int value)    { m_training_points=value;    }
   void                          Training_Restarts(int value)  { m_training_restarts=value;  }
   void                          ActivationType(int value)     { m_activation_type=value;    }
   void                          Hidden_1_Bias(double value)   { m_hidden_1_bias=value;      }
   void                          Hidden_2_Bias(double value)   { m_hidden_2_bias=value;      }
   void                          Output_Bias(double value)     { m_output_bias=value;        }

protected:

   CBdSS                         m_norm;
   CMLPBase                      m_base;
   CMLPTrain                     m_train;
   CMatrixDouble                 m_xy;
   CMLPReport                    m_report;
   CMLPTrainer                   m_trainer;
   CMultilayerPerceptron         m_network;
   
   bool                          m_in_training;
      
   int                           m_inputs;
   int                           m_outputs;
   
   double                        m_last_long_y;
   double                        m_last_short_y;
   
   bool                          ReadWeights();
   bool                          WriteWeights(CRowDouble &Export);
   
   void                          Process(double &Y);
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CSignalPerceptron::CSignalPerceptron(void)  :    m_features(4),
                                                      m_hidden(1),
                                                      m_hidden_1_size(5),
                                                      m_hidden_2_size(3),
                                                      m_training_points(4),
                                                      m_training_restarts(2),
                                                      m_activation_type(1),
                                                      m_hidden_1_bias(0.05),
                                                      m_hidden_2_bias(0.05),
                                                      m_output_bias(0.05)
  {
//--- initialization of protected data
      m_used_series=USE_SERIES_OPEN+USE_SERIES_HIGH+USE_SERIES_LOW+USE_SERIES_CLOSE+USE_SERIES_SPREAD+USE_SERIES_TIME;
      
  }
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalPerceptron::ValidationSettings(void)
  {
   if(!CExpertSignal::ValidationSettings())
      return(false);
//--- initial data checks

   m_inputs=m_features;
   m_outputs=2;
   
   printf(__FUNCSIG__+" model x independents, and y classifiers are: "+IntegerToString(m_inputs)+", and: "+IntegerToString(m_outputs));
   
   m_xy.Resize(m_training_points,m_inputs+m_outputs);
   
   m_train.MLPCreateTrainerCls(m_inputs,m_outputs,m_trainer);
   
   if(m_hidden==0)
   {
      m_base.MLPCreateC0(m_inputs,m_outputs,m_network);
   }
   else if(m_hidden==1)
   {
      m_base.MLPCreateC1(m_inputs,m_hidden_1_size,m_outputs,m_network);
   }
   else if(m_hidden==2)
   {
      m_base.MLPCreateC2(m_inputs,m_hidden_1_size,m_hidden_2_size,m_outputs,m_network);
   }
   else if(m_hidden>2||m_hidden<0)
   {
      printf(__FUNCSIG__+" invalid number of hidden layers should be 0, 1, or 2. ");
      return(false);
   }
      
   if(!ReadWeights())
   {
      m_network.m_weights.Fill(1.0);
   }
      
   int _last_layer=1;
   
   if(m_hidden>0)
   {
      _last_layer++;
      //
      for(int i=0;i<m_hidden_1_size;i++)
      {
         m_base.MLPSetNeuronInfo(m_network,1,i,m_activation_type,m_hidden_1_bias);
      }
      
      if(m_hidden>1)
      {
         _last_layer++;
         //
         for(int i=0;i<m_hidden_2_size;i++)
         {
            m_base.MLPSetNeuronInfo(m_network,2,i,m_activation_type,m_hidden_2_bias);
         }
      }
   }
      
   for(int i=0;i<m_outputs;i++)
   {
      m_base.MLPSetNeuronInfo(m_network,_last_layer,i,m_activation_type,m_output_bias);
   }
   
   //
   
   m_in_training=false;
     
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalPerceptron::InitIndicators(CIndicators *indicators)
  {
//--- check pointer
   if(indicators==NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
   
      
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalPerceptron::LongCondition(void)
  {
      int result=0;
      
      if(!m_symbol.IsSynchronized()){ return(result); }
      
      double _y=0.0; Process(_y);
      
      if(m_last_long_y<_y){ result=100; }
      
      m_last_long_y=_y;
      
      printf(__FUNCSIG__+" result is: "+IntegerToString(result));
      
      return(result);
  }
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalPerceptron::ShortCondition(void)
  {
      int result=0;
      
      if(!m_symbol.IsSynchronized()){ return(result); }
      
      double _y=0.0; Process(_y);
      
      if(m_last_short_y>_y){ result=100;  }
      
      m_last_short_y=_y;
      
      printf(__FUNCSIG__+" result is: "+IntegerToString(result));
      
      return(result);
  }
//+------------------------------------------------------------------+
//| Process function                                                 |
//+------------------------------------------------------------------+
void CSignalPerceptron::Process(double &Y)
   {
      m_high.Refresh(-1);
      m_low.Refresh(-1);
      
      m_close.Refresh(-1);
      
      int _info=-1;
      
      for(int n=0;n<m_training_points;n++)
      {
         for(int i=0;i<m_inputs;i++)
         {
            m_xy.Set(n,i,m_close.GetData(n+1)-m_close.GetData(n+1+i+1));
         }
         
         //set classifier 'weights' for long & short
         m_xy.Set(n,m_inputs,m_close.GetData(n)-fmin(m_low.GetData(n),m_close.GetData(n+1)));
         m_xy.Set(n,m_inputs+1,fmax(m_high.GetData(n),m_close.GetData(n+1))-m_close.GetData(n+1));
      } 
      
      //normalise data
      CRowDouble _means,_sigmas;
      m_norm.DSNormalize(m_xy,m_training_points,m_inputs,_info,_means,_sigmas);
      
      m_train.MLPSetDataset(m_trainer,m_xy,m_training_points);
      //
      if(!m_in_training)
      {
         m_train.MLPStartTraining(m_trainer,m_network,false);
         m_in_training=true;
      }
      else if(m_in_training)
      {
         while(m_train.MLPContinueTraining(m_trainer,m_network))
         {
            //
         }
      }
      //
      double _x_inputs[],_y_outputs[];
      ArrayResize(_x_inputs,m_inputs);
      ArrayResize(_y_outputs,m_outputs);
      
      ArrayInitialize(_x_inputs,0.0);
      ArrayInitialize(_y_outputs,0.0);
      
      for(int i=0;i<m_inputs;i++)
      {
         _x_inputs[i]=((m_close.GetData(0)-m_close.GetData(i+1))-m_network.m_columnmeans[i]);
         if(m_network.m_columnsigmas[i]!=0.0){ _x_inputs[i]/=m_network.m_columnsigmas[i]; }
      }
      
      //
      
      //forward feed...
      m_base.MLPProcess(m_network,_x_inputs,_y_outputs);
      
      //long probability minus short probability
      Y=_y_outputs[0]-_y_outputs[1];
   }
//+------------------------------------------------------------------+
//| Read Weights                                                     |
//+------------------------------------------------------------------+
bool CSignalPerceptron::ReadWeights()
  {
      CRowDouble _p;
      double _p_read[];
      ResetLastError();
      int _read_handle=FileOpen("PERCEPTRON\\_param_"+m_symbol.Name()+"_"+
                                 EnumToString(m_period)+"_"+
                                 ".bin",
                                 FILE_SHARE_READ|FILE_BIN|FILE_COMMON);
      if(_read_handle!=INVALID_HANDLE)
      {
         FileReadArray(_read_handle,_p_read);
         _p.Resize(ArraySize(_p_read));
         for(int i=0;i<_p.Size();i++){
            _p.Set(i,_p_read[i]);}
         
         printf(__FUNCSIG__+" read data = "+IntegerToString(_p.Size()));
         
         FileClose(_read_handle);
         
         m_base.MLPImportTunableParameters(m_network,_p);
         
         return(true);
      }
      else
      {
         printf(__FUNCSIG__+" File read failed, error "+IntegerToString(GetLastError()));
      }
      
      return(false);
  }
//+------------------------------------------------------------------+
//| Write Weights                                                    |
//+------------------------------------------------------------------+
bool CSignalPerceptron::WriteWeights(CRowDouble &Export)
  {
      double _p_write[];
      //--- open the file
      ResetLastError();
      int _write_handle=FileOpen("PERCEPTRON\\_param_"+m_symbol.Name()+"_"+
                                 EnumToString(m_period)+"_"+
                                 ".bin",
                                 FILE_WRITE|FILE_BIN|FILE_COMMON);
      if(_write_handle!=INVALID_HANDLE)
      {
         ArrayResize(_p_write,Export.Size());
         Export.ToArray(_p_write);
         
         FileWriteArray(_write_handle,_p_write);
         
         printf(__FUNCSIG__+" written data = "+IntegerToString(Export.Size()));
         
         FileClose(_write_handle);
         
         return(true);
      }
      else
      {
         printf(__FUNCSIG__+" File write failed, error "+IntegerToString(GetLastError()));
      }
      
      return(false);
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
void CSignalPerceptron::~CSignalPerceptron(void)
  {
      double _result=TesterStatistics(STAT_PROFIT),_criteria=DBL_MIN;
      
      ResetLastError();
      int _read_handle=FileOpen("PERCEPTRON\\_crit_"+m_symbol.Name()+"_"+
                                 EnumToString(m_period)+"_"+
                                 ".bin",
                                 FILE_READ|FILE_BIN|FILE_COMMON);
      
      if(_read_handle!=INVALID_HANDLE)
      {
         _criteria=FileReadDouble(_read_handle);
         FileClose(_read_handle);
         
      }
      else
      {
         printf(__FUNCSIG__+" File read failed, error "+IntegerToString(GetLastError()));
         FileClose(_read_handle);
      }
      
      //
      printf(__FUNCSIG__+" criteria: "+DoubleToString(_criteria)+", result: "+DoubleToString(_result));
      
      if(_criteria<_result)
      {
         int _p_count;
         CRowDouble _p_export;
         m_base.MLPExportTunableParameters(m_network,_p_export,_p_count);
         WriteWeights(_p_export);
   
         ResetLastError();
         int _write_handle=FileOpen("PERCEPTRON\\_crit_"+m_symbol.Name()+"_"+
                                    EnumToString(m_period)+"_"+
                                    ".bin",
                                    FILE_WRITE|FILE_BIN|FILE_COMMON);
         if(_write_handle!=INVALID_HANDLE)
         {
            FileWriteDouble(_write_handle,_result);
            FileClose(_write_handle);
         }
         else
         {
            printf(__FUNCSIG__+" File write failed, error "+IntegerToString(GetLastError()));
            FileClose(_read_handle);
         }
      }
  }
//+------------------------------------------------------------------+
