//+------------------------------------------------------------------+
//|                                                 EquityCurves.mq5 |
//|                                  Copyright 2023, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"
#resource "\\Files\\sp500close.csv" as string sp500close
#include <Math\Stat\Stat.mqh>
#include <EasyAndFastGUI\WndCreate.mqh>
#include <PerformanceRatios.mqh>


//+------------------------------------------------------------------+
//| risk return metrics                                              |
//+------------------------------------------------------------------+
enum ENUM_METRIC
{
 ENUM_SHARPE_RATIO=0,//sharpe ratio
 ENUM_NP_BURKE_RATIO,//net profit burke ratio
 ENUM_M_BURKE_RATIO,//mean return burke ratio
 ENUM_NPMD_RATIO,//net profit to max drawdown ratio
 ENUM_OMEGA_RATIO,//omega ratio
 ENUM_UPSIDE_RATIO,//upside potential ratio
 ENUM_JENSEN_ALPHA//jensen's alpha
};
  
//+------------------------------------------------------------------+
//| Gui application class                                            |
//+------------------------------------------------------------------+
class CApp:public CWndCreate
{
 protected:
  
  CWindow      m_window;                   // main window
  
  
  CTextEdit    m_amean_inc;                //mean coefficient text edit control for first curve
  CTextEdit    m_bmean_inc;                //mean coefficient text edit control for second curve
  
  CTextEdit    m_asd_inc;                  //standard deviation coefficient text edit control for first curve
  CTextEdit    m_bsd_inc;                  //standard deviation coefficient text edit control for second curve
  
  CTextEdit    m_seed_inc;                 //random number seed text edit control
  CTextEdit    m_capital_inc;              //initial capital text edit control
  
  
  CComboBox    m_metric_type;              //performance metric comboxbox control
  
  CGraph       m_graph;                    //graph plot
  
  CCurve       *m_curve1,
               *m_curve2,
               *m_curve3;                  // individual curves on graph plot
  
  double       m_x[],                      //x axis values for all plots
               m_close[],                  //SP 500 close prices
               m_e[],                      //benchmark sp500 equity curve
               m_e1[],                     //first (blue)random equity curve 
               m_e2[],                     //second (green) random equity curve
               m_r[],                      //benchmark sp500 returns
               m_r1[],                     //returns for first (blue) equity curve
               m_r2[],                     //returns for second (green) equity curve
               rnorm[];                    //normally distributed random numbers
  
  double       rmean,                      //benchmark mean returns
               rstand;                     //benchmark standard deviation of returns
  
public:
               CApp(void);                 //constructor
              ~CApp(void);                 //destructor
              
  void        OnInitEvent(void);
  void        OnDeinitEvent(const int reason);
  
  virtual void OnEvent(const int id , const long &lparam, const double &dparam,const string &sparam);
  
  bool        CreateGUI(void);
  
protected:
  bool        CreatePlot(const int x_gap, const int y_gap);
  
private:
  bool        InitArrays(void);
  void        SetBuffers(void);
  void        RecalculatingSeries(void);
  void        TextAdd(void);
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CApp::CApp(void)
{
 rmean=rstand=0.0;
 m_curve1=m_curve2=m_curve3=NULL;
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CApp::~CApp(void)
{
 
} 
//+------------------------------------------------------------------+
//| On initialization                                                |
//+------------------------------------------------------------------+
void CApp::OnInitEvent(void)
{
}
//+------------------------------------------------------------------+
//| On DeInitilization                                               |
//+------------------------------------------------------------------+
void CApp::OnDeinitEvent(const int reason)
{
 CWndEvents::Destroy();
}                 
//+------------------------------------------------------------------+
//| Event handler                                                    |
//+------------------------------------------------------------------+
void CApp::OnEvent(const int id,const long &lparam,const double &dparam,const string &sparam)
{
 if(id==CHARTEVENT_CUSTOM+ON_CLICK_COMBOBOX_ITEM)
  {// on selection of performance metric recalculate values
   if(lparam==m_metric_type.Id())
    {
     RecalculatingSeries();
     return;
    }
   return;  
  }
 if(id==CHARTEVENT_CUSTOM+ON_END_EDIT || id==CHARTEVENT_CUSTOM+ON_CLICK_BUTTON)
  {
   if(lparam==m_amean_inc.Id() || lparam==m_bmean_inc.Id() ||
      lparam==m_asd_inc.Id() ||  lparam==m_bsd_inc.Id()    ||
      lparam==m_seed_inc.Id() ||  lparam==m_capital_inc.Id())
    {//on button click of text box edit recalculate values
     RecalculatingSeries();
     return;
    }  
   return;
  }
}    
//+------------------------------------------------------------------+
//|  Create the graphical user interface                             |
//+------------------------------------------------------------------+
bool CApp::CreateGUI(void)
{
//---first check and initialize main buffer
 if(!InitArrays())
     return(false);
//---initialize window creation     
 if(!CWndCreate::CreateWindow(m_window,"Simulated SP500 Equity Curves",1,1,700,600,true,true,true,false))
     return(false);
//---create text edit for mean coeff 1     
 if(!CWndCreate::CreateTextEdit(m_amean_inc,"Blue Curve: mean coeff",m_window,0,false,7,25,250,70,10,0.01,0.01,2,0.24))
     return(false);
//---create text edit for mean coeff 2     
 if(!CWndCreate::CreateTextEdit(m_bmean_inc,"Green Curve: mean coeff",m_window,0,false,7,50,250,70,10,0.01,0.01,2,0.02))
     return(false);
//---create text edit for stan dev coeff 1     
 if(!CWndCreate::CreateTextEdit(m_asd_inc,"Blue Curve: standard deviation coeff",m_window,0,false,7,75,250,70,10,0.01,0.01,2,2.5))
     return(false);
//---create text edit for stand dev coeff 2
 if(!CWndCreate::CreateTextEdit(m_bsd_inc,"Green Curve: standard deviation coeff",m_window,0,false,7,100,250,70,10,0.01,0.01,2,0.75))
     return(false);  
//---create text edit for seed                
 if(!CWndCreate::CreateTextEdit(m_seed_inc,"Set Seed",m_window,0,false,270,25,280,180,1000000,10,1,0,123))
     return(false);
//---create text edit for initial capital     
 if(!CWndCreate::CreateTextEdit(m_capital_inc,"Initial Capital",m_window,0,false,270,50,280,180,1000000,10,10,0,10000.0))
     return(false);        
//---create combobox for metric selection         
 string items1[]={"sharpe ratio","net profit burke ratio","mean return burke ratio","net profit to max drawdown ratio","omega ratio","upside potential ratio","jensen's alpha"};
   if(!CWndCreate::CreateCombobox(m_metric_type,"RiskReturnMetric: ",m_window,0,false,270,75,280,180,items1,93,1))
      return(false);
//---initialize graph plot
 if(!CreatePlot(2,125))
      return(false);
//---init events      
 CWndEvents::CompletedGUI();
//--- 
 return(true);     
}
//+------------------------------------------------------------------+
//| Draw the equity curves                                           |
//+------------------------------------------------------------------+
bool CApp::CreatePlot(const int x_gap,const int y_gap)
{
//---attach graph object
 m_graph.MainPointer(m_window);
//---set resize properties of graph 
 m_graph.AutoXResizeMode(true);
 m_graph.AutoYResizeMode(true);
 m_graph.AutoXResizeRightOffset(2);
 m_graph.AutoYResizeBottomOffset(23);
//---create graph 
 if(!m_graph.CreateGraph(x_gap,y_gap))
   return(false);
//---set background color  
 CGraphic *graph=m_graph.GetGraphicPointer();
 graph.BackgroundColor(ColorToARGB(clrWhiteSmoke));
 graph.HistoryNameWidth(0);
 graph.HistorySymbolSize(0);
//---fill buffers used in graph plot 
 SetBuffers();
//---set all curves 
 m_curve1=graph.CurveAdd(m_x,m_e1,ColorToARGB(clrBlue),CURVE_LINES,NULL);
 m_curve2=graph.CurveAdd(m_x,m_e2,ColorToARGB(clrGreen),CURVE_LINES,NULL);
 m_curve3=graph.CurveAdd(m_x,m_e,ColorToARGB(clrRed),CURVE_LINES,NULL);
//---set axis properties 
 graph.XAxis().Name("Time Index");
 graph.XAxis().NameSize(20);
 graph.YAxis().Name("Equity");
 graph.YAxis().NameSize(20);
//---set line width for curves 
 m_curve1.LinesWidth(3);
 m_curve2.LinesWidth(3);
 m_curve3.LinesWidth(4);
//---plot the curves 
 graph.CurvePlotAll();
//---add text overlay with calculated performance metrics 
 TextAdd();
//---add graph to window container
 CWndContainer::AddToElementsArray(0,m_graph);
//---
 return true;  
}
//+------------------------------------------------------------------+
//| Fill buffers used in graph plot                                  |
//+------------------------------------------------------------------+
void CApp::SetBuffers(void)
{
//---get value of initial capital from gui control
    double newcap=StringToDouble(m_capital_inc.GetValue());
//---reset initial values for equity curves    
      m_e[0]=m_e1[0]=m_e2[0]=newcap;
//---generate equity curve for benchmark equity curve      
     for(int i = 1; i<ArraySize(m_e);i++)
      {
       m_e[i]=m_e[i-1]*(1+m_r[i-1]);
      }
//---get values for coefficients from gui controls
 double mean1=StringToDouble(m_amean_inc.GetValue()),mean2=StringToDouble(m_bmean_inc.GetValue());
 double sd1=StringToDouble(m_asd_inc.GetValue()),sd2=StringToDouble(m_bsd_inc.GetValue());
//---set seed from gui control
  MathSrand(uint(StringToInteger(m_seed_inc.GetValue())));
//---generate first set of random numbers  
 if(!MathRandomNormal(mean1/double(ArraySize(m_r)),sd1*rstand,ArraySize(m_e),rnorm))
      {
       Print("MathRandomNormal1 failure ",__LINE__," ",GetLastError());
       return;
      }    
//---use random numbers to set returns and equity for first curve   
    for(int i = 1; i<ArraySize(m_e);i++)
     {
      m_r1[i-1]=m_r[i-1]+rnorm[i-1];
      m_e1[i]=m_e1[i-1]*(1+m_r1[i-1]);
     }
//---generate second set of random numbers    
    if(!MathRandomNormal(mean2/double(ArraySize(m_r)),sd2*rstand,ArraySize(m_e),rnorm))
      {
       Print("MathRandomNormal2  failure ",__LINE__," ",GetLastError());
       return;
      }          
//---use random numbers to set returns and equity for second curve
    for(int i = 1; i<ArraySize(m_e);i++)
     {
      m_r2[i-1]=m_r[i-1]+rnorm[i-1];
      m_e2[i]=m_e2[i-1]*(1+m_r2[i-1]);
     }   
//---
}

//+------------------------------------------------------------------+
//| Initialize data buffers                                          |
//+------------------------------------------------------------------+
bool CApp::InitArrays(void)
{
  string strings[];
//---split string values
  int ssize=StringSplit(sp500close,StringGetCharacter("\n",0),strings);
//---resize internal buffer  
  if(ArrayResize(m_close,ssize-2)<ssize-2)
   {
    Print(__FUNCTION__," Resizing error ", GetLastError());
    return false;
   }
//---fill internal buffer with close prices of SP 500   
  for(int i=1;i<ssize-1;i++)
      m_close[i-1]=StringToDouble(strings[i]);  

//---resize internal arrays
    uint NumBars=uint(ssize-2);
//---    
    if(ArrayResize(m_e, int(NumBars)) < int(NumBars)  ||
       ArrayResize(m_e1,int(NumBars)) < int(NumBars)  ||
       ArrayResize(m_e2,int(NumBars)) < int(NumBars)  ||
       ArrayResize(m_r, int(NumBars-1)) < int(NumBars-1)  ||
       ArrayResize(m_r1,int(NumBars-1)) < int(NumBars-1)  ||
       ArrayResize(m_r2,int(NumBars-1)) < int(NumBars-1)  ||
       ArrayResize(m_x ,int(NumBars)) < int(NumBars)  )
    {
     Print("Memory allocation error ", GetLastError());
     return false;;
    }
//---fill benchmark returns and x axis buffers   
    m_x[0]=0.0;
//---    
    for(uint i = 1; i<NumBars;i++)
     {
      m_r[i-1]=(m_close[i]/m_close[i-1])-1.0;
      m_x[i]=double(i);
     }
//---calculate mean and standard deviation of benchmark returns
    rstand=MathStandardDeviation(m_r);
    rmean=MathMean(m_r);
//---    
   return true;
}
//+------------------------------------------------------------------+
//| Recalcualte buffers                                              |
//+------------------------------------------------------------------+
void CApp::RecalculatingSeries(void)
{
//---reset the internal buffers
    SetBuffers();
//---get graph plot
    CGraphic *graph=m_graph.GetGraphicPointer();
    m_curve1=graph.CurveGetByIndex(0);
    m_curve2=graph.CurveGetByIndex(1);
    m_curve3=graph.CurveGetByIndex(2);
//---update graph plot   
    m_curve1.Update(m_x,m_e1);
    m_curve2.Update(m_x,m_e2);
    m_curve3.Update(m_x,m_e);
//---redraw graph plot    
    graph.Redraw(true);
//---redraw and recalculate     
    TextAdd();
//---refresh   
    graph.Update();
//---   
}
//+------------------------------------------------------------------+
//| Text output and calculation of selected performance metric       |
//+------------------------------------------------------------------+
void CApp::TextAdd(void)
{
//---local variables
  double v1,v2,v3,beta;
//---get selected metric from gui control  
  ENUM_METRIC metric = (ENUM_METRIC)m_metric_type.GetListViewPointer().SelectedItemIndex();
//---get raw text representation of metric  
  string mtype = m_metric_type.GetValue();
//---set to upper case  
  StringToUpper(mtype);
//---calculate the selected metric  
  switch(metric)
   {
    case ENUM_SHARPE_RATIO:
     v1=hf_sharpe(m_r1);
     v2=hf_sharpe(m_r2);
     v3=hf_sharpe(m_r);
     break;
   case ENUM_NP_BURKE_RATIO:
     v1=netprofit_burke(m_e1);
     v2=netprofit_burke(m_e2); 
     v3=netprofit_burke(m_e); 
     break;
   case ENUM_M_BURKE_RATIO:
     v1=meanreturns_burke(m_e1);
     v2=meanreturns_burke(m_e2);
     v3=meanreturns_burke(m_e);
     break;
   case ENUM_NPMD_RATIO:
     v1=netProfiMaxDD(m_e1);
     v2=netProfiMaxDD(m_e2); 
     v3=netProfiMaxDD(m_e);
     break;
   case ENUM_OMEGA_RATIO:
     v1=omega(m_r1);
     v2=omega(m_r2); 
     v3=omega(m_r);
     break;
   case ENUM_UPSIDE_RATIO:
     v1=upsidePotentialRatio(m_r1);
     v2=upsidePotentialRatio(m_r2);
     v3=upsidePotentialRatio(m_r); 
     break;
  case ENUM_JENSEN_ALPHA:
     leastsquaresfit(m_r1,m_r,v1,beta);
     leastsquaresfit(m_r2,m_r,v2,beta);
     leastsquaresfit(m_r,m_r,v3,beta);
     break;
  default:
     v1=v2=v3=0.0;              
     break;
   }
//---get pointer for graph plot  
 CGraphic *graph=m_graph.GetGraphicPointer();
//---more local variables for text over lay   
   int  x     = 200;                    // x coordinate
   int  y     = 30;                     // y coordinate
   uint clr   = ColorToARGB(clrBlack); //header text color
//---set text overlay displaying metrics for all curves
  graph.TextAdd(x,y,mtype,clr,TA_CENTER);
  graph.TextAdd(x,y+20,DoubleToString(v1),ColorToARGB(clrBlue),TA_CENTER);
  graph.TextAdd(x,y+40,DoubleToString(v2),ColorToARGB(clrGreen),TA_CENTER);
  graph.TextAdd(x,y+60,DoubleToString(v3),ColorToARGB(clrRed),TA_CENTER);
//---
}

CApp app;
//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit(void)
  {
   ulong tick_counter=::GetTickCount();
//---
   app.OnInitEvent();
//--- 
   if(!app.CreateGUI())
     {
      ::Print(__FUNCTION__," > error");
      return(INIT_FAILED);
     }

   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   app.OnDeinitEvent(reason);
  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick(void)
  {
  }
//+------------------------------------------------------------------+
//| Timer function                                                   |
//+------------------------------------------------------------------+
void OnTimer(void)
  {
   app.OnTimerEvent();
  }
//+------------------------------------------------------------------+
//| Trade function                                                   |
//+------------------------------------------------------------------+
void OnTrade(void)
  {
  }
//+------------------------------------------------------------------+
//| ChartEvent function                                              |
//+------------------------------------------------------------------+
void OnChartEvent(const int    id,
                  const long   &lparam,
                  const double &dparam,
                  const string &sparam)
  {
   app.ChartEvent(id,lparam,dparam,sparam);
  }
//+------------------------------------------------------------------+
