//+------------------------------------------------------------------+
//|                                            PerformanceRatios.mqh |
//|                        Copyright 2023, MetaQuotes Software Corp. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, MetaQuotes Software Corp."
#property link      "https://www.mql5.com"
#include <Math\Stat\Math.mqh>
//+------------------------------------------------------------------+
//|High frequency Sharpe Ratio                                       |
//+------------------------------------------------------------------+
double hf_sharpe(double &rt[])
  {
   double sd=MathStandardDeviation(rt);
   if(sd)
      return MathMean(rt)/sd;
   else
      return 0;
  }
//+------------------------------------------------------------------+
//|Maximum drawdowns function given equity curve                     |
//+------------------------------------------------------------------+
bool MaxNDrawdowns(const int num_drawdowns,double &in_ec[],double &out_dd[])
  {
   ZeroMemory(out_dd);

   ResetLastError();

   if(num_drawdowns<=0)
     {
      Print("Invalid function parameter for num_drawdowns ");
      return false;
     }
   double u[],v[];

   int size = ArraySize(in_ec);

   if((ArrayResize(v,(size*(size-1))/2)< int((size*(size-1))/2))||
      (ArraySize(out_dd)!=num_drawdowns && ArrayResize(out_dd,num_drawdowns)<num_drawdowns))
     {
      Print(__FUNCTION__, " resize error ", GetLastError());
      return false;
     }

   int k=0;
   for(int i=0; i<size-1; i++)
     {
      for(int j=i+1; j<size; j++)
        {
         v[k]=in_ec[i]-in_ec[j];
         k++;
        }
     }

   ArraySort(v);

   for(int i=0; i<k; i++)
     {
      if(v[i]>0)
        {
         if(i)
           {
            if(!ArrayRemove(v,0,i))
              {
               Print(__FUNCTION__, " error , ArrayRemove: ",GetLastError());
               return false;
              }
            else
               break;
           }
         else
            break;
        }
     }

   size=ArraySize(v);

   if(size && size<=num_drawdowns)
     {
      if(ArrayCopy(out_dd,v)<size)
        {
         Print(__FUNCTION__, " error ", GetLastError());
         return false;
        }
      else
         return (true);
     }


   if(ArrayCopy(out_dd,v,0,size-num_drawdowns,num_drawdowns)<num_drawdowns)
     {
      Print(__FUNCTION__, " error ", GetLastError());
      return false;
     }


   return(true);

  }
//+------------------------------------------------------------------+
//|Net profit to maximum drawdown ratio                              |
//+------------------------------------------------------------------+
double netProfiMaxDD(double &in_ec[])
  {
   double outdd[];
   int insize=ArraySize(in_ec);

   if(MaxNDrawdowns(1,in_ec,outdd))
      return ((in_ec[insize-1]-in_ec[0])/outdd[0]);
   else
      return 0;
  }
//+------------------------------------------------------------------+
//|Net profit based Burke ratio                                                       |
//+------------------------------------------------------------------+
double netprofit_burke(double &in_ec[],int n_highestdrawdowns=0)
  {
   double outdd[];
   double sumdd=0;
   int insize=ArraySize(in_ec);

   if(n_highestdrawdowns<=0)
      n_highestdrawdowns=int(insize/20);

   if(MaxNDrawdowns(n_highestdrawdowns,in_ec,outdd))
     {
      for(int i=0; i<ArraySize(outdd); i++)
        {
         sumdd+=(outdd[i]*outdd[i]);
        }
      return (in_ec[insize-1]-in_ec[0])/(MathSqrt((1.0/double(insize)) * sumdd));
     }
   else
      return 0;
  }
//+------------------------------------------------------------------+
//|Mean return based Burke ratio                                     |
//+------------------------------------------------------------------+
double meanreturns_burke(double &in_ec[],int n_highestdrawdowns=0)
  {
   double outdd[];
   double rets[];

   double sumdd=0;
   int insize=ArraySize(in_ec);

   if(ArrayResize(rets,insize-1)<insize-1)
     {
      Print(__FUNCTION__," Memory allocation error ",GetLastError());
      return 0;
     }

   for(int i=1; i<insize; i++)
      rets[i-1] = (in_ec[i]/in_ec[i-1]) - 1.0;

   if(n_highestdrawdowns<=0)
      n_highestdrawdowns=int(insize/20);

   if(MaxNDrawdowns(n_highestdrawdowns,in_ec,outdd))
     {
      for(int i=0; i<ArraySize(outdd); i++)
         sumdd+=(outdd[i]*outdd[i]);
      return MathMean(rets)/(MathSqrt((1.0/double(insize)) * sumdd));
     }
   else
      return 0;
  }
//+------------------------------------------------------------------+
//|Partial Moments                                                   |
//+------------------------------------------------------------------+
double partialmoment(const uint n,double &rt[],double &rtb[],bool upper=false)
  {
   double pm[];
   int insize=ArraySize(rt);

   if(n)
     {
      if(ArrayResize(pm,insize)<insize)
        {
         Print(__FUNCTION__," resize error ", GetLastError());
         return 0;
        }

      for(int i=0; i<insize; i++)
         pm[i] = (!upper)?MathPow(MathMax(rtb[i]-rt[i],0),n):MathPow(MathMax(rt[i]-rtb[i],0),n);
      return MathMean(pm);
     }
   else
     {
      int k=0;
      for(int i=0; i<insize; i++)
        {
         if((!upper && rtb[i]>=rt[i]) || (upper && rt[i]>rtb[i]))
           {
            ArrayResize(pm,k+1,1);
            pm[k]=rt[i];
            ++k;
           }
         else
            continue;
        }

      return MathMean(pm);

     }

  }
//+------------------------------------------------------------------+
//|Upside potential ratio                                            |
//+------------------------------------------------------------------+
double upsidePotentialRatio(double &rt[])
  {
   double rb[];

   if(ArrayResize(rb,ArraySize(rt))<0)
     {
      Print(__FUNCTION__, " Resize error ",GetLastError());
      return 0;
     }

   ArrayInitialize(rb,0.0);

   double pmomentu=MathPow(partialmoment(2,rt,rb,true),0.5);
   double pmomentl=MathPow(partialmoment(2,rt,rb),0.5);
   if(pmomentl)
      return pmomentu/pmomentl;
   else
      return 0;
  }
//+------------------------------------------------------------------+
//|omega ratio                                                       |
//+------------------------------------------------------------------+
double omega(double &rt[])
  {
   double rb[];

   if(ArrayResize(rb,ArraySize(rt))<0)
     {
      Print(__FUNCTION__, " Resize error ",GetLastError());
      return 0;
     }

   ArrayInitialize(rb,0.0);

   double pmomentl=MathPow(partialmoment(2,rt,rb),0.5);

   if(pmomentl)
      return MathMean(rt)/pmomentl;
   else
      return 0;
  }
//+------------------------------------------------------------------+
//|linear model using least squares fit y=a+bx                       |
//+------------------------------------------------------------------+
double leastsquaresfit(double &y[],double &x[], double &alpha,double &beta)
  {
   double esquared=0;

   int ysize=ArraySize(y);
   int xsize=ArraySize(x);

   double sumx=0,sumy=0,sumx2=0,sumxy=0;

   int insize=MathMin(ysize,xsize);

   for(int i=0; i<insize; i++)
     {
      sumx+=x[i];
      sumx2+=x[i]*x[i];
      sumy+=y[i];
      sumxy+=x[i]*y[i];
     }

   beta=((insize*sumxy)-(sumx*sumy))/((insize*sumx2)-(sumx*sumx));
   alpha=(sumy-(beta*sumx))/insize;

   double pred,error;

   for(int i=0; i<insize; i++)
     {
      pred=alpha+(beta*x[i]);
      error=pred-y[i];
      esquared+=(error*error);
     }

   return esquared;

  }
//+------------------------------------------------------------------+
//| Perfect Profit score                                             |
//+------------------------------------------------------------------+
double perfectprofitscore(double &ec[],double initial_capital)
  {
   double pps=0;

   if(initial_capital<=0)
     {
      Print(__FUNCTION__, "Invalid function parameter | initial_captital ",initial_capital);
      return pps;
     }

   double y[];
   int insize=ArraySize(ec);

   if(ArrayResize(y,insize)<insize)
     {
      Print(__FUNCTION__, " resize error ",GetLastError());
      return pps;
     }

   for(int i = 0; i<insize; i++)
      y[i]=ec[i]/initial_capital;

   double a,b;
   double e2=leastsquaresfit(y,ec,a,b);

   if(!e2)
      return pps;
   else
      return ((ec[insize-1]-initial_capital)/initial_capital)*e2;

  }
//+------------------------------------------------------------------+
//| profit factor                                                    |
//+------------------------------------------------------------------+
double profit_factor(double &returns[])
  {
   int size = ArraySize(returns);

   if(size<=0)
      return 0;

   double neg=0,pos=0;
   for(int i=0; i<size; i++)
     {
      if(returns[i]>0.0)
         pos+=returns[i];
      else
         if(returns[i]<0.0)
            neg-=returns[i];
     }

   if(!neg)
      return 0;
   else
      return pos/neg;


  }





//+------------------------------------------------------------------+
