/*
    GetPaneHeights: get heights in pixel of all panes in a window

    FOR WHAT:

        With the use of the built-in functions "WindowPriceMin/Max()"
        together, you can calculate appropriate spacing/margin to
        text/arrow objects (OBJ_TEXT/OBJ_ARROW), whatever size the
        window is.  See the EXAMPLE section below.

    SYNOPSIS:

        #import "GetPaneHeights.ex4"
            int GetPaneHeights(int &height[]);
        #import

        height[i] = the height in pixel of pane "i"
            i = 0: main chart
            i = 1: indicator #1
            i = 2: indicator #2
            ... (in top-to-bottom order)

        The size of "height[]" must be large enough to contain all panes.
        This function returns the number of panes.
        If it fails, the return value is zero.

    NOTE:

        This function scans the device context (DC) of a window,
        so if (the leftmost piece of) the window is covered by another
        window, then this function will fail.

    EXAMPLE:

        #import "GetPaneHeights.ex4"
            int GetPaneHeights(int &height[]);
        #import

        int         i, panes, height[10];
        double      spacing;
        string      sign;

        panes = GetPaneHeights(height);
        for (i = 0; i < panes; i++)
            Print("height[", i, "] = ", height[i], " pixels");

        // Analyzing the chart... Hmm... it's time to sell.

        if (panes == 0) {
            Alert("WARNING: cannot get the pane height!");
            height[0] = 500;
        }
        sign = "sell sign";
        spacing = 5 * (WindowPriceMax(0) - WindowPriceMin(0)) / height[0];
        ObjectCreate(sign, OBJ_ARROW, 0, Time[0], High[0] + spacing);
        ObjectSet(sign, OBJPROP_ARROWCODE, ...);
        ObjectSet(sign, OBJPROP_COLOR, ...);

        // The sell sign will be located 5 pixels above the highest price
        // of the latest bar, whatever size the window is.

    Version 1.01, Copyright (c) 2009, kartz

    (Change Log)
        1.00 -> 1.01: fix a bug of recognizing the date-and-time row as a pane
                      on a machine which doesn't use font anti-aliasing.
*/

#property copyright     "Copyright (c) 2009, kartz"
#property library

#import "user32.dll"
    int     GetDC(int hwnd);                            // HDC
    int     ReleaseDC(int hwnd, int hdc);
    int     GetClientRect(int hwnd, int &rect[4]);      // BOOL
#import "gdi32.dll"
    int     GetPixel(int hdc, int x, int y);            // COLORREF
#import

#define CLR_INVALID     0xFFFFFFFF      // Windows' constant
#define TEST_XY_COORD   10

int GetPaneHeights(int &height[])
{

    int     i, x, y, h, c, cc, frame, hwnd, hdc, rect[4];

    hwnd = WindowHandle(Symbol(), Period());
    if (hwnd == 0)
        return(0);
    if (!GetClientRect(hwnd, rect))
        return(0);
    h = rect[3];
    hdc = GetDC(hwnd);
    if (hdc == 0)
        return(0);

    i = 0;
    c = GetPixel(hdc, 0, TEST_XY_COORD);
    if (c != CLR_INVALID) {
        // scan horizontally.
        for (x = 1; x <= TEST_XY_COORD; x++) {
            frame = GetPixel(hdc, x, TEST_XY_COORD);
            if ((frame == CLR_INVALID) || (frame != c))
                break;
        }
        // "x" must be 2, but scanned just in case.
        if (frame != CLR_INVALID) {
            // scan vertically along the frame.
            for (cc = CLR_INVALID, y = 0; y < h; y++, cc = c) {
                c = GetPixel(hdc, x, y);
                if (c == CLR_INVALID) {
                    i = 0;
                    break;
                }
                if (c == frame) {
                    if (cc != frame)
                        height[i] = y;
                } else if (cc == frame) {
                    height[i] = y - height[i] - 1;
                    i++;
                    if (i == WindowsTotal())
                        break;
                }
            }
        }
    }
    ReleaseDC(hwnd, hdc);
    return(i);
}
