//+------------------------------------------------------------------+
//|                                                   help_plots.mqh |
//|                         Copyright 2026, Paul Carissimo Paschini. |
//|                     https://www.mql5.com/en/users/carissimopaul/ |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| CPlotManager: Automates Buffer and Plot Index Management         |
//+------------------------------------------------------------------+
class CPlotManager
  {
private:
   int               m_buf_idx;  // Current Buffer Index
   int               m_plot_idx; // Current Plot Index

public:
                     CPlotManager(void) : m_buf_idx(0), m_plot_idx(0) {}

   //--- 1. Calculation Buffer (Hidden, no plot)
   void              AddCalc(double &data_buffer[])
     {
      SetIndexBuffer(m_buf_idx, data_buffer, INDICATOR_CALCULATIONS);
      m_buf_idx++;
     }

   //--- 2. Standard Line (Returns Plot Index)
   int               AddLine(double &data_buffer[], string label, color clr, int width = 1, ENUM_LINE_STYLE style = STYLE_SOLID)
     {
      SetIndexBuffer(m_buf_idx, data_buffer, INDICATOR_DATA);

      int current_plot = m_plot_idx;

      PlotIndexSetInteger(m_plot_idx, PLOT_DRAW_TYPE, DRAW_LINE);
      PlotIndexSetString(m_plot_idx, PLOT_LABEL, label);
      PlotIndexSetInteger(m_plot_idx, PLOT_LINE_COLOR, clr);
      PlotIndexSetInteger(m_plot_idx, PLOT_LINE_WIDTH, width);
      PlotIndexSetInteger(m_plot_idx, PLOT_LINE_STYLE, style);

      m_buf_idx++;
      m_plot_idx++;

      return current_plot;
     }

   //--- 3. Color Line (Returns Plot Index)
   int               AddColorLine(double &data_buffer[], double &color_buffer[], string label, int width = 1, ENUM_LINE_STYLE style = STYLE_SOLID)
     {
      SetIndexBuffer(m_buf_idx, data_buffer, INDICATOR_DATA);
      SetIndexBuffer(m_buf_idx + 1, color_buffer, INDICATOR_COLOR_INDEX);

      int current_plot = m_plot_idx;

      PlotIndexSetInteger(m_plot_idx, PLOT_DRAW_TYPE, DRAW_COLOR_LINE);
      PlotIndexSetString(m_plot_idx, PLOT_LABEL, label);
      PlotIndexSetInteger(m_plot_idx, PLOT_LINE_WIDTH, width);
      PlotIndexSetInteger(m_plot_idx, PLOT_LINE_STYLE, style);

      m_buf_idx += 2;
      m_plot_idx++;

      return current_plot;
     }

   //--- 4. Arrows (Returns Plot Index)
   int               AddArrow(double &data_buffer[], string label, uchar arrow_code, int shift_px, color clr, int width = 1)
     {
      SetIndexBuffer(m_buf_idx, data_buffer, INDICATOR_DATA);

      int current_plot = m_plot_idx;

      PlotIndexSetInteger(m_plot_idx, PLOT_DRAW_TYPE, DRAW_ARROW);
      PlotIndexSetString(m_plot_idx, PLOT_LABEL, label);
      PlotIndexSetInteger(m_plot_idx, PLOT_ARROW, arrow_code);
      PlotIndexSetInteger(m_plot_idx, PLOT_ARROW_SHIFT, shift_px);
      PlotIndexSetInteger(m_plot_idx, PLOT_LINE_COLOR, clr);
      PlotIndexSetInteger(m_plot_idx, PLOT_LINE_WIDTH, width);

      m_buf_idx++;
      m_plot_idx++;

      return current_plot;
     }

   //--- 5. Color Candles (Returns Plot Index)
   int               AddColorCandles(double &O[], double &H[], double &L[], double &C[], double &ClrBuf[], string label)
     {
      SetIndexBuffer(m_buf_idx, O, INDICATOR_DATA);
      SetIndexBuffer(m_buf_idx + 1, H, INDICATOR_DATA);
      SetIndexBuffer(m_buf_idx + 2, L, INDICATOR_DATA);
      SetIndexBuffer(m_buf_idx + 3, C, INDICATOR_DATA);
      SetIndexBuffer(m_buf_idx + 4, ClrBuf, INDICATOR_COLOR_INDEX);

      int current_plot = m_plot_idx;

      PlotIndexSetInteger(m_plot_idx, PLOT_DRAW_TYPE, DRAW_COLOR_CANDLES);
      PlotIndexSetString(m_plot_idx, PLOT_LABEL, label);

      m_buf_idx += 5;
      m_plot_idx++;

      return current_plot;
     }

   //--- 6. Filling (Returns Plot Index)
   int               AddFilling(double &top_buf[], double &bot_buf[], string label, color clr)
     {
      SetIndexBuffer(m_buf_idx, top_buf, INDICATOR_DATA);
      SetIndexBuffer(m_buf_idx + 1, bot_buf, INDICATOR_DATA);

      int current_plot = m_plot_idx;

      PlotIndexSetInteger(m_plot_idx, PLOT_DRAW_TYPE, DRAW_FILLING);
      PlotIndexSetString(m_plot_idx, PLOT_LABEL, label);
      PlotIndexSetInteger(m_plot_idx, PLOT_LINE_COLOR, clr);

      m_buf_idx += 2;
      m_plot_idx++;

      return current_plot;
     }

   // Return total buffers used so far
   int               GetBuffers()
     {
      return m_buf_idx;
     }

   // Return total plots used so far
   int               GetPlots()
     {
      return m_plot_idx;
     }

   // Automatic verification
   void              SanityCheck(int declared_buffers, int declared_plots)
     {
      if(m_buf_idx != declared_buffers || m_plot_idx != declared_plots)
        {
         Print("WARNING");
         PrintFormat("Buffers: Used %d / Declared %d", m_buf_idx, declared_buffers);
         PrintFormat("Plots:   Used %d / Declared %d", m_plot_idx, declared_plots);

         if(m_buf_idx > declared_buffers)
            Print("You are using more BUFFERS than declared in #property!");
         if(m_plot_idx > declared_plots)
            Print("You are using more PLOTS than declared in #property!");
         if(m_buf_idx < declared_buffers)
            Print("You are using less BUFFERS than declared in #property!");
         if(m_plot_idx < declared_plots)
            Print("You are using less PLOTS than declared in #property!");
        }
      else
        {
         PrintFormat("Buffers OK: %d Buffers, %d Plots.", m_buf_idx, m_plot_idx);
        }
     }
  };

//+------------------------------------------------------------------+
