//+------------------------------------------------------------------+
//|                                                    CTsLogger.mqh |
//|                                         Copyright 2025, Trefoter |
//|                             https://www.mql5.com/ru/users/useral |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, Trefoter"
#property link      "https://www.mql5.com/ru/users/useral"
#property strict

/* CTsLogger -     , 
        MQL5.
  CTsLogger -   
        
     .
 ,      
   ""   ,
     . */



//+------------------------------------------------------------------+
//|                                 |
//+------------------------------------------------------------------+

//   
enum ENUM_LOG_LEVEL
{
   LOG_LEVEL_ERROR,           //  
   LOG_LEVEL_WARNING,         //   
   LOG_LEVEL_INFO,            //  
   LOG_LEVEL_DEBUG            //  
};

//+------------------------------------------------------------------+
//|                                    |
//+------------------------------------------------------------------+
class CTsLogger
{
private:
   //     
   ENUM_LOG_LEVEL  m_globalLogLevel;     //   
   string          m_logFileName;        //    
   int             m_fileHandle;         //   
   bool            m_logToTerminal;      //    
   bool            m_initialized;        //   
   string          m_debugModules[];     //     
   bool            m_debugModeAll;       //    
   bool            m_debugModulesPaused; //    
   
   //   
   bool     OpenLogFile();                //   
   void     CloseLogFile();               //   
   void     WriteLog(string level, string moduleId, string message); //   
   string   FormatLogMessage(string level, string moduleId, string message); // 
   bool     IsModuleInDebugMode(string moduleId); //     
   int      FindModuleIndex(string moduleId);    //    
   string   GetParentModule(string moduleId);    //   
   bool     IsModuleOrParentInDebugMode(string moduleId); //    
   bool     IsChildModule(string potentialChild, string parentModule); //    
   
public:
   //   
   CTsLogger();
   ~CTsLogger();
   
   //  
   bool     Initialize(string logFileName, bool logToTerminal = true);
   void     SetGlobalLogLevel(ENUM_LOG_LEVEL level);
   
   //  
   void     Debug(string moduleId, string message);
   void     Info(string moduleId, string message);
   void     Warning(string moduleId, string message);
   void     Error(string moduleId, string message);
   
   //   
   void     EnableDebugMode(string moduleId);
   void     DisableDebugMode(string moduleId);
   bool     IsInDebugMode(string moduleId);
   void     EnableDebugModeAll();
   void     DisableDebugModeAll();
   
   //    
   void     PauseDebugMode();              //   
   void     ResumeDebugMode();             //   
   bool     IsDebugModePaused();           //   
   void     ResetDebugModules();           //    
   bool     HasChildDebugModules(string parentModule); //   
};

//+------------------------------------------------------------------+
//|                                                       |
//+------------------------------------------------------------------+
CTsLogger::CTsLogger()
{
   m_globalLogLevel = LOG_LEVEL_INFO;
   m_logFileName = "";
   m_fileHandle = INVALID_HANDLE;
   m_logToTerminal = true;
   m_initialized = false;
   ArrayResize(m_debugModules, 0);
   m_debugModeAll = false;
   m_debugModulesPaused = false;
}

//+------------------------------------------------------------------+
//|                                                        |
//+------------------------------------------------------------------+
CTsLogger::~CTsLogger()
{
   if(m_fileHandle != INVALID_HANDLE)
   {
      CloseLogFile();
   }
}

//+------------------------------------------------------------------+
//|                                              |
//+------------------------------------------------------------------+
bool CTsLogger::Initialize(string logFileName, bool logToTerminal = true)
{
   //    ,   
   if(m_initialized && m_fileHandle != INVALID_HANDLE)
   {
      CloseLogFile();
   }
   
   //   
   m_logFileName = logFileName;
   m_logToTerminal = logToTerminal;
   
   //   ,    
   if(m_logFileName != "")
   {
      if(!OpenLogFile())
      {
         Print(":      (", GetLastError(), "): ", m_logFileName);
         //  ,      
         // (    )
      }
   }
   
   m_initialized = true;
   
   //    
   string initMessage = "TsLogger .  : " + 
                       EnumToString(m_globalLogLevel) + 
                       ",   : " + (m_logToTerminal ? "" : "") +
                       ", : " + (m_logFileName != "" ? m_logFileName : "");
   
   Info("TsLogger", initMessage);
   
   return true;
}

//+------------------------------------------------------------------+
//|                                                 |
//+------------------------------------------------------------------+
bool CTsLogger::OpenLogFile()
{
   //       
   m_fileHandle = FileOpen(m_logFileName, FILE_WRITE|FILE_READ|FILE_TXT|FILE_ANSI, '\t');
   
   if(m_fileHandle == INVALID_HANDLE)
   {
      Print(":      (", GetLastError(), "): ", m_logFileName);
      return false;
   }
   
   //        
   FileSeek(m_fileHandle, 0, SEEK_END);
   
   return true;
}

//+------------------------------------------------------------------+
//|                                                 |
//+------------------------------------------------------------------+
void CTsLogger::CloseLogFile()
{
   if(m_fileHandle != INVALID_HANDLE)
   {
      FileClose(m_fileHandle);
      m_fileHandle = INVALID_HANDLE;
   }
}

//+------------------------------------------------------------------+
//|                             |
//+------------------------------------------------------------------+
void CTsLogger::SetGlobalLogLevel(ENUM_LOG_LEVEL level)
{
   m_globalLogLevel = level;
   
   if(m_initialized)
   {
      Info("TsLogger", "   : " + EnumToString(level));
   }
}

//+------------------------------------------------------------------+
//|                                       |
//+------------------------------------------------------------------+
string CTsLogger::FormatLogMessage(string level, string moduleId, string message)
{
   string timestamp = TimeToString(TimeCurrent(), TIME_DATE|TIME_MINUTES|TIME_SECONDS);
   return timestamp + " [" + level + "] [" + moduleId + "] " + message;
}

//+------------------------------------------------------------------+
//|                                                        |
//+------------------------------------------------------------------+
void CTsLogger::WriteLog(string level, string moduleId, string message)
{
   string formattedMessage = FormatLogMessage(level, moduleId, message);
   
   //   ,  
   if(m_logToTerminal)
   {
      Print(formattedMessage);
   }
   
   //   ,  
   if(m_fileHandle != INVALID_HANDLE)
   {
      FileWrite(m_fileHandle, formattedMessage);
      //         
      FileFlush(m_fileHandle); //    
   }
}

//+------------------------------------------------------------------+
//|        |
//+------------------------------------------------------------------+
string CTsLogger::GetParentModule(string moduleId)
{
   int dotPos = StringFind(moduleId, ".");
   if(dotPos > 0)
      return StringSubstr(moduleId, 0, dotPos);
   
   return moduleId; //   ID,   
}

//+------------------------------------------------------------------+
//| ,                |
//+------------------------------------------------------------------+
bool CTsLogger::IsChildModule(string potentialChild, string parentModule)
{
   // ,        + "."
   if(StringFind(potentialChild, parentModule + ".") == 0)
      return true;
   
   return false;
}

//+------------------------------------------------------------------+
//| ,            |
//+------------------------------------------------------------------+
bool CTsLogger::IsModuleOrParentInDebugMode(string moduleId)
{
   int size = ArraySize(m_debugModules);
   
   //   
   for(int i = 0; i < size; i++)
   {
      if(m_debugModules[i] == moduleId)
         return true;
   }
   
   // ,   -      
   //    (,  moduleId = "TradeModule.OrderExecution",
   //  ,      "TradeModule")
   for(int i = 0; i < size; i++)
   {
      //          
      if(IsChildModule(moduleId, m_debugModules[i]))
         return true;
   }
   
   return false;
}

//+------------------------------------------------------------------+
//| ,                         |
//+------------------------------------------------------------------+
bool CTsLogger::IsModuleInDebugMode(string moduleId)
{
   //      -  true
   if(m_debugModeAll)
      return true;
   
   //      -  false
   if(m_debugModulesPaused)
      return false;
   
   //       
   return IsModuleOrParentInDebugMode(moduleId);
}

//+------------------------------------------------------------------+
//|                              |
//+------------------------------------------------------------------+
int CTsLogger::FindModuleIndex(string moduleId)
{
   int size = ArraySize(m_debugModules);
   
   for(int i = 0; i < size; i++)
   {
      if(m_debugModules[i] == moduleId)
         return i;
   }
   
   return -1;
}

//+------------------------------------------------------------------+
//|                         |
//+------------------------------------------------------------------+
void CTsLogger::EnableDebugMode(string moduleId)
{
   // ,      
   if(FindModuleIndex(moduleId) >= 0)
      return;
   
   //     
   int size = ArraySize(m_debugModules);
   ArrayResize(m_debugModules, size + 1);
   m_debugModules[size] = moduleId;
   
   //      ,  
   if(m_debugModulesPaused)
      m_debugModulesPaused = false;
   
   Info("TsLogger", "    : " + moduleId);
}

//+------------------------------------------------------------------+
//|                        |
//+------------------------------------------------------------------+
void CTsLogger::DisableDebugMode(string moduleId)
{
   bool moduleFound = false;
   int size = ArraySize(m_debugModules);
   
   //       
   string tempModules[];
   int newSize = 0;
   
   //       ,   
   //      
   for(int i = 0; i < size; i++)
   {
      //            
      if(m_debugModules[i] != moduleId && !IsChildModule(m_debugModules[i], moduleId))
      {
         ArrayResize(tempModules, newSize + 1);
         tempModules[newSize++] = m_debugModules[i];
      }
      else
      {
         moduleFound = true;
      }
   }
   
   //      
   if(moduleFound)
   {
      //     
      ArrayResize(m_debugModules, newSize);
      for(int i = 0; i < newSize; i++)
      {
         m_debugModules[i] = tempModules[i];
      }
      
      Info("TsLogger", "       : " + moduleId);
   }
}

//+------------------------------------------------------------------+
//| ,                |
//+------------------------------------------------------------------+
bool CTsLogger::IsInDebugMode(string moduleId)
{
   return IsModuleInDebugMode(moduleId);
}

//+------------------------------------------------------------------+
//|                              |
//+------------------------------------------------------------------+
void CTsLogger::EnableDebugModeAll()
{
   m_debugModeAll = true;
   //      ,  
   if(m_debugModulesPaused)
      m_debugModulesPaused = false;
       
   Info("TsLogger", "      ");
}

//+------------------------------------------------------------------+
//|                             |
//+------------------------------------------------------------------+
void CTsLogger::DisableDebugModeAll()
{
   m_debugModeAll = false;
   m_debugModulesPaused = true; //   
   Info("TsLogger", "    ( )");
}

//+------------------------------------------------------------------+
//|                           |
//+------------------------------------------------------------------+
void CTsLogger::PauseDebugMode()
{
   if(!m_debugModulesPaused && ArraySize(m_debugModules) > 0)
   {
      m_debugModulesPaused = true;
      Info("TsLogger", "      ");
   }
}

//+------------------------------------------------------------------+
//|               |
//+------------------------------------------------------------------+
void CTsLogger::ResumeDebugMode()
{
   if(m_debugModulesPaused)
   {
      m_debugModulesPaused = false;
      Info("TsLogger", "    " + IntegerToString(ArraySize(m_debugModules)) + " ");
   }
}

//+------------------------------------------------------------------+
//| ,                          |
//+------------------------------------------------------------------+
bool CTsLogger::IsDebugModePaused()
{
   return m_debugModulesPaused;
}

//+------------------------------------------------------------------+
//|                               |
//+------------------------------------------------------------------+
void CTsLogger::ResetDebugModules()
{
   int prevSize = ArraySize(m_debugModules);
   ArrayResize(m_debugModules, 0);
   m_debugModulesPaused = false;
   if(prevSize > 0)
      Info("TsLogger", "     (" + IntegerToString(prevSize) + " .)");
}

//+------------------------------------------------------------------+
//|                      |
//+------------------------------------------------------------------+
bool CTsLogger::HasChildDebugModules(string parentModule)
{
   int size = ArraySize(m_debugModules);
   
   for(int i = 0; i < size; i++)
   {
      if(IsChildModule(m_debugModules[i], parentModule))
         return true;
   }
   
   return false;
}

//+------------------------------------------------------------------+
//|     DEBUG                            |
//+------------------------------------------------------------------+
void CTsLogger::Debug(string moduleId, string message)
{
   // ,   
   if(!m_initialized)
   {
      Print(": TsLogger  . : [DEBUG] [", moduleId, "] ", message);
      return;
   }
   
   //   ,   DEBUG     
   if(m_globalLogLevel >= LOG_LEVEL_DEBUG || IsModuleInDebugMode(moduleId))
   {
      WriteLog("DEBUG", moduleId, message);
   }
}

//+------------------------------------------------------------------+
//|     INFO                             |
//+------------------------------------------------------------------+
void CTsLogger::Info(string moduleId, string message)
{
   // ,   
   if(!m_initialized)
   {
      Print(": TsLogger  . : [INFO] [", moduleId, "] ", message);
      return;
   }
   
   //   ,   INFO  
   if(m_globalLogLevel >= LOG_LEVEL_INFO)
   {
      WriteLog("INFO", moduleId, message);
   }
}

//+------------------------------------------------------------------+
//|     WARNING                          |
//+------------------------------------------------------------------+
void CTsLogger::Warning(string moduleId, string message)
{
   // ,   
   if(!m_initialized)
   {
      Print(": TsLogger  . : [WARNING] [", moduleId, "] ", message);
      return;
   }
   
   //  ,   WARNING  
   if(m_globalLogLevel >= LOG_LEVEL_WARNING)
   {
      WriteLog("WARNING", moduleId, message);
   }
}

//+------------------------------------------------------------------+
//|     ERROR                            |
//+------------------------------------------------------------------+
void CTsLogger::Error(string moduleId, string message)
{
   // ,   
   if(!m_initialized)
   {
      Print(": TsLogger  . : [ERROR] [", moduleId, "] ", message);
      return;
   }
   
   //   ,    
   WriteLog("ERROR", moduleId, message);
}

