﻿// GoldRat Order Import
#property copyright "© GoldRat 2016"
#property link "https://www.mql5.com/en/users/goldrat"
#property description "Import trades from file"
#property version "1.1"
#property strict

input int Magic = 20000;
input string Symbol = "";

#include <GRat_OrderImport.mqh>

int Deviation = 50;
ENUM_ORDER_TYPE_FILLING FillingMode = ORDER_FILLING_RETURN;

bool Started;


int OnInit()
{
	Started = false;
	EventSetMillisecondTimer(300);
	return (INIT_SUCCEEDED);
}


void OnDeinit(const int reason)
{
	EventKillTimer();
}


void OnTimer()
{
	MqlTradeRequest aReq[];
	OrderImport(aReq, Magic, Symbol == "" ? _Symbol : Symbol);
	for (int i = 0; i < ArraySize(aReq); i++)
	{
		aReq[i].symbol = _Symbol;
		aReq[i].volume = MathAbs(NormalizeLots(aReq[i].volume));
		aReq[i].price = NormalizeDouble(aReq[i].price, _Digits);
		aReq[i].stoplimit = NormalizeDouble(aReq[i].stoplimit, _Digits);
		aReq[i].sl = NormalizeDouble(aReq[i].sl, _Digits);
		aReq[i].tp = NormalizeDouble(aReq[i].tp, _Digits);
		aReq[i].deviation = Deviation;
		aReq[i].type_filling = GetFillingMode(FillingMode);
      aReq[i].type_time = GetTypeTime();
		MqlTradeResult res = {};
		bool ok = false;
		if (aReq[i].action == TRADE_ACTION_DEAL)
		{
			if (aReq[i].position > 0)
			{
				// Close position
				ENUM_POSITION_TYPE type = (ENUM_ORDER_TYPE)aReq[i].type == ORDER_TYPE_BUY ? POSITION_TYPE_SELL : POSITION_TYPE_BUY;
				if (SelectPos(type, aReq[i].volume, aReq[i].sl, aReq[i].tp))
				{
					aReq[i].position = PositionGetInteger(POSITION_TICKET);
					if (aReq[i].volume > PositionGetDouble(POSITION_VOLUME)) aReq[i].volume = PositionGetDouble(POSITION_VOLUME);
					aReq[i].sl = 0;
					aReq[i].tp = 0;
					ok = true;
				}
			}
			else
			{
				// Open position
				ok = true;
			}
			if (ok)
			{
				if ((ENUM_ORDER_TYPE)aReq[i].type == ORDER_TYPE_BUY)
				{
					aReq[i].price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
				}
				else
				{
					aReq[i].price = SymbolInfoDouble(_Symbol, SYMBOL_BID);
				}
			}
		}
		else if (aReq[i].action == TRADE_ACTION_SLTP)
		{
			// Modify position
			ENUM_POSITION_TYPE type = (ENUM_ORDER_TYPE)aReq[i].type == ORDER_TYPE_BUY ? POSITION_TYPE_BUY : POSITION_TYPE_SELL;
			if (SelectPos(type, aReq[i].volume, -1, -1))
			{
				aReq[i].position = PositionGetInteger(POSITION_TICKET);
				ok = true;
			}
		}
		else if (aReq[i].action == TRADE_ACTION_REMOVE)
		{
			// Delete order
			if (SelectOrd(aReq[i].type, aReq[i].volume, aReq[i].price, aReq[i].sl, aReq[i].tp))
			{
				aReq[i].order = OrderGetInteger(ORDER_TICKET);
				ok = true;
			}
		}
		else if (aReq[i].action == TRADE_ACTION_MODIFY)
		{
			// Modify order
			if (SelectOrd(aReq[i].type, aReq[i].volume, -1, -1, -1))
			{
				aReq[i].order = OrderGetInteger(ORDER_TICKET);
				ok = true;
			}
		}
		else  if (aReq[i].action == TRADE_ACTION_PENDING)
		{
			// Open order
			ok = true;
		}
		if (ok)
		{
			if (!OrderSend(aReq[i], res))
			{
				Print(EnumToString(aReq[i].action) + " error #" + IntegerToString(_LastError));
			}
		}
	}
}


bool SelectPos(ENUM_POSITION_TYPE type, double volume, double sl, double tp)
{
	bool ret = false;
	ulong ticket = 0;
	double vol = 0;
	for (int i = 0; i < PositionsTotal(); i++)
	{
		string symbol = PositionGetSymbol(i);
		if (symbol == "" || symbol == NULL) continue;
		if ((Magic <= 0 || PositionGetInteger(POSITION_MAGIC) == Magic) && symbol == _Symbol && (ENUM_POSITION_TYPE)PositionGetInteger(POSITION_TYPE) == type)		
		{
			if ((sl < 0 || PositionGetDouble(POSITION_SL) == sl) && (tp < 0 || PositionGetDouble(POSITION_TP) == tp))
			{
				if (volume == 0 || volume == PositionGetDouble(POSITION_VOLUME))
				{
					ret = true;
					break;
				}
				else if (PositionGetDouble(POSITION_VOLUME) > vol)
				{
					vol = PositionGetDouble(POSITION_VOLUME);
					ticket = PositionGetInteger(POSITION_TICKET);
				}
			}
		}
	}
	if (!ret && ticket > 0)
	{
		ret = PositionSelectByTicket(ticket);
	}
	return (ret);
}


bool SelectOrd(ENUM_ORDER_TYPE type, double volume, double price, double sl, double tp)
{
	bool ret = false;
	for (int i = 0; i < OrdersTotal(); i++)
	{
		ulong ticket = OrderGetTicket(i);
		if (ticket == 0) continue;
		string symbol = OrderGetString(ORDER_SYMBOL);
		if ((Magic <= 0 || OrderGetInteger(ORDER_MAGIC) == Magic) && symbol == _Symbol && (ENUM_ORDER_TYPE)OrderGetInteger(ORDER_TYPE) == type && OrderGetDouble(ORDER_VOLUME_INITIAL) == volume)
		{
			if ((price < 0 || OrderGetDouble(ORDER_PRICE_OPEN) == price) && (sl < 0 || OrderGetDouble(ORDER_SL) == sl) && (tp < 0 || OrderGetDouble(ORDER_TP) == tp))
			{
				ret = true;
				break;
			}
		}
	}
	return (ret);
}


ENUM_ORDER_TYPE_FILLING GetFillingMode(ENUM_ORDER_TYPE_FILLING type = ORDER_FILLING_FOK)
{
	ENUM_SYMBOL_TRADE_EXECUTION exeMode = (ENUM_SYMBOL_TRADE_EXECUTION)SymbolInfoInteger(_Symbol, SYMBOL_TRADE_EXEMODE);
	ENUM_ORDER_TYPE_FILLING fillingMode = (ENUM_ORDER_TYPE_FILLING)SymbolInfoInteger(_Symbol, SYMBOL_FILLING_MODE);
	if (fillingMode == 0 || type >= ORDER_FILLING_RETURN || (fillingMode & (type + 1)) != type + 1)
	{
		if (exeMode == SYMBOL_TRADE_EXECUTION_EXCHANGE || exeMode == SYMBOL_TRADE_EXECUTION_INSTANT)
		{
			return ORDER_FILLING_RETURN ;
		}
		else
		{
			if (fillingMode == SYMBOL_FILLING_IOC)
			{
				return ORDER_FILLING_IOC;
			}
			else
			{
				return ORDER_FILLING_FOK;
			}
		}
	}
	else
	{
		return type;
	}
}


ENUM_ORDER_TYPE_TIME GetTypeTime()
{
   ENUM_ORDER_TYPE_TIME ret = ORDER_TIME_GTC;
   int exp = (int)SymbolInfoInteger(_Symbol, SYMBOL_EXPIRATION_MODE);
   if ((exp & SYMBOL_EXPIRATION_GTC) != SYMBOL_EXPIRATION_GTC)
   {
      ret = ORDER_TIME_DAY;
   }
	return (ret);
}


double NormalizeLots(double lot)
{
	if (lot <= 0.0)
		return (0.0);
	double min = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MIN);
	double max = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_MAX);
	double step = SymbolInfoDouble(_Symbol, SYMBOL_VOLUME_STEP);
	double _lot = lot < min ? min : (lot > max ? max : lot);
	_lot = MathRound(_lot / step) * step;
	return (_lot);
}