//+------------------------------------------------------------------+
//|                                                    Functions.mqh |
//|                                    2019-2021, dimitri pecheritsa |
//|                                         mql5.com/en/users/dmipec |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| functions for the selection sort example                         |
//+------------------------------------------------------------------+
//| 10 | deals load (deals[], from, to)                              |
//| 20 | deals key symbol (deals[], keys[])                          |
//| 30 | deals print (deals[])                                       |
//| 40 | array sort (keys[], items[], algorithm, order)              |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| 10 | deals load                                                  |
//+------------------------------------------------------------------+
//|  [out] tickets: destination array                                |
//|  [in] from, to: range of dates. by default loads all deals       |
//|available in the account                                          |
//|  [ret] number of copied deals in the specified date range. -1 is |
//|the history select error                                          |
//+------------------------------------------------------------------+
int DealsLoad(ulong& dst_tickets[],datetime from=0,datetime to=0)
  {
   to=(to==0)? TimeCurrent() : to;
   if(!HistorySelect(from,to))
      return -1;
   int total=HistoryDealsTotal(),count=0;
   for(int i=0; i<total; i++)
     {
      ulong iticket=HistoryDealGetTicket(i);
      if(!iticket)
         continue;
      int size=ArraySize(dst_tickets);
      ArrayResize(dst_tickets,size+1,size*2);
      dst_tickets[size]=iticket;
      count++;
     }
   return count;
  }
//+------------------------------------------------------------------+
//| 20 | deals key symbol                                            |
//+------------------------------------------------------------------+
//|  [in] deals: source array with deals                             |
//|  [out] keys: destination array for deal symbol                   |
//|  [ret] number of items copied. the destination 'keys' is filled  |
//|with the deals symbol values which are extracted from the         |
//|elements of the source 'deals'                                    |
//+------------------------------------------------------------------+
void DealsKeySymbol(ulong& deals[],string& keys[])
  {
   int total=ArraySize(deals);
   ArrayResize(keys,total);
   for(int i=0; i<total; i++)
      keys[i]=(!HistoryDealSelect(deals[i]))? "" : HistoryDealGetString(deals[i],DEAL_SYMBOL);
  }
//+------------------------------------------------------------------+
//| 30 | deals print                                                 |
//+------------------------------------------------------------------+
//|  [in] deals: source array with the names of the symbols          |
//|  [ret] void. print a table of deals consiting of two columns:    |
//|deal ticket and symbol                                            |
//+------------------------------------------------------------------+
void DealsPrint(ulong& deals[])
  {
//--- format table columns
   string format="%30s | %s";
//--- create horizontal table line
   string line;
   StringInit(line,68,'-');
//--- header
   printf(line);
   printf(format,"deal","symbol");
   printf(line);
//--- body
   int total=ArraySize(deals);
   for(int i=0; i<total; i++)
     {
      //--- some data for the deal
      string symbol=(!HistoryDealSelect(deals[i]))? "" : HistoryDealGetString(deals[i],DEAL_SYMBOL);
      //--- out
      printf(format,(string)deals[i],symbol);
     }
//--- footer
   printf(line);
  }
//+------------------------------------------------------------------+
//| 40 | array sort                                                  |
//+------------------------------------------------------------------+
//|  [in/out] keys: array which values are compared and sorted       |
//|  [in/out] items: array which items are swapped, and thus sorted  |
//|the same way as the keys array. 'items' are ignored if its length |
//|does not match the length of the 'keys'                           |
//|  [in] algorithm: any compatible sorting algorithm                |
//|  [in] sorting order flag: accending true / descending false      |
//|  [ret] void. sorts 'items' by 'keys' with the specified concrete |
//|sorting algorithm, in accending or descending order               |
//+------------------------------------------------------------------+
#include <Mqh\Algorithms\ASorter.mqh>
template<typename TKey,typename TItem>
void ArraySort(TKey& keys[],TItem& items[],ASorter<TKey,TItem>* algorithm,bool accending_order=true)
  {
//--- prepare algorithm
   ArrayCopy(algorithm.keys,keys);
   ArrayCopy(algorithm.items,items);
//--- execute algorithm sorting
   algorithm.Sort();
//--- save sorted items
   ArrayCopy(keys,algorithm.keys);
   ArrayCopy(items,algorithm.items);
//--- keep sorting order
   if(!accending_order)
     {
      ArrayReverse(keys);
      ArrayReverse(items);
     }
//--- clear
   if(CheckPointer(algorithm)==POINTER_DYNAMIC)
      delete algorithm;
  }
//+------------------------------------------------------------------+
