//=====================================================================
//	Drawing a linear channel on a chart according to three specified points
//=====================================================================

//---------------------------------------------------------------------
#property copyright 	"Dima S., 2010 ."
#property link      	"dimascub@mail.com"
#property version   	"1.01"
#property description "Slide Channel Drawing"
//---------------------------------------------------------------------
#property indicator_chart_window
//---------------------------------------------------------------------

//---------------------------------------------------------------------
//	Version history:
//---------------------------------------------------------------------
//	17.11.2010. - V1.00
//	 - First version;
//
//	29.11.2010. - V1.01
//	 - Changed - names of external parameters that move the position of displaying of
//		 information;
//
//---------------------------------------------------------------------

//---------------------------------------------------------------------
//	Connected libraries:
//---------------------------------------------------------------------
#include	<TextDisplay.mqh>
#include	<SlideChannelClasses.mqh>
//---------------------------------------------------------------------

//=====================================================================
//	External parameters:
//=====================================================================
input string	PrefixString = "MainChannel";
//---------------------------------------------------------------------
input ENUM_TIMEFRAMES		ExtremumTimeFrame = PERIOD_CURRENT;
//---------------------------------------------------------------------
input bool		ShowInfo = true;
input int			UpDownInfoShift = 1;
input int			LeftRightInfoShift = 1;
input color		TitlesColor = LightCyan;
input color		FieldsColor = PaleTurquoise;
//---------------------------------------------------------------------
input color		UpBorderColor = PaleGreen;
input color		DnBorderColor = Pink;
input color		MiddleLineColor = Khaki;
input int			BorderWidth = 2;
input int			MiddleLineWidth = 1;
//---------------------------------------------------------------------
input bool		ShowBorderPercentageLines = true;
input double	PercentageZoneSize = 16.18;
//---------------------------------------------------------------------
input bool		ShowMiddleLine = true;
//---------------------------------------------------------------------
input bool		ShowMiddlePercentageLines = true;
input double	PercentageMiddleZoneSize = 5.0;
//---------------------------------------------------------------------


//---------------------------------------------------------------------
string	ChnTitles_Array[ ] = {"Channel Size:"};
string	BrdTitles_Array[ ] = {"Up Border:", "Middle Line:", "Dn Border:"};
//---------------------------------------------------------------------


//---------------------------------------------------------------------
bool			is_first_init = true;
//---------------------------------------------------------------------
#define		WIDTH			128
#define		HEIGHT		128
#define		FONTSIZE	10
//---------------------------------------------------------------------
//	Variables for displaying of the dynamic information about a current channel:
//---------------------------------------------------------------------
double		channel_size = 0.0;																// size
double		channel_slope_koeff = 0.0;														// coefficient of inclination
double		channel_zone = 0.0;																// size of the percentage zone
double		channel_upborder_price = 0.0;													// current price of the upper border
double		channel_dnborder_price = 0.0;													// current price of the lower border
double		channel_middle_price = 0.0;													// current price of the middle line
double		channel_upborder_distance = 0.0;												// distance from the current price to the upper border
double		channel_dnborder_distance = 0.0;												// distance from the current price to the lower border
double		channel_middle_distance = 0.0;												// distance from the current price to the middle line
//---------------------------------------------------------------------
datetime	curr_left_point = 0;
datetime	curr_middle_point = 0;
datetime	curr_right_point = 0;
//---------------------------------------------------------------------
double		curr_left_price = 0;
double		curr_middle_price = 0;
double		curr_right_price = 0;
//---------------------------------------------------------------------
ENUM_TIMEFRAMES	period_current;
//---------------------------------------------------------------------

//---------------------------------------------------------------------
TableDisplay*					channal_display_Ptr;				// displaying general information about the channel on the screen
TableDisplay*					border_display_Ptr;				// displaying information about the channel borders on the screen
//---------------------------------------------------------------------
TSlideChannelObject*	channel_Ptr;							   // drawing the channel
//---------------------------------------------------------------------

//---------------------------------------------------------------------
//	Handle of the initialization event:
//---------------------------------------------------------------------
int	OnInit()
{
	period_current = ExtremumTimeFrame;
	if(ExtremumTimeFrame == PERIOD_CURRENT)
	{
		period_current = Period();
	}

	if(is_first_init != true)
	{
		DeleteGraphObjects();
	}
	InitGraphObjects();
	is_first_init = false;

	EventSetTimer(1);

	ChartRedraw(0);

	return(0);
}

//---------------------------------------------------------------------
//	Handle of the event of coming of new tick by the current symbol:
//---------------------------------------------------------------------
int	OnCalculate(const int rates_total, const int prev_calculated, const int begin, const double& price[ ])
{
	return(rates_total);
}

//---------------------------------------------------------------------
//	Handler of timer events:
//---------------------------------------------------------------------
void	OnTimer()
{
	if(GetExtremums() == true)
	{
		SetExtremums();	
	}

	if(channel_Ptr.IsChannelCalculated() == true)
	{
		channel_size = channel_Ptr.GetChannelSize() / Point() * 0.1;
		channel_slope_koeff = channel_Ptr.GetKoeffA();
		channel_upborder_price = channel_Ptr.GetUpBorderPrice();
		channel_dnborder_price = channel_Ptr.GetDnBorderPrice();
		channel_middle_price = channel_Ptr.GetMiddleLinePrice();
		double	bid = SymbolInfoDouble(Symbol(), SYMBOL_BID);
		channel_upborder_distance = (bid - channel_upborder_price) / Point() * 0.1;
		channel_dnborder_distance = (bid - channel_dnborder_price) / Point() * 0.1;
		channel_middle_distance = (bid - channel_middle_price) / Point() * 0.1;
	}

	if(ShowInfo == true)
	{
		RefreshInfo();
		ChartRedraw(0);
	}
}

//---------------------------------------------------------------------
//	Handle of deinitialization event:
//---------------------------------------------------------------------
void	OnDeinit(const int _reason)
{
	EventKillTimer();
	DeleteGraphObjects();
}

//---------------------------------------------------------------------
//	Initialization of graphical objects:
//---------------------------------------------------------------------
void	InitGraphObjects()
{
//Print("Creating...");

	if(ShowInfo == true)
	{
		channal_display_Ptr = new TableDisplay();
		channal_display_Ptr.SetParams(0, 0, CORNER_LEFT_UPPER);
		for(int k = 0; k < 3; k++)
		{
			channal_display_Ptr.AddFieldObject(WIDTH, HEIGHT, LeftRightInfoShift + 9, UpDownInfoShift + k * 2 + 8, FieldsColor, "Arial", FONTSIZE);
			channal_display_Ptr.SetAnchor(k, ANCHOR_LEFT);
		}

//	Field headers:
		channal_display_Ptr.AddTitleObject(WIDTH, HEIGHT, LeftRightInfoShift + 16, UpDownInfoShift + 5, PrefixString, TitlesColor, "Arial", FONTSIZE);
		channal_display_Ptr.SetAnchor(3, ANCHOR_RIGHT);
		for(int k = 0; k < 1; k++)
		{
			channal_display_Ptr.AddTitleObject(WIDTH, HEIGHT, LeftRightInfoShift + 8, UpDownInfoShift + k * 2 + 8, ChnTitles_Array[ k ], TitlesColor, "Arial", FONTSIZE);
			channal_display_Ptr.SetAnchor(k + 4, ANCHOR_RIGHT);
		}

		border_display_Ptr = new TableDisplay();
		border_display_Ptr.SetParams(0, 0, CORNER_LEFT_UPPER);
		for(int k = 0; k < 3; k++)
		{
			border_display_Ptr.AddFieldObject(WIDTH, HEIGHT, LeftRightInfoShift + 9, UpDownInfoShift + k * 2 + 10, FieldsColor, "Arial", FONTSIZE);
			border_display_Ptr.SetAnchor(k, ANCHOR_LEFT);
		}

//	Field headers:
		for(int k = 0; k < 3; k++)
		{
			border_display_Ptr.AddTitleObject(WIDTH, HEIGHT, LeftRightInfoShift + 8, UpDownInfoShift + k * 2 + 10, BrdTitles_Array[ k ], TitlesColor, "Arial", FONTSIZE);
			border_display_Ptr.SetAnchor(k + 3, ANCHOR_RIGHT);
		}
	}

	GetExtremums();

	channel_Ptr = new TSlideChannelObject();
	channel_Ptr.CreateChannel(PrefixString, 0, 0, Symbol(), period_current, curr_left_point, curr_middle_point, curr_right_point, curr_left_price, curr_middle_price, curr_right_price);
	channel_Ptr.SetBorderWidth(BorderWidth);
	channel_Ptr.SetMiddleWidth(MiddleLineWidth);
	channel_Ptr.SetUpBorderColor(UpBorderColor);
	channel_Ptr.SetDnBorderColor(DnBorderColor);
	channel_Ptr.SetMiddleColor(MiddleLineColor);
	channel_Ptr.ShowBorderZone(ShowBorderPercentageLines);
	channel_Ptr.BorderZonePercentage(PercentageZoneSize);
	channel_Ptr.ShowMiddle(ShowMiddleLine);
	channel_Ptr.ShowMiddleZone(ShowMiddlePercentageLines);
	channel_Ptr.MiddleZonePercentage(PercentageMiddleZoneSize);
}

//---------------------------------------------------------------------
//	Clear extremum points:
//---------------------------------------------------------------------
bool	GetExtremums()
{
	double	temp;
	string	name;
	bool		changed = false;

	StringConcatenate(name, PrefixString, "_", Symbol(), "_", period_current, "_DTime_Extr1");
	if(GlobalVariableGet(name, temp) != false)
	{
		if(curr_left_point != (datetime)temp)
		{
			curr_left_point = (datetime)temp;
			changed = true;
		}
	}

	StringConcatenate(name, PrefixString, "_", Symbol(), "_", period_current, "_DTime_Extr2");
	if(GlobalVariableGet(name, temp) != false)
	{
		if(curr_middle_point != (datetime)temp)
		{
			curr_middle_point = (datetime)temp;
			changed = true;
		}
	}

	StringConcatenate(name, PrefixString, "_", Symbol(), "_", period_current, "_DTime_Extr3");
	if(GlobalVariableGet(name, temp) != false)
	{
		if(curr_right_point != (datetime)temp)
		{
			curr_right_point = (datetime)temp;
			changed = true;
		}
	}

	StringConcatenate(name, PrefixString, "_", Symbol(), "_", period_current, "_Price_Extr1");
	if(GlobalVariableGet(name, temp) != false)
	{
		if(curr_left_price != (datetime)temp)
		{
			curr_left_price = temp;
			changed = true;
		}
	}

	StringConcatenate(name, PrefixString, "_", Symbol(), "_", period_current, "_Price_Extr2");
	if(GlobalVariableGet(name, temp) != false)
	{
		if(curr_middle_price != (datetime)temp)
		{
			curr_middle_price = temp;
			changed = true;
		}
	}

	StringConcatenate(name, PrefixString, "_", Symbol(), "_", period_current, "_Price_Extr3");
	if(GlobalVariableGet(name, temp) != false)
	{
		if(curr_right_price != (datetime)temp)
		{
			curr_right_price = temp;
			changed = true;
		}
	}

	return(changed);
}

//---------------------------------------------------------------------
//	Setting extremum points of the channel:
//---------------------------------------------------------------------
bool	SetExtremums()
{
// Update the channel position (if necessary):
	return(channel_Ptr.SetExtremums(curr_left_point, curr_middle_point, curr_right_point, curr_left_price, curr_middle_price, curr_right_price));
}

//---------------------------------------------------------------------
//	Deleting graphical objects:
//---------------------------------------------------------------------
void	DeleteGraphObjects()
{
//Print("Deleting...");
	if(ShowInfo == true)
	{
		if(CheckPointer(channal_display_Ptr) == POINTER_DYNAMIC)
		{
			delete(channal_display_Ptr);
		}
		if(CheckPointer(border_display_Ptr) == POINTER_DYNAMIC)
		{
			delete(border_display_Ptr);
		}
	}

	if(CheckPointer(channel_Ptr) == POINTER_DYNAMIC)
	{
		delete(channel_Ptr);
	}
}

//---------------------------------------------------------------------
//	Refreshing dynamical information:
//---------------------------------------------------------------------
void	RefreshInfo()
{
	channal_display_Ptr.SetText(0, DoubleToString(channel_size, 1) + " p.");

	border_display_Ptr.SetText(0, DoubleToString(channel_upborder_price, Digits()) + " ( " + DoubleToString(channel_upborder_distance, 1) + " p. )");
	border_display_Ptr.SetColor(0, UpBorderColor);

	if(ShowMiddleLine == true)
	{
		border_display_Ptr.SetText(1, DoubleToString(channel_middle_price, Digits()) + " ( " + DoubleToString(channel_middle_distance, 1) + " p. )");
		border_display_Ptr.SetColor(1, MiddleLineColor);
	}

	border_display_Ptr.SetText(2, DoubleToString(channel_dnborder_price, Digits()) + " ( " + DoubleToString(channel_dnborder_distance, 1) + " p. )");
	border_display_Ptr.SetColor(2, DnBorderColor);
}

//---------------------------------------------------------------------
//	Getting color depending in the sign of number:
//---------------------------------------------------------------------
color	GetNumberColor(double _precision, double _number, color _plus, color _minus, color _null)
{
	if(_number > _precision)
	{
		return(_plus);
	}

	if(_number < (-_precision))
	{
		return(_minus);
	}

	return(_null);
}
//---------------------------------------------------------------------
