//Version  May 1, 2010
//+X================================================================X+
//|                                                 iPriceSeries.mqh |
//|                        Copyright  2010,        Nikolay Kositsin | 
//|                              Khabarovsk,   farria@mail.redcom.ru | 
//+X================================================================X+
/* 
 * The function iPriceSeries() returns the input price of a bar by its number
 * bar and by the number of the price 'applied_price':
 * 1-CLOSE, 2-OPEN, 3-HIGH, 4-LOW, 5-MEDIAN, 6-TYPICAL, 7-WEIGHTED,
 * 8-SIMPL, 9-QUARTER, 10-TRENDFOLLOW, 11-0.5 * TRENDFOLLOW.
 * 
 * Example: 
 * double dPrice = iPriceSeries("GBPJPY", 240, 5, bar, true)
 *                - iPriceSeries("GBPJPY", 240, 5, bar + 1, true);
 */
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
/*
//---- declaration and initialization of the enumeration of type of price constants
enum Applied_price_ //Type of constant
 {
  PRICE_CLOSE_ = 1,     // 1
  PRICE_OPEN_,          // 2
  PRICE_HIGH_,          // 3
  PRICE_LOW_,           // 4
  PRICE_MEDIAN_,        // 5
  PRICE_TYPICAL_,       // 6
  PRICE_WEIGHTED_,      // 7
  PRICE_SIMPL_,         // 8
  PRICE_QUARTER_,       // 9
  PRICE_TRENDFOLLOW0_, // 10
  PRICE_TRENDFOLLOW1_  // 11
 };
*/
//+X================================================================X+   
//| PriceSeries() function                                           |
//+X================================================================X+ 
double PriceSeries
 (
  uint applied_price, // Price constant
  uint   bar, // Index of shift relative to the current bar for a specified number of periods back or forward).
  const double& Open [],
  const double& Low  [],
  const double& High [],
  const double& Close[]
 )
//PriceSeries(applied_price, bar, open, low, high, close)
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  switch(applied_price) 
   {
    //----+ Price constant from the enumeration ENUM_APPLIED_PRICE
    case  PRICE_CLOSE: return(Close[bar]);
    case  PRICE_OPEN: return(Open [bar]);
    case  PRICE_HIGH: return(High [bar]);
    case  PRICE_LOW: return(Low  [bar]);
    case  PRICE_MEDIAN: return((High[bar] + Low[bar])/2.0);
    case  PRICE_TYPICAL: return((Close[bar] + High[bar] + Low[bar])/3.0);                                                                 
    case  PRICE_WEIGHTED: return((2*Close[bar] + High[bar] + Low[bar])/4.0);
        
    //----+                            
    case  8: return((Open[bar] + Close[bar])/2.0);
    case  9: return((Open[bar] + Close[bar] + High[bar] + Low[bar])/4.0);     
    //----                                
    case 10: 
            {
             if(Close[bar] > Open[bar])return(High[bar]);
             else
              {
               if(Close[bar] < Open[bar])
                    return(Low[bar]);
               else return(Close[bar]);
              } 
            }
    //----         
    case 11: 
            {
             if(Close[bar] > Open[bar])return((High[bar] + Close[bar])/2.0);
             else
              {
               if(Close[bar] < Open[bar])
                    return((Low[bar] + Close[bar])/2.0);
               else return(Close[bar]);
              } 
             break;
            }
    //----
    default: return(Close[bar]);
   }
//----+
  //return(0);
 }
//+X================================================================X+   
//| iPriceSeries() function                                          |
//+X================================================================X+ 
double iPriceSeries
 (
  string          symbol, // Symbol name. NULL means current symbol.
  ENUM_TIMEFRAMES timeframe, // Period. Can be one of the chart periods. 0 means period of the current chart.
  uint            applied_price, // Price constant
  uint            bar, // Index of shift relative to the current bar for a specified number of period back or forward).
  bool            set // Direction of indexing arrays.
 )
//iPriceSeries(symbol, timeframe, applied_price, bar, set)
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  uint Bar;
  double diPriceSeries, price[1];
  //----
  if (!set)
       Bar = Bars(symbol, timeframe) - 1 - bar;
  else Bar = bar;
  //----
  switch(applied_price) 
   {
    case  1: CopyClose(symbol, timeframe, Bar, 1, price); diPriceSeries = price[0]; break;
    case  2: CopyOpen (symbol, timeframe, Bar, 1, price); diPriceSeries = price[0]; break;
    case  3: CopyHigh (symbol, timeframe, Bar, 1, price); diPriceSeries = price[0]; break;
    case  4: CopyLow  (symbol, timeframe, Bar, 1, price); diPriceSeries = price[0]; break;
    //----
    case  5: CopyHigh(symbol, timeframe, Bar, 1, price); diPriceSeries  = price[0]; 
             CopyLow (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             diPriceSeries /= 2.0; 
             break;
    //----
    case  6: CopyClose(symbol, timeframe, Bar, 1, price); diPriceSeries  = price[0]; 
             CopyHigh (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             CopyLow  (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0];  
             diPriceSeries /= 3.0; 
             break;
    //----                                                                  
    case  7: CopyClose(symbol, timeframe, Bar, 1, price); diPriceSeries  = price[0] * 2; 
             CopyHigh (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             CopyLow  (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             diPriceSeries /= 4.0; 
             break;
           
    //----                                 
    case  8: CopyClose(symbol, timeframe, Bar, 1, price); diPriceSeries  = price[0]; 
             CopyOpen (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             diPriceSeries /= 2.0; 
             break;
    //----
    case  9: CopyClose(symbol, timeframe, Bar, 1, price); diPriceSeries  = price[0]; 
             CopyOpen (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             CopyHigh (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             CopyLow  (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             diPriceSeries /= 4.0; 
             break;
    //----                                
    case 10: 
            {
             double Open_[1], Low_[1], High_[1], Close_[1];
             //----
             CopyClose(symbol, timeframe, Bar, 1, Close_);
             CopyOpen (symbol, timeframe, Bar, 1, Open_ );
             CopyHigh (symbol, timeframe, Bar, 1, High_ );
             CopyLow  (symbol, timeframe, Bar, 1, Low_  );
             //----
             if(Close_[0] > Open_[0])diPriceSeries = High_[0];
             else
              {
               if(Close_[0] < Open_[0])
                    diPriceSeries = Low_[0];
               else diPriceSeries = Close_[0];
              } 
             break;
            }
    //----         
    case 11: 
            {
             double Open_[1], Low_[1], High_[1], Close_[1];
             //----
             CopyClose(symbol, timeframe, Bar, 1, Close_);
             CopyOpen (symbol, timeframe, Bar, 1, Open_ );
             CopyHigh (symbol, timeframe, Bar, 1, High_ );
             CopyLow  (symbol, timeframe, Bar, 1, Low_  );
             //----
             if(Close_[0] > Open_[0])diPriceSeries = (High_[0] + Close_[0]) / 2.0;
             else
              {
               if(Close_[0] < Open_[0])
                    diPriceSeries = (Low_[0] + Close_[0]) / 2.0;
               else diPriceSeries = Close_[0];
              } 
             break;
            }
    //----
    default: CopyClose(symbol, timeframe, Bar, 1, price); diPriceSeries = price[0]; break;
   }
//----+
  return(diPriceSeries);
 }
//+X================================================================X+   
//| bPriceSeries() function                                          |
//+X================================================================X+ 
bool bPriceSeries
 (
  string          symbol, // Symbol name. NULL means current symbol.
  ENUM_TIMEFRAMES timeframe, // Period. Can be one of the chart periods. 0 means period of the current chart.
  int             rates_total, // amount of history in bars at the current tick (if the parameter 'set' is equal to 'true', 
                                     // then value of the parameter is not needed in the function calculation and can be equal to 0)
  uint            applied_price, // Price constant
  uint            bar, // Index of shift relative to the current bar for a specified number of period back or forward).
  bool            set, // Direction of indexing arrays.
  double&         Price_ // return the obtained value by the link
 )
//bPriceSeries(symbol, timeframe, int rates_total, applied_price, bar, set, Price_)
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  uint Bar;
  double series[1];
  ArraySetAsSeries(series, true);
  //----
  if (!set)
       Bar = rates_total - 1 - bar;
  else Bar = bar;
  //----
  switch(applied_price) 
   {
    case  1: if (CopyClose(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ = series[0]; break;
    case  2: if (CopyOpen (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ = series[0]; break;
    case  3: if (CopyHigh (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ = series[0]; break;
    case  4: if (CopyLow  (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ = series[0]; break;
    //----
    case  5: if (CopyHigh(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_  = series[0]; 
             if (CopyLow (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             Price_ /= 2.0; 
             break;
    //----
    case  6: if (CopyClose(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_  = series[0]; 
             if (CopyHigh (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             if (CopyLow  (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0];  
             Price_ /= 3.0; 
             break;
    //----                                                                  
    case  7: if (CopyClose(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_  = series[0] * 2; 
             if (CopyHigh (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             if (CopyLow  (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             Price_ /= 4.0; 
             break;
           
    //----                                 
    case  8: if (CopyClose(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_  = series[0]; 
             if (CopyOpen (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             Price_ /= 2.0; 
             break;
    //----
    case  9: if (CopyClose(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_  = series[0]; 
             if (CopyOpen (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             if (CopyHigh (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             if (CopyLow  (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             Price_ /= 4.0; 
             break;
    //----                                
    case 10: 
            {
             double Open_[1], Low_[1], High_[1], Close_[1];
             //----
             if (CopyClose(symbol, timeframe, Bar, 1, Close_) < 0) return(false);
             if (CopyOpen (symbol, timeframe, Bar, 1, Open_ ) < 0) return(false);
             if (CopyHigh (symbol, timeframe, Bar, 1, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, Bar, 1, Low_  ) < 0) return(false);
             //----
             if(Close_[0] > Open_[0])Price_ = High_[0];
             else
              {
               if(Close_[0] < Open_[0])
                    Price_ = Low_[0];
               else Price_ = Close_[0];
              } 
             break;
            }
    //----         
    case 11: 
            {
             double Open_[1], Low_[1], High_[1], Close_[1];
             //----
             if (CopyClose(symbol, timeframe, Bar, 1, Close_) < 0) return(false);
             if (CopyOpen (symbol, timeframe, Bar, 1, Open_ ) < 0) return(false);
             if (CopyHigh (symbol, timeframe, Bar, 1, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, Bar, 1, Low_  ) < 0) return(false);
             //----
             if(Close_[0] > Open_[0])Price_ = (High_[0] + Close_[0]) / 2.0;
             else
              {
               if(Close_[0] < Open_[0])
                    Price_ = (Low_[0] + Close_[0]) / 2.0;
               else Price_ = Close_[0];
              } 
             break;
            }
    //----
    default: if (CopyClose(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ = series[0]; break;
   }
//----+
  return(true);
 }
//+X================================================================X+   
//| bPriceSeriesOnArray() function                                   |
//+X================================================================X+ 
bool bPriceSeriesOnArray
 (
  string          symbol, // Symbol name. NULL means current symbol.
  ENUM_TIMEFRAMES timeframe, // Period. Can be one of the chart periods. 0 means period of the current chart.
  uint            applied_price, // Price constant
  int             start_pos, // Number of copied element
  int             count, // Number of elements to be copied
  double&         series[]  // array, to which the information is copied
 )
//bPriceSeriesOnArray(symbol, timeframe, applied_price, start_pos, count, Array)
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  ArraySetAsSeries(series, true);

  switch(applied_price) 
   {
    case  1: if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false); break;
    case  2: if (CopyOpen (symbol, timeframe, start_pos, count, series) < 0) return(false); break;
    case  3: if (CopyHigh (symbol, timeframe, start_pos, count, series) < 0) return(false); break;
    case  4: if (CopyLow  (symbol, timeframe, start_pos, count, series) < 0) return(false); break;
    //----
    case  5: 
            {
             double Low_[];
             ArraySetAsSeries(Low_, true);
             if (CopyHigh(symbol, timeframe, start_pos, count, series) < 0) return(false); 
             if (CopyLow (symbol, timeframe, start_pos, count,  Low_ ) < 0) return(false);
             
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
                             series[kkk] = (series[kkk] + Low_[kkk]) / 2.0;
             break;
            }
    //----
    case  6: 
            {
             double Low_[], High_[];
             ArraySetAsSeries(Low_,  true);
             ArraySetAsSeries(High_, true);
             if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false); 
             if (CopyHigh (symbol, timeframe, start_pos, count, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, start_pos, count, Low_  ) < 0) return(false);
             
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
                            series[kkk] = (series[kkk] + High_[kkk] + Low_[kkk]) / 3.0;
             break;
            }
    //----                                                                  
    case  7: 
            {
             double Low_[], High_[];
             ArraySetAsSeries(Low_,  true);
             ArraySetAsSeries(High_, true);
             if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false); 
             if (CopyHigh (symbol, timeframe, start_pos, count, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, start_pos, count, Low_  ) < 0) return(false);
             
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
                            series[kkk] = (2 * series[kkk] + High_[kkk] + Low_[kkk]) / 4.0;
             break;
            }
    //----                                 
    case  8: 
            {
             double Open_[];
             ArraySetAsSeries(Open_, true);
             if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false); 
             if (CopyOpen (symbol, timeframe, start_pos, count, Open_ ) < 0) return(false);
             
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
                            series[kkk] = (series[kkk] + Open_[kkk]) / 2.0;
             break;
            }
    //----
    case  9: 
            {
             double Open_[], Low_[], High_[];
             ArraySetAsSeries(Open_, true);
             ArraySetAsSeries(Low_,  true);
             ArraySetAsSeries(High_, true);
             if (CopyOpen (symbol, timeframe, start_pos, count, Open_ ) < 0) return(false);
             if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false);
             if (CopyHigh (symbol, timeframe, start_pos, count, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, start_pos, count, Low_  ) < 0) return(false);
             
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
                            series[kkk] = (Open_[kkk] + series[kkk] + High_[kkk] + Low_[kkk]) / 4.0;
             break;
            }
    //----                                
    case 10: 
            {
             double Open_[], Low_[], High_[];
             ArraySetAsSeries(Open_, true);
             ArraySetAsSeries(Low_,  true);
             ArraySetAsSeries(High_, true);
             if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false);
             if (CopyOpen (symbol, timeframe, start_pos, count, Open_ ) < 0) return(false);
             if (CopyHigh (symbol, timeframe, start_pos, count, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, start_pos, count, Low_  ) < 0) return(false);
             //----
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
              {
               if(series[kkk] > Open_[kkk]) series[kkk] = High_[kkk];
               else
                {
                 if(series[kkk] < Open_[kkk])
                      series[kkk] = Low_[kkk];
                } 
              }
             break;
            }
    //----         
    case 11: 
            {
             double Open_[], Low_[], High_[];
             ArraySetAsSeries(Open_, true);
             ArraySetAsSeries(Low_,  true);
             ArraySetAsSeries(High_, true);
             if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false);
             if (CopyOpen (symbol, timeframe, start_pos, count, Open_ ) < 0) return(false);
             if (CopyHigh (symbol, timeframe, start_pos, count, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, start_pos, count, Low_  ) < 0) return(false);
             //----
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
              {
               if(series[kkk] > Open_[kkk]) series[kkk] = (High_[kkk] + series[kkk]) / 2.0;
               else
                {
                 if(series[kkk] < Open_[kkk])
                    series[kkk] = (Low_[kkk] + series[kkk]) / 2.0;
                } 
              }
             break;
            }
    //----
    default: if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false);
   }
//----+
  return(true);
 }
//+X================================================================X+   
//| iPriceSeriesAlert() function                                     |
//+X================================================================X+
/*
 * The function iPriceSeriesAlert() is intended for indicating an unacceptable 
 * value of the parameter applied_price passed to the function iPriceSeries().
 */
void iPriceSeriesAlert(uchar applied_price)
 {
//----+
  if (applied_price < 1)
   Alert("Parameter applied_price must less than 1. You specified incorrect value", 
                                              applied_price, " value 1 will be used");
  //----+                     
  if (applied_price > 11)
   Alert("The parameter applied_price must not exceed 11. You specified incorrect value",
                                              applied_price, " value 1 will be used");
//----+ 
 }
//+X----------------------+ <<< The End >>> +-----------------------X+