//MQL5 Version  June 21, 2010 Final
//+X================================================================X+
//|                                             SmoothAlgorithms.mqh |
//|                               Copyright  2010, Nikolay Kositsin |
//|                              Khabarovsk,   farria@mail.redcom.ru | 
//+X================================================================X+
#property copyright "2010,   Nikolay Kositsin"
#property link      "farria@mail.redcom.ru"
#property version   "1.00"

//+X================================================================X+
//|  Classes for averaging of prices series                          |
//+X================================================================X+

//+X================================================================X+
//|  Functional utilities for the classes of averaging algorithms    |
//+X================================================================X+
class CMovSeriesTools
 {
public:
  void                   MALengthCheck(string LengthName, int    ExternLength);
  void                   MALengthCheck(string LengthName, double ExternLength);
  
protected: 
  bool                   BarCheck1(int begin, int bar, bool Set);
  bool                   BarCheck2(int begin, int bar, bool Set, int Length);
  bool                   BarCheck3(int begin, int bar, bool Set, int Length);
  
  bool                   BarCheck4(int rates_total, int bar, bool Set); 
  bool                   BarCheck5(int rates_total, int bar, bool Set);
  bool                   BarCheck6(int rates_total, int bar, bool Set);
  
  void                   LengthCheck(int&    ExternLength);
  void                   LengthCheck(double& ExternLength);
  
  void                   Recount_ArrayZeroPos(int& count,
                                               int Length,
                                               int prev_calculated,
                                               double series,
                                               int bar,
                                               double& Array[]
                                               );

  int                    Recount_ArrayNumber(int count, int Length, int Number);
  
  bool                   SeriesArrayResize(string FunctionsName,
                                            int Length,
                                            double& Array[],
                                            int& Size_
                                            );
                                            
  bool                   ArrayResizeErrorPrint(string FunctionsName, int& Size_);
 };
//+X================================================================X+
//|  The function for the classic averaging of price series          |
//+X================================================================X+
class CMoving_Average : public CMovSeriesTools
  {
public:  
   double    MASeries  (uint begin,// number of beginning of bars for reliable calculation
                         uint prev_calculated,// amount of history in bars at the previous tick
                         uint rates_total,// amount of history in bars at the current tick
                         int Length,// period of averaging
                         ENUM_MA_METHOD MA_Method,// method of averaging (MODE_SMA, MODE_EMA, MODE_SMMA, MODE_LWMA)
                         double series,// value of the price series calculated for the new bar with number 'bar'
                         uint bar,// bar number
                         bool set // direction of indexing arrays
                        );

   double    SMASeries (uint begin,// number of beginning of bars for reliable calculation
                         uint prev_calculated,// amount of history in bars at the previous tick
                         uint rates_total,// amount of history in bars at the current tick
                         int Length,// period of averaging
                         double series,// value of the price series calculated for the new bar with number 'bar'
                         uint bar,// bar number
                         bool set // direction of indexing arrays
                        );
                        
   double    EMASeries (uint begin,// number of beginning of bars for reliable calculation
                         uint prev_calculated,// amount of history in bars at the previous tick
                         uint rates_total,// amount of history in bars at the current tick
                         double Length,// period of averaging
                         double series,// value of the price series calculated for the new bar with number 'bar'
                         uint bar,// bar number
                         bool set // direction of indexing arrays
                        );
                        
   double    SMMASeries(uint begin,// number of beginning of bars for reliable calculation
                         uint prev_calculated,// amount of history in bars at the previous tick
                         uint rates_total,// amount of history in bars at the current tick
                         int Length,// period of averaging
                         double series,// value of the price series calculated for the new bar with number 'bar'
                         uint bar,// bar number
                         bool set // direction of indexing arrays
                        );

   double    LWMASeries(uint begin,// number of beginning of bars for reliable calculation
                         uint prev_calculated,// amount of history in bars at the previous tick
                         uint rates_total,// amount of history in bars at the current tick
                         int Length,// period of averaging
                         double series,// value of the price series calculated for the new bar with number 'bar'
                         uint bar,// bar number
                         bool set // direction of indexing arrays
                        );

protected:
   double            m_SeriesArray[];
   int               m_Size_, m_count, m_weight;
   double            m_Moving, m_MOVING, m_Pr;
   double            m_sum, m_SUM, m_lsum, m_LSUM;
  };
//+X================================================================X+
//|  The algorithm of getting the standard deviation                 |
//+X================================================================X+
class CStdDeviation : public CMovSeriesTools
 {
public:
  double StdDevSeries(uint begin, // number of beginning of bars for reliable calculation
                       uint prev_calculated, // amount of history in bars at previous tick
                       uint rates_total, // amount of history in bars at the current tick
                       int Length, // period of averaging
                       double deviation, // deviation
                       double series,  // value of the price series calculated for the new bar with number 'bar'
                       double MovSeries, // Value of a moving average, on which basis the StdDeviation is calculated
                       uint bar,  // bar number
                       bool set // direction of indexing arrays
                       );
  protected:
  int    m_Size_, m_count;
  double m_Sum, m_SUM, m_Sum2, m_SUM2;
  double m_SeriesArray[];  
 };
//+X================================================================X+
//| The JMA algorithm of averaging of price series                   |
//+X================================================================X+
class CJJMA : public CMovSeriesTools
 { 
  public:  
    double JJMASeries(uint begin, // number of beginning of bars for reliable calculation
                       uint prev_calculated, // amount of history in bars at previous tick
                       uint rates_total, // amount of history in bars at the current tick
                       int  Din, // permission to change the parameters Length and Phase at every bar. 
                                  // 0 - prohibition to change the parameters, any other values means permission.
                       double Phase, // parameter that can change withing the range -100 ... +100, it depends on the quality of intermediate process of averaging
                       double Length, // depth of smoothing
                       double series,  // value of the price series calculated for the new bar with number 'bar'
                       uint bar,  // bar number
                       bool set // direction of indexing arrays
                       );
                        
    void JJMALengthChack(string LengthName, int ExternLength);
    void JJMAPhaseChack (string PhaseName,  int ExternPhase);
 
  protected:
    void JJMAInit (uint begin, int Din, double Phase, double Length, double series, uint bar);
  
//----+ Declaration of global variables
    bool m_f0;
    bool m_bRING1[128], m_bRING2[11], m_bLIST[128];
    //----
    double m_f18_, m_f38_;
    double m_buffer[62], m_f90, m_fC0;
    double m_Kg, m_Pf, m_s18_, m_JMA_;   
    double m_f18, m_f38, m_fA8, m_fC8;
    double m_f78, m_f88, m_f98, m_JMA;
    double m_fC0_, m_fC8_, m_fA8_, m_s8_;
    double m_RING1[128], m_RING2[11], m_LIST[128];
    double m_ring1[128], m_ring2[11], m_list[128];
    double m_s8, m_s18, m_v1, m_v2, m_v3;
    //----
    int m_s40_, m_s70_;
    int m_s48_, m_s50, m_s70;
    int m_s38_, m_s48, m_s50_;
    int m_s28, m_s30, m_s38, m_s40;
    int m_LP1_, m_LP1, m_LP2_, m_LP2;
 };
//+X================================================================X+
//| The Tilson's algorithm of averaging of price series              |
//+X================================================================X+
class CT3 : public CMovSeriesTools
 {
public:
  double   T3Series(uint begin, // number of beginning of bars for reliable calculation
                     uint prev_calculated, // amount of history in bars at previous tick
                     uint rates_total, // amount of history in bars at the current tick
                     int  Din, // permission to change the parameter Length at every bar. 
                                // 0 - prohibition to change the parameters, any other values means permission.
                     double Curvature, // coefficient (its value is increased 100 times for convenience!)
                     double Length, // depth of smoothing
                     double series,  // value of the price series calculated for the new bar with number 'bar'
                     uint bar,  // bar number
                     bool set // direction of indexing arrays
                     ); 
protected:
  void     T3Init (uint begin,
                    int Din,
                    double Curvature,
                    double Length,
                    double series,
                    uint bar
                    );

  //----+ Declaration of global variables
  double   m_b2, m_b3;
  //----
  double   m_e1, m_e2, m_e3, m_e4, m_e5, m_e6;
  double   m_E1, m_E2, m_E3, m_E4, m_E5, m_E6;
  double   m_c1, m_c2, m_c3, m_c4, m_w1, m_w2;
 };
//+X================================================================X+
//| The algorithm of ultralinear averaging of price series           |
//+X================================================================X+
class CJurX : public CMovSeriesTools
 {
public:  
  double   JurXSeries(uint begin, // number of beginning of bars for reliable calculation
                       uint prev_calculated, // amount of history in bars at previous tick
                       uint rates_total, // amount of history in bars at the current tick
                       int  Din, // permission to change the parameter Length at every bar. 
                                  // 0 - prohibition to change the parameters, any other values means permission.
                       double Length, // depth of smoothing
                       double series,  // value of the price series calculated for the new bar with number 'bar'
                       uint bar,  // bar number
                       bool set // direction of indexing arrays
                       ); 
protected:
  void     JurXInit(uint begin,
                     int Din,
                     double Length,
                     double series,
                     uint bar
                     );

   //----+ Declaration of global variables
   double   m_AB, m_AC;
   double   m_f1, m_f2, m_f3, m_f4, m_f5;
   double   m_f6, m_Kg, m_Hg, m_F1, m_F2;
   double   m_F3, m_F4, m_F5, m_F6, m_w ;
 };
//+X================================================================X+
//| Tushar Chande's algorithms of averaging for any prices series    |
//+X================================================================X+
class CCMO : public CMovSeriesTools
 {
public: 
  double     VIDYASeries(uint begin, // number of beginning of bars for reliable calculation
                          uint prev_calculated, // amount of history in bars at previous tick
                          uint rates_total, // amount of history in bars at the current tick
                          int CMO_Length, // CMO period
                          double EMA_Length,
                          double series,  // value of the price series calculated for the new bar with number 'bar'
                          uint bar,  // bar number
                          bool set // direction of indexing arrays
                          );
                            
  double    CMOSeries(uint begin, // number of beginning of bars for reliable calculation
                       uint prev_calculated, // amount of history in bars at previous tick
                       uint rates_total, // amount of history in bars at the current tick
                       int CMO_Length, // CMO period
                       double series,
                       uint bar,  // bar number
                       bool set // direction of indexing arrays
                       );
  
protected:  
  double    m_dSeriesArray[];
  int       m_Size_, m_count;
  double    m_UpSum_, m_UpSum, m_DnSum_, m_DnSum, m_Vidya, m_Vidya_;
  double    m_AbsCMO_, m_AbsCMO, m_series1, m_series1_, m_SmoothFactor;
 };
//+X================================================================X+
//| The algorithm of getting the AMA indicator calculated            |
//| on the basis of optional price series                            |
//+X================================================================X+
class CAMA : public CMovSeriesTools
 {
  public:
    double AMASeries(uint begin, // number of beginning of bars for reliable calculation
                      uint prev_calculated, // amount of history in bars at previous tick
                      uint rates_total, // amount of history in bars at the current tick
                      int Length, // period of AMA
                      int Fast_Length, // period of fast moving average
                      int Slow_Length, // period of slow moving average
                      double Rate, // rate of the smoothing constant
                      double series,  // value of the price series calculated for the new bar with number 'bar'
                      uint bar,  // bar number
                      bool set // direction of indexing arrays
                     );
  protected:
//----+  
  double   m_SeriesArray[];
  double   m_dSeriesArray[];
  double   m_NOISE,  m_noise;
  double   m_Ama,  m_AMA_,  m_slowSC,  m_fastSC,  m_dSC;
  int      m_Size_1,  m_Size_2,  m_count;
 };
//+X================================================================X+
//| The algorithm of parabolic averaging of price series             |
//+X================================================================X+
class CParMA : public CMovSeriesTools
 {
public: 
   double  ParMASeries (uint begin,// number of beginning of bars for reliable calculation
                         uint prev_calculated,// amount of history in bars at the previous tick
                         uint rates_total,// amount of history in bars at the current tick
                         int Length,// period of averaging
                         double series,// value of the price series calculated for the new bar with number 'bar'
                         uint bar,// bar number
                         bool set // direction of indexing arrays
                        );
protected:
   void    ParMAInit(double Length);
   
   double  m_SeriesArray[];
   int     m_Size_, m_count;
   int     m_sum_x, m_sum_x2, m_sum_x3, m_sum_x4;
 };
//Version  May 1, 2010
//+X================================================================X+
//|                                                 iPriceSeries.mqh |
//|                        Copyright  2010,        Nikolay Kositsin | 
//|                              Khabarovsk,   farria@mail.redcom.ru | 
//+X================================================================X+
/* 
 * The function iPriceSeries() returns the input price of a bar by its number
 * bar and by the number of the price 'applied_price':
 * 1-CLOSE, 2-OPEN, 3-HIGH, 4-LOW, 5-MEDIAN, 6-TYPICAL, 7-WEIGHTED,
 * 8-SIMPL, 9-QUARTER, 10-TRENDFOLLOW, 11-0.5 * TRENDFOLLOW.
 * 
 * Example: 
 * double dPrice = iPriceSeries("GBPJPY", 240, 5, bar, true)
 *                - iPriceSeries("GBPJPY", 240, 5, bar + 1, true);
 */
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
/*
//---- declaration and initialization of the enumeration of type of price constants
enum Applied_price_ //Type of constant
 {
  PRICE_CLOSE_ = 1,     // 1
  PRICE_OPEN_,          // 2
  PRICE_HIGH_,          // 3
  PRICE_LOW_,           // 4
  PRICE_MEDIAN_,        // 5
  PRICE_TYPICAL_,       // 6
  PRICE_WEIGHTED_,      // 7
  PRICE_SIMPL_,         // 8
  PRICE_QUARTER_,       // 9
  PRICE_TRENDFOLLOW0_, // 10
  PRICE_TRENDFOLLOW1_  // 11
 };
*/
//+X================================================================X+   
//| PriceSeries() function                                           |
//+X================================================================X+ 
double PriceSeries
 (
  uint applied_price, // Price constant
  uint   bar, // Index of shift relative to the current bar for a specified number of periods back or forward).
  const double& Open [],
  const double& Low  [],
  const double& High [],
  const double& Close[]
 )
//PriceSeries(applied_price, bar, open, low, high, close)
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  switch(applied_price) 
   {
    //----+ Price constant from the enumeration ENUM_APPLIED_PRICE
    case  PRICE_CLOSE: return(Close[bar]);
    case  PRICE_OPEN: return(Open [bar]);
    case  PRICE_HIGH: return(High [bar]);
    case  PRICE_LOW: return(Low  [bar]);
    case  PRICE_MEDIAN: return((High[bar] + Low[bar])/2.0);
    case  PRICE_TYPICAL: return((Close[bar] + High[bar] + Low[bar])/3.0);                                                                 
    case  PRICE_WEIGHTED: return((2*Close[bar] + High[bar] + Low[bar])/4.0);
        
    //----+                            
    case  8: return((Open[bar] + Close[bar])/2.0);
    case  9: return((Open[bar] + Close[bar] + High[bar] + Low[bar])/4.0);     
    //----                                
    case 10: 
            {
             if(Close[bar] > Open[bar])return(High[bar]);
             else
              {
               if(Close[bar] < Open[bar])
                    return(Low[bar]);
               else return(Close[bar]);
              } 
            }
    //----         
    case 11: 
            {
             if(Close[bar] > Open[bar])return((High[bar] + Close[bar])/2.0);
             else
              {
               if(Close[bar] < Open[bar])
                    return((Low[bar] + Close[bar])/2.0);
               else return(Close[bar]);
              } 
             break;
            }
    //----
    default: return(Close[bar]);
   }
//----+
  //return(0);
 }
//+X================================================================X+   
//| iPriceSeries() function                                          |
//+X================================================================X+ 
double iPriceSeries
 (
  string          symbol, // Symbol name. NULL means current symbol.
  ENUM_TIMEFRAMES timeframe, // Period. Can be one of the chart periods. 0 means period of the current chart.
  uint            applied_price, // Price constant
  uint            bar, // Index of shift relative to the current bar for a specified number of period back or forward).
  bool            set // Direction of indexing arrays.
 )
//iPriceSeries(symbol, timeframe, applied_price, bar, set)
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  uint Bar;
  double diPriceSeries, price[1];
  //----
  if (!set)
       Bar = Bars(symbol, timeframe) - 1 - bar;
  else Bar = bar;
  //----
  switch(applied_price) 
   {
    case  1: CopyClose(symbol, timeframe, Bar, 1, price); diPriceSeries = price[0]; break;
    case  2: CopyOpen (symbol, timeframe, Bar, 1, price); diPriceSeries = price[0]; break;
    case  3: CopyHigh (symbol, timeframe, Bar, 1, price); diPriceSeries = price[0]; break;
    case  4: CopyLow  (symbol, timeframe, Bar, 1, price); diPriceSeries = price[0]; break;
    //----
    case  5: CopyHigh(symbol, timeframe, Bar, 1, price); diPriceSeries  = price[0]; 
             CopyLow (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             diPriceSeries /= 2.0; 
             break;
    //----
    case  6: CopyClose(symbol, timeframe, Bar, 1, price); diPriceSeries  = price[0]; 
             CopyHigh (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             CopyLow  (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0];  
             diPriceSeries /= 3.0; 
             break;
    //----                                                                  
    case  7: CopyClose(symbol, timeframe, Bar, 1, price); diPriceSeries  = price[0] * 2; 
             CopyHigh (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             CopyLow  (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             diPriceSeries /= 4.0; 
             break;
           
    //----                                 
    case  8: CopyClose(symbol, timeframe, Bar, 1, price); diPriceSeries  = price[0]; 
             CopyOpen (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             diPriceSeries /= 2.0; 
             break;
    //----
    case  9: CopyClose(symbol, timeframe, Bar, 1, price); diPriceSeries  = price[0]; 
             CopyOpen (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             CopyHigh (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             CopyLow  (symbol, timeframe, Bar, 1, price); diPriceSeries += price[0]; 
             diPriceSeries /= 4.0; 
             break;
    //----                                
    case 10: 
            {
             double Open_[1], Low_[1], High_[1], Close_[1];
             //----
             CopyClose(symbol, timeframe, Bar, 1, Close_);
             CopyOpen (symbol, timeframe, Bar, 1, Open_ );
             CopyHigh (symbol, timeframe, Bar, 1, High_ );
             CopyLow  (symbol, timeframe, Bar, 1, Low_  );
             //----
             if(Close_[0] > Open_[0])diPriceSeries = High_[0];
             else
              {
               if(Close_[0] < Open_[0])
                    diPriceSeries = Low_[0];
               else diPriceSeries = Close_[0];
              } 
             break;
            }
    //----         
    case 11: 
            {
             double Open_[1], Low_[1], High_[1], Close_[1];
             //----
             CopyClose(symbol, timeframe, Bar, 1, Close_);
             CopyOpen (symbol, timeframe, Bar, 1, Open_ );
             CopyHigh (symbol, timeframe, Bar, 1, High_ );
             CopyLow  (symbol, timeframe, Bar, 1, Low_  );
             //----
             if(Close_[0] > Open_[0])diPriceSeries = (High_[0] + Close_[0]) / 2.0;
             else
              {
               if(Close_[0] < Open_[0])
                    diPriceSeries = (Low_[0] + Close_[0]) / 2.0;
               else diPriceSeries = Close_[0];
              } 
             break;
            }
    //----
    default: CopyClose(symbol, timeframe, Bar, 1, price); diPriceSeries = price[0]; break;
   }
//----+
  return(diPriceSeries);
 }
//+X================================================================X+   
//| bPriceSeries() function                                          |
//+X================================================================X+ 
bool bPriceSeries
 (
  string          symbol, // Symbol name. NULL means current symbol.
  ENUM_TIMEFRAMES timeframe, // Period. Can be one of the chart periods. 0 means period of the current chart.
  int             rates_total, // amount of history in bars at the current tick (if the parameter 'set' is equal to 'true', 
                                     // then value of the parameter is not needed in the function calculation and can be equal to 0)
  uint            applied_price, // Price constant
  uint            bar, // Index of shift relative to the current bar for a specified number of period back or forward).
  bool            set, // Direction of indexing arrays.
  double&         Price_ // return the obtained value by the link
 )
//bPriceSeries(symbol, timeframe, int rates_total, applied_price, bar, set, Price_)
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  uint Bar;
  double series[1];
  ArraySetAsSeries(series, true);
  //----
  if (!set)
       Bar = rates_total - 1 - bar;
  else Bar = bar;
  //----
  switch(applied_price) 
   {
    case  1: if (CopyClose(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ = series[0]; break;
    case  2: if (CopyOpen (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ = series[0]; break;
    case  3: if (CopyHigh (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ = series[0]; break;
    case  4: if (CopyLow  (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ = series[0]; break;
    //----
    case  5: if (CopyHigh(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_  = series[0]; 
             if (CopyLow (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             Price_ /= 2.0; 
             break;
    //----
    case  6: if (CopyClose(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_  = series[0]; 
             if (CopyHigh (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             if (CopyLow  (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0];  
             Price_ /= 3.0; 
             break;
    //----                                                                  
    case  7: if (CopyClose(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_  = series[0] * 2; 
             if (CopyHigh (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             if (CopyLow  (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             Price_ /= 4.0; 
             break;
           
    //----                                 
    case  8: if (CopyClose(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_  = series[0]; 
             if (CopyOpen (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             Price_ /= 2.0; 
             break;
    //----
    case  9: if (CopyClose(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_  = series[0]; 
             if (CopyOpen (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             if (CopyHigh (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             if (CopyLow  (symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ += series[0]; 
             Price_ /= 4.0; 
             break;
    //----                                
    case 10: 
            {
             double Open_[1], Low_[1], High_[1], Close_[1];
             //----
             if (CopyClose(symbol, timeframe, Bar, 1, Close_) < 0) return(false);
             if (CopyOpen (symbol, timeframe, Bar, 1, Open_ ) < 0) return(false);
             if (CopyHigh (symbol, timeframe, Bar, 1, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, Bar, 1, Low_  ) < 0) return(false);
             //----
             if(Close_[0] > Open_[0])Price_ = High_[0];
             else
              {
               if(Close_[0] < Open_[0])
                    Price_ = Low_[0];
               else Price_ = Close_[0];
              } 
             break;
            }
    //----         
    case 11: 
            {
             double Open_[1], Low_[1], High_[1], Close_[1];
             //----
             if (CopyClose(symbol, timeframe, Bar, 1, Close_) < 0) return(false);
             if (CopyOpen (symbol, timeframe, Bar, 1, Open_ ) < 0) return(false);
             if (CopyHigh (symbol, timeframe, Bar, 1, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, Bar, 1, Low_  ) < 0) return(false);
             //----
             if(Close_[0] > Open_[0])Price_ = (High_[0] + Close_[0]) / 2.0;
             else
              {
               if(Close_[0] < Open_[0])
                    Price_ = (Low_[0] + Close_[0]) / 2.0;
               else Price_ = Close_[0];
              } 
             break;
            }
    //----
    default: if (CopyClose(symbol, timeframe, Bar, 1, series) < 0) return(false); Price_ = series[0]; break;
   }
//----+
  return(true);
 }
//+X================================================================X+   
//| bPriceSeriesOnArray() function                                   |
//+X================================================================X+ 
bool bPriceSeriesOnArray
 (
  string          symbol, // Symbol name. NULL means current symbol.
  ENUM_TIMEFRAMES timeframe, // Period. Can be one of the chart periods. 0 means period of the current chart.
  uint            applied_price, // Price constant
  int             start_pos, // Number of copied element
  int             count, // Number of elements to be copied
  double&         series[]  // array, to which the information is copied
 )
//bPriceSeriesOnArray(symbol, timeframe, applied_price, start_pos, count, Array)
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  ArraySetAsSeries(series, true);

  switch(applied_price) 
   {
    case  1: if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false); break;
    case  2: if (CopyOpen (symbol, timeframe, start_pos, count, series) < 0) return(false); break;
    case  3: if (CopyHigh (symbol, timeframe, start_pos, count, series) < 0) return(false); break;
    case  4: if (CopyLow  (symbol, timeframe, start_pos, count, series) < 0) return(false); break;
    //----
    case  5: 
            {
             double Low_[];
             ArraySetAsSeries(Low_, true);
             if (CopyHigh(symbol, timeframe, start_pos, count, series) < 0) return(false); 
             if (CopyLow (symbol, timeframe, start_pos, count,  Low_ ) < 0) return(false);
             
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
                             series[kkk] = (series[kkk] + Low_[kkk]) / 2.0;
             break;
            }
    //----
    case  6: 
            {
             double Low_[], High_[];
             ArraySetAsSeries(Low_,  true);
             ArraySetAsSeries(High_, true);
             if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false); 
             if (CopyHigh (symbol, timeframe, start_pos, count, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, start_pos, count, Low_  ) < 0) return(false);
             
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
                            series[kkk] = (series[kkk] + High_[kkk] + Low_[kkk]) / 3.0;
             break;
            }
    //----                                                                  
    case  7: 
            {
             double Low_[], High_[];
             ArraySetAsSeries(Low_,  true);
             ArraySetAsSeries(High_, true);
             if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false); 
             if (CopyHigh (symbol, timeframe, start_pos, count, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, start_pos, count, Low_  ) < 0) return(false);
             
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
                            series[kkk] = (2 * series[kkk] + High_[kkk] + Low_[kkk]) / 4.0;
             break;
            }
    //----                                 
    case  8: 
            {
             double Open_[];
             ArraySetAsSeries(Open_, true);
             if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false); 
             if (CopyOpen (symbol, timeframe, start_pos, count, Open_ ) < 0) return(false);
             
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
                            series[kkk] = (series[kkk] + Open_[kkk]) / 2.0;
             break;
            }
    //----
    case  9: 
            {
             double Open_[], Low_[], High_[];
             ArraySetAsSeries(Open_, true);
             ArraySetAsSeries(Low_,  true);
             ArraySetAsSeries(High_, true);
             if (CopyOpen (symbol, timeframe, start_pos, count, Open_ ) < 0) return(false);
             if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false);
             if (CopyHigh (symbol, timeframe, start_pos, count, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, start_pos, count, Low_  ) < 0) return(false);
             
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
                            series[kkk] = (Open_[kkk] + series[kkk] + High_[kkk] + Low_[kkk]) / 4.0;
             break;
            }
    //----                                
    case 10: 
            {
             double Open_[], Low_[], High_[];
             ArraySetAsSeries(Open_, true);
             ArraySetAsSeries(Low_,  true);
             ArraySetAsSeries(High_, true);
             if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false);
             if (CopyOpen (symbol, timeframe, start_pos, count, Open_ ) < 0) return(false);
             if (CopyHigh (symbol, timeframe, start_pos, count, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, start_pos, count, Low_  ) < 0) return(false);
             //----
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
              {
               if(series[kkk] > Open_[kkk]) series[kkk] = High_[kkk];
               else
                {
                 if(series[kkk] < Open_[kkk])
                      series[kkk] = Low_[kkk];
                } 
              }
             break;
            }
    //----         
    case 11: 
            {
             double Open_[], Low_[], High_[];
             ArraySetAsSeries(Open_, true);
             ArraySetAsSeries(Low_,  true);
             ArraySetAsSeries(High_, true);
             if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false);
             if (CopyOpen (symbol, timeframe, start_pos, count, Open_ ) < 0) return(false);
             if (CopyHigh (symbol, timeframe, start_pos, count, High_ ) < 0) return(false);
             if (CopyLow  (symbol, timeframe, start_pos, count, Low_  ) < 0) return(false);
             //----
             for (int kkk = start_pos; kkk < start_pos + count; kkk++)
              {
               if(series[kkk] > Open_[kkk]) series[kkk] = (High_[kkk] + series[kkk]) / 2.0;
               else
                {
                 if(series[kkk] < Open_[kkk])
                    series[kkk] = (Low_[kkk] + series[kkk]) / 2.0;
                } 
              }
             break;
            }
    //----
    default: if (CopyClose(symbol, timeframe, start_pos, count, series) < 0) return(false);
   }
//----+
  return(true);
 }
//+X================================================================X+   
//| iPriceSeriesAlert() function                                     |
//+X================================================================X+
/*
 * The function iPriceSeriesAlert() is intended for indicating an unacceptable 
 * value of the parameter applied_price passed to the function iPriceSeries().
 */
void iPriceSeriesAlert(uchar applied_price)
 {
//----+
  if (applied_price < 1)
   Alert("Parameter applied_price must less than 1. You specified incorrect value", 
                                              applied_price, " value 1 will be used");
  //----+                     
  if (applied_price > 11)
   Alert("The parameter applied_price must not exceed 11. You specified incorrect value",
                                              applied_price, " value 1 will be used");
//----+ 
 }
//+X================================================================X+
//|  The classic algorithms of averaging                             |
//+X================================================================X+    
double CMoving_Average::MASeries
(
 uint begin,// number of beginning of bars for reliable calculation
 uint prev_calculated,// amount of history in bars at the previous tick
 uint rates_total,// amount of history in bars at the current tick
 int Length,// period of averaging
 ENUM_MA_METHOD MA_Method,// method of averaging (MODE_SMA, MODE_EMA, MODE_SMMA, MODE_LWMA)
 double series,// value of the price series calculated for the new bar with number 'bar'
 uint bar,// bar number
 bool set // direction of indexing arrays
 )
// MASeries(begin, prev_calculated, Length, MA_Method, series, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
  {
//----+   
   switch(MA_Method)
     {
      case MODE_SMA:  return(SMASeries (begin, prev_calculated, rates_total, Length, series, bar, set));
      case MODE_EMA:  return(EMASeries (begin, prev_calculated, rates_total, Length, series, bar, set));
      case MODE_SMMA: return(SMMASeries(begin, prev_calculated, rates_total, Length, series, bar, set));
      case MODE_LWMA: return(LWMASeries(begin, prev_calculated, rates_total, Length, series, bar, set));
      default:
        {
         if(bar==begin)
           {
            string word;
            StringConcatenate(word,__FUNCTION__,"():",
                              " The parameter MA_Method must be withing the range from MODE_SMA to MODE_LWMA.",
                              " You specified unacceptable value ",MA_Method," value MODE_SMA will be used!");
            Print(word);
           }
         return(SMASeries(begin,prev_calculated,rates_total,Length,series,bar,set));
        }
     }
//----+
  }
//+X================================================================X+
//|  Simple averaging                                                |
//+X================================================================X+    
double CMoving_Average::SMASeries
(
 uint begin,// number of beginning of bars for reliable calculation
 uint prev_calculated,// amount of history in bars at the previous tick
 uint rates_total,// amount of history in bars at the current tick
 int Length,// period of averaging
 double series,// value of the price series calculated for the new bar with number 'bar'
 uint bar,// bar number
 bool set // direction of indexing arrays
 )
// SMASeries(begin, prev_calculated, rates_total, Length, series, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
  { 
//----+  Checking the beginning of bars for reliable calculation 
   if(BarCheck1(begin,bar,set)) return(EMPTY_VALUE);

//----+ Declaration of local variables
   int iii,kkk;
   double sma;
   
//----+ checking for correctness the external parameter Length
   LengthCheck(Length);	

//----+ Changing the size of array of variables
   if(bar==begin && !SeriesArrayResize(__FUNCTION__, Length, m_SeriesArray, m_Size_))
      return(EMPTY_VALUE);

//----+ rearrangement and initialization of cells of the array m_SeriesArray
   Recount_ArrayZeroPos(m_count, Length, prev_calculated, series, bar, m_SeriesArray);

//----+ Initialization of zero
   if(BarCheck2(begin, bar, set, Length))
     {
      m_sum=0.0;

      for(iii=1; iii<Length; iii++)
        {
         kkk = Recount_ArrayNumber(m_count,Length,iii);
         m_sum+= m_SeriesArray[kkk];
        }
     }
   else if(BarCheck3(begin,bar,set,Length)) return(EMPTY_VALUE);

//----+  SMA
   m_sum+= series;
   sma = m_sum / Length;
   kkk = Recount_ArrayNumber(m_count, Length, Length - 1);
   m_sum-=m_SeriesArray[kkk];

//----+ saving values of variables
   if(BarCheck4(rates_total,bar,set))
     {
      m_SUM=m_sum;
     }

//----+ restoring values of variables
   if(BarCheck5(rates_total,bar,set))
     {
      m_sum=m_SUM;
     }
//----+
   return(sma);
  }
//+X================================================================X+
//|  Exponential averaging                                           |
//+X================================================================X+    
double CMoving_Average::EMASeries
(
 uint begin,// number of beginning of bars for reliable calculation
 uint prev_calculated,// amount of history in bars at the previous tick
 uint rates_total,// amount of history in bars at the current tick
 double Length, // period of averaging
 double series, // value of the price series calculated for the new bar with number 'bar'
 uint bar,// bar number
 bool set // direction of indexing arrays
 )
// EMASeries(begin, prev_calculated, Length, series, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
  {
//----+  Checking the beginning of bars for reliable calculation 
   if(BarCheck1(begin,bar,set)) return(EMPTY_VALUE);

//----+ Declaration of local variables
   double ema;
   
//----+ checking for correctness the external parameter Length
   LengthCheck(Length);	

//----+ Initialization of zero
   if(bar==begin)
     {
      m_Pr=2.0/(Length+1.0);
      m_Moving=series;
     }

//----+  EMA 
   m_Moving=series*m_Pr+m_Moving *(1-m_Pr);
   ema=m_Moving;

//----+ saving values of variables
   if(BarCheck4(rates_total,bar,set))
     {
      m_MOVING=m_Moving;
     }

//----+ restoring values of variables
   if(BarCheck5(rates_total,bar,set))
     {
      m_Moving=m_MOVING;
     }
//----+
   return(ema);
  }
//+X================================================================X+
//|  Smoothed averaging                                              |
//+X================================================================X+
double CMoving_Average::SMMASeries
(
 uint begin,// number of beginning of bars for reliable calculation
 uint prev_calculated,// amount of history in bars at the previous tick
 uint rates_total,// amount of history in bars at the current tick
 int Length,// period of averaging
 double series,// value of the price series calculated for the new bar with number 'bar'
 uint bar,// bar number
 bool set // direction of indexing arrays
 )
// SMMASeries(begin, prev_calculated, Length, series, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
  {
//----+  Checking the beginning of bars for reliable calculation 
   if(BarCheck1(begin,bar,set)) return(EMPTY_VALUE);
   
//----+ Declaration of local variables
   int iii;
   double smma;
   
//----+ checking for correctness the external parameter Length
   LengthCheck(Length);	

//----+ Changing the size of array of variables
   if(bar==begin && !SeriesArrayResize(__FUNCTION__,Length,m_SeriesArray,m_Size_))
      return(EMPTY_VALUE);
      
//----+ rearrangement and initialization of cells of the array m_SeriesArray
   Recount_ArrayZeroPos(m_count, Length, prev_calculated, series, bar, m_SeriesArray);

//----+ Initialization of zero
   if(BarCheck2(begin,bar,set,Length))
     {
      m_sum=0.0;
      for(iii=0; iii<Length; iii++)
         m_sum+= m_SeriesArray[iii];

      m_Moving=(m_sum-series)/(Length-1);
     }
   else if(BarCheck3(begin,bar,set,Length)) return(EMPTY_VALUE);

//----+ Calculation of SMMA
   m_sum=m_Moving *(Length-1)+series;
   m_Moving=m_sum/Length;
   smma=m_Moving;

//----+ saving values of variables
   if(BarCheck4(rates_total,bar,set))
     {
      m_SUM=m_sum;
      m_MOVING=m_Moving;
     }

//----+ restoring values of variables
   if(BarCheck5(rates_total,bar,set))
     {
      m_sum=m_SUM;
      m_Moving=m_MOVING;
     }
//----+
   return(smma);
  }
//+X================================================================X+
//|  Linear weighted averaging                                       |
//+X================================================================X+
double CMoving_Average::LWMASeries
(
 uint begin,// number of beginning of bars for reliable calculation
 uint prev_calculated,// amount of history in bars at the previous tick
 uint rates_total,// amount of history in bars at the current tick
 int Length,// period of averaging
 double series,// value of the price series calculated for the new bar with number 'bar'
 uint bar,// bar number
 bool set // direction of indexing arrays
 )
// LWMASeries(begin, prev_calculated, Length, series, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
  {
//----+  Checking the beginning of bars for reliable calculation 
   if(BarCheck1(begin,bar,set)) return(EMPTY_VALUE);
   
//----+ Declaration of local variables
   double lwma;
   int iii,kkk,Length_=Length+1;
   
//----+ checking for correctness the external parameter Length
   LengthCheck(Length);	

//----+ Changing the size of array of variables
   if(bar==begin && !SeriesArrayResize(__FUNCTION__, Length_, m_SeriesArray, m_Size_))
      return(EMPTY_VALUE);

//----+ rearrangement and initialization of cells of the array m_SeriesArray
   Recount_ArrayZeroPos(m_count, Length_, prev_calculated, series, bar, m_SeriesArray);

//----+ Initialization of zero
   if(BarCheck2(begin,bar,set,Length_))
     {
      m_sum=0.0;
      m_lsum=0.0;
      m_weight=0;
      int rrr=Length;

      for(iii=1; iii<=Length; iii++,rrr--)
        {
         kkk = Recount_ArrayNumber(m_count,Length_,iii);
         m_sum+= m_SeriesArray[kkk] * rrr;
         m_lsum+=m_SeriesArray[kkk];
         m_weight+=iii;
        }
     }
   else if(BarCheck3(begin,bar,set,Length_)) return(EMPTY_VALUE);

//----+  LWMA
   m_sum+= series * Length - m_lsum;
   kkk = Recount_ArrayNumber(m_count,Length_,Length);
   m_lsum+= series - m_SeriesArray[kkk];
   lwma = m_sum / m_weight;

//----+ saving values of variables
   if(BarCheck4(rates_total,bar,set))
     {
      m_SUM  = m_sum;
      m_LSUM = m_lsum;
     }

//----+ restoring values of variables
   if(BarCheck5(rates_total,bar,set))
     {
      m_sum=m_SUM;
      m_lsum=m_LSUM;
     }
//----+
   return(lwma);
  }
//+X================================================================X+
//|  Calculation of the standard deviation                           |
//+X================================================================X+    
double CStdDeviation::StdDevSeries
 (
  uint begin, // number of beginning of bars for reliable calculation
  uint prev_calculated, // amount of history in bars at previous tick
  uint rates_total, // amount of history in bars at the current tick
  int Length, // period of averaging
  double deviation, // deviation
  double series,  // value of the price series calculated for the new bar with number 'bar'
  double MovSeries, // Value of a moving average, on which basis the StdDeviation is calculated
  uint bar,  // bar number
  bool set // direction of indexing arrays
 )
// StdDevSeries(begin, prev_calculated, rates_total, period, deviation, Series, MovSeries, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  //---- Checking the beginning of bars for reliable calculation 
  if (BarCheck1(begin, bar, set)) return(EMPTY_VALUE);
  
  //----+ Declaration of local variables
  int iii, kkk;
  double StdDev, m_SumX2;

  //----+ Changing the size of array of variables
   if(bar==begin && !SeriesArrayResize(__FUNCTION__, Length, m_SeriesArray, m_Size_))
      return(EMPTY_VALUE);
  
  //----+ checking for correctness the external parameter Length
  LengthCheck(Length);
      
  //----+ rearrangement and initialization of cells of the array m_SeriesArray
  Recount_ArrayZeroPos(m_count, Length, prev_calculated, series, bar, m_SeriesArray);
  
  //----+ Initialization of zero
  if(BarCheck2(begin, bar, set, Length))
   {
    m_Sum = 0.0;
    m_Sum2 = 0.0;
    for(iii = 1; iii < Length; iii++)
     {
      kkk = Recount_ArrayNumber(m_count, Length, iii); 
      m_Sum += m_SeriesArray[kkk];
      m_Sum2 += MathPow(m_SeriesArray[kkk], 2);
     }
   }              
  else if(BarCheck3(begin, bar, set, Length)) return(EMPTY_VALUE);
   
  //----+  StdDev
  m_Sum += series;
  m_Sum2 += MathPow(series, 2);
  m_SumX2 = Length * MathPow(MovSeries, 2) - 2 * MovSeries * m_Sum + m_Sum2;
  
  kkk = Recount_ArrayNumber(m_count, Length, Length - 1);
  m_Sum2 -=  MathPow(m_SeriesArray[kkk], 2);
  m_Sum -= m_SeriesArray[kkk];
  
  StdDev = deviation * MathSqrt(m_SumX2 / Length);
  
  //----+ restoring values of variables
  if (BarCheck5(rates_total, bar, set))
   { 
    m_Sum = m_SUM;
    m_Sum2 = m_SUM2;
   }
   
  //----+ saving values of variables
  if (BarCheck4(rates_total, bar, set)) 
   {
    m_SUM = m_Sum;
    m_SUM2 = m_Sum2;
   }  
//----+
  return(StdDev); 
 }
//+X================================================================X+
//| JMA Averaging                                                    |
//+X================================================================X+
double CJJMA::JJMASeries
 (
  uint begin, // number of beginning of bars for reliable calculation
  uint prev_calculated, // amount of history in bars at previous tick
  uint rates_total, // amount of history in bars at the current tick
  int  Din, // permission to change the parameters Length and Phase at every bar. 
                 // 0 - prohibition to change the parameters, any other values means permission.
  double Phase, // parameter that can change withing the range -100 ... +100, it depends on the quality of intermediate process of averaging
  double Length, // depth of smoothing
  double series,  // value of the price series calculated for the new bar with number 'bar'
  uint bar,  // bar number
  bool set // direction of indexing arrays
 )
// JMASeries(begin, prev_calculated, rates_total, Din, Phase, Length, series, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  //---- Checking the beginning of bars for reliable calculation 
  if (BarCheck1(begin, bar, set)) return(EMPTY_VALUE);
  
  //----+ Initialization of coefficients
  JJMAInit(begin, Din, Phase, Length, series, bar); 
  
  //----+ Declaration of local variables
  int s60, s68, s58, fD8;
  int kkk, ddd;
  //----
  double fA0, fAPow;
  double dSer5, dSer6, jjma;
  double v6, f30, f40, fE0, fE8;
  double v5, f48, f58, f68, f28;
  double fB0, fD0, f8, f60, f20;
  double vv, v4, f70, s10, s20;
  //----
  string word;
     
  //----+
  if (m_LP1 < 61)
   {
    m_LP1++; 
    m_buffer[m_LP1] = series;
   }
  //-x-x-x-x-x-x-x-+  <<< Calculation of the function JMASeries() >>> 
  if (m_LP1 > 30)
   {
    if (!m_f0)
     {
      m_f0 = true; 
      v5 = 1;
      fD8 = 29;
      //----
      m_f38 = m_buffer[1];
      m_f18 = m_buffer[1];
     }
    else fD8 = 0;
    //-S-S-S-S-+
    for(int rrr = fD8; rrr >= 0; rrr--)
     {
      if (rrr == 0) 
           f8 = series; 
      else f8 = m_buffer[31 - rrr];
      //----
      f28 = f8 - m_f18;
      f48 = f8 - m_f38;
      //----
      if (MathAbs(f28) > MathAbs(f48))
           m_v2 = MathAbs(f28); 
      else m_v2 = MathAbs(f48);
      //----
      fA0 = m_v2; 
      vv = fA0 + 0.0000000001; //{1.0e-10;}
      
      if (m_s48 <= 1) 
           m_s48 = 127; 
      else m_s48--;
      //----
      if (m_s50 <= 1)
           m_s50 = 10;  
      else m_s50--;
      //----
      if (m_s70 < 128) 
            m_s70++;
      //----      
      m_s8 += vv - m_ring2[m_s50];
      //----
      m_ring2[m_s50] = vv;
      m_bRING2[m_s50] = true;
      //----
      if (m_s70 > 10)
           s20 = m_s8 / 10.0; 
      else s20 = m_s8 / m_s70;
      //----
      if (m_s70 > 127)
       {
        s10 = m_ring1[m_s48];
        m_ring1[m_s48] = s20;
        m_bRING1[m_s48] = true;
        s68 = 64; 
        s58 = s68;
        //----
        while (s68 > 1)
         {
          if (m_list[s58] < s10)
           {
            s68 /= 2.0; 
            s58 += s68;
           }
          else 
           if (m_list[s58]<= s10)s68 = 1;
           else
            {
             s68 /= 2.0; 
             s58 -= s68;
            }
         }
       }
      else
       {
        m_ring1[m_s48] = s20;
        m_bRING1[m_s48] = true;
        //----
        if (m_s28 + m_s30 > 127)
         {
          m_s30--; 
          s58 = m_s30;
         }
        else
         {
          m_s28++; 
          s58 = m_s28;
         }
        //----
        if (m_s28 > 96)
             m_s38 = 96;
        else m_s38 = m_s28;
        //----
        if (m_s30 < 32) 
             m_s40 = 32; 
        else m_s40 = m_s30;
       }
      //----
      s68 = 64; 
      s60 = s68;
      //----
      while (s68 > 1)
       {
        if (m_list[s60] >= s20)
         {
          if (m_list[s60 - 1] <= s20)s68 = 1; 
          else 
           {
            s68 /= 2.0; 
            s60 -= s68;
           }
         }
        else
         {
          s68 /= 2.0; 
          s60 += s68;
         }
        //----
        if (s60 == 127)
         if (s20 > m_list[127]) s60 = 128;
       }
      //----
      if (m_s70 > 127)
       {
        if (s58 >= s60)
         {
          if (m_s38 + 1 > s60)
           if (m_s40 - 1 < s60) m_s18 += s20;
          //----     
          else if (m_s40 + 0 > s60)
                if (m_s40 - 1 < s58) 
                   m_s18 += m_list[m_s40 - 1];
         }
        else 
         if (m_s40 >= s60) 
          {
           if (m_s38 + 1 < s60) 
             if (m_s38 + 1 > s58)
               m_s18  += m_list[m_s38 + 1]; 
          }                
         else if (m_s38 + 2 > s60) m_s18 += s20; 
              //----           
              else if (m_s38 + 1 < s60)
                    if (m_s38 + 1 > s58) 
                      m_s18 += m_list[m_s38 + 1];
        //----    
        if (s58 > s60)
         {
          if (m_s40 - 1 < s58)
           if (m_s38 + 1 > s58) 
            m_s18 = m_s18 - m_list[s58];
          //----
          else if (m_s38 < s58)
                if (m_s38 + 1 > s60) 
                  m_s18 -= m_list[m_s38];
         }
        else
         {
          if (m_s38 + 1 > s58 && m_s40 - 1 < s58)
           m_s18 -= m_list[s58];
          //----                
          else if (m_s40 + 0 > s58)
                if (m_s40 - 0 < s60) 
                  m_s18 -= m_list[m_s40];
         }
       }
      //----
      if (s58 <= s60)
       {
        if (s58 >= s60)
         {
          m_list[s60] = s20;
          m_bLIST[s60] = true;
         }
        else
         {
          for(ddd = s58 + 1; ddd <= s60 - 1; ddd++)
           {
            m_list[ddd - 1] = m_list[ddd];
            m_bLIST[ddd - 1] = true;
           }
          //----             
          m_list[s60 - 1] = s20;
          m_bLIST[s60 - 1] = true;
         }
       }
      else
       {
        for(ddd = s58 - 1; ddd >= s60; ddd--)
         {
          m_list[ddd + 1] = m_list[ddd];
          m_bLIST[ddd + 1] = true;
         }
        //----                
        m_list[s60] = s20;
        m_bLIST[s60] = true;
       }
      //---- 
      if (m_s70 <= 127)
       {
        m_s18 = 0;
        for( ddd = m_s40; ddd <= m_s38; ddd++) 
         m_s18 += m_list[ddd];
       }
      //---- 
      f60 = m_s18 / (m_s38 - m_s40 + 1.0);
      //----
      if (m_LP2 > 30) 
           m_LP2 = 31; 
      else m_LP2++;
      //----
      if (m_LP2 <= 30)
       {
        if (f28 > 0.0) 
             m_f18 = f8; 
        else m_f18 = f8 - f28 * m_f90;
        //----
        if (f48 < 0.0) 
             m_f38 = f8; 
        else m_f38 = f8 - f48 * m_f90;
        //----
        m_JMA = series;
        //----
        if (m_LP2 != 30) continue;               
        else
         {
          m_fC0 = series;
          if (MathCeil(m_f78) >= 1) 
               v4 = MathCeil(m_f78);
          else v4 = 1.0;
          //----
          if (v4 > 0)fE8 = MathFloor(v4);
          else
           {
            if (v4 < 0)
                 fE8 = MathCeil(v4);
            else fE8 = 0.0;
           }
          //----
          if (MathFloor(m_f78) >= 1) 
               m_v2 = MathFloor(m_f78); 
          else m_v2 = 1.0;
          //----
          if (m_v2 > 0)fE0 = MathFloor(m_v2);
          else
           {
            if (m_v2 < 0)
                 fE0 = MathCeil(m_v2);
            else fE0 = 0.0;
           }
          //----
          if (fE8 == fE0)f68 = 1.0; 
          else 
           {
            v4 = fE8 - fE0; 
            f68 = (m_f78 - fE0) / v4;
           }
          //---- 
          if (fE0 <= 29)
               v5 = fE0; 
          else v5 = 29;
          //----
          if (fE8 <= 29) 
               v6 = fE8; 
          else v6 = 29;
          
          dSer5 = series - m_buffer[m_LP1 - v5];
          dSer6 = series - m_buffer[m_LP1 - v6]; 
          //----
          m_fA8 = dSer5 * (1.0 - f68) / fE0 + dSer6 * f68 / fE8;
         }
       }
      else
       {
        fAPow = MathPow(fA0/f60, m_f88); 
        //----
        if (m_f98 >= fAPow) 
             m_v1 = fAPow;
        else m_v1 = m_f98;
        //----
        if (m_v1 < 1.0)m_v2 = 1.0;
        else
         {
          if (m_f98 >= fAPow) 
               m_v3 = fAPow;             
          else m_v3 = m_f98;
          
          m_v2 = m_v3;
         }
        //---- 
        f58 = m_v2; 
        f70 = MathPow(m_f90, MathSqrt(f58));
        //----
        if (f28 > 0.0) 
             m_f18 = f8;
        else m_f18 = f8 - f28 * f70;
        //----
        if (f48 < 0.0)
             m_f38 = f8; 
        else m_f38 = f8 - f48 * f70;
       }
     }
    //---- 
    if (m_LP2 > 30)
     {
      f30 = MathPow(m_Kg, f58);
      //----
      m_fC0 *= f30;
      m_fC0 += (1.0 - f30) * series;
      //----
      m_fC8 *= m_Kg;
      m_fC8 += (series - m_fC0) * (1.0 - m_Kg);
      //----
      fD0 = m_Pf * m_fC8 + m_fC0;
      f20 = f30 *(-2.0);
      //----
      f40 = f30 * f30;
      fB0 = f20 + f40 + 1.0;
      //----
      m_fA8 *= f40;
      m_fA8 += (fD0 - m_JMA) * fB0;
      //----
      m_JMA += m_fA8;
     }
   }
  //-x-x-x-x-x-x-x-+
  
  if (m_LP1 <= 30) return(EMPTY_VALUE);
  jjma = m_JMA;
  
  //----+ restoring values of variables
  if (BarCheck5(rates_total, bar, set))
   { 
    //---- Restoring modified cells of arrays from memory
    for(kkk = 0; kkk < 128; kkk++) if (m_bLIST [kkk]) m_list [kkk] = m_LIST [kkk];
    for(kkk = 0; kkk < 128; kkk++) if (m_bRING1[kkk]) m_ring1[kkk] = m_RING1[kkk]; 
    for(kkk = 0; kkk < 11;  kkk++) if (m_bRING2[kkk]) m_ring2[kkk] = m_RING2[kkk];
    
    //---- Zeroizing numbers of modified cells of arrays
    ArrayInitialize(m_bRING1, false);
    ArrayInitialize(m_bRING2, false);
    ArrayInitialize(m_bLIST,  false);
    
    //---- writing values of variables from the memory
     m_fC0 = m_fC0_; 
     m_fC8 = m_fC8_; 
     m_fA8 = m_fA8_;
     m_s8  = m_s8_ ; 
     m_f18 = m_f18_; 
     m_f38 = m_f38_;
     m_s18 = m_s18_; 
     m_JMA = m_JMA_; 
     m_s38 = m_s38_;
     m_s48 = m_s48_; 
     m_s50 = m_s50_; 
     m_LP1 = m_LP1_;
     m_LP2 = m_LP2_; 
     m_s40 = m_s40_; 
     m_s70 = m_s70_;
    }

  //----+ saving values of variables
  if (BarCheck4(rates_total, bar, set)) 
   {   
    //---- writing modified cells of arrays to the memory
    for(kkk = 0; kkk < 128; kkk++) if (m_bLIST [kkk]) m_LIST [kkk] = m_list [kkk];
    for(kkk = 0; kkk < 128; kkk++) if (m_bRING1[kkk]) m_RING1[kkk] = m_ring1[kkk]; 
    for(kkk = 0; kkk < 11;  kkk++) if (m_bRING2[kkk]) m_RING2[kkk] = m_ring2[kkk];   
    
    //---- Zeroizing numbers of modified cells of arrays
    ArrayInitialize(m_bRING1, false);
    ArrayInitialize(m_bRING2, false);
    ArrayInitialize(m_bLIST,  false); 
    
    //---- writing values of variables to the memory
    m_fC0_ = m_fC0; 
    m_fC8_ = m_fC8; 
    m_fA8_ = m_fA8;
    m_s8_  = m_s8 ; 
    m_f18_ = m_f18; 
    m_f38_ = m_f38;
    m_s18_ = m_s18; 
    m_JMA_ = m_JMA; 
    m_s38_ = m_s38;
    m_s48_ = m_s48; 
    m_s50_ = m_s50; 
    m_LP1_ = m_LP1;
    m_LP2_ = m_LP2; 
    m_s40_ = m_s40; 
    m_s70_ = m_s70;
   } 
   
//----+  End of calculations of the function JMASeries()
  return(jjma);
 }
//+X================================================================X+
//|  Initialization of variables of the JMA algorithm                |
//+X================================================================X+    
void CJJMA::JJMAInit
 (
  uint begin,
  int  Din,
  double Phase,
  double Length,
  double series,
  uint bar
 )
// Jm_JMAInit(begin, Din, Phase, Length, series, bar) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+  
  //----+ <<< Calculation of coefficients >>> 
  if (bar == begin || Din != 0) 
   {
    if (bar == begin)
     {
      m_s28 = 63;
      m_s30 = 64;
      m_f0 = false;
      
      //----
      for(int kkk = 0;   kkk <= m_s28; kkk++) m_list[kkk] = -1000000.0;                     
      for(int kkk = m_s30; kkk <= 127; kkk++) m_list[kkk] = +1000000.0;
      
      //---- All cells of arrays must be overwritten
      ArrayInitialize(m_bRING1, true);
      ArrayInitialize(m_bRING2, true);
      ArrayInitialize(m_bLIST,  true);
      
      //----+ Deleting trash from arrays at repeated initializations 
      ArrayInitialize(m_RING1,  0.0);
      ArrayInitialize(m_RING2,  0.0);
      ArrayInitialize(m_ring1,  0.0);
      ArrayInitialize(m_ring2,  0.0);
      ArrayInitialize(m_buffer, 0.0);
      //----
      m_fC0 = 0.0; 
      m_fC8 = 0.0; 
      m_fA8 = 0.0;
      m_s8  = 0.0; 
      m_f18 = 0.0; 
      m_f38 = 0.0;
      m_s18 = 0.0; 
      m_JMA = series; 
      m_s38 = 0.0;
      m_s48 = 0.0; 
      m_s50 = 0.0; 
      m_LP1 = 0.0;
      m_LP2 = 0.0; 
      m_s40 = 0.0; 
      m_s70 = 0.0;
     }
    
    double Dr, Ds, Dl;
    //----
    if (Length >= 1.0000000002) 
         Dr = (Length - 1.0) / 2.0;
    else Dr = 0.0000000001;
    //----
    if (Phase >= -100 && Phase <= 100)
      m_Pf = Phase / 100.0 + 1.5;
    //----        
    if (Phase > +100) m_Pf = 2.5;
    if (Phase < -100) m_Pf = 0.5;
    //----
    Dr *= 0.9; 
    m_Kg = Dr / (Dr + 2.0);
    Ds = MathSqrt(Dr); 
    Dl = MathLog(Ds); 
    m_v1= Dl;
    m_v2 = m_v1;
    //----
    Dr = MathLog(2.0);
    if (m_v1 / Dr + 2.0 < 0.0) 
         m_v3= 0.0;
    else m_v3 = (m_v2 / Dr) + 2.0;
    //----
    m_f98 = m_v3;
    //----
    if (m_f98 >= 2.5) 
         m_f88 = m_f98 - 2.0;
    else m_f88= 0.5;
    //----
    m_f78 = Ds * m_f98; 
    m_f90= m_f78 / (m_f78 + 1.0);
   }
//----+
 }
//+X================================================================X+
//| Checking the depth of averaging Length for correctness           |
//+X================================================================X+
void CJJMA::JJMALengthChack(string LengthName, int ExternLength)

// Jm_JMALengthChack(LengthName, ExternLength)
 {
//----+
  //---- writing messages about unacceptable values of input parameters  
  if (ExternLength < 1)
   {
    string word;
    StringConcatenate
      (word, __FUNCTION__, " (): Parameter ", LengthName,
            " must be no less than 1. You specified incorrect value",
                            ExternLength, " value 1 will be used");
    Print(word);
    return;
   }
//----+
 }
//+X================================================================X+
//| Checking the correctness of the Phase parameter of averaging     |
//+X================================================================X+
void CJJMA::JJMAPhaseChack(string PhaseName, int ExternPhase)

// Jm_JMAPhaseChack(PhaseName, ExternPhase)
 {
//----+
  //---- writing messages about unacceptable values of input parameters 
  if (ExternPhase < -100)
   {
    string word;
    StringConcatenate
      (word, __FUNCTION__, " (): Parameter ", PhaseName,
            " must be no less than -100. You specified incorrect value",
                            ExternPhase, " value -100 will be used");
    Print(word);
    return;
   }
  //----
  if (ExternPhase > +100)
   {
    string word;
    StringConcatenate
      (word, __FUNCTION__, " (): Parameter ", PhaseName,
            " must not exceed +100. You specified incorrect value",
                            ExternPhase, " value +100 will be used");
    Print(word);
    return;
   }
//----+
 }
//+X================================================================X+
//|  T3 averaging                                                    |
//+X================================================================X+
double CT3::T3Series
 (
  uint begin, // number of beginning of bars for reliable calculation
  uint prev_calculated, // amount of history in bars at previous tick
  uint rates_total, // amount of history in bars at the current tick
  int  Din, // permission to change the parameter Length at every bar. 
                    // 0 - prohibition to change the parameters, any other values means permission.
  double Curvature, // coefficient (its value is increased 100 times for convenience!)
  double Length, // depth of smoothing
  double series,  // value of the price series calculated for the new bar with number 'bar'
  uint bar,  // bar number
  bool set // direction of indexing arrays
 )
// T3Series(begin, prev_calculated, rates_total, Din, Curvature, Length, series, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  //---- Checking the beginning of bars for reliable calculation 
  if (BarCheck1(begin, bar, set)) return(EMPTY_VALUE);
  
  //----+ Declaration of local variables
  double e0, T3_;
  
  //----+ checking for correctness the external parameter Length
  LengthCheck(Length);	 
  
  //----+ Calculation of coefficients
  T3Init(begin, Din, Curvature, Length, series, bar);
   
  e0 = series; 
  //----+ <<< calculation of T3 >>> 
  m_e1 = m_w1 * e0 + m_w2 * m_e1;
  m_e2 = m_w1 * m_e1 + m_w2 * m_e2;
  m_e3 = m_w1 * m_e2 + m_w2 * m_e3;
  m_e4 = m_w1 * m_e3 + m_w2 * m_e4;
  m_e5 = m_w1 * m_e4 + m_w2 * m_e5;
  m_e6 = m_w1 * m_e5 + m_w2 * m_e6;
  //----  
  T3_ = m_c1 * m_e6 + m_c2 * m_e5 + m_c3 * m_e4 + m_c4 * m_e3;
  
  //----+ restoring values of variables
  if (BarCheck5(rates_total, bar, set))
   { 
    m_e1 = m_E1; 
    m_e2 = m_E2; 
    m_e3 = m_E3; 
    m_e4 = m_E4; 
    m_e5 = m_E5;
    m_e6 = m_E6;
   }
  
  //----+ saving values of variables
  if (BarCheck4(rates_total, bar, set))
   {

    m_E1 = m_e1; 
    m_E2 = m_e2; 
    m_E3 = m_e3; 
    m_E4 = m_e4; 
    m_E5 = m_e5; 
    m_E6 = m_e6;
   }
 
//----+ End of calculation of value of the function T3Series()
  return(T3_);
 }
//+X================================================================X+
//|  Initialization of variables of the T3 algorithm                 |
//+X================================================================X+    
void CT3::T3Init
 (
  uint begin,
  int Din,
  double Curvature,
  double Length,
  double series,
  uint bar
 )
// T3InitInit(begin, Din, Curvature, Length, series, bar) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+  
  //----+ <<< Calculation of coefficients >>>
  if (bar == begin || Din != 0) 
   {
    double b = Curvature / 100.0;
    m_b2 = b * b;
    m_b3 = m_b2 * b;
    m_c1 = -m_b3;
    m_c2 = (3 * (m_b2 + m_b3));
    m_c3 = -3 * (2 * m_b2 + b + m_b3);
    m_c4 = (1 + 3 * b + m_b3 + 3 * m_b2);
    double n  = 1 + 0.5 * (Length - 1);
    m_w1 = 2 / (n + 1);
    m_w2 = 1 - m_w1;
    
    if (bar == begin)
     {
      m_e1 = series; 
      m_e2 = series; 
      m_e3 = series; 
      m_e4 = series; 
      m_e5 = series;
      m_e6 = series;
     }
   }
//----+
 }
//+X================================================================X+
//| Ultralinear averaging                                            |
//+X================================================================X+    
double CJurX::JurXSeries
 (
  uint begin, // number of beginning of bars for reliable calculation
  uint prev_calculated, // amount of history in bars at previous tick
  uint rates_total, // amount of history in bars at the current tick
  int  Din, // permission to change the parameter Length at every bar 
                 // 0 - prohibition to change the parameters, any other values means permission.
  double Length, // depth of smoothing
  double series,  // value of the price series calculated for the new bar with number 'bar'
  uint bar,  // bar number
  bool set // direction of indexing arrays
 )
// JurXSeries(begin, prev_calculated, rates_total, Din, Length, series, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  //---- Checking the beginning of bars for reliable calculation 
  if (BarCheck1(begin, bar, set)) return(EMPTY_VALUE);
  
  //----+ Declaration of local variables
  double V1, V2, JurX_;

  //----+ checking for correctness the external parameter Length
  LengthCheck(Length);	 
	 
	 //----+ Initialization of coefficients
  JurXInit(begin, Din, Length, series, bar);
	 
 	//---- <<< calculation of JurX >>> 
 	m_f1   =  m_Hg * m_f1 + m_Kg * series;
	 m_f2   =  m_Kg * m_f1 + m_Hg * m_f2;
	 V1     =  m_AC * m_f1 - m_AB * m_f2;
	 m_f3   =  m_Hg * m_f3 + m_Kg * V1;
	 m_f4   =  m_Kg * m_f3 + m_Hg * m_f4;
 	V2     =  m_AC * m_f3 - m_AB * m_f4;
	 m_f5   =  m_Hg * m_f5 + m_Kg * V2;
	 m_f6   =  m_Kg * m_f5 + m_Hg * m_f6;
	 JurX_  =  m_AC * m_f5 - m_AB * m_f6;
	 
	 //----+ restoring values of variables
  if (BarCheck5(rates_total, bar, set))
   { 
	   m_f1 = m_F1; 
	   m_f2 = m_F2; 
	   m_f3 = m_F3; 
	   m_f4 = m_F4; 
	   m_f5 = m_F5; 
	   m_f6 = m_F6;
   }
    
	 //----+ saving values of variables
  if (BarCheck4(rates_total, bar, set)) 
	  {
	   m_F1 = m_f1; 
	   m_F2 = m_f2; 
	   m_F3 = m_f3; 
	   m_F4 = m_f4; 
	   m_F5 = m_f5; 
	   m_F6 = m_f6;
	  }
	 
//----+ End of calculation of value of the functions JurX.Series
  return(JurX_);
	
 }
//+X================================================================X+
//|  Initialization of variables of the algorithm JurX               |
//+X================================================================X+    
void CJurX::JurXInit
 (
  uint begin,
  int Din,
  double Length,
  double series,
  uint bar
 )
// JurXInit(begin, Din, Length, series, bar) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+  
  if (bar == begin || Din != 0) 
   {
	   if (Length >= 6) 
	        m_w = Length - 1;        
	   else m_w = 5; 
	  
	   m_Kg = 3 / (Length + 2.0); 
	   m_Hg = 1.0 - m_Kg;
	   //----
	   if (bar == begin)
	    {
	     m_f1 = series; 
	     m_f2 = series; 
	     m_f3 = series; 
	     m_f4 = series; 
	     m_f5 = series; 
	     m_f6 = series;
	     
	     m_AB = 0.5;
	     m_AC = 1.5;
	    }
	 }
//----+
 }
//+X================================================================X+
//|  CMOSeries() function                                            |
//+X================================================================X+    
double CCMO::CMOSeries
 (
  uint begin, // number of beginning of bars for reliable calculation
  uint prev_calculated, // amount of history in bars at previous tick
  uint rates_total, // amount of history in bars at the current tick
  int CMO_Length, // CMO period
  double series,  // value of the price series calculated for the new bar with number 'bar'
  uint bar,  // bar number
  bool set // direction of indexing arrays
 )
// CMOSeries(begin, prev_calculated, rates_total, CMO_Length, series, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  //---- Checking the beginning of bars for reliable calculation 
  if (BarCheck1(begin, bar, set)) return(EMPTY_VALUE);
  
  //----+ Declaration of local variables;
  double dseries, abcmo;
  int iii, rrr, size = CMO_Length + 1;
  
  //----+ Changing the size of array of variables
  if (bar == begin && !SeriesArrayResize(__FUNCTION__, size, m_dSeriesArray, m_Size_))
    return(EMPTY_VALUE);
  
  //----+ checking the external parameter CMO_Length for correctness
  LengthCheck(CMO_Length);

  //---- checking whether there are enough bars 
  if (BarCheck1(begin + 1, bar, set)) return(EMPTY_VALUE);
  
  //----+ rearrangement of cells of the array SeriesArray
  Recount_ArrayZeroPos(m_count, size, prev_calculated, series - m_series1, bar, m_dSeriesArray);
   
  //----+ Initialization of zero
  if (BarCheck2(begin, bar, set, CMO_Length + 3))
   {         
    m_UpSum = 0.0;
    m_DnSum = 0.0;
    
    for(iii = 1; iii < CMO_Length; iii++)
     {
      rrr = Recount_ArrayNumber(m_count, size, iii);      
      dseries = m_dSeriesArray[rrr];
     
      if (dseries > 0) m_UpSum += dseries;        
      if (dseries < 0) m_DnSum -= dseries;
     } 

    m_AbsCMO = 0.000000001;
   }
  else if (BarCheck3(begin, bar, set, CMO_Length + 3))
   {
    m_series1 = series;
    return(EMPTY_VALUE);
   }             

	 dseries = m_dSeriesArray[m_count];
  if (dseries > 0) m_UpSum += dseries;        
  if (dseries < 0) m_DnSum -= dseries;
  if (m_UpSum + m_DnSum > 0) 
   m_AbsCMO = MathAbs((m_UpSum - m_DnSum) / (m_UpSum + m_DnSum));
   abcmo = m_AbsCMO;                                      
  //----
  rrr = Recount_ArrayNumber(m_count, size, CMO_Length - 1);      
  dseries = m_dSeriesArray[rrr];
  if (dseries > 0) m_UpSum -= dseries;        
  if (dseries < 0) m_DnSum += dseries;
  
  //----+ restoring values of variables
  if (BarCheck5(rates_total, bar, set))
   { 
    m_AbsCMO = m_AbsCMO_;
    m_UpSum = m_UpSum_;
    m_DnSum = m_DnSum_;
    m_series1 = m_series1_;
   }
  else m_series1 = series;
  
  //----+ saving values of variables
  if (BarCheck4(rates_total, bar, set))
   {
     m_AbsCMO_ = m_AbsCMO;
    m_UpSum_ = m_UpSum;
    m_DnSum_ = m_DnSum;
    m_series1_ = m_series1;
   }
//----+
  return(abcmo); 
 }
//+X================================================================X+
//|  VIDYASeries() function                                          |
//+X================================================================X+    
double CCMO::VIDYASeries
 (
  uint begin, // number of beginning of bars for reliable calculation
  uint prev_calculated, // amount of history in bars at previous tick
  uint rates_total, // amount of history in bars at the current tick
  int CMO_Length, // CMO period
  double EMA_Length, // EMA period
  double series,  // value of the price series calculated for the new bar with number 'bar'
  uint bar,  // bar number
  bool set // direction of indexing arrays
 )
// VIDYASeries(begin, prev_calculated, rates_total, CMO_Length, EMA_Length, series, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  //----+ Declaration of local variables
  double vidya, CMO_ = CMOSeries(begin, prev_calculated, rates_total, CMO_Length, series, bar, set);
 
  //----+ Initialization of zero
  if (BarCheck2(begin, bar, set, CMO_Length + 3))
   {             	   
	   m_Vidya = series;
	   //---- Initialization of the factor of smoothing of 
    m_SmoothFactor = 2.0 / (EMA_Length + 1.0);
   }
  else if (BarCheck3(begin, bar, set, CMO_Length + 3)) return(EMPTY_VALUE);             

  //----
  CMO_ *= m_SmoothFactor;                                            
  m_Vidya = CMO_ * series + (1 - CMO_) * m_Vidya;
  vidya = m_Vidya;
  
  //----+ restoring values of variables
  if (BarCheck5(rates_total, bar, set))
   { 
    m_Vidya = m_Vidya_;
   }
   
  //----+ saving values of variables
  if (BarCheck4(rates_total, bar, set))
   {
     m_Vidya_ = m_Vidya;
   } 
//----+
  return(vidya); 
 }
//+X================================================================X+
//|  Kaufman's averaging                                             |
//+X================================================================X+    
double CAMA::AMASeries
 (
  uint begin, // number of beginning of bars for reliable calculation
  uint prev_calculated, // amount of history in bars at previous tick
  uint rates_total, // amount of history in bars at the current tick
  int Length, // period of AMA
  int Fast_Length, // period of fast moving average
  int Slow_Length, // period of slow moving average
  double Rate, // rate of the smoothing constant
  double series,  // value of the price series calculated for the new bar with number 'bar'
  uint bar,  // bar number
  bool set // direction of indexing arrays
 )
// AMASeries(symbol, timeframe, begin, limit, Length, series, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  //---- Checking the beginning of bars for reliable calculation 
  if (BarCheck1(begin, bar, set)) return(EMPTY_VALUE);
  
  //----+ Declaration of local variables
  double signal, ER, ERSC, SSC, dprice, ama;
  int iii, kkk, rrr, size = Length + 1;
  
  //----+ Changing size of arrays of variables
  if (bar == begin)
   if (!SeriesArrayResize(__FUNCTION__, size, m_SeriesArray,  m_Size_1)    
    || !SeriesArrayResize(__FUNCTION__, size, m_dSeriesArray,  m_Size_2))
     return(EMPTY_VALUE);
  
  //----+ checking for correctness the external parameter Length
  LengthCheck(Length);
  
  //----+ rearrangement and initialization of cells of the array m_SeriesArray
  Recount_ArrayZeroPos( m_count, size, prev_calculated, series, bar, m_SeriesArray);
 
  //---- checking whether there are enough bars 
  if (BarCheck1(begin + 1, bar, set)) return(EMPTY_VALUE);
  
  kkk = Recount_ArrayNumber( m_count, size, 1);
  dprice = series - m_SeriesArray[kkk];
  m_dSeriesArray[ m_count] = dprice;
  
  //----+ Initialization of zero
  if (BarCheck2(begin, bar, set, Length + 3))
   {
    //---- initialization of constants
    rrr = Recount_ArrayNumber( m_count, size, 1); 
     m_Ama = m_SeriesArray[rrr];
     m_slowSC = (2.0 / (Slow_Length + 1));
     m_fastSC = (2.0 / (Fast_Length + 1)); 
     m_dSC =  m_fastSC -  m_slowSC; 
     m_noise = 0.000000001;
 
    for(iii = 1; iii < Length; iii++)
     {
      rrr = Recount_ArrayNumber( m_count, size, iii);      
       m_noise += MathAbs(m_dSeriesArray[rrr]);
     }
   }
  else if (BarCheck3(begin, bar, set, Length + 3)) return(EMPTY_VALUE);             

  //----
   m_noise += MathAbs(dprice);
  rrr = Recount_ArrayNumber( m_count, size, Length); 
  signal = MathAbs(series - m_SeriesArray[rrr]);    
  //----                     
  ER = signal /  m_noise;
  ERSC = ER *  m_dSC;
  SSC = ERSC +  m_slowSC;
   m_Ama =  m_Ama + (MathPow(SSC, Rate) * (series -  m_Ama));
  ama =  m_Ama;
  kkk = Recount_ArrayNumber( m_count, size, Length - 1);      
   m_noise -= MathAbs(m_dSeriesArray[kkk]);
   
  //----+ restoring values of variables
  if (BarCheck5(rates_total, bar, set))
   { 
     m_noise =  m_NOISE;
     m_Ama =  m_AMA_;
   }
   
  //----+ saving values of variables
  if (BarCheck4(rates_total, bar, set))
   {
     m_AMA_ =  m_Ama;
     m_NOISE =  m_noise;
   }
//----+
  return(ama); 
 }
//+X================================================================X+
//| Parabolic averaging                                              |
//+X================================================================X+    
double CParMA::ParMASeries
 (
 uint begin,// number of beginning of bars for reliable calculation
 uint prev_calculated,// amount of history in bars at the previous tick
 uint rates_total,// amount of history in bars at the current tick
 int Length,// period of averaging
 double series,// value of the price series calculated for the new bar with number 'bar'
 uint bar,// bar number
 bool set // direction of indexing arrays
 )
// SMASeries(Number, symbol, timeframe, begin, limit, Length, series, bar, set) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  //---- Checking the beginning of bars for reliable calculation 
  if (BarCheck1(begin, bar, set)) return(EMPTY_VALUE);
  
  //----+ Declaration of local variables
  int iii, kkk;
  //----
  double S, B0, B1, B2, parma;
  double A, B, C, D, E, F; 
  double K, L, M, P, Q, R;
  double sum_y, sum_xy, sum_x2y, var_tmp; 

  //----+ Changing the size of array of variables
   if(bar==begin && !SeriesArrayResize(__FUNCTION__, Length, m_SeriesArray, m_Size_))
      return(EMPTY_VALUE);
    
  //----+ checking for correctness the external parameter Length
  LengthCheck(Length);
  
  //----+ rearrangement and initialization of cells of the array m_SeriesArray
  Recount_ArrayZeroPos(m_count, Length, prev_calculated, series, bar, m_SeriesArray);
  
  //----+ Initialization of zero
  if(BarCheck2(begin, bar, set, Length)) ParMAInit(Length);          
  else if(BarCheck3(begin, bar, set, Length)) return(EMPTY_VALUE);
  
  //----+ Calculation ParMA
  sum_y   = 0.0;
  sum_xy  = 0.0;
  sum_x2y = 0.0;
  //----
  for (iii = 1; iii <= Length; iii++)
   {
    kkk = Recount_ArrayNumber(m_count, Length, Length - iii);
    var_tmp  = m_SeriesArray[kkk];  
    sum_y   += var_tmp;
    sum_xy  += iii * var_tmp;    
    sum_x2y += iii * iii * var_tmp; 
   }
      
 // the difference between two adjacent bars for sum_x2y: Sum(i=0; i<Length){(2*Length* - 1)*Series[i] + 2*i*Series[i]}
   
  // initialization
  A = Length;
  B = m_sum_x;  
  C = m_sum_x2; 
  F = m_sum_x3; 
  M = m_sum_x4; 
  P = sum_y;                   
  R = sum_xy; 
  S = sum_x2y;
  // intermediates
  D = B;  
  E = C;  
  K = C;  
  L = F;
  Q = D / A;  
  E = E - Q * B; 
  F = F - Q * C;  
  R = R - Q * P;  
  Q = K / A;  
  L = L - Q * B;
  M = M - Q * C; 
  S = S - Q * P; 
  Q = L / E;
  // calculate regression coefficients
  B2 = (S - R * Q) / (M - F * Q);
  B1 = (R - F * B2) / E;
  B0 = (P - B * B1 - C * B2) / A;
  // value to be returned - parabolic MA
  parma = B0 + (B1 + B2 * A) * A;
//----+
  return(parma); 
 }
//+X================================================================X+
//|  Initialization of variables of the algorithm of                 |
//|  parabolic averaging                                             |
//+X================================================================X+    
void CParMA::ParMAInit(double Length)
// ParMAInit(Length)
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  uint var_tmp;
  m_sum_x = 0;
  m_sum_x2 = 0;
  m_sum_x3 = 0;
  m_sum_x4 = 0;
    
  for(uint iii = 1; iii <= Length; iii++)
   {
    var_tmp  = iii;
    m_sum_x   += var_tmp; 
    var_tmp *= iii;
    m_sum_x2  += var_tmp; 
    var_tmp *= iii;
    m_sum_x3  += var_tmp; 
    var_tmp *= iii;
    m_sum_x4  += var_tmp; 
   }
//----+
 }
//+X================================================================X+
//| Checking correctness of the period of averaging                  |
//+X================================================================X+
void CMovSeriesTools::MALengthCheck(string LengthName, int ExternLength)

// MALengthCheck(LengthName, ExternLength)
 {
//----+ 
  if (ExternLength < 1)
   {
    string word;
    StringConcatenate
      (word, __FUNCTION__, " (): Parameter ", LengthName,
            " must be no less than 1. You specified incorrect value",
                            ExternLength, " value 1 will be used");
    Print(word);
    return;
   }
//----+
 }
//+X================================================================X+
//| Checking correctness of the period of averaging                  |
//+X================================================================X+
void CMovSeriesTools::MALengthCheck(string LengthName, double ExternLength)

// MALengthCheck(LengthName, ExternLength)
 {
//----+ 
  if (ExternLength < 1)
   {
    string word;
    StringConcatenate
      (word, __FUNCTION__, " (): Parameter ", LengthName,
            " must be no less than 1. You specified incorrect value",
                            ExternLength, " value 1 will be used");
    Print(word);
    return;
   }
//----+
 }
//+X================================================================X+
//| Checking if a bar withing the range of calculation               |
//+X================================================================X+
bool CMovSeriesTools::BarCheck1(int begin, int bar, bool Set)

// BarCheck1(begin, bar, Set)
 {
//----+
  if (!Set && bar < begin || Set && bar > begin) return(true);
//----+
  return(false);
 }
//+X================================================================X+
//| Checking bar for the start of calculation                        |
//+X================================================================X+
bool CMovSeriesTools::BarCheck2(int begin, int bar, bool Set, int Length)

// BarCheck2(begin, bar, Set, Length)
 {
//----+
  if (!Set&&bar == begin + Length - 1 || Set&&bar == begin - Length + 1)
   return(true);
//----+
  return(false);
 }
//+X================================================================X+
//| Checking bar for absence of bars for averaging                   |
//+X================================================================X+
bool CMovSeriesTools::BarCheck3(int begin, int bar, bool Set, int Length)

// BarCheck3(begin, bar, Set, Length)
 {
//----+
  if (!Set&&bar < begin + Length - 1 || Set&&bar > begin - Length + 1)
   return(true);
//----+
  return(false);
 }
//+X================================================================X+
//| Checking bar at the moment of saving information                 |
//+X================================================================X+
bool CMovSeriesTools::BarCheck4(int rates_total, int bar, bool Set)

// BarCheck4(rates_total, bar, Set)
 {
//----+
  //----+ saving values of variables
  if (!Set&&bar == rates_total - 2 || Set&&bar == 1) return(true); 
//----+
  return(false);
 }
//+X================================================================X+
//| Checking bar at the moment of restoring information              |
//+X================================================================X+
bool CMovSeriesTools::BarCheck5(int rates_total, int bar, bool Set)

// BarCheck5(rates_total, begin, bar, set)
 {
//----+
  //----+ restoring values of variables
  if (!Set&&bar == rates_total - 1 || Set&&bar == 0) return(true);
//----+
  return(false);
 }
//+X================================================================X+
//| Changing incorrect period of averaging                           |
//+X================================================================X+
void CMovSeriesTools::LengthCheck(int& ExternLength)

// LengthCheck(LengthName, ExternLength)
 {
//----+
  if (ExternLength < 1) ExternLength = 1;  
//----+
 }
//+X================================================================X+
//| Changing incorrect period of averaging                           |
//+X================================================================X+
void CMovSeriesTools::LengthCheck(double& ExternLength)

// LengthCheck(ExternLength)
 {
//----+
  if (ExternLength < 1) ExternLength = 1;  
//----+
 }
//+X================================================================X+
//|  Recalculation of position of a newest element in the array      |
//+X================================================================X+    
void CMovSeriesTools::Recount_ArrayZeroPos
 (
  int& count, // Return the current value of the price series by the link
  int Length,
  int prev_calculated,
  double series, // value of the price series calculated for the new bar with number 'bar'
  int bar,
  double& Array[]
 )
// Recount_ArrayZeroPos(count, Length, prev_calculated, series, bar, Array[])
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  if (bar != prev_calculated - 1)
   {
    count--;
    if (count < 0) count = Length - 1;
   }

  Array[count] = series;
//----+
 }
//+X================================================================X+
//|  Transformation of a timeseries number into an array position    |
//+X================================================================X+    
int CMovSeriesTools::Recount_ArrayNumber
 (
  int count,  // Number of the current value of the price series
  int Length,
  int Number // Position of the requested value relatively to the current bar 'bar'
  )
// Recount_ArrayNumber(count, Length, Number)
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+
  int ArrNumber = Number + count;
  
  if (ArrNumber > Length - 1) ArrNumber -= Length;
//----+
  return(ArrNumber);
 }
//+X================================================================X+
//|  Changing size of the array Array[]                              |
//+X================================================================X+    
bool CMovSeriesTools::SeriesArrayResize
 (
  string FunctionsName, // Name of function, in which the size is changed
  int Length, // new size of array
  double& Array[], // array that is changed
  int& Size_ // new size of the array
 )
// SeriesArrayResize(FunctionsName, Length, Array, Size_) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+    
  
  //----+ Changing the size of array of variables
  if (Length > Size_)
   {
    uint Size = Length + 1;
    
    if (ArrayResize(Array, Size) == -1)
     {
      ArrayResizeErrorPrint(FunctionsName, Size_);
      return(false);
     }

    Size_ = Size;
   }
//----+
  return(true);
 }
//+X================================================================X+
//|  Writing the error of changing size of the array into log file   |
//+X================================================================X+    
bool CMovSeriesTools::ArrayResizeErrorPrint
 (
  string FunctionsName,
  int& Size_
 )
// ArrayResizeErrorPrint(FunctionsName, Size_) 
//+ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -+
 {
//----+    
  string lable, word;
  StringConcatenate(lable, FunctionsName, "():");
  StringConcatenate(word, lable, " Error!!! Failed to change",
            " the size of array of variables of the function ", FunctionsName, "()!");
  Print(word);
  //----             
  int error = GetLastError();
  ResetLastError();
  //----
  if (error > 4000)
   {
    StringConcatenate(word, lable, "(): Error code ", error);
    Print(word);
   } 
	   
   Size_ = -2;
   return(false);
//----+
  return(true);
 }
//+X----------------------+ <<< The End >>> +-----------------------X+
