//+------------------------------------------------------------------+
//|                                                      gammaDF.mqh |
//|                                 (c) 2010.03.22, Vladimir Gomonov |
//|                                            MetaDriver@rambler.ru |
//+------------------------------------------------------------------+
#property copyright "(c) 2010.03.22, Vladimir Gomonov"
#property link      "MetaDriver@rambler.ru"

//#include <math.h>
//#include <assert.h>

//#include "gammaDF.h"
#include <Math\logGamma.mqh>

class GammaDF {
    public:
       GammaDF() {}
       void Init(double _shape, double _scale=1);
       double value(double x);    //   Gamma(x|shape,scale)
       double inv(double p);      //  : Gamma(x|shape,scale)=p
    private:
       double a, shape, scale, lga;
       double fraction(double x);
       double series(double x);
 };

const double zero = 0.0;
const double one = 1.0;
const double two = 2.0;

void GammaDF::Init(double _shape, double _scale)
//  :  a(_shape), shape(_shape), scale(_scale), lga(logGamma(_shape))

 {
//   assert(shape > 0 && scale > 0);
     if (!(_shape > 0 && _scale > 0)) 
     {
     Print(". () - <= 0 // GammaDF::Init(double _shape, double _scale);"); 
//     return -DBL_MAX; 
     }
   a=_shape;
   shape=_shape;
   scale=_scale;
   lga=logGamma(_shape);
 }


double GammaDF::series(double x)
// . Abramowitz & Stegun,
//      Handbook of Mathematical Functions, 1964 [6.5.29]
//  ., .
//          (: , 1979)
//
//      P(a,x)
//      .
//
{
   double sum, prev_sum, term, aa = a;
   long i = 0;

   term = sum = one / a;
   do {
      aa += one;
      term *= x / aa;
      prev_sum = sum; sum += term;
      i++;
   } while(fabs(prev_sum) != fabs(sum));
   return sum *= exp(-x + a * log(x) - lga);
}/* incGamma_series */

double GammaDF::fraction(double x)
// . Abramowitz & Stegun,
//      Handbook of Mathematical Functions, 1964 [6.5.31]
//  ., .
//      (: , 1979)
//
//      P(a,x)
//       
//
//  P(a,x)=exp(-x +x*ln(a))*CF/logGamma(a),            
//                                                                      
//  
//
//        1    1-a   1   2-a   2
//  CF = ---  ----- --- ----- --- ....
//       x+    1+   x+   1+   x+
//
//     CF(n) = A(n) / B(n)
//
//  
//        A(n) = (s(n) * A(n-1) + r(n) * A(n-2)) * factor
//        B(n) = (s(n) * B(n-1) + r(n) * B(n-2)) * factor
//                                                                   
//        A(-1) = 1, B(-1) = 0, A(0) = s(0), B(0) = 1.                  
//                                                                      
//  
//                                                                      
//        s(0) = 0, s(1) = x, r(0) = 0, r(1) = 1,                       
//                                                                      
//   
//                                                                      
//        A(1) = one * factor, B(1) = r * factor                        
//                                                                      
//  , ,
//                                                                      
//        r(i) = k - a  if i = 2k,   k > 0                              
//        r(i) = k      if i = 2k+1,                                    
//        s(i) = 1      if i = 2k,                                      
//        s(i) = x      if i = 2k+1                                     
//                                                                      
//  factor -  
//
{
   double old_sum=zero, factor=one;
   double A0=zero, A1=one, B0=one, B1=x;
   double sum=one/x, z=zero, ma=zero-a, rfact;

   do {
      z += one;
      ma += one;
      /* two steps of recurrence replace A's & B's */
      A0 = (A1 + ma * A0) * factor;	/* i even */
      B0 = (B1 + ma * B0) * factor;
      rfact = z * factor;
      A1 = x * A0 + rfact * A1;	/* i odd, A0 already rescaled */
      B1 = x * B0 + rfact * B1;
      if (B1) {
	 factor = one / B1;
	 old_sum = sum;
	 sum = A1 * factor;
      }/* if */
   } while (fabs(sum) != fabs(old_sum));
   return exp(-x + a * log(x) - lga) * sum;
}/*fraction*/

double GammaDF::value(double x)
//  Gamma(x|a):
//       ,   ,
//        -   'a',
//         'x'.
{
   x *= scale;
   if(x <= zero)
      return zero;            /*  ! */
   else if(x < (a + one))
      return series(x);
   else
      return one - fraction(x);
}/*value*/

double GammaDF::inv(double p)
//    'x',   Gamma(x|a) = p,
//      ..  'p'  ,   ,
//        -   'a',
//         'x'.
{
   double fx, l = 0, r = 1, x;

   if (p == 0) return 0;
//   assert(p > 0 && p < 1);
   if(!(p > 0 && p < 1))
     {
     Print(".   -     (0..1) // GammaDF::inv(double p);"); 
     return -DBL_MAX; 
     }

   for(l=0, r=a/2; value(r) < p; r+=a/2) l=r;
   x=(l+r)/2;
   do {
      fx = value(x);
      if (fx > p) r = x; else
      if (fx < p) l = x; else
         break;
      x = (l + r)* 0.5;
    } while ((l!=x) && (r!=x));
    return x;

}/*inv*/

//************************************************************************//


double GammaDistribution(double x, double a, double b)
//  Gamma(x|a):
//       ,   ,
//        -   'a',
//         'x'.
{
  GammaDF gd;    gd.Init(a,b);
  return gd.value(x);
}
double InvGammaDistribution(double p, double a, double b)
//    'x',   Gamma(x|a) = p,
//      ..  'p'  ,   ,
//        -   'a',
//         'x'.
{
  GammaDF gd;    gd.Init(a,b);
  return gd.inv(p);
}

double ExcelGammaDistr(double x, double a, double b)
//  Gamma(x|a):
//       ,   ,
//        -   'a',
//         'x'.
{
  GammaDF gd;    gd.Init(a,1.0/b);
  return gd.value(x);
}
