//+------------------------------------------------------------------+
//|                                                      testpso.mq5 |
//|                                 Copyright © 2016-2020, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//| Parallel Particle Swarm Optimization: test script                |
//|                           https://www.mql5.com/ru/articles/8321/ |
//+------------------------------------------------------------------+

#include <ParticleSwarmParallel.mqh>

class PSOTests
{
  class BaseFunctor: public Functor
  {
    protected:
      const int params;
      double max[], min[], steps[];
      
    public:
      BaseFunctor(const int p): params(p)
      {
        ArrayResize(max, params);
        ArrayResize(min, params);
        ArrayResize(steps, params);
        ArrayInitialize(steps, 0);
        
        PSOTests::register(&this);
      }
      
      virtual void test(const int loop)
      {
        Swarm swarm(params, max, min, steps);
        swarm.optimize(this, loop);
        double result[];
        swarm.getSolution(result);
        for(int i = 0; i < params; i++)
        {
          Print(i, " ", result[i]);
        }
      }
    
  };

  static BaseFunctor *testCases[];
  
  public:
    class Sphere: public BaseFunctor
    {
      public:
        Sphere(): BaseFunctor(3) // expected global minimum (0, 0, 0)
        {
          for(int i = 0; i < params; i++)
          {
            max[i] = 100;
            min[i] = -100;
          }
        }
        
        virtual void test(const int loop)
        {
          Print("Optimizing " + typename(this));
          BaseFunctor::test(loop);
        }
        
        virtual double calculate(const double &vec[])
        {
          int dim = ArraySize(vec);
          double sum = 0;
          for(int i = 0; i < dim; i++) sum += pow(vec[i], 2);
          return -sum;
        }
    };
    
    class Griewank: public BaseFunctor
    {
      public:
        Griewank(): BaseFunctor(2) // expected global minimum (0, 0)
        {
          for(int i = 0; i < params; i++)
          {
            max[i] = 600;
            min[i] = -600;
          }
        }
        
        virtual void test(const int loop)
        {
          Print("Optimizing " + typename(this));
          BaseFunctor::test(loop);
        }
        
        virtual double calculate(const double &vec[])
        {
          int dim = ArraySize(vec);
          
          double sum = 0.;
          double prod = 1.;
          for(int i = 0; i < dim; i++)
          {
        	  sum += pow(vec[i], 2);
        	  prod *= cos(vec[i] / sqrt(i + 1));
          }
          return -(sum / 4000 - prod + 1);
        }
    };
    
    class Rosenbrock: public BaseFunctor
    {
      public:
        Rosenbrock(): BaseFunctor(2) // expected global minimum (1, 1)
        {
          for(int i = 0; i < params; i++)
          {
            max[i] = +2.048;
            min[i] = -2.048;
          }
        }
        
        virtual void test(const int loop)
        {
          Print("Optimizing " + typename(this));
          BaseFunctor::test(loop);
        }
        
        virtual double calculate(const double &vec[])
        {
          int dim = ArraySize(vec);
          double sum = 0.;
          for(int i = 0; i < dim - 1; i++)
          {
            sum += 100 * pow((vec[i + 1] - pow(vec[i], 2)), 2) + pow((1 - vec[i]), 2);
          }
          return -sum;
        }
    };
  
    static void register(BaseFunctor *f)
    {
      int n = ArraySize(testCases);
      ArrayResize(testCases, n + 1);
      testCases[n] = f;
    }
    
    static void run(const int loop = 100)
    {
      for(int i = 0; i < ArraySize(testCases); i++)
      {
        testCases[i].test(loop);
      }
    }
};

static PSOTests::BaseFunctor *PSOTests::testCases[];


void OnStart()
{
  PSOTests::Sphere sphere;
  PSOTests::Griewank griewank;
  PSOTests::Rosenbrock rosenbrock;
  PSOTests::run();
}
