#define RESERVE_HISTORY_ORDERS 2000

#include "Order.mqh"

class HISTORYORDERS
{
private:
  int StartPos;
  int LastPos;

  int PrevAmountHistoryOrders;

  int TradesProfit;
  int TradesLoss;

  int TradesShortTotal;
  int TradesLongTotal;

  int TradesShortProfit;
  int TradesLongProfit;

  double InitialDeposit;
  double Withdrawal;

  double GrossProfit;
  double GrossLoss;

  double MaxProfit;
  double MaxLoss;

  double Balance;
  double MaxBalance;
  double BalanceDD;

  void InitStatistic()
  {
    this.LastPos = this.StartPos;

    this.TradesProfit = 0;
    this.TradesLoss = 0;

    this.TradesShortTotal = 0;
    this.TradesLongTotal = 0;

    this.TradesShortProfit = 0;
    this.TradesLongProfit = 0;

    this.InitialDeposit = ((this.StartPos < this.AmountHistoryOrders) &&
                           (this.HistoryOrders[this.LastPos] == OP_BALANCE)) ? this.HistoryOrders[this.LastPos++].OrderProfit() : 0;
    this.Withdrawal = 0;

    this.GrossProfit = 0;
    this.GrossLoss = 0;

    this.MaxProfit = 0;
    this.MaxLoss = 0;

    this.Balance = 0;
    this.MaxBalance = 0;
    this.BalanceDD = 0;

    return;
  }

  void SetOrder( const double Profit, int &TradesLongShortProfit, int &TradesLongShortTotal )
  {
    if (Profit > 0)
    {
      if (Profit > this.MaxProfit)
        this.MaxProfit = Profit;

      this.GrossProfit += Profit;

      this.TradesProfit++;
      TradesLongShortProfit++;
    }
    else if (Profit < 0)
    {
      if (Profit < this.MaxLoss)
        this.MaxLoss = Profit;

      this.GrossLoss += Profit;

      this.TradesLoss++;
    }

    this.Balance += Profit;

    if (this.Balance > this.MaxBalance)
      this.MaxBalance = this.Balance;
    else if (this.MaxBalance - this.Balance > this.BalanceDD)
      this.BalanceDD = this.MaxBalance - this.Balance;

    TradesLongShortTotal++;

    return;
  }

  void Refresh( void )
  {
    while (this.LastPos < this.AmountHistoryOrders)
    {
      switch (this.HistoryOrders[this.LastPos].OrderType())
      {
      case ORDER_TYPE_BUY:
        this.SetOrder(this.HistoryOrders[this.LastPos].OrderProfit(), this.TradesLongProfit, this.TradesLongTotal);

        break;
      case ORDER_TYPE_SELL:
        this.SetOrder(this.HistoryOrders[this.LastPos].OrderProfit(), this.TradesShortProfit, this.TradesShortTotal);

        break;
      case OP_BALANCE:
        if (this.LastPos != this.StartPos)
          this.Withdrawal += this.HistoryOrders[this.LastPos].OrderProfit();
        else
          this.InitialDeposit = this.HistoryOrders[this.LastPos].OrderProfit();
      }

      this.LastPos++;
    }

    return;
  }

  static string DoubleToString( const double Num, const int digits )
  {
    return((Num == DBL_MAX) ? "Max" : ::DoubleToString(Num, digits));
  }

protected:
  ORDER HistoryOrders[];

  int AmountHistoryOrders;
  int ReserveHistoryOrders;

  void AddHistoryOrder( const ORDER &Order )
  {
    if (this.AmountHistoryOrders == this.ReserveHistoryOrders)
      this.ReserveHistoryOrders = ::ArrayResize(this.HistoryOrders, this.ReserveHistoryOrders + RESERVE_HISTORY_ORDERS);

    if (Order.IsNotNull() && this.AmountHistoryOrders < this.ReserveHistoryOrders)
    {
      this.HistoryOrders[this.AmountHistoryOrders] = Order;

      this.HistoryOrders[this.AmountHistoryOrders++].ToHistory();
    }

    return;
  }

  bool CopyFrom( const HISTORYORDERS* SourceOrders )
  {
    this.AmountHistoryOrders = SourceOrders.AmountHistoryOrders;
    this.ReserveHistoryOrders = ::ArrayCopy(this.HistoryOrders, SourceOrders.HistoryOrders);

    this.InitStat();

    return(this.ReserveHistoryOrders == ::ArraySize(SourceOrders.HistoryOrders));
  }

public:
  HISTORYORDERS() : AmountHistoryOrders(0), ReserveHistoryOrders(0), PrevAmountHistoryOrders(0), StartPos(0)
  {
    this.InitStatistic();
  }

  bool ResetTickets( void )
  {
    long Tickets[][2];
    const int Size = ::ArrayResize(Tickets, this.AmountHistoryOrders);

  #define VALUE 0
  #define INDEX 1
    for (int i = 0; i < Size; i++)
    {
      Tickets[i][VALUE] = this.HistoryOrders[i].GetTicket();
      Tickets[i][INDEX] = i;
    }

    ::ArraySort(Tickets);

    for (int i = 0; i < Size; i++)
      this.HistoryOrders[(int)Tickets[i][INDEX]].SetTicket(i + 1);
  #undef INDEX
  #undef VALUE

    return(true);
  }

  bool InitStat( const long StartTime = 0 )
  {
    if (this.AmountHistoryOrders)
    {
      if (this.StartPos < this.AmountHistoryOrders)
      {
        if (this.HistoryOrders[this.StartPos] != StartTime)
        {
          if (this.HistoryOrders[this.StartPos] < StartTime)
          {
            while ((++this.StartPos < this.AmountHistoryOrders) && (this.HistoryOrders[this.StartPos] < StartTime))
              ;

            this.InitStatistic();
          }
          else if (this.StartPos && (this.HistoryOrders[0] <= StartTime))
          {
            const int PrevStartPos = this.StartPos;

            while ((--this.StartPos >= 0) && (this.HistoryOrders[this.StartPos] >= StartTime))
              ;

            this.StartPos++;

            if (this.StartPos != PrevStartPos)
              this.InitStatistic();
          }
        }
      }
      else if (this.HistoryOrders[this.StartPos - 1] > StartTime)
      {
        while ((--this.StartPos >= 0) && (this.HistoryOrders[this.StartPos] >= StartTime))
          ;

        this.StartPos++;

        this.InitStatistic();
      }
    }

    return(true);
  }

  double TesterStatistics( const ENUM_STATISTICS StatID )
  {
    double Res = 0;

    this.Refresh();

    switch (StatID)
    {
    case STAT_INITIAL_DEPOSIT: //   
      Res = this.InitialDeposit;

      break;
    case STAT_WITHDRAWAL: //     
      Res = this.Withdrawal;

      break;
    case STAT_PROFIT: //     
      Res = this.GrossLoss + this.GrossProfit;

      break;
    case STAT_GROSS_PROFIT: //  ,    () .     
      Res = this.GrossProfit;

      break;
    case STAT_GROSS_LOSS: //  ,    () .     
      Res = this.GrossLoss;

      break;
    case STAT_MAX_PROFITTRADE: //         .     
      Res = this.MaxProfit;

      break;
    case STAT_MAX_LOSSTRADE: //         .     
      Res = this.MaxLoss;

      break;
    case STAT_BALANCE_DD: //     .        ,   .
      Res = this.BalanceDD;

      break;
    case STAT_EXPECTED_PAYOFF: //   
      {
        const int Total = this.TradesLongTotal + this.TradesShortTotal;

        Res = Total ? (this.GrossLoss + this.GrossProfit) / Total : 0;
      }

      break;
    case STAT_PROFIT_FACTOR: //    STAT_GROSS_PROFIT/STAT_GROSS_LOSS.  STAT_GROSS_LOSS=0,     DBL_MAX
      Res = this.GrossLoss ? -this.GrossProfit / this.GrossLoss : DBL_MAX;

      break;
    case STAT_RECOVERY_FACTOR: //     STAT_PROFIT/STAT_BALANCE_DD
      Res = this.BalanceDD ? (this.GrossLoss + this.GrossProfit) / this.BalanceDD : 0;

      break;
    case STAT_DEALS: //   
      Res = (this.TradesLongTotal + this.TradesShortTotal) << 1;

      break;
    case STAT_TRADES: //  
      Res = this.TradesLongTotal + this.TradesShortTotal;

      break;
    case STAT_PROFIT_TRADES: //  
      Res = this.TradesProfit;

      break;
    case STAT_LOSS_TRADES: //  
      Res = this.TradesLoss;

      break;
    case STAT_SHORT_TRADES: //  
      Res = this.TradesShortTotal;

      break;
    case STAT_LONG_TRADES: //  
      Res = this.TradesLongTotal;

      break;
    case STAT_PROFIT_SHORTTRADES: //   
      Res = this.TradesShortProfit;

      break;
    case STAT_PROFIT_LONGTRADES: //   
      Res = this.TradesLongProfit;

      break;
    }

    return(Res);
  }

  bool IsHistoryChange( void )
  {
    const bool Res = (this.AmountHistoryOrders != this.PrevAmountHistoryOrders);

    if (Res)
      this.PrevAmountHistoryOrders = this.AmountHistoryOrders;

    return(Res);
  }

  string StatToString( void )
  {
    return("Profit = " + HISTORYORDERS::DoubleToString(this.TesterStatistics(STAT_PROFIT), 2) + ", " +
           "Trades = " + HISTORYORDERS::DoubleToString(this.TesterStatistics(STAT_TRADES), 0) + ", " +
           "PF = " + HISTORYORDERS::DoubleToString(this.TesterStatistics(STAT_PROFIT_FACTOR), 2) + ", " +
           "Mean = " + HISTORYORDERS::DoubleToString(this.TesterStatistics(STAT_EXPECTED_PAYOFF), 2) + ", " +
           "MaxDD = " + HISTORYORDERS::DoubleToString(-this.TesterStatistics(STAT_BALANCE_DD), 2) + ", " +
           "RF = " + HISTORYORDERS::DoubleToString(this.TesterStatistics(STAT_RECOVERY_FACTOR), 2));
  }
};

#undef RESERVE_HISTORY_ORDERS
