//+------------------------------------------------------------------+
//|                                                     Settings.mqh |
//|                                 Copyright © 2016-2020, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//| Parallel Particle Swarm Optimization                             |
//|                           https://www.mql5.com/ru/articles/8321/ |
//+------------------------------------------------------------------+
#include <Filer.mqh>

#define __SETTINGS_H__

#define MAX_COLUMN 4
#define SET_COLUMN_NAME  0
#define SET_COLUMN_START 1
#define SET_COLUMN_STEP  2
#define SET_COLUMN_STOP  3

class Settings: public Feed
{
  private:
    string data[][MAX_COLUMN]; // name, start, step, stop
    double numbers[][MAX_COLUMN];
    long dimensions[];
    long cursor;
    long total;

    void buildIndex()
    {
      const int n = ArrayRange(data, 0);
      ArrayResize(numbers, n);
      ArrayResize(dimensions, n + 1);
      dimensions[0] = 1;
      total = 1;
      for(int i = 0; i < n; i++)
      {
        numbers[i][SET_COLUMN_START] = (double)data[i][SET_COLUMN_START];
        numbers[i][SET_COLUMN_STOP] = (double)data[i][SET_COLUMN_STOP];
        numbers[i][SET_COLUMN_STEP] = (double)data[i][SET_COLUMN_STEP];
        dimensions[i + 1] = (int)((numbers[i][SET_COLUMN_STOP] - numbers[i][SET_COLUMN_START] + DBL_EPSILON) / numbers[i][SET_COLUMN_STEP]) + 1;
        total *= dimensions[i + 1];
        dimensions[i + 1] *= dimensions[i];
      }
      cursor = 0;
      Print("Total optimization space " , n, " dimensions: ", total);
    }

  public:
    Settings(): cursor(-1), total(0) {}
    
    long getTotalSpace()
    {
      if(total == 0) buildIndex();
      return total;
    }
    
    bool getNames(string &names[]) const
    {
      const int n = ArrayRange(data, 0);
      ArrayResize(names, n);
      for(int i = 0; i < n; i++)
      {
        names[i] = data[i][SET_COLUMN_NAME];
      }
      return true;
    }
    
    // brute force loop through all combinations of parameters
    bool getNext(double &parameters[])
    {
      if(cursor == -1) buildIndex();
      else if(cursor >= total) return false;
      
      const int n = ArrayRange(data, 0);
      ArrayResize(parameters, n);
      long _cursor = cursor;
      for(int i = 0; i < n; i++)
      {
        int k = n - i - 1;
        long index = _cursor / dimensions[k];
        parameters[k] = numbers[k][SET_COLUMN_START] + index * numbers[k][SET_COLUMN_STEP];
        _cursor -= index * dimensions[k];
      }
      cursor++;
      return true;
    }
    
    virtual bool feed(const int dump) override
    {
      string temp = FileReadString(dump);
      StringTrimLeft(temp);
      StringTrimRight(temp);
      if(StringLen(temp) == 0) return true;
      
      const int size = ArrayRange(data, 0);
      ArrayResize(data, size + 1);

      int i = 0;
      
      data[size][i++] = temp;

      while(!FileIsLineEnding(dump))
      {
        if(i == MAX_COLUMN)
        {
          Print("Unexpected data: ", FileReadString(dump));
          return false;
        }
        data[size][i++] = FileReadString(dump);
      }
      
      return true;
    }
    
    template<typename T>
    T get(const string name, const int type) const
    {
      if(type < SET_COLUMN_NAME && type > SET_COLUMN_STOP) return NULL;
      for(int i = 0; i < ArrayRange(data, 0); i++)
      {
        if(data[i][0] == name)
        {
          return (T)data[i][type];
        }
      }
      return NULL;
    }

    template<typename T>
    T get(const int i, const int type) const
    {
      const int n = ArrayRange(data, 0);
      if(i < 0 || i >= n) return NULL; // TODO: throw exception
      return (T)data[i][type];
    }
    
    int size() const
    {
      return ArrayRange(data, 0);
    }
    
    bool isVoid() const
    {
      const int n = ArrayRange(data, 0);
      if(n == 0) return true;
      for(int i = 0; i < n; i++)
      {
        if((float)data[i][SET_COLUMN_STEP] == 0
        || ((float)data[i][SET_COLUMN_START] >= (float)data[i][SET_COLUMN_STOP]))
        {
          return true;
        }
      }
      return false;
    }
    
    void print()
    {
      ArrayPrint(data);
    }
};
