//+------------------------------------------------------------------+
//|                                        ParticleSwarmParallel.mqh |
//|                                 Copyright © 2016-2020, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//| Parallel Particle Swarm Optimization                             |
//|                           https://www.mql5.com/ru/articles/8321/ |
//+------------------------------------------------------------------+

#ifndef _PSO_H_
#define _PSO_H_

#define AUTO_SIZE_FACTOR 5
#define PPSO_FILE_PREFIX "PPSO-"

#include <crc64ttb.mqh>
#include <filer.mqh>

string sharedName(const int id, const string prefix = PPSO_FILE_PREFIX, const string ext = ".csv")
{
  ushort array[];
  StringToShortArray(TerminalInfoString(TERMINAL_PATH), array);
  const string program = MQLInfoString(MQL_PROGRAM_NAME) + "-";
  if(id != -1)
  {
    return prefix + program + StringFormat("%08I64X-%04d", crc64(array), id) + ext;
  }
  return prefix + program + StringFormat("%08I64X-*", crc64(array)) + ext;
}

template<typename T>
class Visitor
{
  public:
    virtual void visit(TreeNode<T> *node) = 0;
};

template<typename T>
class TreeNode
{
  private:
    TreeNode *left;
    TreeNode *right;
    T value;

  public:
    TreeNode(T t)
    {
      value = t;
    }
    
    bool add(T t)
    {
      if(t < value)
      {
        if(left == NULL)
        {
          left = new TreeNode<T>(t);
          return false;
        }
        else
        {
          return left.add(t);
        }
      }
      else if(t > value)
      {
        if(right == NULL)
        {
          right = new TreeNode<T>(t);
          return false;
        }
        else
        {
          return right.add(t);
        }
      }
      return true;
    }
    
    ~TreeNode()
    {
      if(left != NULL) delete left;
      if(right != NULL) delete right;
    }
    
    TreeNode *getLeft(void) const
    {
      return left;
    }

    TreeNode *getRight(void) const
    {
      return right;
    }

    T getValue(void) const
    {
      return value;
    }
};


template<typename T>
class Exporter: public Visitor<T>
{
  private:
    int file;
    uint level;
    
  public:
    Exporter(const string name): level(0)
    {
      file = FileOpen(name, FILE_READ | FILE_WRITE | FILE_CSV | FILE_ANSI | FILE_SHARE_READ| FILE_SHARE_WRITE | FILE_COMMON, ',');
    }
    
    ~Exporter()
    {
      FileClose(file);
    }
    
    virtual void visit(TreeNode<T> *node) override
    {
      #ifdef PSO_DEBUG_BINTREE
      if(node.getLeft()) visit(node.getLeft());
      FileWrite(file, node.getValue());
      if(node.getRight()) visit(node.getRight());
      #else
      const T v = node.getValue();
      FileWrite(file, v);
      level++;
      if((level | (uint)v) % 2 == 0)
      {
        if(node.getLeft()) visit(node.getLeft());
        if(node.getRight()) visit(node.getRight());
      }
      else
      {
        if(node.getRight()) visit(node.getRight());
        if(node.getLeft()) visit(node.getLeft());
      }
      level--;
      #endif
    }
};

template<typename T>
class BinaryTree
{
  private:
    TreeNode<T> *root;
    
  public:
    bool add(T t)
    {
      if(root == NULL)
      {
        root = new TreeNode<T>(t);
        return false;
      }
      else
      {
        return root.add(t);
      }
    }
    
    ~BinaryTree()
    {
      if(root != NULL) delete root;
    }
    
    void visit(Visitor<T> *visitor)
    {
      visitor.visit(root);
    }
};

class Particle
{
  public:
    double position[];
    double best[];
    double velocity[];
    
    double positionValue;
    double bestValue;
    int    group;
    
    Particle(const int params)
    {
      ArrayResize(position, params);
      ArrayResize(best, params);
      ArrayResize(velocity, params);
      bestValue = -DBL_MAX;
      group = -1;
    }
};

class Group
{
  private:
    double result;
  
  public:
    double optimum[];
    
    Group(const int params)
    {
      ArrayResize(optimum, params);
      ArrayInitialize(optimum, 0);
      result = -DBL_MAX;
    }
    
    void assign(const double x)
    {
      result = x;
    }
    
    double getResult() const
    {
      return result;
    }
    
    bool isAssigned() const
    {
      return result != -DBL_MAX;
    }
};

class Functor
{
  public:
    virtual double calculate(const double &vector[]) = 0;
};

class Swarm: public Feed
{
  public:
    class Stats
    {
      public:
        int done;
        int planned;
        Stats(): done(0), planned(0) {}
    };
    
  private:
    Particle *particles[];
    Group *groups[];
    int _size;
    int _globals;
    int _params;
    long _total;
    
    double highs[];
    double lows[];
    double steps[];
    double solution[];
    
    int _read;
    int _unique;
    int _restored;
    BinaryTree<ulong> index;
    BinaryTree<ulong> merge;
    
    Stats stats;
    
    bool calcSpace()
    {
      _total = 1;
      for(int i = 0; i < _params; i++)
      {
        if(steps[i] == 0) return false; // the problem size is unknown, space is continuous
        _total *= ((int)((highs[i] - lows[i] + DBL_EPSILON) / steps[i]) + 1);
      }
      return true;
    }
    
    void init(const int size, const int globals, const int params, const double &max[], const double &min[], const double &inc[])
    {
      _size = size;
      _globals = globals;
      _params = params;

      // globals (number of groups) is assumed much less than size
      
      ArrayCopy(highs, max);
      ArrayCopy(lows, min);
      ArrayCopy(steps, inc);
      
      // check that product of all ((max - min)/inc), that is total, is much less than size
      if(calcSpace())
      {
        if(_size > _total) // swarm is too large
        {
          _size = (int)_total;
          Print("Swarm is reduced to optimization space size: ", _size);
        }
      }
      
      ArrayResize(solution, _params);

      ArrayResize(particles, _size);
      for(int i = 0; i < _size; i++)
      {
        particles[i] = new Particle(_params);
        
        do
        {
          for(int p = 0; p < _params; p++)
          {
            particles[i].position[p] = (MathRand() * 1.0 / 32767) * (highs[p] - lows[p]) + lows[p];
            if(steps[p] != 0)
            {
              particles[i].position[p] = ((int)MathRound((particles[i].position[p] - lows[p]) / steps[p])) * steps[p] + lows[p];
            }
            particles[i].best[p] = particles[i].position[p];
            particles[i].velocity[p] = (MathRand() * 1.0 / 32767) * 2 * (highs[p] - lows[p]) - (highs[p] - lows[p]);
          }
        }
        while(index.add(crc64(particles[i].position)) && !IsStopped());
      }
      
      ArrayResize(groups, _globals);
      for(int i = 0; i < _globals; i++)
      {
        groups[i] = new Group(_params);
      }
      
      for(int i = 0; i < _size; i++)
      {
        particles[i].group = (_globals > 1) ? (int)MathMin(MathRand() * 1.0 / 32767 * _globals, _globals - 1) : 0;
      }
      
      Print("PSO[", _params, "] created: ", _size, "/", _globals);
    }
    
  public:
    Swarm(const int params, const double &max[], const double &min[], const double &step[])
    {
      init(params * AUTO_SIZE_FACTOR, (int)MathSqrt(params * AUTO_SIZE_FACTOR), params, max, min, step);
    }
    
    Swarm(const int size, const int globals, const int params, const double &max[], const double &min[], const double &step[])
    {
      if(MQLInfoInteger(MQL_OPTIMIZATION))
      {
        init(size == 0 ? params * AUTO_SIZE_FACTOR : size, 1, params, max, min, step);
      }
      else
      if(size && globals)
      {
        init(size, globals, params, max, min, step);
      }
      else
      {
        init(params * AUTO_SIZE_FACTOR, (int)MathSqrt(params * AUTO_SIZE_FACTOR), params, max, min, step);
      }
    }
    
    ~Swarm()
    {
      for(int i = 0; i < _size; i++)
      {
        delete particles[i];
      }
      for(int i = 0; i < _globals; i++)
      {
        delete groups[i];
      }
    }
    
    static void removeIndex()
    {
      string name;
      int count = 0;
      const string filter = sharedName(-1);
      long h = FileFindFirst(filter, name, FILE_COMMON);
      if(h != INVALID_HANDLE)
      {
        do
        {
          Print(name);
          count += FileDelete(name, FILE_COMMON);
        }
        while(FileFindNext(h, name));
        FileFindClose(h);
      }
      if(count > 0) Print(count, " tmp-files deleted");
    }
    
    bool restoreIndex()
    {
      string name;
      _read = 0;
      _unique = 0;
      _restored = 0;
      const string filter = sharedName(-1); // use wildcards to merge multiple indices for all cores
      long h = FileFindFirst(filter, name, FILE_COMMON);
      if(h != INVALID_HANDLE)
      {
        int count = 0;
        do
        {
          Print(name);
          FileReader reader(name, FILE_COMMON);
          reader.read(this);
          count++;
        }
        while(FileFindNext(h, name));
        FileFindClose(h);
        PrintFormat("%d files read, %d old records, %d restored", count, _unique, _restored);
        if(_unique >= _total)
        {
          Print("All particles are already in logs, skip the pass");
          return false;
        }
      }
      return true;
    }
    
    virtual bool feed(const int dump) override
    {
      const ulong value = (ulong)FileReadString(dump);
      _read++;
      if(!index.add(value)) _restored++;
      else if(!merge.add(value)) _unique++;
      return true;
    }
    
    void exportIndex(const int id)
    {
      int _id = 0;
      string parts[];
      const int n = StringSplit(TerminalInfoString(TERMINAL_DATA_PATH), '-', parts);
      if(n > 2) _id = (int)StringToInteger(parts[n - 1]);

      const string name = sharedName(_id ? _id : id);
      Exporter<ulong> exporter(name);
      index.visit(&exporter);
    }
    
    int getSize()
    {
      return _size;
    }
    
    double optimize(Functor &f, const int cycles, const double inertia = 0.8, const double selfBoost = 0.4, const double groupBoost = 0.4)
    {
      double result = -DBL_MAX;
      ArrayInitialize(solution, 0);
      
      double next[];
      ArrayResize(next, _params);
      
      int actualCount = 0;
      
      Print("PSO Processing...");
      for(int c = 0; c < cycles && !IsStopped(); c++)
      {
        int skipped = 0;
        for(int i = 0; i < _size && !IsStopped(); i++)
        {
          if(c > 0)
          {
            const bool assigned = groups[particles[i].group].isAssigned();
            for(int p = 0; p < _params; p++)
            {
              const double r1 = MathRand() * 1.0 / 32767;
              const double rg = assigned ? MathRand() * 1.0 / 32767 : 0;
              particles[i].velocity[p] = inertia * particles[i].velocity[p] + selfBoost * r1 * (particles[i].best[p] - particles[i].position[p]) + groupBoost * rg * (groups[particles[i].group].optimum[p] - particles[i].position[p]);
              next[p] = particles[i].position[p] + particles[i].velocity[p];
              if(next[p] < lows[p]) next[p] = lows[p];
              else if(next[p] > highs[p]) next[p] = highs[p];
              if(steps[p] != 0)
              {
                next[p] = ((int)MathRound((next[p] - lows[p]) / steps[p])) * steps[p] + lows[p];
              }
            }
  
            if(index.add(crc64(next))) // already processed
            {
              skipped++;
              continue;
            }

            ArrayCopy(particles[i].position, next);
          }
          
          actualCount++;
          
          particles[i].positionValue = f.calculate(particles[i].position);
          if(particles[i].positionValue > particles[i].bestValue)
          {
            particles[i].bestValue = particles[i].positionValue;
            ArrayCopy(particles[i].best, particles[i].position);
          }
    
          if(particles[i].positionValue > groups[particles[i].group].getResult())
          {
            groups[particles[i].group].assign(particles[i].positionValue);
            ArrayCopy(groups[particles[i].group].optimum, particles[i].position);
            
            if(particles[i].positionValue > result)
            {
              result = particles[i].positionValue;
              ArrayCopy(solution, particles[i].position);
            }
          }
        }
        Print("Cycle ", c, " done, skipped ", skipped, " of ", _size, " / ", result);
        if(skipped == _size) break; // full coverage
      }
      Print("PSO Finished ", actualCount, " of ", (cycles * _size), " planned calculations: ", !IsStopped());
      stats.done = actualCount;
      stats.planned = cycles * _size;
      return result;
    }
    
    void getStats(Stats &s) const
    {
      s = stats;
    }
    
    bool getSolution(double &result[])
    {
      ArrayCopy(result, solution);
      return !IsStopped();
    }
    
};

#endif