//+------------------------------------------------------------------+
//|                                                    MT5Bridge.mqh |
//|                                 Copyright © 2017-2020, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//|                                                                  |
//| MT5 backward wrapper for MT4Orders trading API                   |
//| and virtual testing/optimization library Virtual.mqh             |
//| Based on the article:                                            |
//|      Ready-made Expert Advisors from the MQL5 Wizard work in MT4 |
//|                            https://www.mql5.com/en/articles/3068 |
//|                                                                  |
//|                                     version: mt5-1.2, 17/08/2020 |
//| status: PREVIEW                                                  |
//+------------------------------------------------------------------+

// Virtual.mqh or MT4Orders.mqh is required for MT5Bridge.mqh

#ifndef __VIRTUAL__
  #ifndef __MT4ORDERS__
    #include "!!!  Virtual.mqh OR MT4Orders.mqh is required  !!!"
  #endif
#endif


//+------------------------------------------------------------------+
//| Implementation                                                   |
//+------------------------------------------------------------------+


TICKET_TYPE historyDeals[], historyOrders[];

bool MT5HistorySelect(datetime from_date, datetime to_date)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return HistorySelect(from_date, to_date);
  }
#endif

  int deals = 0, orders = 0;
  
  ArrayResize(historyDeals, 0);
  ArrayResize(historyOrders, 0);
  
  for(int i = 0; i < OrdersHistoryTotal(); i++)
  {
    if(OrderSelect(i, SELECT_BY_POS, MODE_HISTORY))
    {
      if(OrderOpenTime() >= from_date || OrderCloseTime() <= to_date)
      {
        if(OrderType() <= OP_SELL) // deal
        {
          ArrayResize(historyDeals, deals + 1);
          historyDeals[deals] = OrderTicket();
          deals++;
        }
        // pending orders processed as is,
        // market orders doubled as deals (enter+exit for a single trade)
        ArrayResize(historyOrders, orders + 1);
        historyOrders[orders] = OrderTicket();
        orders++;
      }
    }
  }
  return true;
}

#define HistorySelect MT5HistorySelect

int EnumOrderType2Code(int action) // ENUM_ORDER_TYPE
{
  // ORDER_TYPE_BUY/ORDER_TYPE_SELL and derivatives
  return (action % 2 == 0) ? OP_BUY : OP_SELL;
}

ENUM_POSITION_TYPE Order2Position(int type)
{
  return type == OP_BUY ? POSITION_TYPE_BUY : POSITION_TYPE_SELL;
}

ENUM_ORDER_STATE OrderState()
{
  return OrderType() > OP_SELL ? ORDER_STATE_PLACED : ORDER_STATE_FILLED;
}

ENUM_DEAL_TYPE OrderType2DealType(const int type)
{
  static ENUM_DEAL_TYPE types[] = {DEAL_TYPE_BUY, DEAL_TYPE_SELL, -1, -1, -1, -1, DEAL_TYPE_BALANCE};
  return types[type];
}

ENUM_ORDER_TYPE OrderType2OrderType(const int type)
{
  static ENUM_ORDER_TYPE types[] = {ORDER_TYPE_BUY, ORDER_TYPE_SELL, ORDER_TYPE_BUY_LIMIT, ORDER_TYPE_SELL_LIMIT, ORDER_TYPE_BUY_STOP, ORDER_TYPE_SELL_STOP, -1};
  return types[type];
}


bool MT5OrderSend(const MqlTradeRequest &request, MqlTradeResult &result)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
  #ifdef OrderSend
    #define MACROS_ORDERSEND  
    
    #undef OrderSend
  #endif // OrderSend
  
    return OrderSend(request, result);
    
  #ifdef MACROS_ORDERSEND  
    #define OrderSend VIRTUAL::VirtualOrderSend
    
    #undef MACROS_ORDERSEND
  #endif // MACROS_ORDERSEND  
  }    
#endif

  int cmd;
  
  result.retcode = 0;

  switch(request.type)
  {
    case ORDER_TYPE_BUY:
      cmd = OP_BUY;
      break;
    case ORDER_TYPE_SELL:
      cmd = OP_SELL;
      break;
    case ORDER_TYPE_BUY_LIMIT:
      cmd = OP_BUYLIMIT;
      break;
    case ORDER_TYPE_SELL_LIMIT:
      cmd = OP_SELLLIMIT;
      break;
    case ORDER_TYPE_BUY_STOP:
      cmd = OP_BUYSTOP;
      break;
    case ORDER_TYPE_SELL_STOP:
      cmd = OP_SELLSTOP;
      break;
    default:
      Print("Unsupported request type:", request.type);
      return false;
  }
  
  ResetLastError();

  if(request.action == TRADE_ACTION_DEAL || request.action == TRADE_ACTION_PENDING)
  {
    if(request.price == 0)
    {
      if(cmd == OP_BUY)
      {
        ((MqlTradeRequest)request).price = SymbolInfoDouble(request.symbol, SYMBOL_ASK);
      }
      else
      if(cmd == OP_SELL)
      {
        ((MqlTradeRequest)request).price = SymbolInfoDouble(request.symbol, SYMBOL_BID);
      }
    }
    
    if(request.position > 0)
    {
      if(!OrderClose(request.position, request.volume, request.price, (int)request.deviation))
      {
        result.retcode = GetLastError();
      }
      else
      {
        result.retcode = TRADE_RETCODE_DONE;
        result.deal = request.position | 0x8000000000000000;
        result.order = request.position | 0x8000000000000000;
        result.volume = request.volume;
        result.price = request.price;
      }
    }
    else
    {
      TICKET_TYPE ticket = OrderSend(request.symbol, cmd, request.volume, request.price, (int)request.deviation, request.sl, request.tp, request.comment, request.magic, request.expiration);
      if(ticket == -1)
      {
        result.retcode = GetLastError();
      }
      else
      {
        result.retcode = TRADE_RETCODE_DONE;
        result.deal = ticket;
        result.order = ticket;
        result.request_id = (uint)ticket;
        
        if(OrderSelect(ticket, SELECT_BY_TICKET))
        {
          result.volume = OrderLots();
          result.price = OrderOpenPrice() > 0 ? OrderOpenPrice() : request.price;
          result.comment = OrderComment();
          result.ask = SymbolInfoDouble(OrderSymbol(), SYMBOL_ASK);
          result.bid = SymbolInfoDouble(OrderSymbol(), SYMBOL_BID);
        }
        else
        {
          result.volume = request.volume;
          result.price = request.price;
          result.comment = "";
        }
      }
    }
  }
  else if(request.action == TRADE_ACTION_SLTP) // change opened position
  {
    if(OrderSelect(request.position, SELECT_BY_TICKET))
    {
      if(!OrderModify(request.position, OrderOpenPrice(), request.sl, request.tp, 0))
      {
        result.retcode = GetLastError();
      }
      else
      {
        result.retcode = TRADE_RETCODE_DONE;
        result.deal = OrderTicket();
        result.order = OrderTicket();
        result.request_id = (uint)OrderTicket();
        result.volume = OrderLots();
        result.comment = OrderComment();
      }
    }
    else
    {
      result.retcode = TRADE_RETCODE_POSITION_CLOSED;
    }
  }
  else if(request.action == TRADE_ACTION_MODIFY) // change pending order
  {
    if(OrderSelect(request.order, SELECT_BY_TICKET))
    {
      if(!OrderModify(request.order, request.price, request.sl, request.tp, request.expiration))
      {
        result.retcode = GetLastError();
      }
      else
      {
        result.retcode = TRADE_RETCODE_DONE;
        result.deal = OrderTicket();
        result.order = OrderTicket();
        result.request_id = (uint)OrderTicket();
        result.price = request.price;
        result.volume = OrderLots();
        result.comment = OrderComment();
      }
    }
    else
    {
      result.retcode = TRADE_RETCODE_INVALID_ORDER;
    }
  }
  else if(request.action == TRADE_ACTION_REMOVE)
  {
    if(!OrderDelete(request.order))
    {
      result.retcode = GetLastError();
    }
    else
    {
      result.retcode = TRADE_RETCODE_DONE;
    }
  }
  else if(request.action == TRADE_ACTION_CLOSE_BY)
  {
    if(!OrderCloseBy(request.position, request.position_by))
    {
      result.retcode = GetLastError();
    }
    else
    {
      result.retcode = TRADE_RETCODE_DONE;
    }
  }
  return true;
}

long MT5PositionGetInteger(ENUM_POSITION_PROPERTY_INTEGER property_id)
{
  long temp;
  if(MT5PositionGetInteger(property_id, temp))
  {
    return temp;
  }
  return 0;
}

bool MT5PositionGetInteger(ENUM_POSITION_PROPERTY_INTEGER property_id, long &long_var)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return PositionGetInteger(property_id, long_var);
  }
#endif
  switch(property_id)
  {
    case POSITION_TICKET:
    case POSITION_IDENTIFIER:
      long_var = OrderTicket();
      return true;
    case POSITION_TIME:
    case POSITION_TIME_UPDATE:
      long_var = OrderOpenTime();
      return true;
    case POSITION_TIME_MSC:
    case POSITION_TIME_UPDATE_MSC:
      long_var = OrderOpenTime() * 1000;
      return true;
    case POSITION_TYPE:
      long_var = Order2Position(OrderType());
      return true;
    case POSITION_MAGIC:
      long_var = OrderMagicNumber();
      return true;
  }

  return false;
}

#define PositionGetInteger MT5PositionGetInteger


double MT5PositionGetDouble(ENUM_POSITION_PROPERTY_DOUBLE property_id)
{
  double temp;
  if(MT5PositionGetDouble(property_id, temp))
  {
    return temp;
  }
  return 0;
}

bool MT5PositionGetDouble(ENUM_POSITION_PROPERTY_DOUBLE property_id, double &double_var)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return PositionGetDouble(property_id, double_var);
  }
#endif
  switch(property_id)
  {
    case POSITION_VOLUME:
      double_var = OrderLots();
      return true;
    case POSITION_PRICE_OPEN:
      double_var = OrderOpenPrice();
      return true;
    case POSITION_SL:
      double_var = OrderStopLoss();
      return true;
    case POSITION_TP:
      double_var = OrderTakeProfit();
      return true;
    case POSITION_PRICE_CURRENT:
      double_var = SymbolInfoDouble(OrderSymbol(), OrderType() == OP_BUY ? SYMBOL_BID : SYMBOL_ASK);
      return true;
    case POSITION_COMMISSION:
      double_var = OrderCommission();
      return true;
    case POSITION_SWAP:
      double_var = OrderSwap();
      return true;
    case POSITION_PROFIT:
      double_var = OrderProfit();
      return true;
  }

  return false;
}

#define PositionGetDouble MT5PositionGetDouble

string MT5PositionGetString(ENUM_POSITION_PROPERTY_STRING property_id)
{
  string temp;
  if(MT5PositionGetString(property_id, temp))
  {
    return temp;
  }
  return "";
}

bool MT5PositionGetString(ENUM_POSITION_PROPERTY_STRING property_id, string &string_var)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return PositionGetString(property_id, string_var);
  }
#endif
  switch(property_id)
  {
    case POSITION_SYMBOL:
      string_var = OrderSymbol();
      return true;
    case POSITION_COMMENT:
      string_var = OrderComment();
      return true;
  }
  return false;
}

#define PositionGetString MT5PositionGetString

bool MT5PositionSelect(string symbol)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return PositionSelect(symbol);
  }
#endif
  for(int i = 0; i < ::OrdersTotal(); i++)
  {
    if(OrderSelect(i, SELECT_BY_POS))
    {
      if(OrderSymbol() == symbol && (OrderType() <= OP_SELL))
      {
        return true;
      }
    }
  }
  return false;
}

#define PositionSelect MT5PositionSelect

bool MT5PositionSelectByTicket(ulong ticket)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return PositionSelectByTicket(ticket);
  }
#endif
  if(OrderSelect(ticket, SELECT_BY_TICKET))
  {
    if(OrderType() <= OP_SELL)
    {
      return true;
    }
  }
  return false;
}

#define PositionSelectByTicket MT5PositionSelectByTicket

int MT5PositionsTotal()
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return PositionsTotal();
  }
#endif
  int count = 0;
  for(int i = 0; i < ::OrdersTotal(); i++)
  {
    if(OrderSelect(i, SELECT_BY_POS))
    {
      if(OrderType() <= OP_SELL)
      {
        count++;
      }
    }
  }
  return count;
}

#define PositionsTotal MT5PositionsTotal

string MT5PositionGetSymbol(int index)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return PositionGetSymbol(index);
  }
#endif
  int count = 0;
  for(int i = 0; i < ::OrdersTotal(); i++)
  {
    if(OrderSelect(i, SELECT_BY_POS))
    {
      if(OrderType() <= OP_SELL)
      {
        if(index == count)
        {
          return OrderSymbol();
        }
        count++;
      }
    }
  }
  return "";
}

#define PositionGetSymbol MT5PositionGetSymbol

ulong MT5PositionGetTicket(int index)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return PositionGetTicket(index);
  }
#endif
  int count = 0;
  for(int i = 0; i < ::OrdersTotal(); i++)
  {
    if(OrderSelect(i, SELECT_BY_POS))
    {
      if(OrderType() <= OP_SELL)
      {
        if(index == count)
        {
          return OrderTicket();
        }
        count++;
      }
    }
  }
  return 0;
}

#define PositionGetTicket MT5PositionGetTicket

int MT5HistoryDealsTotal()
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return HistoryDealsTotal();
  }
#endif
  return ArraySize(historyDeals) * 2;
}

#define HistoryDealsTotal MT5HistoryDealsTotal

int MT5HistoryOrdersTotal()
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return HistoryOrdersTotal();
  }
#endif
  return ArraySize(historyOrders) * 2;
}

#define HistoryOrdersTotal MT5HistoryOrdersTotal

ulong MT5HistoryDealGetTicket(int index)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return HistoryDealGetTicket(index);
  }
#endif
  if(::OrderSelect(historyDeals[index / 2], SELECT_BY_TICKET, MODE_HISTORY))
  {
    // odd - enter - positive, even - exit - negative
    return (index % 2 == 0) ? OrderTicket() : (OrderTicket() | 0x8000000000000000);
  }
  return 0;
}

#define HistoryDealGetTicket MT5HistoryDealGetTicket

ulong MT5HistoryOrderGetTicket(int index)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return HistoryOrderGetTicket(index);
  }
#endif
  if(::OrderSelect(historyOrders[index / 2], SELECT_BY_TICKET, MODE_HISTORY))
  {
    if(OrderType() <= OP_SELL)
    {
      return (index % 2 == 0) ? OrderTicket() : (OrderTicket() | 0x8000000000000000);
    }
    else if(index % 2 == 0) // pending order is returned once
    {
      return OrderTicket();
    }
    else
    {
      Print("History order ", OrderType(), " ticket[", index, "]=", OrderTicket(), " -> 0");
    }
  }
  return 0;
}

#define HistoryOrderGetTicket MT5HistoryOrderGetTicket

long MT5HistoryDealGetInteger(ulong ticket_number, ENUM_DEAL_PROPERTY_INTEGER property_id)
{
  long temp;
  if(MT5HistoryDealGetInteger(ticket_number, property_id, temp))
  {
    return temp;
  }
  return 0;
}

#define REVERSE(type) ((type + 1) % 2)

bool MT5HistoryDealGetInteger(ulong ticket_number, ENUM_DEAL_PROPERTY_INTEGER property_id, long &long_var)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return HistoryDealGetInteger(ticket_number, property_id, long_var);
  }
#endif
  bool exit = ((ticket_number & 0x8000000000000000) != 0);
  ticket_number &= ~0x8000000000000000;
  if(::OrderSelect(ticket_number, SELECT_BY_TICKET, MODE_HISTORY))
  {
    switch(property_id)
    {
      case DEAL_TICKET:
      case DEAL_ORDER:
      case DEAL_POSITION_ID:
        long_var = OrderTicket();
        return true;
      case DEAL_TIME:
        long_var = exit ? OrderCloseTime() : OrderOpenTime();
        return true;
      case DEAL_TIME_MSC:
        long_var = (exit ? OrderCloseTime() : OrderOpenTime()) * 1000;
        return true;
      case DEAL_TYPE:
        long_var = OrderType2DealType(exit ? REVERSE(OrderType()) : OrderType());
        return true;
      case DEAL_ENTRY:
        long_var = exit ? DEAL_ENTRY_OUT : DEAL_ENTRY_IN;
        return true;
      case DEAL_MAGIC:
        long_var = OrderMagicNumber();
        return true;
    }
  }
  return false;
}

#define HistoryDealGetInteger MT5HistoryDealGetInteger

double MT5HistoryDealGetDouble(ulong ticket_number, ENUM_DEAL_PROPERTY_DOUBLE property_id)
{
  double temp;
  if(MT5HistoryDealGetDouble(ticket_number, property_id, temp))
  {
    return temp;
  }
  return 0;
}

bool MT5HistoryDealGetDouble(ulong ticket_number, ENUM_DEAL_PROPERTY_DOUBLE property_id, double &double_var)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return HistoryDealGetDouble(ticket_number, property_id, double_var);
  }
#endif
  bool exit = ((ticket_number & 0x8000000000000000) != 0);
  ticket_number &= ~0x8000000000000000;
  if(::OrderSelect(ticket_number, SELECT_BY_TICKET, MODE_HISTORY))
  {
    switch(property_id)
    {
      case DEAL_VOLUME:
        double_var = OrderLots();
        return true;
      case DEAL_PRICE:
        double_var = exit ? OrderClosePrice() : OrderOpenPrice();
        return true;
      case DEAL_COMMISSION:
        double_var = exit? 0 : OrderCommission();
        return true;
      case DEAL_SWAP:
        double_var = exit ? OrderSwap() : 0;
        return true;
      case DEAL_PROFIT:
        double_var = exit ? OrderProfit() : 0;
        return true;
    }
  }
  return false;
}

#define HistoryDealGetDouble MT5HistoryDealGetDouble

string MT5HistoryDealGetString(ulong ticket_number, ENUM_DEAL_PROPERTY_STRING property_id)
{
  string temp;
  if(MT5HistoryDealGetString(ticket_number, property_id, temp))
  {
    return temp;
  }
  return "";
}

bool MT5HistoryDealGetString(ulong ticket_number, ENUM_DEAL_PROPERTY_STRING property_id, string &string_var)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return HistoryDealGetString(ticket_number, property_id, string_var);
  }
#endif
  bool exit = ((ticket_number & 0x8000000000000000) != 0);
  ticket_number &= ~0x8000000000000000;
  if(::OrderSelect(ticket_number, SELECT_BY_TICKET, MODE_HISTORY))
  {
    switch(property_id)
    {
      case DEAL_SYMBOL:
        string_var = OrderSymbol();
        return true;
      case DEAL_COMMENT:
        string_var = OrderComment();
        return true;
    }
  }
  return false;
}

#define HistoryDealGetString MT5HistoryDealGetString

double MT5HistoryOrderGetDouble(ulong ticket_number, ENUM_ORDER_PROPERTY_DOUBLE property_id)
{
  double temp;
  if(MT5HistoryOrderGetDouble(ticket_number, property_id, temp))
  {
    return temp;
  }
  return 0;
}

bool MT5HistoryOrderGetDouble(ulong ticket_number, ENUM_ORDER_PROPERTY_DOUBLE property_id, double &double_var)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return HistoryOrderGetDouble(ticket_number, property_id, double_var);
  }
#endif
  bool exit = ((ticket_number & 0x8000000000000000) != 0);
  ticket_number &= ~0x8000000000000000;
  if(::OrderSelect(ticket_number, SELECT_BY_TICKET, MODE_HISTORY))
  {
    switch(property_id)
    {
      case ORDER_VOLUME_INITIAL:
      case ORDER_VOLUME_CURRENT:
        double_var = OrderLots();
        return true;
      case ORDER_PRICE_OPEN:
        double_var = exit ? OrderClosePrice() : OrderOpenPrice();
        return true;
      case ORDER_SL:
        double_var = OrderStopLoss();
        return true;
      case ORDER_TP:
        double_var = OrderTakeProfit();
        return true;
      case ORDER_PRICE_CURRENT:
        double_var = SymbolInfoDouble(OrderSymbol(), EnumOrderType2Code(OrderType()) == OP_BUY ? SYMBOL_BID : SYMBOL_ASK);
        return true;
    }
  }
  return false;
}

#define HistoryOrderGetDouble MT5HistoryOrderGetDouble

long MT5HistoryOrderGetInteger(ulong ticket_number, ENUM_ORDER_PROPERTY_INTEGER property_id)
{
  long temp = 0;
  if(MT5HistoryOrderGetInteger(ticket_number, property_id, temp))
  {
    return temp;
  }
  return 0;
}

bool MT5HistoryOrderGetInteger(ulong ticket_number, ENUM_ORDER_PROPERTY_INTEGER property_id, long &long_var)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return HistoryOrderGetInteger(ticket_number, property_id, long_var);
  }
#endif
  bool exit = ((ticket_number & 0x8000000000000000) != 0);
  ticket_number &= ~0x8000000000000000;
  if(::OrderSelect(ticket_number, SELECT_BY_TICKET, MODE_HISTORY))
  {
    switch(property_id)
    {
      case ORDER_TICKET:
        long_var = OrderTicket();
        return true;
      case ORDER_TIME_SETUP:
        long_var = exit ? OrderCloseTime() : OrderOpenTime();
        return true;
      case ORDER_TIME_SETUP_MSC:
        long_var = (exit ? OrderCloseTime() : OrderOpenTime()) * 1000;
        return true;
      case ORDER_TYPE:
        if(OrderType() <= OP_SELL)
        {
          long_var = OrderType2OrderType(exit ? REVERSE(OrderType()) : OrderType());
        }
        else
        {
          long_var = OrderType2OrderType(OrderType());
        }
        return true;
      case ORDER_STATE:
        long_var = OrderState();
        return true;
      case ORDER_TIME_EXPIRATION:
        long_var = OrderExpiration();
        return true;
      case ORDER_TIME_DONE:
        long_var = exit ? OrderCloseTime() : OrderOpenTime();
        return true;
      case ORDER_TIME_DONE_MSC:
        long_var = (exit ? OrderCloseTime() : OrderOpenTime()) * 1000;
        return true;
      case ORDER_TYPE_FILLING:
        long_var = ORDER_FILLING_FOK;
        return true;
      case ORDER_TYPE_TIME:
        long_var = OrderExpiration() ? ORDER_TIME_SPECIFIED : ORDER_TIME_GTC;
        return true;
      case ORDER_MAGIC:
        long_var = OrderMagicNumber();
        return true;
    }
  }
  return false;
}

#define HistoryOrderGetInteger MT5HistoryOrderGetInteger

string MT5HistoryOrderGetString(ulong ticket_number, ENUM_ORDER_PROPERTY_STRING property_id)
{
  string temp;
  if(MT5HistoryOrderGetString(ticket_number, property_id, temp))
  {
    return temp;
  }
  return "";
}

bool MT5HistoryOrderGetString(ulong ticket_number, ENUM_ORDER_PROPERTY_STRING property_id, string &string_var)
{
#ifdef __VIRTUAL__
  if(VIRTUAL::GetHandle() == 0)
  {
    return HistoryOrderGetString(ticket_number, property_id, string_var);
  }
#endif
  bool exit = ((ticket_number & 0x8000000000000000) != 0);
  ticket_number &= ~0x8000000000000000;
  if(::OrderSelect(ticket_number, SELECT_BY_TICKET, MODE_HISTORY))
  {
    switch(property_id)
    {
      case ORDER_SYMBOL:
        string_var = OrderSymbol();
        return true;
      case ORDER_COMMENT:
        string_var = OrderComment();
        return true;
    }
  }
  return false;
}

#define HistoryOrderGetString MT5HistoryOrderGetString


#ifdef __VIRTUAL__

#define MT5COPYFUNC(NAME, TYPE) \
int MT5Copy##NAME(string symbol_name, ENUM_TIMEFRAMES timeframe, int start_pos, int count, TYPE &array[]) \
{ \
  if(VIRTUAL::GetHandle() == 0) \
  { \
    return Copy##NAME(symbol_name, timeframe, start_pos, count, array); \
  } \
  else \
  { \
    const int b = iBarShift(symbol_name, timeframe, TimeCurrent()); \
    return Copy##NAME(symbol_name, timeframe, b + start_pos, count, array); \
  } \
} \
int MT5Copy##NAME(string symbol_name, ENUM_TIMEFRAMES timeframe, datetime start_pos, int count, TYPE &array[]) \
{ \
  return Copy##NAME(symbol_name, timeframe, start_pos, count, array); \
} \
int MT5Copy##NAME(string symbol_name, ENUM_TIMEFRAMES timeframe, datetime start_pos, datetime count, TYPE &array[]) \
{ \
  return Copy##NAME(symbol_name, timeframe, start_pos, count, array); \
} \

MT5COPYFUNC(Rates, MqlRates)
MT5COPYFUNC(Open, double)
MT5COPYFUNC(High, double)
MT5COPYFUNC(Low, double)
MT5COPYFUNC(Close, double)
MT5COPYFUNC(Time, datetime)
MT5COPYFUNC(TickVolume, long)
MT5COPYFUNC(RealVolume, long)
MT5COPYFUNC(Spread, int)

#define CopyRates MT5CopyRates
#define CopyOpen MT5CopyOpen
#define CopyHigh MT5CopyHigh
#define CopyLow MT5CopyLow
#define CopyClose MT5CopyClose
#define CopyTime MT5CopyTime
#define CopyTickVolume MT5CopyTickVolume
#define CopyRealVolume MT5CopyRealVolume
#define CopySpread MT5CopySpread


int MT5CopyBuffer(int indicator_handle, int buffer_num, int start_pos, int count, double &buffer[])
{
  if(VIRTUAL::GetHandle() == 0)
  {
    return CopyBuffer(indicator_handle, buffer_num, start_pos, count, buffer);
  }
  else
  {
    // this is a hack, we don't know symbol/timeframe by the handle
    const int b = iBarShift(_Symbol, _Period, TimeCurrent());
    return CopyBuffer(indicator_handle, buffer_num, b + start_pos, count, buffer);
  }
}

int MT5CopyBuffer(int indicator_handle, int buffer_num, datetime start_pos, int count, double &buffer[])
{
  return CopyBuffer(indicator_handle, buffer_num, start_pos, count, buffer);
}

int MT5CopyBuffer(int indicator_handle, int buffer_num, datetime start_pos, datetime count, double &buffer[])
{
  return CopyBuffer(indicator_handle, buffer_num, start_pos, count, buffer);
}

#define CopyBuffer MT5CopyBuffer
#define _0_ iBarShift(_Symbol, _Period, TimeCurrent())

#endif // __VIRTUAL__

#ifdef OrderSend
  #undef OrderSend
#endif // OrderSend

#ifdef OrderSelect
  #undef OrderSelect
#endif // OrderSelect

#ifdef OrderModify
  #undef OrderModify
#endif // OrderModify

#ifdef OrderDelete
  #undef OrderDelete 
#endif // OrderDelete

#ifdef OrdersTotal
  #undef OrdersTotal 
#endif // OrdersTotal

#ifdef OrderType
  #undef OrderType          
#endif // OrderType

#define OrderSend MT5OrderSend