//+------------------------------------------------------------------+
//|                                                         PPSO.mq5 |
//|                                    Copyright (c) 2020, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//| Parallel Particle Swarm Optimization using the tester math calcs |
//|                           https://www.mql5.com/ru/articles/8321/ |
//+------------------------------------------------------------------+

#property tester_no_cache


enum TEST
{
  Sphere,
  Griewank,
  Rosenbrock
};

sinput int Cycles = 100;
sinput TEST TestCase = Sphere;
sinput int SwarmSize = 0;
input int GroupCount = 0;


#include <ParticleSwarmParallel.mqh>

class BaseFunctor: public Functor
{
  protected:
    const int params;
    double max[], min[], steps[];
    
    double optimum;
    double result[];

  public:
    BaseFunctor(const int p): params(p)
    {
      ArrayResize(max, params);
      ArrayResize(min, params);
      ArrayResize(steps, params);
      ArrayInitialize(steps, 0);
      
      optimum = DBL_EPSILON; // TODO: set to NaN
    }
    
    virtual void test(void)
    {
      Swarm swarm(SwarmSize, GroupCount, params, max, min, steps);
      optimum = swarm.optimize(this, Cycles);
      swarm.getSolution(result);
    }
    
    double getSolution(double &output[]) const
    {
      ArrayCopy(output, result);
      return optimum;
    }
};

class PSOTests
{
  public:
    class Sphere: public BaseFunctor
    {
      public:
        Sphere(): BaseFunctor(3) // expected global minimum (0, 0, 0)
        {
          for(int i = 0; i < params; i++)
          {
            max[i] = 100;
            min[i] = -100;
          }
        }
        
        virtual double calculate(const double &vec[])
        {
          int dim = ArraySize(vec);
          double sum = 0;
          for(int i = 0; i < dim; i++) sum += pow(vec[i], 2);
          return -sum;
        }
    };
    
    class Griewank: public BaseFunctor
    {
      public:
        Griewank(): BaseFunctor(2) // expected global minimum (0, 0)
        {
          for(int i = 0; i < params; i++)
          {
            max[i] = 600;
            min[i] = -600;
          }
        }
        
        virtual double calculate(const double &vec[])
        {
          int dim = ArraySize(vec);
          
          double sum = 0.;
          double prod = 1.;
          for(int i = 0; i < dim; i++)
          {
        	  sum += pow(vec[i], 2);
        	  prod *= cos(vec[i] / sqrt(i + 1));
          }
          return -(sum / 4000 - prod + 1);
        }
    };
    
    class Rosenbrock: public BaseFunctor
    {
      public:
        Rosenbrock(): BaseFunctor(2) // expected global minimum (1, 1)
        {
          for(int i = 0; i < params; i++)
          {
            max[i] = +2.048;
            min[i] = -2.048;
          }
        }
        
        virtual double calculate(const double &vec[])
        {
          int dim = ArraySize(vec);
          double sum = 0.;
          for(int i = 0; i < dim - 1; i++)
          {
            sum += 100 * pow((vec[i + 1] - pow(vec[i], 2)), 2) + pow((1 - vec[i]), 2);
          }
          return -sum;
        }
    };
};


int OnInit()
{
  if(!MQLInfoInteger(MQL_TESTER))
  {
    EventSetTimer(1);
  }
  Print("Successive PSO of ", EnumToString(TestCase));
  return INIT_SUCCEEDED;
}

void OnTimer()
{
  EventKillTimer();
  OnTester();
}


double OnTester()
{
  MathSrand(GroupCount); // reproducable randomization
  
  BaseFunctor *functor = NULL;
  
  switch(TestCase)
  {
    case Sphere:
      functor = new PSOTests::Sphere();
      break;
    case Griewank:
      functor = new PSOTests::Griewank();
      break;
    case Rosenbrock:
      functor = new PSOTests::Rosenbrock();
      break;
  }
  
  functor.test();
  
  double output[];
  double result = functor.getSolution(output);
  if(MQLInfoInteger(MQL_OPTIMIZATION))
  {
    FrameAdd("PSO", 0xC0DE, result, output);
  }
  else
  {
    Print("Solution: ", result);
    for(int i = 0; i < ArraySize(output); i++)
    {
      Print(i, " ", output[i]);
    }
  }
  
  delete functor;
  return result;
}


int passcount = 0;
double best = -DBL_MAX;
double location[];

void OnTesterInit()
{
  Print("Parallel PSO of ", EnumToString(TestCase));
}

void OnTesterPass()
{
  ulong pass;
  string name;
  long id;
  double r;
  double data[];
  
  while(FrameNext(pass, name, id, r, data))
  {
    // compare r with all other passes results
    if(r > best)
    {
      best = r;
      ArrayCopy(location, data);
    }
    
    const int n = ArraySize(data);
    ArrayResize(data, n + 1);
    data[n] = r;
    ArrayPrint(data, 12);
  
    passcount++;
  }
}

void OnTesterDeinit()
{
  Print("Solution: ", best, " (after ", passcount, " passes)");
  ArrayPrint(location);
}
