//+------------------------------------------------------------------+
//|                                                   ExprBotPSO.mq5 |
//|                                    Copyright (c) 2020, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//| Parallel Particle Swarm Optimization                             |
//|                           https://www.mql5.com/ru/articles/8321/ |
//+------------------------------------------------------------------+
#property copyright "Copyright (c) 2020, Marketeer"
#property link      "https://www.mql5.com/en/users/marketeer"
#property version   "1.0"

#define PPSO_SHARED_SETTINGS __FILE__ + ".csv"
#property tester_no_cache
#property tester_file PPSO_SHARED_SETTINGS

#define INDICATOR_FUNCTORS
#define PSO_LOG_VERBOSE

#include <ParticleSwarmParallel.mqh>
#include <Filer.mqh>
#include <Settings.mqh>
#include <fxsaber/MT4Orders.mqh>
#include <fxsaber/Expert.mqh>
#include <fxsaber/Virtual/Virtual.mqh>
#include <ExpresSParserS/v1.1/ExpressionCompiler.mqh>
#include <ExpresSParserS/v1.1/TesterStats.mqh>


sinput string GroupTradingStrategy = ""; // >>  T R A D I N G   S T R A T E G Y
input string SignalBuy = "M_EMA_OPEN_9_6_27(Fast,Bar)/M_EMA_OPEN_9_6_27(Slow,Bar) > 1 + T";
input string SignalSell = "M_EMA_OPEN_9_6_27(Fast,Bar)/M_EMA_OPEN_9_6_27(Slow,Bar) < 1 - T";
input int Fast = 9;
input int Slow = 21;
input double T = 0.0; // Threshold
input double Lot = 0.1;


sinput string GroupVirtualLib = ""; // >>  V I R T U A L
sinput bool VirtualTester = false;
sinput ENUM_STATISTICS Estimator = STAT_PROFIT;


sinput string GroupPSO = ""; // >>  P S O
sinput bool InternalOptimization = false;
sinput bool PSO_Enable = false;
sinput int PSO_Cycles = 100;
sinput int PSO_SwarmSize = 0;
input int PSO_GroupCount = 0;
sinput int PSO_RandomSeed = 0;
sinput string Formula = "";


VariableTable vts, vtf; // trade signals, target function
ExpressionCompiler ecb(vts), ecs(vts), ece(vtf);
Promise *p1, *p2, *p3;
Settings settings;
MqlTick _ticks[];


class BaseFunctor: public Functor
{
  protected:
    const int params;
    double max[], min[], steps[];
    
    double optimum;
    double result[];

  public:
    BaseFunctor(const int p): params(p)
    {
      ArrayResize(max, params);
      ArrayResize(min, params);
      ArrayResize(steps, params);
      ArrayInitialize(steps, 0);
      
      optimum = DBL_EPSILON; // TODO: set to NaN
    }
    
    virtual bool test(Swarm::Stats *pstats = NULL)
    {
      Swarm swarm(PSO_SwarmSize, PSO_GroupCount, params, max, min, steps);
      if(MQLInfoInteger(MQL_OPTIMIZATION))
      {
        if(!swarm.restoreIndex()) return false;
      }
      optimum = swarm.optimize(this, PSO_Cycles);
      swarm.getSolution(result);
      if(MQLInfoInteger(MQL_OPTIMIZATION))
      {
        swarm.exportIndex(PSO_GroupCount);
      }
      if(pstats)
      {
        swarm.getStats(*pstats);
      }
      return true;
    }
    
    double getSolution(double &output[]) const
    {
      ArrayCopy(output, result);
      return optimum;
    }
};

class WorkerFunctor: public BaseFunctor
{
   string names[];

  public:
    WorkerFunctor(const Settings &s): BaseFunctor(s.size())
    {
      s.getNames(names);
      for(int i = 0; i < params; i++)
      {
        max[i] = s.get<double>(i, SET_COLUMN_STOP);
        min[i] = s.get<double>(i, SET_COLUMN_START);
        steps[i] = s.get<double>(i, SET_COLUMN_STEP);
      }
    }
    
    virtual double calculate(const double &vec[])
    {
      // assert ArraySize(vec) == params
      string msg = "";
      for(int i = 0; i < params; i++)
      {
        vts.set(names[i], vec[i]);
        msg += names[i] + ":" + (string)(float)vec[i] + ", ";
      }
      
      VIRTUAL::Tester(_ticks, OnTick, TesterStatistics(STAT_INITIAL_DEPOSIT));
      VIRTUAL::ResetTickets();
      const double r = p3 ? p3.with(EmbedTesterStats).resolve() : TesterStatistics(Estimator);
      msg += (string)(float)r;
      Print(msg);
      VIRTUAL::Delete();
      return r;
    }
};


int OnInit()
{
  if(InternalOptimization)
  {
    if(!VirtualTester)
    {
      Print("InternalOptimization mode is ignored: it's available with VirtualTester only");
    }
    else
    {
      FileReader f(PPSO_SHARED_SETTINGS);
      if(f.isReady() && f.read(settings))
      {
        const int n = settings.size();
        Print("Got settings: ", n);
      }
      else
      {
        if(MQLInfoInteger(MQL_OPTIMIZATION))
        {
          Print("FileLoad error: ", GetLastError());
          return INIT_PARAMETERS_INCORRECT;
        }
        else
        {
          Print("WARNING! Virtual optimization inside single pass - slowest mode, debugging only");
        }
      }
    }
  }

  vts.set("Fast", Fast);
  vts.set("Slow", Slow);
  vts.set("T", T);
  vts.set("Bar", 0);

  p1 = ecb.evaluate(SignalBuy, true);
  if(!ecb.success())
  {
    Print("Syntax error in Buy signal:");
    p1.print();
    return INIT_FAILED;
  }

  p2 = ecs.evaluate(SignalSell, true);
  if(!ecs.success())
  {
    Print("Syntax error in Sell signal:");
    p2.print();
    return INIT_FAILED;
  }

  if(StringLen(Formula) > 0)
  {
    EmbedTesterStats(vtf);
    p3 = ece.evaluate(Formula);
    if(!ece.success())
    {
      Print("Syntax error in Formula:");
      p3.print();
      return INIT_FAILED;
    }
  }
  else
  {
    p3 = NULL;
  }
  
  return INIT_SUCCEEDED;
}


bool IsNewBar(datetime now)
{
  static datetime lastBar = 0;
  
  now = now / PeriodSeconds() * PeriodSeconds();

  if(lastBar != now)
  {
    lastBar = now;
    return true;
  }
  return false;
}


#define _Ask SymbolInfoDouble(_Symbol, SYMBOL_ASK)
#define _Bid SymbolInfoDouble(_Symbol, SYMBOL_BID)

bool OnTesterCalled = false;

void OnTick()
{
  if(VirtualTester && !OnTesterCalled)
  {
    MqlTick _tick;
    SymbolInfoTick(_Symbol, _tick);
    const int n = ArraySize(_ticks);
    ArrayResize(_ticks, n + 1, n / 2);
    _ticks[n] = _tick;
    
    return; // skip all time scope and collect ticks
  }

  if(!IsNewBar(TimeCurrent())) return;
  
  const int bar = iBarShift(_Symbol, PERIOD_CURRENT, TimeCurrent());

  bool buy = p1.with("Bar", bar).resolve();
  bool sell = p2.with("Bar", bar).resolve();
  
  if(buy && sell)
  {
    buy = false;
    sell = false;
  }
  
  if(buy)
  {
    OrdersCloseAll(_Symbol, OP_SELL);
    if(OrdersTotalByType(_Symbol, OP_BUY) == 0)
    {
      OrderSend(_Symbol, OP_BUY, Lot, _Ask, 100, 0, 0);
    }
  }
  else if(sell)
  {
    OrdersCloseAll(_Symbol, OP_BUY);
    if(OrdersTotalByType(_Symbol, OP_SELL) == 0)
    {
      OrderSend(_Symbol, OP_SELL, Lot, _Bid, 100, 0, 0);
    }
  }
  else
  {
    OrdersCloseAll();
  }
}


void OrdersCloseAll(const string symbol = NULL, const int type = -1) // OP_BUY or OP_SELL
{
  for(int i = OrdersTotal() - 1; i >= 0; i--)
  {
    if(OrderSelect(i, SELECT_BY_POS))
    {
      if(OrderType() <= OP_SELL
      && (type == -1 || OrderType() == type)
      && (symbol == NULL || symbol == OrderSymbol()))
      {
        OrderClose(OrderTicket(), OrderLots(), OrderClosePrice(), 100);
      }
    }
  }
}

int OrdersTotalByType(const string symbol = NULL, const int type = -1) // OP_BUY or OP_SELL
{
  int count = 0;
  for(int i = OrdersTotal() - 1; i >= 0; i--)
  {
    if(OrderSelect(i, SELECT_BY_POS))
    {
      if(OrderType() <= OP_SELL
      && (type == -1 || OrderType() == type)
      && (symbol == NULL || symbol == OrderSymbol()))
      {
        count++;
      }
    }
  }
  return count;
}


template<typename T>
bool ResetOptimizableParam(const string name, const int h)
{
  bool enabled;
  T value, start, step, stop;
  if(ParameterGetRange(name, enabled, value, start, step, stop))
  {
    // disable all native optimization except for PSO-related params
    // preserve original settings in the file h
    if((StringFind(name, "PSO_") != 0) && enabled)
    {
      ParameterSetRange(name, false, value, start, step, stop);
      FileWrite(h, name, start, step, stop); // 5007
      return true;
    }
  }
  return false;
}

void ExportSettings(const bool existence = false)
{
  if(existence && FileIsExist(PPSO_SHARED_SETTINGS)) return;
  
  int h = FileOpen(PPSO_SHARED_SETTINGS, FILE_ANSI|FILE_WRITE|FILE_CSV, ',');
  if(h == INVALID_HANDLE)
  {
    Print("FileSave error: ", GetLastError());
  }
  
  MqlParam parameters[];
  string names[];
  int count = 0;
  
  EXPERT::Parameters(0, parameters, names);
  for(int i = 0; i < ArraySize(names); i++)
  {
    if(ResetOptimizableParam<double>(names[i], h))
    {
      const int n = ArraySize(header);
      ArrayResize(header, n + 1);
      header[n] = names[i];
      count++;
    }
  }
  Print(count, " optimized parameters found");
  FileClose(h); // 5008
}


string resultfile;
string header[];
int passcount = 0;
double best = -DBL_MAX;
double location[];

void OnTesterInit()
{
  if(VirtualTester && InternalOptimization)
  {
    if(!PSO_Enable)
    {
      Print("Internal optimization in brute force mode makes no sense within built-in optimization");
      Print("Either enable PSO mode (PSO_Enable=true): fast, multiple cores");
      Print("or disable built-in optimization: slow, single core optimization withing single pass, both PSO and brute force");
      ExpertRemove();
      return;
    }
    
    ExportSettings();
  
    bool enabled;
    long value, start, step, stop;
    if(ParameterGetRange("PSO_GroupCount", enabled, value, start, step, stop))
    {
      if(!enabled)
      {
        const int cores = TerminalInfoInteger(TERMINAL_CPU_CORES);
        Print("PSO_GroupCount is set to default (number of cores): ", cores);
        ParameterSetRange("PSO_GroupCount", true, 0, 1, 1, cores);
      }
    }
    resultfile = PPSO_FILE_PREFIX + FileUtils::escape(MQLInfoString(MQL_PROGRAM_NAME)) + "-" + TimeStamp() + ".csv";
  }
  
  // remove CRC indices from previous optimization runs
  Swarm::removeIndex();
}

string TimeStamp(const datetime now = 0)
{
  string timestamp = TimeToString(now ? now : TimeLocal());
  StringReplace(timestamp, ".", "");
  StringReplace(timestamp, ":", "");
  StringReplace(timestamp, " ", "");
  return timestamp;
}

void OnTesterPass()
{
  ulong pass;
  string name;
  long id;
  double r;
  double data[];
  
  while(FrameNext(pass, name, id, r, data))
  {
    #ifdef PSO_LOG_VERBOSE
    Print(pass, " ", name, " ", id, " ", r);
    #endif
    // compare r with all other passes results
    if(r > best)
    {
      best = r;
      ArrayCopy(location, data);
    }
    
    int h = FileOpen(resultfile, FILE_READ | FILE_WRITE | FILE_ANSI | FILE_CSV | FILE_SHARE_WRITE);
    if(h != INVALID_HANDLE)
    {
      if(FileSize(h) == 0)
      {
        FileWriteString(h, "N,G,P,R,");
        for(int i = 0; i < ArraySize(data); i++)
        {
          FileWriteString(h, (i < ArraySize(header) ? header[i] : "#" + (string)(i + 1)) + ",");
        }
        FileWriteString(h, "\n");
      }
      else
      {
        FileSeek(h, 0, SEEK_END);
      }
      
      FileWriteString(h, (string)passcount + ",");
      FileWriteString(h, (string)id + ",");
      FileWriteString(h, (string)pass + ",");
      FileWriteString(h, (string)r + ",");
      
      for(int i = 0; i < ArraySize(data); i++)
      {
        FileWriteString(h, (string)(float)data[i] + ",");
      }
      
      FileWriteString(h, "\n");
      FileClose(h);
      passcount++;
    }
  }
}

void OnTesterDeinit()
{
  if(best != -DBL_MAX)
  {
    Print("Solution: ", best);
    ArrayPrint(location);
  }
}

double OnTester()
{
  if(VirtualTester)
  {
    OnTesterCalled = true;

    // MQL API implies some limitations for this function, so ticks are collected in OnTick
    // const int size = CopyTicksRange(_Symbol, _ticks, COPY_TICKS_ALL);

    const int size = ArraySize(_ticks);
    PrintFormat("Ticks size=%d error=%d", size, GetLastError());
    if(size <= 0) return 0;
    
    if(settings.isVoid() || !InternalOptimization)
    {
      Print("Fallback to single test: void settings - ", settings.isVoid(), " io - ", InternalOptimization);
      VIRTUAL::Tester(_ticks, OnTick, TesterStatistics(STAT_INITIAL_DEPOSIT));
      Print(VIRTUAL::ToString(INT_MAX));
      Print("Trades: ", VIRTUAL::VirtualOrdersHistoryTotal());
      return p3 ? p3.with(EmbedTesterStats).resolve() : TesterStatistics(Estimator);
    }

    settings.print();
    const int n = settings.size();
    
    if(PSO_Enable)
    {
      MathSrand(PSO_GroupCount + PSO_RandomSeed); // reproducable randomization
      WorkerFunctor worker(settings);
      Swarm::Stats stats;
      if(worker.test(&stats))
      {
        double output[];
        double result = worker.getSolution(output);
        if(MQLInfoInteger(MQL_OPTIMIZATION))
        {
          FrameAdd(StringFormat("PSO%d/%d", stats.done, stats.planned), PSO_GroupCount, result, output);
        }
        #ifdef PSO_LOG_VERBOSE
        ArrayResize(output, n + 1);
        output[n] = result;
        ArrayPrint(output);
        #endif
        return result;
      }
    }
    else // brute force
    {
      string names[];
      double output[];
      settings.getNames(names);
      double solution = -DBL_MAX;
      while(settings.getNext(output))
      {
        for(int i = 0; i < n; i++)
        {
          vts.set(names[i], output[i]);
        }
        VIRTUAL::Tester(_ticks, OnTick, TesterStatistics(STAT_INITIAL_DEPOSIT));
        VIRTUAL::ResetTickets();
        double result = p3 ? p3.with(EmbedTesterStats).resolve() : TesterStatistics(Estimator);
        VIRTUAL::Delete();
        solution = MathMax(result, solution);
        if(MQLInfoInteger(MQL_OPTIMIZATION))
        {
          FrameAdd("RAW", PSO_GroupCount, result, output);
        }
        #ifdef PSO_LOG_VERBOSE
        ArrayResize(output, n + 1);
        output[n] = result;
        ArrayPrint(output);
        #endif
      }
      return solution;
    }
    return 0;
  }
  return p3 ? p3.with(EmbedTesterStats).resolve() : TesterStatistics(Estimator);
}

void OnDeinit(const int)
{
}