#include <Controls/Rect.mqh>
#include "LayoutStdLib.mqh"

class LayoutExporter
{
  protected:
    CWndContainer *owner;
    StdLayoutCache *cache;
    int handle;

    string extractClass(const string name) const
    {
      string array[];
      if(StringSplit(name, ' ', array) > 0)
      {
        return array[0];
      }
      return name;
    }

    string extractName(const string id) const
    {
      const int n = StringLen(id);
      for(int i = 0; i < n; i++)
      {
        ushort c = StringGetCharacter(id, i);
        if(c > '9')
        {
          return StringSubstr(id, i);
        }
      }
      return id;
    }

    bool isNonAligned(CWnd *element)
    {
      const ENUM_WND_ALIGN_FLAGS f = element.Alignment();
      return f == WND_ALIGN_NONE;
    }

    string extractAlign(CWnd *element) const
    {
      ENUM_WND_ALIGN_FLAGS f = element.Alignment();
      string s = EnumToString(f);
      string result = "";
      if(StringFind(s, "::") > 0)
      {
        int count = 0;
        for(int i = 15; i >= 0; i--)
        {
          int mask = ((int)f) & (i);
          if(mask != 0)
          {
            s = EnumToString((ENUM_WND_ALIGN_FLAGS)mask);
            if(StringFind(s, "::") == -1) // exact element of enum
            {
              f = (ENUM_WND_ALIGN_FLAGS)((int)f & ~mask); // clear bits
              if(result != "") result += "|";
              result += s;
              count++;
            }
          }
        }
        if(count > 1)
        {
          result = "(ENUM_WND_ALIGN_FLAGS)(" + result + ")";
        }
      }
    
      if(f == WND_ALIGN_CONTENT)
      {
        if(StringLen(result) > 0)
        {
          return "(ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|" + result + ")";
        }
        else
        {
          return "(ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT)";
        }
      }
      
      if(StringLen(result) > 0)
      {
        return result;
      }
      
      return EnumToString(f);
    }

    string extractMargins(CWnd *element) const
    {
      CRect r = element.Margins();
      return StringFormat(" <= PackedRect(%d, %d, %d, %d)", r.left, r.top, r.right, r.bottom);
    }

    // TODO: need to detect non-default colors per control type
    string extractColors(CWnd *element) const
    {
      CWndObj *obj = dynamic_cast<CWndObj *>(element);
      if(obj != NULL)
      {
        obj.ColorBackground();
        obj.ColorBorder();
        obj.Color();
      }
      else
      {
        CWndClient *client = dynamic_cast<CWndClient *>(element);
        if(client != NULL)
        {
          client.ColorBorder();
          client.ColorBackground();
        }
      }
      return NULL;
    }

    int saveElement(CWnd *element)
    {
      static string level = "";
      int count = 0;
      
      if(CheckPointer(element) != POINTER_INVALID)
      {
        if(element == owner)
        {
          FileWriteString(handle, level + "// GUI Layout for MQL app (standard controls library)\n");
          FileWriteString(handle, level + "// Don't forget to keep a pointer to the main container to call Pack() on refresh\n");
        }
      
        CWndContainer *container = dynamic_cast<CWndContainer *>(element);
        if(container != NULL)
        {
          const int index = cache.indexOf(container);
          if(index > -1)
          {
            FileWriteString(handle, level + "{\n");
            level += "  ";
          
          
            FileWriteString(handle, level);
            
            const string name = extractName(container.Name());
            
            if(isNonAligned(container))
            {
              FileWriteString(handle, "_layout<" + extractClass(container._rtti) + "> " + name
                + "(\"" + name + "\", " + (string)container.Left() + ", " + (string)container.Top()
                + ", " + (string)container.Right() + ", " + (string)container.Bottom() + ");\n");
            }
            else
            {
              FileWriteString(handle, "_layout<" + extractClass(container._rtti) + "> " + name
                + "(\"" + name + "\", " + (string)container.Width() + ", " + (string)container.Height()
                + ", " + extractAlign(container) + ");\n");
            }
              
            CBox *box = dynamic_cast<CBox *>(container);
            if(box != NULL)
            {
              if(box.LayoutStyle() != LAYOUT_STYLE_HORIZONTAL)
              {
                FileWriteString(handle, level);
                FileWriteString(handle, name + " <= " + EnumToString(box.LayoutStyle()) + ";\n");
              }
            }
    
            FileWriteString(handle, level + name + extractMargins(container) + ";\n");
    
            count++;
          
            FileWriteString(handle, level + "{\n");
    
            level += "  ";
          }
      
          int children = 0;
          for(int j = 0; j < container.ControlsTotal(); j++)
          {
            children += saveElement(container.Control(j));
          }
          
          count += children;
            
          if(index > -1)
          {
            if(children == 0)
            {
              FileWriteString(handle, level + "// dummy (feel free to delete)\n");
            }
            
            StringSetLength(level, StringLen(level) - 2);
            FileWriteString(handle, level + "}\n");
            StringSetLength(level, StringLen(level) - 2);
            FileWriteString(handle, level + "}\n");
          }
          
        }
        else
        {
          if(cache.indexOf(element) > -1)
          {
            FileWriteString(handle, level);
            
            const string name = extractName(element.Name());
            if(isNonAligned(element))
            {
              FileWriteString(handle, "_layout<" + extractClass(element._rtti) + "> " + name
                + "(\"" + name + "\", " + (string)element.Left() + ", " + (string)element.Top()
                + ", " + (string)element.Right() + ", " + (string)element.Bottom() + ");\n");
            }
            else
            {
              FileWriteString(handle, "_layout<" + extractClass(element._rtti) + "> " + name
                + "(\"" + name + "\", " + (string)element.Width() + ", " + (string)element.Height()
                + ", " + extractAlign(element) + ");\n");
            }
            FileWriteString(handle, level + name + extractMargins(element) + ";\n");
            count++;
          }
        }
      }
    
      return count;
    }

  public:
    LayoutExporter(CWndContainer *ptr): owner(ptr) {}

    int saveToFile(StdLayoutCache *c, const string name)
    {
      cache = c;
      handle = FileOpen(name, FILE_TXT | FILE_ANSI | FILE_WRITE);
      const int result = saveElement(owner);
      FileClose(handle);
      return result;
    }
};
