//+------------------------------------------------------------------+
//|                                              ControlsDialog4.mqh |
//|                               Copyright (c) 2019-2020, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//|                           https://www.mql5.com/ru/articles/7734/ |
//+------------------------------------------------------------------+

#include <ControlsPlus/Dialog.mqh>
#include <ControlsPlus/Button.mqh>
#include <ControlsPlus/Edit.mqh>
#include <ControlsPlus/DatePicker.mqh>
#include <Layouts/Box.mqh>
#include <Layouts/GridTkEx.mqh>
#include <Layouts/SpinEditResizable.mqh>
#include <Layouts/ComboBoxResizable.mqh>
#include <Layouts/ListViewResizable.mqh>
#include <Layouts/CheckGroupResizable.mqh>
#include <Layouts/RadioGroupResizable.mqh>
#include <Layouts/AppDialogResizable.mqh>
#include <Layouts/LayoutStdLib.mqh>

//#include <Layouts/TraceObjects.mqh>

//+------------------------------------------------------------------+
//| defines                                                          |
//+------------------------------------------------------------------+
// buttons
#define BUTTON_WIDTH (100)
#define BUTTON_HEIGHT (20)
// editbox
#define EDIT_HEIGHT (20)
// groups
#define GROUP_WIDTH (150)
#define LIST_HEIGHT (179)
#define RADIO_HEIGHT (56)
#define CHECK_HEIGHT (93)
#define DEFAULT_MARGIN 2


class CControlsDialog;

class NotifiableButton: public Notifiable<CButton>
{
  public:
    virtual bool onEvent(const int event, /* i.e. CControlsDialog * */ void *anything) override
    {
      CControlsDialog *parent = dynamic_cast<CControlsDialog *>(anything);
      parent.SetCallbackText(__FUNCTION__ + " " + this.Name() + " " + (string)this.StateFlags());
      this.StateFlagsReset(7);
      return true;
    }
};

class MyLayoutStyleable: public StdLayoutStyleable
{
  public:
    virtual void apply(CWnd *control, const STYLER_PHASE phase) override
    {
      CButton *button = dynamic_cast<CButton *>(control);
      if(button != NULL)
      {
        if(phase == STYLE_PHASE_BEFORE_INIT)
        {
          button.Font("Arial Black");
        }
      }
      else
      {
        CEdit *edit = dynamic_cast<CEdit *>(control);
        if(edit != NULL && edit.ReadOnly())
        {
          if(phase == STYLE_PHASE_AFTER_INIT)
          {
            edit.ColorBackground(clrLightGray);
          }
        }
      }

      if(phase == STYLE_PHASE_BEFORE_INIT)
      {
        control.Margins(DEFAULT_MARGIN);
      }
    }
};

class MyStdLayoutCache: public StdLayoutCache
{
  protected:
    MyLayoutStyleable styler;
    CControlsDialog *parent;

  public:
    MyStdLayoutCache(CControlsDialog *owner): parent(owner) {}

    virtual StdLayoutStyleable *getStyler() const override
    {
      return (StdLayoutStyleable *)&styler;
    }
    
    virtual bool onEvent(const int event, CWnd *control) override
    {
      if(control != NULL)
      {
        parent.SetCallbackText(__FUNCTION__ + " " + control.Name());
        return true;
      }
      return false;
    }
};

enum LIST_TYPE_TEST
{
  LIST_OF_OPTIONS, // combo, radio, checkbox
  LIST_LISTVIEW
};

//+-----------------------------------------------------------------------+
//| CControlsDialog                                                       |
//| Main dialog window with controls                                      |
//+-----------------------------------------------------------------------+
class CControlsDialog: public AppDialogResizable
{
  private:
    CBox *m_main;
    //CBox m_edit_row;
    CEdit m_edit;
    //CBox m_button_row;
    //CButton m_button1;
    //CButton m_button2;
    CButton m_button3;
    //CBox m_spin_date_row;
    SpinEditResizable m_spin_edit;
    CDatePicker m_date;
    //CBox m_lists_row;
    CBox m_lists_column1;
    ComboBoxResizable m_combo_box;
    RadioGroupResizable m_radio_group;
    CheckGroupResizable m_check_group;
    CBox m_lists_column2;
    ListViewResizable m_list_view;
    
    long button1index;
    MyStdLayoutCache *cache;
    
  public:
    CControlsDialog(void);
    ~CControlsDialog(void);

    bool CreateLayout(const long chart, const string name, const int subwin, const int x1, const int y1, const int x2, const int y2);

    // general event handler
    virtual bool OnEvent(const int id, const long &lparam, const double &dparam, const string &sparam);

    void SetCallbackText(const string text) { m_edit.Text(text); }

  protected:
    // specific handlers
    void OnClickButton1(void);
    void OnClickButton3(void);
    void OnChangeSpinEdit(void);
    void OnChangeDate(void);
    void OnChangeListView(void);
    void OnChangeComboBox(void);
    void OnChangeRadioGroup(void);
    void OnChangeCheckGroup(void);

    // containers
    void createSubList(CBox *container, const LIST_TYPE_TEST type);
    
    virtual void SelfAdjustment(const bool restore = false) override;
};

#include <Layouts/LayoutDefines.mqh>


//+------------------------------------------------------------------+
//| Event handling                                                   |
//+------------------------------------------------------------------+

EVENT_MAP_BEGIN(CControlsDialog)
  ON_EVENT(ON_CLICK, m_button3, OnClickButton3)
  ON_EVENT(ON_CHANGE, m_spin_edit, OnChangeSpinEdit)
  ON_EVENT(ON_CHANGE, m_date, OnChangeDate)
  ON_EVENT(ON_CHANGE, m_list_view, OnChangeListView)
  ON_EVENT(ON_CHANGE, m_combo_box, OnChangeComboBox)
  ON_EVENT(ON_CHANGE, m_radio_group, OnChangeRadioGroup)
  ON_EVENT(ON_CHANGE, m_check_group, OnChangeCheckGroup)
  ON_EVENT_LAYOUT_CTRL_DLG(ON_CLICK, cache, NotifiableButton)
  ON_EVENT_LAYOUT_INDEX(ON_CLICK, cache, button1index, OnClickButton1)
  ON_EVENT_LAYOUT_ARRAY(ON_CLICK, cache)
EVENT_MAP_END(AppDialogResizable)

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
CControlsDialog::CControlsDialog(void)
{
  cache = new MyStdLayoutCache(&this);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
CControlsDialog::~CControlsDialog(void)
{
  delete cache;
}

void CControlsDialog::createSubList(CBox *container, const LIST_TYPE_TEST type)
{
  if(type == LIST_OF_OPTIONS)
  {
    _layout<CBox> listColumnLeft(container, "listscolumn1", GROUP_WIDTH, LIST_HEIGHT, LAYOUT_STYLE_VERTICAL);
    listColumnLeft <= clrLightBlue;
    listColumnLeft["align"] <= (WND_ALIGN_CONTENT|WND_ALIGN_TOP|WND_ALIGN_BOTTOM);
    
    {
      _layout<ComboBoxResizable> combo(m_combo_box, "ComboBox", GROUP_WIDTH, EDIT_HEIGHT);
      combo["align"] <= (WND_ALIGN_TOP|WND_ALIGN_LEFT|WND_ALIGN_RIGHT);
      combo < new StdItemGenerator<ComboBoxResizable>(16);
    }
    
    {
      _layout<RadioGroupResizable> radio(m_radio_group, "RadioGroup", GROUP_WIDTH, RADIO_HEIGHT);
      radio <= WND_ALIGN_WIDTH;
      radio < new StdGroupItemGenerator<RadioGroupResizable>(5, "Radio");
    }
    
    {
      _layout<CheckGroupResizable> check(m_check_group, "CheckGroup", GROUP_WIDTH, CHECK_HEIGHT);
      check <= WND_ALIGN_WIDTH;
      check < new StdGroupItemGenerator<CheckGroupResizable>(10, "Item");
    }
  }
  else
  if(type == LIST_LISTVIEW)
  {
    _layout<CBox> listColumnRight(container, "listscolumn2", GROUP_WIDTH, LIST_HEIGHT);
    listColumnRight <= WND_ALIGN_HEIGHT;

    {
      _layout<ListViewResizable> list(m_list_view, "ListView", GROUP_WIDTH, LIST_HEIGHT);
      list <= WND_ALIGN_CLIENT < new SymbolsItemGenerator<ListViewResizable>();
    }
  }
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CControlsDialog::CreateLayout(const long chart, const string name, const int subwin, const int x1, const int y1, const int x2, const int y2)
{
  StdLayoutBase::setCache(cache); // assign the cache object to store implicit objects

  {
    _layout<CControlsDialog> dialog(this, name, x1, y1, x2, y2);

    {
      // example of implicit object in the cache
      _layout<CBox> clientArea("main", ClientAreaWidth(), ClientAreaHeight(), LAYOUT_STYLE_VERTICAL);
      m_main = clientArea.get(); // we can get the pointer to the object from cache (if required)
      clientArea <= WND_ALIGN_CLIENT <= 0.0; // double type is important

      {
        // another implicit container (we need no access it directly)
        _layout<CBox> editRow("editrow", ClientAreaWidth(), EDIT_HEIGHT * 1.5, (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_TOP|WND_ALIGN_WIDTH));

        {
          // for editboxes default boolean property is ReadOnly
          _layout<CEdit> edit(m_edit, "Edit", ClientAreaWidth(), EDIT_HEIGHT, true);
          
          // double type is important for margins assignment with operator <=
          edit <= 4.0 * DEFAULT_MARGIN <= WND_ALIGN_WIDTH;
        }
      }
      
      {
        _layout<CBox> buttonRow("buttonrow", ClientAreaWidth(), BUTTON_HEIGHT * 1.5);
        buttonRow["align"] <= (WND_ALIGN_CONTENT|WND_ALIGN_WIDTH);

        {
          _layout<CButton> button1("Button1");
          button1index = cache.cacheSize() - 1; // or button1index = button1.get().Id();
          button1["width"] <= BUTTON_WIDTH;
          button1["height"] <= BUTTON_HEIGHT;
        }
        
        {
          _layout<NotifiableButton> button2("Button2", BUTTON_WIDTH, BUTTON_HEIGHT);
        }
        
        {
          _layout<CButton> button3(m_button3, "Button3", BUTTON_WIDTH, BUTTON_HEIGHT, "Locked");
          button3 <= true; // for buttons default boolean property is Locking
        }
        // if buttons not wrapped by "personal" braces they are added in reverse order
        // because destructors are called in the reverse order of creation
      }
      
      {
        _layout<CBox> spinDateRow("spindaterow", ClientAreaWidth(), BUTTON_HEIGHT * 1.5);
        spinDateRow["align"] <= (WND_ALIGN_CONTENT|WND_ALIGN_WIDTH);

        {
          _layout<SpinEditResizable> spin(m_spin_edit, "SpinEdit", GROUP_WIDTH, EDIT_HEIGHT);
          spin["min"] <= 10;
          spin["max"] <= 1000;
          spin["value"] <= 100; // can set value only after limits (this is how SpinEdits work)
        }

        {
          _layout<CDatePicker> date(m_date, "Date", GROUP_WIDTH, EDIT_HEIGHT, TimeCurrent());
        }
      }
      
      {
        _layout<CBox> listRow("listsrow", ClientAreaWidth(), LIST_HEIGHT);
        listRow["top"] <= (int)(EDIT_HEIGHT * 1.5 * 3);
        listRow["align"] <= (WND_ALIGN_CONTENT|WND_ALIGN_CLIENT);
        (listRow <= clrDarkBlue)["border"] <= clrBlue;
        // for containers (such as CBox) default color sets background
        // clrMagenta is just to make it clear it works
        
        createSubList(&m_lists_column1, LIST_OF_OPTIONS);
        createSubList(&m_lists_column2, LIST_LISTVIEW);
        // or vice versa (changed order gives swapped left/right side location)
        // createSubList(&m_lists_column1, LIST_LISTVIEW);
        // createSubList(&m_lists_column2, LIST_OF_OPTIONS);
      }
    }
  }

  // m_main.Pack();
  SelfAdjustment();

  // debug log: mql class instances (controls)
  // TraverseWindows(&this);
  // debug log: chart objects
  // TraceObjects();

  return true;
}

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CControlsDialog::OnClickButton1(void)
{
    m_edit.Text(__FUNCTION__);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CControlsDialog::OnClickButton3(void)
{
    if(m_button3.Pressed())
        m_edit.Text(__FUNCTION__ + " On");
    else
        m_edit.Text(__FUNCTION__ + " Off");
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CControlsDialog::OnChangeSpinEdit()
{
    m_edit.Text(__FUNCTION__ + " : Value=" + IntegerToString(m_spin_edit.Value()));
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CControlsDialog::OnChangeDate(void)
{
    m_edit.Text(__FUNCTION__ + " \"" + TimeToString(m_date.Value(), TIME_DATE) + "\"");
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CControlsDialog::OnChangeListView(void)
{
    m_edit.Text(__FUNCTION__ + " \"" + m_list_view.Select() + "\"");
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CControlsDialog::OnChangeComboBox(void)
{
    m_edit.Text(__FUNCTION__ + " \"" + m_combo_box.Select() + "\"");
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CControlsDialog::OnChangeRadioGroup(void)
{
    m_edit.Text(__FUNCTION__ + " : Value=" + IntegerToString(m_radio_group.Value()));
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CControlsDialog::OnChangeCheckGroup(void)
{
    m_edit.Text(__FUNCTION__ + " : Value=" + IntegerToString(m_check_group.Value()));
}

//+------------------------------------------------------------------+

void CControlsDialog::SelfAdjustment(const bool restore = false)
{
  m_main.Pack();
  Rebound(Rect());
}
