//+---------------------------------------------------------------------+
//|                                                        multiweb.mqh |
//|                                       Copyright (c) 2018, Marketeer |
//|                             https://www.mql5.com/en/users/marketeer |
//| Asynchronous Non-Blocking WebRequest Implemenation on Helper Charts |
//|                               https://www.mql5.com/ru/articles/5337 |
//+---------------------------------------------------------------------+

#include <fxsaber\ResourceData.mqh>

sinput uint MessageBroadcast = 1;

#define MSG_DEINIT   1 // tear down (manager <-> worker)
#define MSG_WEB      2 // start request (client -> manager -> worker)
#define MSG_DONE     3 // request is completed (worker -> client, worker -> manager)
#define MSG_ERROR    4 // request has failed (manager -> client, worker -> client)
#define MSG_DISCOVER 5 // find the manager (client -> manager)
#define MSG_ACCEPTED 6 // request is in progress (manager -> client)
#define MSG_HELLO    7 // the manager is found (manager -> client)

#define MSG(x) (x - MessageBroadcast - CHARTEVENT_CUSTOM)
#define TO_MSG(X) ((ushort)(MessageBroadcast + X))

#define ERROR_NO_IDLE_WORKER  -1
#define ERROR_MQL_WEB_REQUEST -2

#define LEADSIZE (sizeof(int)*5)


class ResourceMediator
{
  private:
    const RESOURCEDATA<uchar> *resource;
    
    union lead
    {
      struct _l
      {
        int m; // method
        int u; // url
        int h; // headers
        int t; // timeout
        int b; // body
      }
      lengths;
      
      uchar sizes[LEADSIZE];
      
      int total()
      {
        return lengths.m + lengths.u + lengths.h + lengths.t + lengths.b;
      }
    }
    metadata;
  
    union _s
    {
      int x;
      uchar b[sizeof(int)];
    }
    int2chars;
    
    
  public:
    ResourceMediator(const RESOURCEDATA<uchar> *r): resource(r)
    {
    }
    
    void packRequest(const string method, const string url, const string headers, const int timeout, const uchar &body[])
    {
      metadata.lengths.m = StringLen(method) + 1;
      metadata.lengths.u = StringLen(url) + 1;
      metadata.lengths.h = StringLen(headers) + 1;
      metadata.lengths.t = sizeof(int);
      metadata.lengths.b = ArraySize(body);
      
      uchar data[];
      ArrayResize(data, LEADSIZE + metadata.total());
      
      ArrayCopy(data, metadata.sizes);
      
      int cursor = LEADSIZE;
      uchar temp[];
      StringToCharArray(method, temp);
      ArrayCopy(data, temp, cursor);
      ArrayResize(temp, 0);
      cursor += metadata.lengths.m;
      
      StringToCharArray(url, temp);
      ArrayCopy(data, temp, cursor);
      ArrayResize(temp, 0);
      cursor += metadata.lengths.u;
      
      StringToCharArray(headers, temp);
      ArrayCopy(data, temp, cursor);
      ArrayResize(temp, 0);
      cursor += metadata.lengths.h;
      
      int2chars.x = timeout;
      ArrayCopy(data, int2chars.b, cursor);
      cursor += metadata.lengths.t;
      
      ArrayCopy(data, body, cursor);
      
      resource = data;
    }
    
    void unpackRequest(string &method, string &url, string &headers, int &timeout, uchar &body[])
    {
      uchar array[];  
      int n = resource.Get(array);
      Print(ChartID(), ": Got ", n, " bytes in request");
      
      ArrayCopy(metadata.sizes, array, 0, 0, LEADSIZE);
      int cursor = LEADSIZE;
      
      method = CharArrayToString(array, cursor, metadata.lengths.m);
      cursor += metadata.lengths.m;
      url = CharArrayToString(array, cursor, metadata.lengths.u);
      cursor += metadata.lengths.u;
      headers = CharArrayToString(array, cursor, metadata.lengths.h);
      cursor += metadata.lengths.h;
      
      ArrayCopy(int2chars.b, array, 0, cursor, metadata.lengths.t);
      timeout = int2chars.x;
      cursor += metadata.lengths.t;
      
      if(metadata.lengths.b > 0)
      {
        ArrayCopy(body, array, 0, cursor, metadata.lengths.b);
      }
    }
    
    void packResponse(const string source, const uchar &result[], const string &result_headers)
    {
      uchar src[];
      uchar meta[];
      uchar data[];
      
      int c = StringToCharArray(source, src);
      int m = StringToCharArray(result_headers, meta);
      int n = ArraySize(result);
      ArrayResize(data, c + m + n + 3 * sizeof(int2chars));

      int2chars.x = c;
      ArrayCopy(data, int2chars.b);
      int2chars.x = m;
      ArrayCopy(data, int2chars.b, sizeof(int2chars));
      int2chars.x = n;
      ArrayCopy(data, int2chars.b, 2 * sizeof(int2chars));
      
      ArrayCopy(data, src, 3 * sizeof(int2chars));
      ArrayCopy(data, meta, 3 * sizeof(int2chars) + c);
      ArrayCopy(data, result, 3 * sizeof(int2chars) + c + m);
      
      resource = data;
    }
    
    void unpackResponse(uchar &initiator[], uchar &headers[], uchar &text[])
    {
      uchar array[];
      int n = resource.Get(array);
      Print(ChartID(), ": Got ", n, " bytes in response");
    
      ArrayCopy(int2chars.b, array, 0, 0, sizeof(int2chars));
      int c = int2chars.x; // original resource name
      ArrayCopy(int2chars.b, array, 0, sizeof(int2chars), sizeof(int2chars));
      int h = int2chars.x; // headers
      ArrayCopy(int2chars.b, array, 0, 2 * sizeof(int2chars), sizeof(int2chars));
      int d = int2chars.x; // document
      ArrayCopy(initiator, array, 0, 3 * sizeof(int2chars), c);
      ArrayCopy(headers, array, 0, 3 * sizeof(int2chars) + c, h);
      ArrayCopy(text, array, 0, 3 * sizeof(int2chars) + c + h, d);
    }
};

class WebWorker
{
  protected:
    long chartID;
    bool busy;
    const RESOURCEDATA<uchar> *resource;
    const string prefix;
    string fullname;
    
    const RESOURCEDATA<uchar> *allocate()
    {
      release();
      resource = new RESOURCEDATA<uchar>(prefix + (string)chartID);
      fullname = StringSubstr(MQLInfoString(MQL_PROGRAM_PATH), StringLen(TerminalInfoString(TERMINAL_PATH)) + 5) + prefix + (string)chartID;
      return resource;
    }
    
  public:
    WebWorker(const long id, const string p = "WRP_"): chartID(id), busy(false), resource(NULL), prefix("::" + p)
    {
    }

    ~WebWorker()
    {
      release();
    }
    
    long getChartID() const
    {
      return chartID;
    }
    
    bool isBusy() const
    {
      return busy;
    }
    
    string getFullName() const
    {
      return fullname;
    }
    
    virtual void release()
    {
      busy = false;
      if(CheckPointer(resource) == POINTER_DYNAMIC) delete resource;
      resource = NULL;
      fullname = NULL;
    }
    
    static void broadcastEvent(ushort msg, long lparam = 0, double dparam = 0.0, string sparam = NULL)
    {
      long currChart = ChartFirst(); 
      while(currChart != -1)
      {
        if(currChart != ChartID())
        {
          EventChartCustom(currChart, msg, lparam, dparam, sparam); 
        }
        currChart = ChartNext(currChart);
      }
    }
  
};

class ClientWebWorker : public WebWorker
{
  protected:
    bool accepted;
    long retry;

    string _method;
    string _url;
    
  public:
    ClientWebWorker(const long id, const string p = "WRP_"): WebWorker(id, p), accepted(false), retry(0)
    {
    }

    virtual void release() override
    {
      WebWorker::release();
      accepted = false;
      retry = 0;
    }
  
    void accept()
    {
      accepted = true;
      Print("Accepted: ", getFullName(), " after ", retry, " retries");
      retry = 0;
    }
    
    string getMethod() const
    {
      return _method;
    }

    string getURL() const
    {
      return _url;
    }
    
    bool request(const string method, const string url, const string headers, const int timeout, const uchar &body[], const long managerChartID)
    {
      _method = method;
      _url = url;
      retry = 0;

      ResourceMediator mediator(allocate());
      mediator.packRequest(method, url, headers, timeout, body);
    
      busy = EventChartCustom(managerChartID, TO_MSG(MSG_WEB), chartID, 0.0, getFullName());
      return busy;
    }
    
    bool resend(const long managerChartID)
    {
      if(busy && !accepted)
      {
        retry++;
        return EventChartCustom(managerChartID, TO_MSG(MSG_WEB), chartID, 0.0, getFullName());
      }
      return false;
    }
    
    static void receiveResult(const string resname, uchar &initiator[], uchar &headers[], uchar &text[])
    {
      Print(ChartID(), ": Reading result ", resname);
      const RESOURCEDATA<uchar> resource(resname);
      ResourceMediator mediator(&resource);
      mediator.unpackResponse(initiator, headers, text);
    }
    
    virtual void onResult(const long code, const uchar &headers[], const uchar &text[]) = 0;
    virtual void onError(const long code) = 0;
    
};

class ServerWebWorker : public WebWorker
{
  public:
    ServerWebWorker(const long id, const string p = "WRP_"): WebWorker(id, p)
    {
    }
    
    bool transfer(const string resname, const long clientChartID)
    {
      busy = EventChartCustom(clientChartID, TO_MSG(MSG_ACCEPTED), chartID, 0.0, resname)
          && EventChartCustom(chartID, TO_MSG(MSG_WEB), clientChartID, 0.0, resname);
      return busy;
    }
    
    void receive(const string source, const uchar &result[], const string &result_headers)
    {
      ResourceMediator mediator(allocate());
      mediator.packResponse(source, result, result_headers);
    }
    
};

template<typename T>
class WebWorkersPool
{
  protected:
    T *workers[];
    
  public:
    WebWorkersPool() {}
    
    WebWorkersPool(const uint size)
    {
      // allocate workers; in clients they are used to store request parameters in resources
      ArrayResize(workers, size);
      for(int i = 0; i < ArraySize(workers); i++)
      {
        workers[i] = NULL;
      }
    }
    
    ~WebWorkersPool()
    {
      for(int i = 0; i < ArraySize(workers); i++)
      {
        if(CheckPointer(workers[i]) == POINTER_DYNAMIC) delete workers[i];
      }
    }
    
    int size() const
    {
      return ArraySize(workers);
    }
    
    void operator<<(T *worker)
    {
      const int n = ArraySize(workers);
      ArrayResize(workers, n + 1);
      workers[n] = worker;
    }
    
    T *findWorker(const string resname) const
    {
      for(int i = 0; i < ArraySize(workers); i++)
      {
        if(workers[i] != NULL)
        {
          if(workers[i].getFullName() == resname)
          {
            return workers[i];
          }
        }
      }
      return NULL;
    }
    
    T *getIdleWorker() const
    {
      for(int i = 0; i < ArraySize(workers); i++)
      {
        if(workers[i] != NULL)
        {
          if(ChartPeriod(workers[i].getChartID()) > 0) // check if exist
          {
            if(!workers[i].isBusy())
            {
              return workers[i];
            }
          }
        }
      }
      return NULL;
    }
    
    T *findWorker(const long id) const
    {
      for(int i = 0; i < ArraySize(workers); i++)
      {
        if(workers[i] != NULL)
        {
          if(workers[i].getChartID() == id)
          {
            return workers[i];
          }
        }
      }
      return NULL;
    }
    
    bool revoke(const long id)
    {
      for(int i = 0; i < ArraySize(workers); i++)
      {
        if(workers[i] != NULL)
        {
          if(workers[i].getChartID() == id)
          {
            if(CheckPointer(workers[i]) == POINTER_DYNAMIC) delete workers[i];
            workers[i] = NULL;
            return true;
          }
        }
      }
      return false;
    }
    
    int available() const
    {
      int count = 0;
      for(int i = 0; i < ArraySize(workers); i++)
      {
        if(workers[i] != NULL)
        {
          count++;
        }
      }
      return count;
    }
    
    T *operator[](int i) const
    {
      return workers[i];
    }
    
};

template<typename T>
class ClientWebWorkersPool: public WebWorkersPool<T>
{
  protected:
    long   managerChartID;
    short  managerPoolSize;
    string name;
    
  public:
    ClientWebWorkersPool(const uint size, const string prefix): WebWorkersPool(size)
    {
      name = prefix;
      // try to find WebRequest manager chart
      WebWorker::broadcastEvent(TO_MSG(MSG_DISCOVER), ChartID());
    }
    
    bool WebRequestAsync(const string method, const string url, const string headers, int timeout, const char &data[])
    {
      T *worker = getIdleWorker();
      if(worker != NULL)
      {
        return worker.request(method, url, headers, timeout, data, managerChartID);
      }
      return false;
    }
    
    void onChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam)
    {
      if(MSG(id) == MSG_DONE) // async request is completed with result or error
      {
        Print(ChartID(), ": Result code ", (long)dparam);
    
        if(sparam != NULL)
        {
          // read data from the resource with name in sparam
          uchar initiator[], headers[], text[];
          ClientWebWorker::receiveResult(sparam, initiator, headers, text);
          string resname = CharArrayToString(initiator);
          
          T *worker = findWorker(resname);
          if(worker != NULL)
          {
            worker.onResult((long)dparam, headers, text);
            worker.release();
          }
          else
          {
            Print("No worker found for ", resname);
          }
        }
        else
        {
          Print("MSG_DONE sparam is NULL"); // abnormal
        }
      }
      else
      if(MSG(id) == MSG_ERROR) // some error happens other than request processing
      {
        if(sparam != NULL)
        {
          T *worker = findWorker(sparam);
          if(worker != NULL)
          {
            if(lparam == ERROR_NO_IDLE_WORKER)
            {
              // we have an option to resubmit request if previous attempt was declined
              // because the pool has run out of capacity, some workers may become idle
              if(worker.resend(managerChartID))
              {
                // Print("Retry for ", sparam);
                // TODO: But it's probably a good idea to count retries and stop after a number or time limit
              }
            }
            else
            if(lparam == ERROR_MQL_WEB_REQUEST)
            {
              worker.onError((long)dparam);
              worker.release();
            }
            else
            {
              Print("MSG_ERROR Unknown ", lparam);
            }
          }
          else
          {
            Print("No worker found for ", sparam);
          }
        }
        else
        {
          Print("MSG_ERROR sparam is NULL"); // abnormal
        }
      }
      else
      if(MSG(id) == MSG_ACCEPTED) // request is being processed by the manager
      {
        T *worker = findWorker(sparam);
        if(worker != NULL)
        {
          worker.accept();
        }
      }
      else
      if(MSG(id) == MSG_HELLO) // manager is found as a result of MSG_DISCOVER broadcast
      {
        if(managerChartID == 0 && lparam != 0)
        {
          if(ChartPeriod(lparam) > 0)
          {
            managerChartID = lparam;
            managerPoolSize = (short)dparam;
            for(int i = 0; i < ArraySize(workers); i++)
            {
              workers[i] = new T(ChartID(), name + (string)(i + 1) + "_");
            }
          }
        }
      }
    }
    
    bool isManagerBound() const
    {
      return managerChartID != 0;
    }
    
    long getManagerID() const
    {
      return managerChartID;
    }
    
    short getManagerPoolSize() const
    {
      return managerPoolSize;
    }

};
