//+---------------------------------------------------------------------+
//|                                                        multiweb.mqh |
//|                                       Copyright (c) 2018, Marketeer |
//|                             https://www.mql5.com/en/users/marketeer |
//| Asynchronous Non-Blocking WebRequest Implemenation on Helper Charts |
//|                               https://www.mql5.com/ru/articles/5337 |
//+---------------------------------------------------------------------+
#property copyright   "Copyright (c) 2018, Marketeer"
#property link        "https://www.mql5.com/en/users/marketeer"
#property description "WebRequest pool manager and workers\n"


sinput uint WebRequestPoolSize = 3;
sinput ulong ManagerChartID = 0;

#include <multiweb.mqh>
#include <fxsaber\Expert.mqh>

bool manager;
WebWorkersPool<ServerWebWorker> pool;

#define FAILED(T) Print(#T, " failed: ", GetLastError());
#define MAX_WORKERS 10
#define OBJ_LABEL_NAME "WebRequestWorker"

const string GVTEMP = "WRP_GV_TEMP";

template<typename T1,typename T2>
class Converter
{
  private:
    union _L2D
    {
      T1 L;
      T2 D;
    }
    L2D;
  
  public:
    T2 operator[](const T1 L)
    {
      L2D.L = L;
      return L2D.D;
    }

    T1 operator[](const T2 D)
    {
      L2D.D = D;
      return L2D.L;
    }
};

Converter<long,double> converter;

int OnInit()
{
  Print("OnInit ", ChartID());
  manager = false;
  
  if(!GlobalVariableCheck(GVTEMP))
  {
    if(WebRequestPoolSize > MAX_WORKERS)
    {
      Alert("Too many workers, maximum is ", MAX_WORKERS);
      return INIT_PARAMETERS_INCORRECT;
    }

    if(WebRequestPoolSize == 0)
    {
      Alert("Number of workers must be non-zero");
      return INIT_PARAMETERS_INCORRECT;
    }

    // it's hardly can go wrong, but we need to check coherent state
    if(ManagerChartID != 0)
    {
      // this looks like an orphaned chart from previous MT session,
      // because manager is not yet registered the global variable,
      // yet this instance is already started with a manager ID
      // NB: do not launch worker instances manually, the manager will do it automatically
      Print("Orphaned chart ", ChartID(), " is being closed; wanted missing manager ID: ", ManagerChartID);
      Print(ChartClose());
      return INIT_FAILED;
    }
    
    // when first instance of multiweb is started, it's treated as manager
    // the global variable is a flag that the manager is present
    if(!GlobalVariableTemp(GVTEMP))
    {
      FAILED(GlobalVariableTemp);
      return INIT_FAILED;
    }
    
    manager = true;
    GlobalVariableSet(GVTEMP, converter[ChartID()]);
    Print("WebRequest Pool Manager started in ", ChartID());
    Comment("WebRequest Pool Manager ", ChartID());
  }
  else
  {
    // it's hardly can go wrong, but we need to compare IDs
    const long id = converter[GlobalVariableGet(GVTEMP)];
    if(id != ManagerChartID)
    {
      // this looks like an orphaned chart from previous MT session,
      // because manager ID in the global variable differs to the passed ManagerChartID
      // NB: do not launch worker instances manually, the manager will do it automatically
      Print("Orphaned chart ", ChartID(), " is being closed; wanted manager ID: ", ManagerChartID, ", registered manager ID: ", id);
      Print(ChartClose());
      return INIT_FAILED;
    }
    // all next instances of multiweb are workers/helpers
    Print("WebRequest Worker started in ", ChartID(), "; manager in ", ManagerChartID);
    Comment("WebRequest Worker ", ChartID(), "; Manager ", ManagerChartID);
    
    ObjectCreate(0, OBJ_LABEL_NAME, OBJ_LABEL, 0, 0, 0);
    ObjectSetInteger(0, OBJ_LABEL_NAME, OBJPROP_COLOR, clrRed);
    ObjectSetInteger(0, OBJ_LABEL_NAME, OBJPROP_CORNER, CORNER_RIGHT_LOWER);
    ObjectSetInteger(0, OBJ_LABEL_NAME, OBJPROP_ANCHOR, ANCHOR_RIGHT_LOWER);
    ObjectSetInteger(0, OBJ_LABEL_NAME, OBJPROP_FONTSIZE, 12);
    ObjectSetInteger(0, OBJ_LABEL_NAME, OBJPROP_XDISTANCE, 12);
    ObjectSetInteger(0, OBJ_LABEL_NAME, OBJPROP_YDISTANCE, 12);
    ObjectSetInteger(0, OBJ_LABEL_NAME, OBJPROP_SELECTABLE, false);
    ObjectSetString(0, OBJ_LABEL_NAME, OBJPROP_TEXT, "This is a helper chart of WebRequest pool. Don't work on it!");
  }
  
  // use the timer for delayed instantiation of workers
  EventSetTimer(1);
  return INIT_SUCCEEDED;
}

void OnDeinit(const int reason)
{
  if(manager)
  {
    Print("WebRequest Pool Manager closed, ", ChartID());
    for(int i = 0; i < pool.size(); i++)
    {
      if(CheckPointer(pool[i]) == POINTER_DYNAMIC)
      {
        // let know workers they are not needed anymore
        EventChartCustom(pool[i].getChartID(), TO_MSG(MSG_DEINIT), ChartID(), 0.0, NULL);
      }
    }
    GlobalVariableDel(GVTEMP);
  }
  else
  {
    Print("WebRequest Worker closed, ", ChartID());
    // let know the manager that this worker is not available anymore
    EventChartCustom(ManagerChartID, TO_MSG(MSG_DEINIT), ChartID(), 0.0, NULL);
    ObjectDelete(0, OBJ_LABEL_NAME);
    ChartClose(0);
  }
  Comment("");
}

void OnTimer()
{
  EventKillTimer();
  if(manager)
  {
    if(!instantiateWorkers())
    {
      Alert("Workers not initialized");
    }
    else
    {
      Comment("WebRequest Pool Manager ", ChartID(), "\nWorkers available: ", pool.available());
    }
  }
  else // worker
  {
    // this is used as a host of resource storing response headers and data
    pool << new ServerWebWorker(ChartID(), "WRR_");
  }
}

bool instantiateWorkers()
{
  MqlParam Params[4];
  
  const string path = MQLInfoString(MQL_PROGRAM_PATH);
  const string experts = "\\MQL5\\";
  const int pos = StringFind(path, experts);
  
  // start itself again (in another role as helper EA)
  Params[0].string_value = StringSubstr(path, pos + StringLen(experts));
  
  Params[1].type = TYPE_UINT;
  Params[1].integer_value = 1; // 1 worker inside new helper EA instance for returning results to the manager or client

  Params[2].type = TYPE_LONG;
  Params[2].integer_value = ChartID(); // this chart is the manager

  Params[3].type = TYPE_UINT;
  Params[3].integer_value = MessageBroadcast; // use the same custom event base number
  
  for(uint i = 0; i < WebRequestPoolSize; ++i)
  {
    long chart = ChartOpen(_Symbol, _Period);
    if(chart == 0)
    {
      FAILED(ChartOpen);
      return false;
    }
    if(!EXPERT::Run(chart, Params))
    {
      FAILED(EXPERT::Run);
      return false;
    }
    pool << new ServerWebWorker(chart);
  }
  return true;
}


void OnChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam) 
{
  if(MSG(id) == MSG_DEINIT) // other instance of this EA is removed or its chart is closed
  {
    if(manager) // the removed copy of EA was a worker, lets remove it from the array
    {
      if(pool.revoke(lparam))
      {
        Print("Worker killed: ", lparam, ", available: ", pool.available());
        Comment("WebRequest Pool Manager ", ChartID(), "\nWorkers available: ", pool.available());
      }
    }
    else // the removed copy of EA was the manager, so the worker is not required anymore
    {
      Print("Worker self-removal, ", ChartID());
      ExpertRemove();
    }
  }
  else
  if(MSG(id) == MSG_DISCOVER) // a worker EA on new client chart is initialized and wants to bind to this manager
  {
    if(manager && (lparam != 0))
    {
      // only manager responds with its chart ID, lparam is the client chart ID
      EventChartCustom(lparam, TO_MSG(MSG_HELLO), ChartID(), pool.available(), NULL);
    }
  }
  else
  if(MSG(id) == MSG_WEB) // a client has requested a web download
  {
    if(lparam != 0)
    {
      if(manager)
      {
        // the manager delegates the work to an idle worker
        // lparam is the client chart ID, sparam is the client resource
        if(!transfer(lparam, sparam))
        {
          EventChartCustom(lparam, TO_MSG(MSG_ERROR), ERROR_NO_IDLE_WORKER, 0.0, sparam);
        }
      }
      else
      {
        // the worker does actually process the web request
        startWebRequest(lparam, sparam);
      }
    }
    else
    {
      Print("MSG_WEB: required client chart ID is missing in lparam");
    }
  }
  else
  if(MSG(id) == MSG_DONE) // a worker identified by chart ID in lparam has finished its job
  {
    WebWorker *worker = pool.findWorker(lparam);
    if(worker != NULL)
    {
      // here we're in the manager, and the pool hold stub workers without resources
      // so this release is intended solely to clean up busy state
      worker.release();

      // result code in dparam
      Print("Result code from ", lparam, ": ", (long)dparam, ", now idle");
    }
  }
}


void startWebRequest(const long returnChartID, const string resname)
{
  Print(ChartID(), ": Reading request ", resname);
  
  const RESOURCEDATA<uchar> resource(resname);
  ResourceMediator mediator(&resource);

  string method, url, headers;
  int timeout;
  uchar body[];

  mediator.unpackRequest(method, url, headers, timeout, body);

  Print(ChartID(), ": ", method, " ", url, " ", headers, " ", timeout, " ", CharArrayToString(body));
  Comment(method, " ", url, "\n", headers);

  uint startTime = GetTickCount();

  char result[];
  string result_headers;
  
  int code = WebRequest(method, url, headers, timeout, body, result, result_headers);
  if(code != -1)
  {
    // create resource with results to pass back to the client via custom event
    ((ServerWebWorker *)pool[0]).receive(resname, result, result_headers);
    // first, send MSG_DONE to the client with resulting resource
    EventChartCustom(returnChartID, TO_MSG(MSG_DONE), ChartID(), (double)code, pool[0].getFullName());
    // second, send MSG_DONE to the manager to set corresponding worker to idle state
    EventChartCustom(ManagerChartID, TO_MSG(MSG_DONE), ChartID(), (double)code, NULL);
    Print(ChartID(), ": ", "Done in ", (GetTickCount() - startTime), "ms");
  }
  else
  {
    // error code in dparam
    EventChartCustom(returnChartID, TO_MSG(MSG_ERROR), ERROR_MQL_WEB_REQUEST, (double)GetLastError(), resname);
    EventChartCustom(ManagerChartID, TO_MSG(MSG_DONE), ChartID(), (double)GetLastError(), NULL);
    Print(ChartID(), ": ", "Failed with code ", GetLastError());
  }
}

bool transfer(const long returnChartID, const string resname)
{
  ServerWebWorker *worker = pool.getIdleWorker();
  if(worker == NULL)
  {
    // uncommenting this leads to potentially bulky logs
    // Print("No idle workers");
    return false;
  }
  Print("Pool manager transfers ", resname);
  return worker.transfer(resname, returnChartID);
}
