//+------------------------------------------------------------------+
//|                       PatternAssociationRulesExpertAdvisor.mq5   |
//|                                Copyright 2023, Evgeniy Koshtenko |
//|                          https://www.mql5.com/ru/users/koshtenko |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, Evgeniy Koshtenko"
#property link      "https://www.mql5.com/ru/users/koshtenko"
#property version   "3.00"

#include <Trade\Trade.mqh>

//--- input parameters
input int      InpPatternLength   = 5;    // Pattern Length (3-10)
input int      InpLookback        = 1000; // Lookback Period (100-5000)
input int      InpForecastHorizon = 6;    // Forecast Horizon (1-20)
input double   InpLotSize         = 0.1;  // Lot Size (minimum)
input double   InpTopPercent      = 10.0; // Top % strongest patterns (1-50)

//--- global variables
struct AssociationRule
  {
  string antecedent;  // pattern
  string consequent;  // "UP" or "DOWN"
  double bayes_prob;  // Bayesian probability
  double confidence;  // classic confidence
  double lift;        // lift
  double strength;    // |bayes_prob - 0.5| * 2  — deviation from random [0..1]
  };

int             g_pattern_length;
int             g_lookback;
int             g_forecast_horizon;
string          g_patterns[];
int             g_pattern_count;
AssociationRule g_rules[];
int             g_rule_count;
double          g_strength_threshold; // strength threshold for top X%
CTrade          trade;
double          g_point;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
   if(InpPatternLength < 3 || InpPatternLength > 10)
     {
      Print("Invalid Pattern Length. Must be between 3 and 10.");
      return INIT_PARAMETERS_INCORRECT;
     }
   if(InpLookback < 100 || InpLookback > 5000)
     {
      Print("Invalid Lookback Period. Must be between 100 and 5000.");
      return INIT_PARAMETERS_INCORRECT;
     }
   if(InpForecastHorizon < 1 || InpForecastHorizon > 20)
     {
      Print("Invalid Forecast Horizon. Must be between 1 and 20.");
      return INIT_PARAMETERS_INCORRECT;
     }
   if(InpTopPercent < 1.0 || InpTopPercent > 50.0)
     {
      Print("Invalid TopPercent. Must be between 1 and 50.");
      return INIT_PARAMETERS_INCORRECT;
     }

   g_pattern_length     = InpPatternLength;
   g_lookback           = InpLookback;
   g_forecast_horizon   = InpForecastHorizon;
   g_point              = SymbolInfoDouble(_Symbol, SYMBOL_POINT);
   g_strength_threshold = 0.0;

   if(!GeneratePatterns())
     {
      Print("Failed to generate patterns.");
      return INIT_FAILED;
     }

   return INIT_SUCCEEDED;
  }

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
  }

//+------------------------------------------------------------------+
//| Collects ALL patterns from the dictionary without any thresholds|
//| Rule strength = |bayes_prob - 0.5| * 2                          |
//| After collection, calculates threshold for top InpTopPercent%   |
//+------------------------------------------------------------------+
void UpdateAssociationRules()
  {
   int total_bars = iBars(_Symbol, PERIOD_CURRENT);
   if(total_bars <= g_lookback)
      return;

   int start = total_bars - g_lookback;
   if(start < g_pattern_length + g_forecast_horizon)
      start = g_pattern_length + g_forecast_horizon;

//--- get close prices
   double close[];
   ArraySetAsSeries(close, true);
   CopyClose(_Symbol, PERIOD_CURRENT, 0, total_bars, close);

//--- build arrays of historical patterns and outcomes
   string hist_patterns[];
   string hist_outcomes[];
   ArrayResize(hist_patterns, total_bars);
   ArrayResize(hist_outcomes, total_bars);

   int valid_count = 0;
   for(int i = 0; i < total_bars - g_pattern_length; i++)
     {
      hist_patterns[i] = "";
      for(int j = 0; j < g_pattern_length; j++)
        {
         hist_patterns[i] += (close[i + j] - close[i + j + 1] > 0) ? "U" : "D";
        }
      hist_outcomes[i] = (i >= g_forecast_horizon && close[i - g_forecast_horizon] > close[i]) ?
                         "UP" : "DOWN";
      if(i >= start)
         valid_count++;
     }

//--- base probabilities
   int n_up = 0, n_down = 0;
   for(int i = start; i >= g_pattern_length + g_forecast_horizon; i--)
     {
      if(hist_outcomes[i] == "UP")
         n_up++;
      else
         n_down++;
     }
   double p_up   = (valid_count > 0) ? double(n_up) / valid_count : 0.5;
   double p_down = (valid_count > 0) ? double(n_down) / valid_count : 0.5;

//--- count ALL patterns from dictionary — without thresholds
   g_rule_count = 0;
   ArrayResize(g_rules, g_pattern_count * 2);

   double alpha = 1.0; // Laplace smoothing

   for(int i = 0; i < g_pattern_count; i++)
     {
      int cnt = 0, cnt_up = 0, cnt_down = 0;

      for(int j = start; j >= g_pattern_length + g_forecast_horizon; j--)
        {
         if(hist_patterns[j] == g_patterns[i])
           {
            cnt++;
            if(hist_outcomes[j] == "UP")
               cnt_up++;
            else
               cnt_down++;
           }
        }

      // Bayesian probability with Laplace prior
      // (works even when cnt=0: returns 0.5)
      double denom      = cnt + 2.0 * alpha;
      double bayes_up   = (cnt_up + alpha) / denom;
      double bayes_down = (cnt_down + alpha) / denom;

      double conf_up    = (cnt > 0) ? double(cnt_up) / cnt : 0.5;
      double conf_down  = (cnt > 0) ? double(cnt_down) / cnt : 0.5;

      double lift_up    = (p_up > 0) ? conf_up / p_up : 1.0;
      double lift_down  = (p_down > 0) ? conf_down / p_down : 1.0;

      // strength = deviation of bayes_prob from 0.5, normalized to [0..1]
      double str_up   = MathAbs(bayes_up - 0.5) * 2.0;
      double str_down = MathAbs(bayes_down - 0.5) * 2.0;

      // UP rule
      g_rules[g_rule_count].antecedent = g_patterns[i];
      g_rules[g_rule_count].consequent = "UP";
      g_rules[g_rule_count].bayes_prob = bayes_up;
      g_rules[g_rule_count].confidence = conf_up;
      g_rules[g_rule_count].lift       = lift_up;
      g_rules[g_rule_count].strength   = str_up;
      g_rule_count++;

      // DOWN rule
      g_rules[g_rule_count].antecedent = g_patterns[i];
      g_rules[g_rule_count].consequent = "DOWN";
      g_rules[g_rule_count].bayes_prob = bayes_down;
      g_rules[g_rule_count].confidence = conf_down;
      g_rules[g_rule_count].lift       = lift_down;
      g_rules[g_rule_count].strength   = str_down;
      g_rule_count++;
     }

   ArrayResize(g_rules, g_rule_count);

//--- compute strength threshold for top InpTopPercent%
   double strengths[];
   ArrayResize(strengths, g_rule_count);
   for(int i = 0; i < g_rule_count; i++)
      strengths[i] = g_rules[i].strength;

   ArraySort(strengths); // ascending sort

   int cutoff_idx = int(MathFloor(g_rule_count * (1.0 - InpTopPercent / 100.0)));
   cutoff_idx = MathMax(0, MathMin(cutoff_idx, g_rule_count - 1));
   g_strength_threshold = strengths[cutoff_idx];

   Print(StringFormat("Rules=%d | Top=%.1f%% | strength_threshold=%.4f",
         g_rule_count, InpTopPercent, g_strength_threshold));
  }

//+------------------------------------------------------------------+
//| Voting among rules for current pattern from top X%              |
//| Returns: 1=UP  -1=DOWN  0=no signal / tie                       |
//+------------------------------------------------------------------+
int GetMajoritySignal(const string &pattern)
  {
   int votes_up = 0, votes_down = 0;

   for(int i = 0; i < g_rule_count; i++)
     {
      if(g_rules[i].antecedent != pattern)
         continue;
      if(g_rules[i].strength < g_strength_threshold)
         continue; // not in top X%

      if(g_rules[i].consequent == "UP")
         votes_up++;
      else
         votes_down++;
     }

   if(votes_up == 0 && votes_down == 0)
      return 0; // pattern not in top
   if(votes_up == votes_down)
      return 0; // tie — skip

   return (votes_up > votes_down) ? 1 : -1;
  }

//+------------------------------------------------------------------+
//| OnTick: enter by top%, exit by opposite signal                  |
//+------------------------------------------------------------------+
void OnTick()
  {
   if(!IsNewBar())
      return;

   UpdateAssociationRules();
   if(g_strength_threshold <= 0)
      return;

   string current_pattern = GetCurrentPattern();
   int    signal          = GetMajoritySignal(current_pattern);

   double ask = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   double bid = SymbolInfoDouble(_Symbol, SYMBOL_BID);

   bool has_pos  = PositionSelect(_Symbol);
   long pos_type = has_pos ? PositionGetInteger(POSITION_TYPE) : -1;

//--- close by opposite signal
   if(has_pos)
     {
      if(pos_type == POSITION_TYPE_BUY && signal == -1)
         trade.PositionClose(_Symbol);
      if(pos_type == POSITION_TYPE_SELL && signal == 1)
         trade.PositionClose(_Symbol);
     }

   has_pos = PositionSelect(_Symbol); // re-read after possible close

//--- open by signal
   if(!has_pos && signal != 0)
     {
      // lot = average strength of top rules for current pattern × base lot × 10
      double total_str = 0.0;
      int    cnt       = 0;
      for(int i = 0; i < g_rule_count; i++)
        {
         if(g_rules[i].antecedent == current_pattern &&
            g_rules[i].strength   >= g_strength_threshold)
           {
            total_str += g_rules[i].strength;
            cnt++;
           }
        }
      double avg_str = (cnt > 0) ? total_str / cnt : 0.1;
      double lot     = MathMax(InpLotSize * avg_str * 10.0, InpLotSize);

      if(signal == 1)
         trade.Buy(lot, _Symbol, ask, 0, 0,
                   StringFormat("BUY top%.0f%% | pat=%s str=%.3f lot=%.2f",
                                InpTopPercent, current_pattern, avg_str, lot));
      else
         trade.Sell(lot, _Symbol, bid, 0, 0,
                    StringFormat("SELL top%.0f%% | pat=%s str=%.3f lot=%.2f",
                                 InpTopPercent, current_pattern, avg_str, lot));
     }
  }

//+------------------------------------------------------------------+
//| Generates all possible patterns                                  |
//+------------------------------------------------------------------+
bool GeneratePatterns()
  {
   g_pattern_count = int(MathPow(2, g_pattern_length));
   if(!ArrayResize(g_patterns, g_pattern_count))
      return false;

   for(int i = 0; i < g_pattern_count; i++)
     {
      string pattern = "";
      for(int j = 0; j < g_pattern_length; j++)
         pattern += ((i >> j) & 1) ? "U" : "D";
      g_patterns[i] = pattern;
     }
   return true;
  }

//+------------------------------------------------------------------+
//| Returns current price pattern                                    |
//+------------------------------------------------------------------+
string GetCurrentPattern()
  {
   double close[];
   ArraySetAsSeries(close, true);
   CopyClose(_Symbol, PERIOD_CURRENT, 0, g_pattern_length + 1, close);

   string pattern = "";
   for(int i = 0; i < g_pattern_length; i++)
      pattern += (close[i] > close[i + 1]) ? "U" : "D";

   return pattern;
  }

//+------------------------------------------------------------------+
//| Checks if a new bar has appeared                                 |
//+------------------------------------------------------------------+
bool IsNewBar()
  {
   static datetime last_time = 0;
   datetime current_time = iTime(_Symbol, PERIOD_CURRENT, 0);
   if(current_time != last_time)
     {
      last_time = current_time;
      return true;
     }
   return false;
  }
//+------------------------------------------------------------------+