﻿//+------------------------------------------------------------------+
//|                       PatternAssociationRulesExpertAdvisor.mq5   |
//|                                Copyright 2023, Evgeniy Koshtenko |
//|                          https://www.mql5.com/ru/users/koshtenko |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, Evgeniy Koshtenko"
#property link      "https://www.mql5.com/ru/users/koshtenko"
#property version   "2.01"

#include <Trade\Trade.mqh>

//--- input parameters
input int      InpPatternLength    = 5;     // Pattern Length (3-10)
input int      InpLookback         = 1000;  // Lookback Period (100-5000)
input int      InpForecastHorizon  = 6;     // Forecast Horizon (1-20)
input double   InpLotSize          = 0.1;   // Lot Size
input int      InpMinOccurrences   = 30;    // Minimum Pattern Occurrences
input double   InpMinSupport       = 0.05;  // Minimum Support (0.01-0.5)
input double   InpMinConfidence    = 0.6;   // Minimum Confidence (0.5-0.95)
input int      InpATRPeriod        = 14;    // ATR Period (not used for SL/TP)
input double   InpATRMultiplier    = 0.5;   // ATR Multiplier (not used for SL/TP)
input int      InpStopLoss         = 2000;  // Stop Loss (pips)
input int      InpTakeProfit       = 200;   // Take Profit (pips)

//--- global variables
struct AssociationRule
  {
  string antecedent;          // pattern
  string consequent;          // "UP" or "DOWN"
  double support;             // support
  double confidence;          // confidence (frequency)
  double lift;                // lift
  double chi_square;          // χ² statistic
  double expected_confidence; // expected probability
  double bayes_prob;          // Bayesian probability
  double p_value;             // significance level (approx.)
  double mutual_info;         // mutual information (MI)
  };

int            g_pattern_length;
int            g_lookback;
int            g_forecast_horizon;
string         g_patterns[];
int            g_pattern_count;
AssociationRule g_rules[];
int            g_rule_count;
CTrade         trade;
double         g_point;
int            g_atr_handle; // ATR indicator handle

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
//--- validate inputs
   if(InpPatternLength < 3 || InpPatternLength > 10)
     {
      Print("Invalid Pattern Length. Must be between 3 and 10.");
      return INIT_PARAMETERS_INCORRECT;
     }

   if(InpLookback < 100 || InpLookback > 5000)
     {
      Print("Invalid Lookback Period. Must be between 100 and 5000.");
      return INIT_PARAMETERS_INCORRECT;
     }

   if(InpForecastHorizon < 1 || InpForecastHorizon > 20)
     {
      Print("Invalid Forecast Horizon. Must be between 1 and 20.");
      return INIT_PARAMETERS_INCORRECT;
     }

   if(InpMinSupport < 0.01 || InpMinSupport > 0.5)
     {
      Print("Invalid Minimum Support. Must be between 0.01 and 0.5.");
      return INIT_PARAMETERS_INCORRECT;
     }

   if(InpMinConfidence < 0.5 || InpMinConfidence > 0.95)
     {
      Print("Invalid Minimum Confidence. Must be between 0.5 and 0.95.");
      return INIT_PARAMETERS_INCORRECT;
     }

//--- set global variables
   g_pattern_length = InpPatternLength;
   g_lookback = InpLookback;
   g_forecast_horizon = InpForecastHorizon;
   g_point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);

//--- initialize ATR indicator (kept for compatibility)
   g_atr_handle = iATR(_Symbol, PERIOD_D1, InpATRPeriod);
   if(g_atr_handle == INVALID_HANDLE)
     {
      Print("Failed to initialize ATR indicator");
      return INIT_FAILED;
     }

//--- generate all possible patterns
   if(!GeneratePatterns())
     {
      Print("Failed to generate patterns.");
      return INIT_FAILED;
     }

   return INIT_SUCCEEDED;
  }

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   if(g_atr_handle != INVALID_HANDLE)
      IndicatorRelease(g_atr_handle);
  }

//+------------------------------------------------------------------+
//| OnTick with probabilistic trading rules                          |
//+------------------------------------------------------------------+
void OnTick()
  {
   if(!IsNewBar())
      return;

   UpdateAssociationRules();
   string current_pattern = GetCurrentPattern();

//--- fixed SL and TP in pips
//--- for 5‑digit quotes (EURUSD, GBPUSD etc.) 1 pip = 10 * Point
//--- for 4‑digit (JPY pairs) 1 pip = Point
//--- determine pip size automatically based on digits
   double pip_size = g_point;
   if(int(SymbolInfoInteger(_Symbol, SYMBOL_DIGITS)) == 5 ||
      int(SymbolInfoInteger(_Symbol, SYMBOL_DIGITS)) == 3)
      pip_size = g_point * 10;

   double sl = InpStopLoss * pip_size;
   double tp = InpTakeProfit * pip_size;
   double price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);

   for(int i = 0; i < g_rule_count; i++)
     {
      if(current_pattern == g_rules[i].antecedent &&
         g_rules[i].bayes_prob >= InpMinConfidence)
        {
         //--- dynamic lot based on signal strength
         double signal_strength = (g_rules[i].confidence - g_rules[i].expected_confidence) * 10;
         double lot = MathMax(InpLotSize * signal_strength, InpLotSize);

         if(g_rules[i].consequent == "UP")
           {
            if(!PositionSelect(_Symbol))
              {
               trade.Buy(lot, _Symbol, price, price - sl, price + tp,
                         StringFormat("BUY: %s->UP | Conf=%.2f Bayes=%.2f Lift=%.2f",
                                      g_rules[i].antecedent,
                                      g_rules[i].confidence * 100,
                                      g_rules[i].bayes_prob * 100,
                                      g_rules[i].lift));
              }
           }
         else if(g_rules[i].consequent == "DOWN")
           {
            if(!PositionSelect(_Symbol))
              {
               trade.Sell(lot, _Symbol, price, price + sl, price - tp,
                          StringFormat("SELL: %s->DOWN | Conf=%.2f Bayes=%.2f Lift=%.2f",
                                       g_rules[i].antecedent,
                                       g_rules[i].confidence * 100,
                                       g_rules[i].bayes_prob * 100,
                                       g_rules[i].lift));
              }
           }
        }
     }
  }

//+------------------------------------------------------------------+
//| Generates all possible patterns                                  |
//+------------------------------------------------------------------+
bool GeneratePatterns()
  {
   g_pattern_count = int(MathPow(2, g_pattern_length));
   if(!ArrayResize(g_patterns, g_pattern_count))
     {
      Print("Failed to resize g_patterns array.");
      return false;
     }

   for(int i = 0; i < g_pattern_count; i++)
     {
      string pattern = "";
      for(int j = 0; j < g_pattern_length; j++)
        {
         pattern += ((i >> j) & 1) ? "U" : "D";
        }
      g_patterns[i] = pattern;
     }

   return true;
  }

//+------------------------------------------------------------------+
//| Updates association rules (improved probabilistic model)         |
//+------------------------------------------------------------------+
void UpdateAssociationRules()
  {
   int total_bars = iBars(_Symbol, PERIOD_CURRENT);
   if(total_bars <= g_lookback)
      return;

   int start = total_bars - g_lookback;
   if(start < g_pattern_length + g_forecast_horizon)
      start = g_pattern_length + g_forecast_horizon;

//--- get close prices
   double close[];
   ArraySetAsSeries(close, true);
   CopyClose(_Symbol, PERIOD_CURRENT, 0, total_bars, close);

   string patterns[];
   string outcomes[];
   ArrayResize(patterns, total_bars);
   ArrayResize(outcomes, total_bars);

   int valid_patterns = 0;

//--- generate patterns and outcomes
   for(int i = 0; i < total_bars - g_pattern_length; i++)
     {
      patterns[i] = "";
      for(int j = 0; j < g_pattern_length; j++)
        {
         double diff = close[i + j] - close[i + j + 1];
         patterns[i] += (diff > 0) ? "U" : "D";
        }
      outcomes[i] = (i >= g_forecast_horizon && close[i - g_forecast_horizon] > close[i]) ? "UP" : "DOWN";
      if(i >= start)
         valid_patterns++;
     }

//--- count total UP/DOWN outcomes
   int outcome_up_count = 0, outcome_down_count = 0;
   for(int i = start; i >= g_pattern_length + g_forecast_horizon; i--)
     {
      if(outcomes[i] == "UP")
         outcome_up_count++;
      else
         outcome_down_count++;
     }
   double p_up = double(outcome_up_count) / valid_patterns;
   double p_down = double(outcome_down_count) / valid_patterns;

//--- calculate rules
   g_rule_count = 0;
   ArrayResize(g_rules, g_pattern_count * 2);

   for(int i = 0; i < g_pattern_count; i++)
     {
      int count_pattern = 0, count_up = 0, count_down = 0;

      for(int j = start; j >= g_pattern_length + g_forecast_horizon; j--)
        {
         if(patterns[j] == g_patterns[i])
           {
            count_pattern++;
            if(outcomes[j] == "UP")
               count_up++;
            else
               count_down++;
           }
        }

      if(count_pattern >= InpMinOccurrences)
        {
         double support = double(count_pattern) / valid_patterns;

         if(support >= InpMinSupport)
           {
            //--- Bayesian probabilities with soft prior
            double alpha = 1.0;
            double bayes_up = (count_up + alpha) / (count_pattern + 2 * alpha);
            double bayes_down = (count_down + alpha) / (count_pattern + 2 * alpha);

            //--- chi-square and approximate p-value
            double exp_up = count_pattern * p_up;
            double exp_down = count_pattern * p_down;
            double chi2 = 0;
            if(exp_up > 0)
               chi2 += MathPow(count_up - exp_up, 2) / exp_up;
            if(exp_down > 0)
               chi2 += MathPow(count_down - exp_down, 2) / exp_down;
            double p_value = MathExp(-0.5 * chi2); // approximate p-value

            //--- mutual information (approximate)
            double mi = 0;
            if(count_up > 0)
               mi += (double(count_up) / valid_patterns) *
                     MathLog(double(count_up) / (count_pattern * p_up) + 1e-8);
            if(count_down > 0)
               mi += (double(count_down) / valid_patterns) *
                     MathLog(double(count_down) / (count_pattern * p_down) + 1e-8);

            //--- UP rule
            double conf_up = double(count_up) / count_pattern;
            if(conf_up >= InpMinConfidence)
              {
               g_rules[g_rule_count].antecedent = g_patterns[i];
               g_rules[g_rule_count].consequent = "UP";
               g_rules[g_rule_count].support = support;
               g_rules[g_rule_count].confidence = conf_up;
               g_rules[g_rule_count].lift = conf_up / p_up;
               g_rules[g_rule_count].chi_square = chi2;
               g_rules[g_rule_count].expected_confidence = p_up;
               g_rules[g_rule_count].p_value = p_value;
               g_rules[g_rule_count].mutual_info = mi;
               g_rules[g_rule_count].bayes_prob = bayes_up;
               g_rule_count++;
              }

            //--- DOWN rule
            double conf_down = double(count_down) / count_pattern;
            if(conf_down >= InpMinConfidence)
              {
               g_rules[g_rule_count].antecedent = g_patterns[i];
               g_rules[g_rule_count].consequent = "DOWN";
               g_rules[g_rule_count].support = support;
               g_rules[g_rule_count].confidence = conf_down;
               g_rules[g_rule_count].lift = conf_down / p_down;
               g_rules[g_rule_count].chi_square = chi2;
               g_rules[g_rule_count].expected_confidence = p_down;
               g_rules[g_rule_count].p_value = p_value;
               g_rules[g_rule_count].mutual_info = mi;
               g_rules[g_rule_count].bayes_prob = bayes_down;
               g_rule_count++;
              }
           }
        }
     }
   ArrayResize(g_rules, g_rule_count);

//--- log top rules
   for(int i = 0; i < g_rule_count; i++)
     {
      if(g_rules[i].lift > 1.2 && g_rules[i].p_value < 0.05)
        {
         Print(StringFormat("Rule %d: %s -> %s | Sup=%.2f%% Conf=%.2f%% Bayes=%.2f%% Lift=%.2f p=%.4f MI=%.4f",
                            i + 1, g_rules[i].antecedent, g_rules[i].consequent,
                            g_rules[i].support * 100, g_rules[i].confidence * 100,
                            g_rules[i].bayes_prob * 100, g_rules[i].lift,
                            g_rules[i].p_value, g_rules[i].mutual_info));
        }
     }
  }

//+------------------------------------------------------------------+
//| Returns current price pattern                                    |
//+------------------------------------------------------------------+
string GetCurrentPattern()
  {
   double close[];
   ArraySetAsSeries(close, true);
   CopyClose(_Symbol, PERIOD_CURRENT, 0, g_pattern_length + 1, close);

   string pattern = "";
   for(int i = 0; i < g_pattern_length; i++)
     {
      pattern += (close[i] > close[i + 1]) ? "U" : "D";
     }
   return pattern;
  }

//+------------------------------------------------------------------+
//| Searches for a string in an array                                |
//+------------------------------------------------------------------+
int ArraySearch(const string &arr[], string value)
  {
   for(int i = 0; i < ArraySize(arr); i++)
     {
      if(arr[i] == value)
         return i;
     }
   return -1;
  }

//+------------------------------------------------------------------+
//| Checks if a new bar has appeared                                 |
//+------------------------------------------------------------------+
bool IsNewBar()
  {
   static datetime last_time = 0;
   datetime current_time = iTime(_Symbol, PERIOD_CURRENT, 0);
   if(current_time != last_time)
     {
      last_time = current_time;
      return true;
     }
   return false;
  }
//+------------------------------------------------------------------+