# ================================================
# ai_trader_ultra_with_finetune.py
# Версия с SEAL RL, балансировкой классов и форвардным тестом
# ================================================
import os
import re
import time
import json
import logging
import subprocess
from datetime import datetime, timedelta
from collections import deque
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt

try:
    import MetaTrader5 as mt5
except ImportError:
    mt5 = None

try:
    import ollama
except ImportError:
    ollama = None

# ====================== КОНФИГ ======================
MODEL_NAME = "koshtenco/shtencoaitrader-3b-analyst-v3"
BASE_MODEL = "llama3.2:3b"
SYMBOLS = ["EURUSD", "GBPUSD", "USDCHF", "USDCAD", "AUDUSD", "NZDUSD", "EURGBP", "AUDCHF"]
TIMEFRAME = mt5.TIMEFRAME_M15 if mt5 else None
LOOKBACK = 400
INITIAL_BALANCE = 140.0
RISK_PER_TRADE = 0.08
MIN_PROB = 60
LIVE_LOT = 0.02
MAGIC = 20251121
SLIPPAGE = 10

# Файнтьюн параметры
FINETUNE_SAMPLES = 2000
FINETUNE_EPOCHS = 3
BACKTEST_DAYS = 30 
PREDICTION_HORIZON = 96  # 24 часа (96 баров по 15 минут)

# SEAL RL параметры (MIT methodology)
SEAL_LEARNING_RATE = 0.001
SEAL_GAMMA = 0.95  # discount factor
SEAL_EPSILON = 0.1  # exploration rate
SEAL_MEMORY_SIZE = 10000
SEAL_BATCH_SIZE = 64
SEAL_UPDATE_FREQUENCY = 100

# Форвардный тест
FORWARD_TEST_DAYS = 7  # дней для форвардного теста

# Целевой баланс классов (1.0 = идеальный 50/50)
BALANCE_RATIO = 1.0

os.makedirs("logs", exist_ok=True)
os.makedirs("dataset", exist_ok=True)
os.makedirs("models", exist_ok=True)
os.makedirs("charts", exist_ok=True)
os.makedirs("seal_checkpoints", exist_ok=True)

logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s | %(message)s",
    handlers=[
        logging.FileHandler("logs/ai_trader_ultra.log", encoding="utf-8"),
        logging.StreamHandler(),
    ],
)
log = logging.getLogger(__name__)


# ====================== SEAL RL AGENT (MIT METHODOLOGY) ======================
class SEALAgent:
    """
    Self-Evolving Adversarial Learning Agent (MIT)
    
    Основано на исследованиях MIT CSAIL:
    - Adversarial self-play для улучшения робастности
    - Curriculum learning с адаптивной сложностью
    - Meta-learning для быстрой адаптации к новым рынкам
    - Evolutionary strategies для оптимизации политики
    """
    
    def __init__(self, state_dim=10, action_dim=3, learning_rate=SEAL_LEARNING_RATE):
        self.state_dim = state_dim
        self.action_dim = action_dim  # 0=HOLD, 1=BUY, 2=SELL
        self.lr = learning_rate
        self.gamma = SEAL_GAMMA
        self.epsilon = SEAL_EPSILON
        
        # Adversarial components (MIT SEAL)
        self.protagonist_policy = {}  # Основная торговая политика
        self.adversary_policy = {}    # Adversarial политика для генерации сложных сценариев
        
        # Experience replay с приоритизацией
        self.memory = deque(maxlen=SEAL_MEMORY_SIZE)
        self.priorities = deque(maxlen=SEAL_MEMORY_SIZE)
        
        # Curriculum learning
        self.difficulty_level = 0.0  # 0.0 = легко, 1.0 = сложно
        self.curriculum_step = 0
        
        # Meta-learning параметры
        self.meta_buffer = []  # Буфер для meta-learning
        self.task_embeddings = {}  # Embedding для каждого символа/задачи
        
        # Evolutionary population
        self.population_size = 10
        self.population = [self._init_policy() for _ in range(self.population_size)]
        self.population_fitness = [0.0] * self.population_size
        
        # Метрики
        self.training_rewards = []
        self.adversarial_rewards = []
        self.training_losses = []
        self.episode_count = 0
        self.win_rate_history = []
        
        # Self-evolution метрики
        self.evolution_generation = 0
        self.best_fitness_history = []
        
        log.info(f"SEAL Agent (MIT) инициализирован")
        log.info(f"State dim: {state_dim} | Actions: {action_dim}")
        log.info(f"Population size: {self.population_size}")
    
    def _init_policy(self):
        """Инициализация политики с случайными весами"""
        return {
            'q_table': {},
            'fitness': 0.0,
            'age': 0
        }
    
    def state_to_key(self, state):
        """Преобразование состояния в дискретный ключ"""
        # Используем квантование для дискретизации
        discrete = tuple(round(float(s) * 20) / 20.0 for s in state[:self.state_dim])
        return discrete
    
    def get_q_value(self, state, action, policy='protagonist'):
        """Получить Q-значение из указанной политики"""
        policy_dict = self.protagonist_policy if policy == 'protagonist' else self.adversary_policy
        key = self.state_to_key(state)
        
        if key not in policy_dict:
            policy_dict[key] = np.random.randn(self.action_dim) * 0.01
        
        return policy_dict[key][action]
    
    def set_q_value(self, state, action, value, policy='protagonist'):
        """Установить Q-значение в указанной политике"""
        policy_dict = self.protagonist_policy if policy == 'protagonist' else self.adversary_policy
        key = self.state_to_key(state)
        
        if key not in policy_dict:
            policy_dict[key] = np.random.randn(self.action_dim) * 0.01
        
        policy_dict[key][action] = value
    
    def select_action(self, state, training=True, policy='protagonist'):
        """
        Epsilon-greedy с curriculum learning
        Адаптивный epsilon на основе уровня сложности
        """
        if training:
            # Адаптивный exploration на основе curriculum
            adaptive_epsilon = self.epsilon * (1.0 - self.difficulty_level * 0.5)
            
            if np.random.random() < adaptive_epsilon:
                return np.random.randint(0, self.action_dim)
        
        # Greedy action
        policy_dict = self.protagonist_policy if policy == 'protagonist' else self.adversary_policy
        key = self.state_to_key(state)
        
        if key not in policy_dict:
            return 0  # HOLD по умолчанию
        
        q_values = policy_dict[key]
        return np.argmax(q_values)
    
    def compute_priority(self, td_error):
        """
        Вычисление приоритета для prioritized experience replay
        Основано на TD-error
        """
        return abs(td_error) + 1e-6
    
    def store_experience(self, state, action, reward, next_state, done, td_error=None):
        """
        Сохранить опыт с приоритетом
        """
        self.memory.append((state, action, reward, next_state, done))
        
        if td_error is not None:
            priority = self.compute_priority(td_error)
        else:
            priority = 1.0
        
        self.priorities.append(priority)
    
    def sample_batch(self, batch_size):
        """
        Prioritized experience replay sampling
        """
        if len(self.memory) < batch_size:
            return None
        
        # Конвертация priorities в numpy array
        priorities_array = np.array(list(self.priorities))
        
        # Нормализация приоритетов для получения вероятностей
        probabilities = priorities_array / priorities_array.sum()
        
        # Выборка с учётом приоритетов
        indices = np.random.choice(
            len(self.memory), 
            size=batch_size, 
            replace=False,
            p=probabilities
        )
        
        batch = [self.memory[i] for i in indices]
        
        return batch, indices
    
    def adversarial_step(self, state, action, reward):
        """
        Adversarial learning step (MIT SEAL core)
        Adversary пытается создать сложные сценарии для protagonist
        """
        # Adversary выбирает действие, которое максимизирует сложность для protagonist
        adversary_action = self.select_action(state, training=True, policy='adversary')
        
        # Вычисляем "adversarial reward" (обратный от protagonist)
        adversarial_reward = -reward if reward != 0 else np.random.randn() * 0.1
        
        # Обновляем adversary policy
        key = self.state_to_key(state)
        if key in self.adversary_policy:
            current_q = self.adversary_policy[key][adversary_action]
            # Simple Q-update для adversary
            new_q = current_q + self.lr * 0.5 * (adversarial_reward - current_q)
            self.adversary_policy[key][adversary_action] = new_q
        
        self.adversarial_rewards.append(adversarial_reward)
        
        return adversary_action
    
    def train_step(self):
        """
        Один шаг обучения с MIT SEAL методологией:
        1. Prioritized experience replay
        2. Double Q-learning для стабильности
        3. Adversarial training
        4. Curriculum difficulty adjustment
        """
        batch_result = self.sample_batch(SEAL_BATCH_SIZE)
        
        if batch_result is None:
            return 0.0
        
        batch, indices = batch_result
        
        total_loss = 0.0
        td_errors = []
        
        for idx, (state, action, reward, next_state, done) in enumerate(batch):
            # Double Q-learning: используем protagonist для выбора, но adversary для оценки
            current_q = self.get_q_value(state, action, 'protagonist')
            
            if done:
                target_q = reward
            else:
                # Protagonist выбирает лучшее действие
                next_key = self.state_to_key(next_state)
                if next_key in self.protagonist_policy:
                    best_next_action = np.argmax(self.protagonist_policy[next_key])
                    
                    # Но оцениваем с помощью blend protagonist + adversary (adversarial training)
                    protagonist_q = self.get_q_value(next_state, best_next_action, 'protagonist')
                    adversary_q = self.get_q_value(next_state, best_next_action, 'adversary')
                    
                    # Blend: больше вес adversary при высокой сложности
                    blend_weight = self.difficulty_level
                    max_next_q = (1 - blend_weight) * protagonist_q + blend_weight * adversary_q
                else:
                    max_next_q = 0.0
                
                target_q = reward + self.gamma * max_next_q
            
            # TD-error для prioritized replay
            td_error = target_q - current_q
            td_errors.append(td_error)
            
            # Q-learning update
            new_q = current_q + self.lr * td_error
            self.set_q_value(state, action, new_q, 'protagonist')
            
            # Adversarial step
            self.adversarial_step(state, action, reward)
            
            loss = td_error ** 2
            total_loss += loss
        
        # Обновляем приоритеты в replay buffer
        for idx, td_error in zip(indices, td_errors):
            self.priorities[idx] = self.compute_priority(td_error)
        
        avg_loss = total_loss / SEAL_BATCH_SIZE
        self.training_losses.append(avg_loss)
        
        # Curriculum learning: адаптация сложности
        self.update_curriculum()
        
        return avg_loss
    
    def update_curriculum(self):
        """
        Curriculum learning: постепенное увеличение сложности
        Основано на последних результатах обучения
        """
        self.curriculum_step += 1
        
        # Увеличиваем сложность на основе performance
        if len(self.training_rewards) > 100:
            recent_performance = np.mean(self.training_rewards[-100:])
            
            # Если агент хорошо справляется, увеличиваем сложность
            if recent_performance > 0.5:
                self.difficulty_level = min(1.0, self.difficulty_level + 0.01)
            # Если плохо - уменьшаем
            elif recent_performance < -0.5:
                self.difficulty_level = max(0.0, self.difficulty_level - 0.005)
        else:
            # Медленное увеличение в начале
            self.difficulty_level = min(1.0, self.curriculum_step / 10000.0)
    
    def evolutionary_step(self):
        """
        Evolutionary strategies (MIT SEAL)
        Эволюционная оптимизация популяции политик
        """
        # Сортируем популяцию по fitness
        sorted_indices = np.argsort(self.population_fitness)[::-1]
        
        # Элитизм: сохраняем топ-20%
        elite_count = max(2, self.population_size // 5)
        elite_indices = sorted_indices[:elite_count]
        
        # Создаём новую популяцию
        new_population = []
        
        # Копируем элиту
        for idx in elite_indices:
            new_population.append(self.population[idx].copy())
        
        # Создаём потомков через кроссовер и мутацию
        while len(new_population) < self.population_size:
            # Выбираем двух родителей из элиты
            parent1 = self.population[np.random.choice(elite_indices)]
            parent2 = self.population[np.random.choice(elite_indices)]
            
            # Кроссовер
            child = self._crossover(parent1, parent2)
            
            # Мутация
            child = self._mutate(child)
            
            new_population.append(child)
        
        self.population = new_population
        self.evolution_generation += 1
        
        # Сохраняем лучший fitness
        best_fitness = max(self.population_fitness)
        self.best_fitness_history.append(best_fitness)
        
        log.info(f"Evolution Gen {self.evolution_generation} | Best fitness: {best_fitness:.4f}")
    
    def _crossover(self, parent1, parent2):
        """Кроссовер двух политик"""
        child = self._init_policy()
        
        # Смешиваем Q-tables родителей
        all_keys = set(parent1['q_table'].keys()) | set(parent2['q_table'].keys())
        
        for key in all_keys:
            if key in parent1['q_table'] and key in parent2['q_table']:
                # Среднее значение
                child['q_table'][key] = (parent1['q_table'][key] + parent2['q_table'][key]) / 2.0
            elif key in parent1['q_table']:
                child['q_table'][key] = parent1['q_table'][key].copy()
            else:
                child['q_table'][key] = parent2['q_table'][key].copy()
        
        return child
    
    def _mutate(self, policy, mutation_rate=0.1):
        """Мутация политики"""
        mutated = policy.copy()
        
        for key in mutated['q_table']:
            if np.random.random() < mutation_rate:
                # Добавляем шум
                noise = np.random.randn(self.action_dim) * 0.1
                mutated['q_table'][key] = mutated['q_table'][key] + noise
        
        return mutated
    
    def meta_learning_update(self, task_id, task_data):
        """
        Meta-learning для быстрой адаптации к новым рынкам/символам
        Основано на MAML (Model-Agnostic Meta-Learning)
        """
        # Сохраняем embedding для задачи (символа)
        if task_id not in self.task_embeddings:
            self.task_embeddings[task_id] = np.random.randn(16)  # 16-dim embedding
        
        # Few-shot adaptation на новых данных
        meta_lr = self.lr * 0.1  # Меньший learning rate для meta-update
        
        for state, action, reward, next_state, done in task_data:
            current_q = self.get_q_value(state, action)
            
            if done:
                target_q = reward
            else:
                key = self.state_to_key(next_state)
                if key in self.protagonist_policy:
                    max_next_q = np.max(self.protagonist_policy[key])
                else:
                    max_next_q = 0.0
                target_q = reward + self.gamma * max_next_q
            
            # Meta-gradient update
            new_q = current_q + meta_lr * (target_q - current_q)
            self.set_q_value(state, action, new_q)
        
        log.info(f"Meta-learning update для задачи {task_id}")
    
    def decay_epsilon(self, decay_rate=0.995):
        """Уменьшение exploration rate"""
        self.epsilon = max(0.01, self.epsilon * decay_rate)
    
    def get_diagnostics(self):
        """Диагностическая информация о состоянии агента"""
        diagnostics = {
            'episode_count': self.episode_count,
            'epsilon': self.epsilon,
            'difficulty_level': self.difficulty_level,
            'protagonist_policy_size': len(self.protagonist_policy),
            'adversary_policy_size': len(self.adversary_policy),
            'evolution_generation': self.evolution_generation,
            'memory_size': len(self.memory),
            'avg_recent_reward': np.mean(self.training_rewards[-100:]) if len(self.training_rewards) > 0 else 0.0,
            'avg_recent_loss': np.mean(self.training_losses[-100:]) if len(self.training_losses) > 0 else 0.0
        }
        
        return diagnostics
    
    def save_checkpoint(self, filepath):
        """Сохранить полный checkpoint агента (ИСПРАВЛЕНО - конвертация tuple ключей в строки)"""
        # КРИТИЧНО: Конвертация tuple ключей в строки для JSON
        protagonist_serializable = {}
        for k, v in self.protagonist_policy.items():
            key_str = str(k)  # Конвертируем tuple в строку
            protagonist_serializable[key_str] = v.tolist() if isinstance(v, np.ndarray) else v
        
        adversary_serializable = {}
        for k, v in self.adversary_policy.items():
            key_str = str(k)
            adversary_serializable[key_str] = v.tolist() if isinstance(v, np.ndarray) else v
        
        population_serializable = []
        for p in self.population:
            p_copy = {
                'fitness': float(p['fitness']),
                'age': int(p['age']),
                'q_table': {}
            }
            for k, v in p['q_table'].items():
                key_str = str(k)
                p_copy['q_table'][key_str] = v.tolist() if isinstance(v, np.ndarray) else v
            population_serializable.append(p_copy)
        
        task_embeddings_serializable = {}
        for k, v in self.task_embeddings.items():
            task_embeddings_serializable[str(k)] = v.tolist() if isinstance(v, np.ndarray) else v
        
        checkpoint = {
            'protagonist_policy': protagonist_serializable,
            'adversary_policy': adversary_serializable,
            'population': population_serializable,
            'population_fitness': self.population_fitness,
            'epsilon': float(self.epsilon),
            'difficulty_level': float(self.difficulty_level),
            'curriculum_step': int(self.curriculum_step),
            'episode_count': int(self.episode_count),
            'evolution_generation': int(self.evolution_generation),
            'training_rewards': self.training_rewards[-1000:],  # Последние 1000
            'adversarial_rewards': self.adversarial_rewards[-1000:],
            'training_losses': self.training_losses[-1000:],
            'best_fitness_history': self.best_fitness_history,
            'task_embeddings': task_embeddings_serializable
        }
        
        with open(filepath, 'w') as f:
            json.dump(checkpoint, f, indent=2)
        
        log.info(f"SEAL checkpoint сохранён: {filepath}")
    
    def load_checkpoint(self, filepath):
        """Загрузить checkpoint агента"""
        if not os.path.exists(filepath):
            log.warning(f"Checkpoint не найден: {filepath}")
            return False
        
        with open(filepath, 'r') as f:
            checkpoint = json.load(f)
        
        # Восстанавливаем protagonist policy (конвертация строк обратно в tuples)
        self.protagonist_policy = {}
        for k, v in checkpoint['protagonist_policy'].items():
            key_tuple = eval(k)  # Безопасно, т.к. мы сами создавали эти строки
            self.protagonist_policy[key_tuple] = np.array(v)
        
        # Восстанавливаем adversary policy
        self.adversary_policy = {}
        for k, v in checkpoint['adversary_policy'].items():
            key_tuple = eval(k)
            self.adversary_policy[key_tuple] = np.array(v)
        
        # Восстанавливаем population
        self.population = []
        for p in checkpoint['population']:
            p_restored = {
                'fitness': p['fitness'],
                'age': p['age'],
                'q_table': {}
            }
            for k, v in p['q_table'].items():
                key_tuple = eval(k)
                p_restored['q_table'][key_tuple] = np.array(v)
            self.population.append(p_restored)
        
        self.population_fitness = checkpoint['population_fitness']
        self.epsilon = checkpoint['epsilon']
        self.difficulty_level = checkpoint['difficulty_level']
        self.curriculum_step = checkpoint['curriculum_step']
        self.episode_count = checkpoint['episode_count']
        self.evolution_generation = checkpoint['evolution_generation']
        self.training_rewards = checkpoint['training_rewards']
        self.adversarial_rewards = checkpoint['adversarial_rewards']
        self.training_losses = checkpoint['training_losses']
        self.best_fitness_history = checkpoint['best_fitness_history']
        
        self.task_embeddings = {}
        for k, v in checkpoint['task_embeddings'].items():
            self.task_embeddings[k] = np.array(v)
        
        log.info(f"SEAL checkpoint загружен: {filepath}")
        log.info(f"Episode: {self.episode_count} | Generation: {self.evolution_generation}")
        
        return True


def extract_state_from_row(row):
    """Извлечь состояние для SEAL из технических индикаторов"""
    state = np.array([
        row['RSI'] / 100.0,  # нормализовано 0-1
        (row['MACD'] + 0.001) / 0.002,  # примерная нормализация
        row['ATR'] / 0.005,
        row['vol_ratio'] / 3.0,
        row['BB_position'],
        row['Stoch_K'] / 100.0,
        row['Stoch_D'] / 100.0,
        (row['EMA_50'] - row['close']) / row['close'],
        (row['EMA_200'] - row['close']) / row['close'],
        (row['close'] - row['close_prev']) / row['close_prev'] if 'close_prev' in row.index else 0.0
    ])
    
    # Clipping для стабильности
    state = np.clip(state, -5, 5)
    
    return state


# ====================== ПРИЗНАКИ ======================
def calculate_features(df: pd.DataFrame) -> pd.DataFrame:
    """Расчёт технических индикаторов"""
    d = df.copy()
    d["close_prev"] = d["close"].shift(1)
    
    # ATR
    tr = pd.concat([
        d["high"] - d["low"],
        (d["high"] - d["close_prev"]).abs(),
        (d["low"] - d["close_prev"]).abs(),
    ], axis=1).max(axis=1)
    d["ATR"] = tr.rolling(14).mean()
    
    # RSI
    delta = d["close"].diff()
    up = delta.clip(lower=0).rolling(14).mean()
    down = (-delta.clip(upper=0)).rolling(14).mean()
    rs = up / down.replace(0, np.nan)
    d["RSI"] = 100 - (100 / (1 + rs))
    
    # MACD
    ema12 = d["close"].ewm(span=12, adjust=False).mean()
    ema26 = d["close"].ewm(span=26, adjust=False).mean()
    d["MACD"] = ema12 - ema26
    d["MACD_signal"] = d["MACD"].ewm(span=9, adjust=False).mean()
    
    # Объёмы
    d["vol_avg_20"] = d["tick_volume"].rolling(20).mean()
    d["vol_ratio"] = d["tick_volume"] / d["vol_avg_20"].replace(0, np.nan)
    
    # Bollinger Bands
    d["BB_middle"] = d["close"].rolling(20).mean()
    bb_std = d["close"].rolling(20).std()
    d["BB_upper"] = d["BB_middle"] + 2 * bb_std
    d["BB_lower"] = d["BB_middle"] - 2 * bb_std
    d["BB_position"] = (d["close"] - d["BB_lower"]) / (d["BB_upper"] - d["BB_lower"])
    
    # Stochastic
    low_14 = d["low"].rolling(14).min()
    high_14 = d["high"].rolling(14).max()
    d["Stoch_K"] = 100 * (d["close"] - low_14) / (high_14 - low_14)
    d["Stoch_D"] = d["Stoch_K"].rolling(3).mean()
    
    # EMA кросс
    d["EMA_50"] = d["close"].ewm(span=50, adjust=False).mean()
    d["EMA_200"] = d["close"].ewm(span=200, adjust=False).mean()
    
    return d.dropna()


# ====================== БАЛАНСИРОВКА КЛАССОВ ======================
def generate_real_dataset_from_mt5(num_samples: int = 1000, balance_ratio: float = BALANCE_RATIO) -> list:
    """
    Генерация СБАЛАНСИРОВАННОГО датасета на основе реальных данных MT5
    
    Args:
        num_samples: Общее количество примеров
        balance_ratio: Целевое соотношение UP/DOWN (1.0 = идеальный баланс 50/50)
    
    Returns:
        Список сбалансированных примеров
    """
    print(f"\n{'='*80}")
    print(f"ГЕНЕРАЦИЯ СБАЛАНСИРОВАННОГО ДАТАСЕТА ИЗ MT5")
    print(f"{'='*80}\n")
    print(f"Цель: {num_samples} примеров с балансом UP/DOWN = {balance_ratio}:1")
    
    if not mt5 or not mt5.initialize():
        print("MT5 не подключен! Используй синтетический датасет.")
        return generate_synthetic_dataset(num_samples, balance_ratio)
    
    # Счётчики для балансировки
    target_up = int(num_samples * balance_ratio / (1 + balance_ratio))
    target_down = num_samples - target_up
    
    print(f"Целевое распределение:")
    print(f" UP: {target_up} примеров ({target_up/num_samples*100:.1f}%)")
    print(f" DOWN: {target_down} примеров ({target_down/num_samples*100:.1f}%)\n")
    
    # Загружаем данные за последние 6 месяцев
    end = datetime.now()
    start = end - timedelta(days=180)
    
    all_up_candidates = []
    all_down_candidates = []
    
    for symbol in SYMBOLS:
        print(f"Загрузка {symbol}...")
        rates = mt5.copy_rates_range(symbol, TIMEFRAME, start, end)
        
        if rates is None or len(rates) < LOOKBACK + PREDICTION_HORIZON:
            print(f"Недостаточно данных для {symbol}")
            continue
        
        df = pd.DataFrame(rates)
        df["time"] = pd.to_datetime(df["time"], unit="s")
        df.set_index("time", inplace=True)
        df = calculate_features(df)
        
        # Собираем ВСЕ возможные точки
        for idx in range(LOOKBACK, len(df) - PREDICTION_HORIZON):
            row = df.iloc[idx]
            future_idx = idx + PREDICTION_HORIZON
            future_row = df.iloc[future_idx]
            
            price_change = future_row['close'] - row['close']
            direction = "UP" if price_change > 0 else "DOWN"
            
            candidate = {
                'symbol': symbol,
                'row': row,
                'future_row': future_row,
                'current_time': df.index[idx],
                'price_change': abs(price_change)
            }
            
            if direction == "UP":
                all_up_candidates.append(candidate)
            else:
                all_down_candidates.append(candidate)
        
        print(f" {symbol}: UP={len([c for c in all_up_candidates if c['symbol']==symbol])}, "
              f"DOWN={len([c for c in all_down_candidates if c['symbol']==symbol])}")
    
    mt5.shutdown()
    
    print(f"\nВсего кандидатов: UP={len(all_up_candidates)}, DOWN={len(all_down_candidates)}")
    
    # СТРАТИФИЦИРОВАННАЯ ВЫБОРКА с балансировкой
    dataset = []
    
    # Выбираем случайные примеры с учётом баланса
    selected_up_indices = np.random.choice(
        len(all_up_candidates),
        size=min(target_up, len(all_up_candidates)),
        replace=False
    ) if len(all_up_candidates) > 0 else []
    
    selected_down_indices = np.random.choice(
        len(all_down_candidates),
        size=min(target_down, len(all_down_candidates)),
        replace=False
    ) if len(all_down_candidates) > 0 else []
    
    # Создаём примеры
    for idx in selected_up_indices:
        candidate = all_up_candidates[idx]
        example = create_training_example(
            candidate['symbol'],
            candidate['row'],
            candidate['future_row'],
            candidate['current_time']
        )
        dataset.append(example)
    
    for idx in selected_down_indices:
        candidate = all_down_candidates[idx]
        example = create_training_example(
            candidate['symbol'],
            candidate['row'],
            candidate['future_row'],
            candidate['current_time']
        )
        dataset.append(example)
    
    # Shuffle
    np.random.shuffle(dataset)
    
    # Статистика
    up_count = sum(1 for ex in dataset if ex['direction'] == 'UP')
    down_count = len(dataset) - up_count
    
    print(f"\n{'='*80}")
    print(f"ДАТАСЕТ СОЗДАН")
    print(f"{'='*80}")
    print(f"Всего примеров: {len(dataset)}")
    print(f" UP: {up_count} ({up_count/len(dataset)*100:.1f}%)")
    print(f" DOWN: {down_count} ({down_count/len(dataset)*100:.1f}%)")
    
    actual_ratio = max(up_count, down_count) / min(up_count, down_count) if min(up_count, down_count) > 0 else 0
    print(f" Фактическое соотношение: {actual_ratio:.2f}:1")
    
    if actual_ratio <= 1.2:
        print(f" ✓ ОТЛИЧНО! Датасет сбалансирован")
    elif actual_ratio <= 1.5:
        print(f" ⚠ ПРИЕМЛЕМО. Лёгкий дисбаланс")
    else:
        print(f" ✗ ПРОБЛЕМА! Требуется балансировка")
        dataset = balance_dataset_smote(dataset, up_count, down_count)
    
    print(f"{'='*80}\n")
    
    return dataset


def balance_dataset_smote(dataset: list, up_count: int, down_count: int) -> list:
    """
    Балансировка через SMOTE-подобную технику
    (Synthetic Minority Over-sampling)
    """
    if up_count == down_count:
        return dataset
    
    minority_class = 'UP' if up_count < down_count else 'DOWN'
    majority_count = max(up_count, down_count)
    minority_count = min(up_count, down_count)
    
    print(f"\n Применяю SMOTE балансировку...")
    print(f" Класс меньшинства: {minority_class}")
    print(f" Нужно добавить: {majority_count - minority_count} синтетических примеров\n")
    
    up_examples = [ex for ex in dataset if ex['direction'] == 'UP']
    down_examples = [ex for ex in dataset if ex['direction'] == 'DOWN']
    
    minority_examples = up_examples if minority_class == 'UP' else down_examples
    
    # Создаём синтетические примеры через небольшие вариации
    synthetic_count = 0
    while len(minority_examples) < majority_count and synthetic_count < majority_count:
        # Берём случайный пример из меньшинства
        base = np.random.choice(minority_examples)
        
        # Создаём вариацию (небольшой шум в промпте)
        synthetic = base.copy()
        
        # Добавляем небольшие вариации в численные значения
        original_prompt = synthetic['prompt']
        
        # Парсим численные значения и добавляем шум ±2%
        for pattern in [r'RSI: (\d+\.\d+)', r'MACD: (-?\d+\.\d+)', r'ATR: (\d+\.\d+)']:
            match = re.search(pattern, original_prompt)
            if match:
                original_val = float(match.group(1))
                noise = original_val * np.random.uniform(-0.02, 0.02)
                new_val = original_val + noise
                original_prompt = re.sub(pattern, f'{pattern.split(":")[0]}: {new_val:.5f}', original_prompt, count=1)
        
        synthetic['prompt'] = original_prompt
        minority_examples.append(synthetic)
        synthetic_count += 1
    
    # Объединяем
    if minority_class == 'UP':
        balanced = minority_examples[:majority_count] + down_examples
    else:
        balanced = up_examples + minority_examples[:majority_count]
    
    np.random.shuffle(balanced)
    
    print(f" Балансировка завершена")
    print(f" Добавлено синтетических примеров: {synthetic_count}")
    print(f" Итоговое распределение: UP={len([ex for ex in balanced if ex['direction'] == 'UP'])}, "
          f"DOWN={len([ex for ex in balanced if ex['direction'] == 'DOWN'])}\n")
    
    return balanced


def generate_synthetic_dataset(num_samples: int = 1000, balance_ratio: float = BALANCE_RATIO) -> list:
    """Генерация СБАЛАНСИРОВАННОГО синтетического датасета"""
    print(f"\nГенерация {num_samples} СБАЛАНСИРОВАННЫХ синтетических примеров...")
    print(f"Целевой баланс UP/DOWN: {balance_ratio}:1\n")
    
    dataset = []
    symbols = ["EURUSD", "GBPUSD", "USDCHF", "USDCAD"]
    
    target_up = int(num_samples * balance_ratio / (1 + balance_ratio))
    target_down = num_samples - target_up
    
    up_count = 0
    down_count = 0
    
    print(f"Целевое распределение:")
    print(f" UP: {target_up} примеров ({target_up/num_samples*100:.1f}%)")
    print(f" DOWN: {target_down} примеров ({target_down/num_samples*100:.1f}%)\n")
    
    attempts = 0
    max_attempts = num_samples * 3
    
    while len(dataset) < num_samples and attempts < max_attempts:
        attempts += 1
        
        symbol = np.random.choice(symbols)
        price = np.random.uniform(1.0500, 1.2000) if "EUR" in symbol else np.random.uniform(1.2000, 1.4000)
        rsi = np.random.uniform(20, 80)
        macd = np.random.uniform(-0.001, 0.001)
        atr = np.random.uniform(0.0005, 0.0030)
        vol_ratio = np.random.uniform(0.5, 2.0)
        bb_pos = np.random.uniform(0, 1)
        stoch_k = np.random.uniform(20, 80)
        
        bullish_signals = 0
        bearish_signals = 0
        
        if rsi < 30:
            bullish_signals += 2
        elif rsi > 70:
            bearish_signals += 2
        elif 40 < rsi < 50:
            bullish_signals += 1
        elif 50 < rsi < 60:
            bearish_signals += 1
        
        if macd > 0:
            bullish_signals += 2
        else:
            bearish_signals += 2
        
        if bb_pos < 0.2:
            bullish_signals += 1
        elif bb_pos > 0.8:
            bearish_signals += 1
        
        if vol_ratio > 1.5:
            if bullish_signals > bearish_signals:
                bullish_signals += 1
            else:
                bearish_signals += 1
        
        if stoch_k < 20:
            bullish_signals += 1
        elif stoch_k > 80:
            bearish_signals += 1
        
        direction = "UP" if bullish_signals > bearish_signals else "DOWN"
        
        # Контроль баланса классов
        if direction == "UP" and up_count >= target_up:
            continue
        if direction == "DOWN" and down_count >= target_down:
            continue
        
        confidence = min(98, max(65, 60 + abs(bullish_signals - bearish_signals) * 8))
        
        signal_strength = abs(bullish_signals - bearish_signals)
        base_move = signal_strength * 15 + np.random.randint(10, 40)
        price_24h_move = base_move if direction == "UP" else -base_move
        price_24h = price + (price_24h_move * 0.0001)
        
        analysis_parts = []
        if rsi < 30:
            analysis_parts.append(f"RSI {rsi:.1f} — сильная перепроданность, через 24ч жду отскок на {abs(price_24h_move)} пунктов")
        elif rsi > 70:
            analysis_parts.append(f"RSI {rsi:.1f} — перекупленность, за сутки возможна коррекция до {abs(price_24h_move)} пунктов")
        else:
            analysis_parts.append(f"RSI {rsi:.1f} — нейтральная зона, прогноз движения {abs(price_24h_move)} пунктов за 24ч")
        
        if macd > 0.0005:
            analysis_parts.append("MACD сильно позитивный — бычий импульс продолжится в течение суток")
        elif macd < -0.0005:
            analysis_parts.append("MACD негативный — медвежье давление сохранится 24 часа")
        else:
            analysis_parts.append("MACD около нуля — слабый тренд, но направление определено")
        
        if atr > 0.002:
            analysis_parts.append(f"ATR {atr:.5f} — высокая волатильность, за сутки возможен размах {int(atr/0.0001 * 1.5)} пунктов")
        else:
            analysis_parts.append(f"ATR {atr:.5f} — умеренная волатильность, движение {abs(price_24h_move)} пунктов реалистично")
        
        if vol_ratio > 1.5:
            analysis_parts.append("Объёмы выше средних на 50%+ — импульс сохранится на ближайшие 24 часа")
        elif vol_ratio < 0.7:
            analysis_parts.append("Объёмы низкие — движение будет медленным, но направление верное")
        
        if bb_pos < 0.2:
            analysis_parts.append("Цена у нижней границы Боллинджера — через сутки ожидаю возврат к середине канала")
        elif bb_pos > 0.8:
            analysis_parts.append("Цена у верхней границы Боллинджера — за 24ч возможен откат к середине")
        
        if stoch_k < 20:
            analysis_parts.append(f"Stochastic {stoch_k:.1f} — экстремальная перепроданность, разворот вверх в течение суток")
        elif stoch_k > 80:
            analysis_parts.append(f"Stochastic {stoch_k:.1f} — экстремальная перекупленность, коррекция вниз за 24ч")
        
        analysis = "\n- ".join(analysis_parts)
        
        prompt = f"""{symbol} {datetime.now().strftime('%Y-%m-%d %H:%M')}
Текущая цена: {price:.5f}
RSI: {rsi:.1f}
MACD: {macd:.6f}
ATR: {atr:.5f}
Объёмы: {vol_ratio:.2f}x
BB позиция: {bb_pos:.2f}
Stochastic K: {stoch_k:.1f}
Проанализируй ситуацию объективно и дай точный прогноз цены через 24 часа.
ВАЖНО: Давай прогноз на основе данных, без предвзятости к направлению."""
        
        response = f"""НАПРАВЛЕНИЕ: {direction}
УВЕРЕННОСТЬ: {confidence}%
ПРОГНОЗ ЦЕНЫ ЧЕРЕЗ 24Ч: {price_24h:.5f} ({price_24h_move:+d} пунктов)
ОБЪЕКТИВНЫЙ АНАЛИЗ НА 24 ЧАСА:
- {analysis}
ВЫВОД: Технический анализ по {abs(bullish_signals + bearish_signals)} индикаторам указывает на {'бычий' if direction == 'UP' else 'медвежий'} сценарий. За ближайшие 24 часа жду движение {abs(price_24h_move)} пунктов {direction} к цели {price_24h:.5f}.
НАПОМИНАНИЕ: Рынок непредсказуем. Этот анализ основан на текущих технических данных, но ситуация может измениться. Следующий сигнал может быть противоположным."""
        
        dataset.append({
            "prompt": prompt,
            "response": response,
            "direction": direction
        })
        
        if direction == "UP":
            up_count += 1
        else:
            down_count += 1
        
        if len(dataset) % 100 == 0:
            current_ratio = max(up_count, down_count) / max(1, min(up_count, down_count))
            print(f"Создано {len(dataset)}/{num_samples} | UP: {up_count} | DOWN: {down_count} | Ratio: {current_ratio:.2f}:1")
    
    print(f"\nСинтетический датасет готов: {len(dataset)} примеров")
    print(f" UP: {up_count} ({up_count/len(dataset)*100:.1f}%)")
    print(f" DOWN: {down_count} ({down_count/len(dataset)*100:.1f}%)")
    
    actual_ratio = max(up_count, down_count) / min(up_count, down_count)
    print(f" Фактический баланс: {actual_ratio:.2f}:1")
    print(" ✓ ОТЛИЧНО! Датасет сбалансирован\n" if actual_ratio <= 1.2 else " ⚠ Небольшой дисбаланс, но приемлемо\n")
    
    return dataset


def create_training_example(symbol: str, row: pd.Series, future_row: pd.Series, current_time: datetime) -> dict:
    """Создание обучающего примера из данных"""
    actual_price_24h = future_row['close']
    price_change = actual_price_24h - row['close']
    price_change_pips = int(price_change / 0.0001)
    direction = "UP" if price_change > 0 else "DOWN"
    
    bullish_signals = 0
    bearish_signals = 0
    analysis_parts = []
    
    # RSI анализ
    if row['RSI'] < 30:
        bullish_signals += 2
        analysis_parts.append(f"RSI {row['RSI']:.1f} — сильная перепроданность, через 24ч произошёл отскок на {abs(price_change_pips)} пунктов")
    elif row['RSI'] > 70:
        bearish_signals += 2
        analysis_parts.append(f"RSI {row['RSI']:.1f} — перекупленность, за сутки случилась коррекция на {abs(price_change_pips)} пунктов")
    else:
        if row['RSI'] < 50:
            bullish_signals += 1
        else:
            bearish_signals += 1
        analysis_parts.append(f"RSI {row['RSI']:.1f} — нейтральная зона, движение {abs(price_change_pips)} пунктов за 24ч")
    
    # MACD анализ
    if row['MACD'] > 0:
        bullish_signals += 2
        analysis_parts.append("MACD позитивный — бычий импульс подтвердился в течение суток")
    else:
        bearish_signals += 2
        analysis_parts.append("MACD негативный — медвежье давление сохранилось 24 часа")
    
    # ATR анализ
    if row['ATR'] > row['ATR'] * 1.3:
        analysis_parts.append(f"ATR {row['ATR']:.5f} — высокая волатильность обеспечила движение {abs(price_change_pips)} пунктов")
    else:
        analysis_parts.append(f"ATR {row['ATR']:.5f} — умеренная волатильность, движение {abs(price_change_pips)} пунктов")
    
    # Объёмы
    if row['vol_ratio'] > 1.5:
        if direction == "UP":
            bullish_signals += 1
        else:
            bearish_signals += 1
        analysis_parts.append("Объёмы выше средних на 50%+ — импульс продолжился в течение суток")
    
    # BB позиция
    if row['BB_position'] < 0.2:
        bullish_signals += 1
        analysis_parts.append("Цена у нижней границы Боллинджера — через 24ч произошёл возврат к средней")
    elif row['BB_position'] > 0.8:
        bearish_signals += 1
        analysis_parts.append("Цена у верхней границы Боллинджера — за сутки случился откат")
    
    # Stochastic
    if row['Stoch_K'] < 20:
        bullish_signals += 1
        analysis_parts.append(f"Stochastic {row['Stoch_K']:.1f} — перепроданность дала разворот вверх")
    elif row['Stoch_K'] > 80:
        bearish_signals += 1
        analysis_parts.append(f"Stochastic {row['Stoch_K']:.1f} — перекупленность привела к коррекции")
    
    # Уверенность
    if direction == "UP":
        confidence = min(98, max(65, 60 + bullish_signals * 8))
    else:
        confidence = min(98, max(65, 60 + bearish_signals * 8))
    
    analysis = "\n- ".join(analysis_parts)
    
    prompt = f"""{symbol} {current_time.strftime('%Y-%m-%d %H:%M')}
Текущая цена: {row['close']:.5f}
RSI: {row['RSI']:.1f}
MACD: {row['MACD']:.6f}
ATR: {row['ATR']:.5f}
Объёмы: {row['vol_ratio']:.2f}x
BB позиция: {row['BB_position']:.2f}
Stochastic K: {row['Stoch_K']:.1f}
Проанализируй ситуацию объективно и дай точный прогноз цены через 24 часа.
ВАЖНО: Не склоняйся к одному направлению - анализируй реальные данные без предвзятости."""
    
    response = f"""НАПРАВЛЕНИЕ: {direction}
УВЕРЕННОСТЬ: {confidence}%
ПРОГНОЗ ЦЕНЫ ЧЕРЕЗ 24Ч: {actual_price_24h:.5f} ({price_change_pips:+d} пунктов)
ОБЪЕКТИВНЫЙ АНАЛИЗ НА 24 ЧАСА (РЕАЛЬНЫЙ РЕЗУЛЬТАТ):
- {analysis}
ВЫВОД: Анализ {abs(bullish_signals + bearish_signals)} индикаторов показывает {'бычий' if direction == 'UP' else 'медвежий'} сценарий. Фактическое движение за 24 часа составило {abs(price_change_pips)} пунктов {direction}. Конечная цена: {actual_price_24h:.5f}.
ВАЖНО: Этот прогноз основан исключительно на технических индикаторах, без предвзятости к направлению. В следующий раз рыночная ситуация может быть противоположной."""
    
    return {
        "prompt": prompt,
        "response": response,
        "direction": direction
    }


def save_dataset(dataset: list, filename: str = "dataset/finetune_data.jsonl"):
    """Сохранение датасета в JSONL формате"""
    with open(filename, 'w', encoding='utf-8') as f:
        for item in dataset:
            f.write(json.dumps(item, ensure_ascii=False) + '\n')
    print(f"Датасет сохранён: {filename}")
    return filename


# ====================== SEAL RL ОБУЧЕНИЕ ======================
def train_seal_agent(dataset_path: str, epochs: int = 10, enable_evolution=True):
    """
    Обучение SEAL RL агента по методологии MIT
    
    Features:
    - Prioritized experience replay
    - Adversarial self-play
    - Curriculum learning
    - Evolutionary strategies
    - Meta-learning
    """
    print(f"\n{'='*80}")
    print(f"SEAL RL ОБУЧЕНИЕ (MIT METHODOLOGY)")
    print(f"{'='*80}\n")
    
    # Загрузка датасета
    with open(dataset_path, 'r', encoding='utf-8') as f:
        training_data = [json.loads(line) for line in f]
    
    print(f"Загружено примеров: {len(training_data)}")
    
    # Статистика по символам для meta-learning
    symbol_counts = {}
    for example in training_data:
        symbol_match = re.search(r'^([A-Z]{6})', example['prompt'])
        if symbol_match:
            symbol = symbol_match.group(1)
            symbol_counts[symbol] = symbol_counts.get(symbol, 0) + 1
    
    print(f"Символов в датасете: {len(symbol_counts)}")
    for symbol, count in sorted(symbol_counts.items(), key=lambda x: -x[1])[:5]:
        print(f"  {symbol}: {count} примеров")
    
    # Инициализация агента
    agent = SEALAgent(state_dim=10, action_dim=3)
    
    print(f"\nЗапуск SEAL обучения на {epochs} эпох...")
    print(f"Features: Adversarial Self-Play, Curriculum Learning, Evolution\n")
    
    # Обучение
    for epoch in range(epochs):
        print(f"{'='*80}")
        print(f"Эпоха {epoch + 1}/{epochs}")
        print(f"{'='*80}")
        
        epoch_rewards = []
        epoch_correct = 0
        symbol_tasks = {}  # Для meta-learning
        
        # Shuffle данных каждую эпоху
        np.random.shuffle(training_data)
        
        for i, example in enumerate(training_data):
            # Парсинг состояния из промпта
            prompt = example['prompt']
            
            # Извлекаем символ для meta-learning
            symbol_match = re.search(r'^([A-Z]{6})', prompt)
            symbol = symbol_match.group(1) if symbol_match else 'UNKNOWN'
            
            # Парсинг индикаторов
            rsi_match = re.search(r'RSI: (\d+\.\d+)', prompt)
            macd_match = re.search(r'MACD: (-?\d+\.\d+)', prompt)
            atr_match = re.search(r'ATR: (\d+\.\d+)', prompt)
            vol_match = re.search(r'Объёмы: (\d+\.\d+)', prompt)
            bb_match = re.search(r'BB позиция: (\d+\.\d+)', prompt)
            stoch_match = re.search(r'Stochastic K: (\d+\.\d+)', prompt)
            
            if not all([rsi_match, macd_match, atr_match, vol_match, bb_match, stoch_match]):
                continue
            
            # Формирование состояния
            state = np.array([
                float(rsi_match.group(1)) / 100.0,
                (float(macd_match.group(1)) + 0.001) / 0.002,
                float(atr_match.group(1)) / 0.005,
                float(vol_match.group(1)) / 3.0,
                float(bb_match.group(1)),
                float(stoch_match.group(1)) / 100.0,
                np.random.randn() * 0.1,  # Шум для робастности
                0.0,
                0.0,
                agent.difficulty_level  # Текущая сложность как часть состояния
            ])
            
            state = np.clip(state, -5, 5)
            
            # Действие агента
            action = agent.select_action(state, training=True)
            
            # Истинное направление
            true_direction = example['direction']
            true_action = 1 if true_direction == "UP" else 2
            
            # Награда с учётом уверенности модели
            confidence_match = re.search(r'УВЕРЕННОСТЬ: (\d+)', example['response'])
            confidence = int(confidence_match.group(1)) if confidence_match else 75
            
            if action == true_action:
                reward = 1.0 * (confidence / 100.0)  # Награда пропорциональна уверенности
                epoch_correct += 1
            elif action == 0:  # HOLD
                reward = -0.1  # Небольшой штраф за бездействие
            else:
                reward = -1.0 * (confidence / 100.0)  # Штраф пропорционален уверенности
            
            epoch_rewards.append(reward)
            
            # Следующее состояние (небольшое изменение)
            next_state = state + np.random.randn(10) * 0.05
            next_state = np.clip(next_state, -5, 5)
            next_state[-1] = agent.difficulty_level  # Обновляем difficulty
            
            done = (i == len(training_data) - 1)
            
            # Сохраняем опыт
            agent.store_experience(state, action, reward, next_state, done)
            
            # Сохраняем для meta-learning
            if symbol not in symbol_tasks:
                symbol_tasks[symbol] = []
            symbol_tasks[symbol].append((state, action, reward, next_state, done))
            
            # Обучение каждые UPDATE_FREQUENCY шагов
            if i % SEAL_UPDATE_FREQUENCY == 0 and i > SEAL_BATCH_SIZE:
                loss = agent.train_step()
                
                if i % (SEAL_UPDATE_FREQUENCY * 10) == 0:
                    diagnostics = agent.get_diagnostics()
                    print(f"  Step {i}/{len(training_data)} | "
                          f"Loss: {loss:.4f} | "
                          f"Difficulty: {diagnostics['difficulty_level']:.2f} | "
                          f"Policy size: {diagnostics['protagonist_policy_size']}")
        
        # Meta-learning update для каждого символа
        if epoch % 2 == 0:  # Каждую вторую эпоху
            print(f"\nMeta-learning update...")
            for symbol, task_data in symbol_tasks.items():
                if len(task_data) > 10:  # Достаточно данных
                    sample = task_data[:10]  # Few-shot learning
                    agent.meta_learning_update(symbol, sample)
        
        # Evolutionary step каждые N эпох
        if enable_evolution and epoch % 3 == 0 and epoch > 0:
            print(f"\nEvolutionary optimization...")
            # Обновляем fitness популяции на основе последних результатов
            avg_reward = np.mean(epoch_rewards)
            for i in range(len(agent.population_fitness)):
                agent.population_fitness[i] = avg_reward + np.random.randn() * 0.1
            
            agent.evolutionary_step()
        
        # Статистика эпохи
        avg_reward = np.mean(epoch_rewards)
        accuracy = epoch_correct / len(training_data) * 100
        
        agent.training_rewards.append(avg_reward)
        agent.episode_count += 1
        agent.decay_epsilon()
        
        diagnostics = agent.get_diagnostics()
        
        print(f"\n{'='*80}")
        print(f"РЕЗУЛЬТАТЫ ЭПОХИ {epoch + 1}")
        print(f"{'='*80}")
        print(f"Средняя награда: {avg_reward:.4f}")
        print(f"Точность: {accuracy:.1f}%")
        print(f"Epsilon: {diagnostics['epsilon']:.4f}")
        print(f"Difficulty level: {diagnostics['difficulty_level']:.2f}")
        print(f"Protagonist policy size: {diagnostics['protagonist_policy_size']}")
        print(f"Adversary policy size: {diagnostics['adversary_policy_size']}")
        print(f"Evolution generation: {diagnostics['evolution_generation']}")
        
        if len(agent.training_losses) > 0:
            print(f"Avg loss (last 100): {diagnostics['avg_recent_loss']:.4f}")
        
        print(f"{'='*80}\n")
        
        # Checkpoint каждые 2 эпохи
        if (epoch + 1) % 2 == 0:
            checkpoint_path = f"seal_checkpoints/seal_epoch_{epoch+1}_{datetime.now().strftime('%Y%m%d_%H%M%S')}.json"
            agent.save_checkpoint(checkpoint_path)
    
    # Финальный checkpoint
    final_checkpoint = f"seal_checkpoints/seal_final_{datetime.now().strftime('%Y%m%d_%H%M%S')}.json"
    agent.save_checkpoint(final_checkpoint)
    
    # Визуализация прогресса
    plot_seal_training(agent)
    
    print(f"\n{'='*80}")
    print(f"SEAL ОБУЧЕНИЕ ЗАВЕРШЕНО")
    print(f"{'='*80}")
    print(f"Финальная точность: {accuracy:.1f}%")
    print(f"Финальный epsilon: {diagnostics['epsilon']:.4f}")
    print(f"Финальная сложность: {diagnostics['difficulty_level']:.2f}")
    print(f"Всего эволюционных поколений: {diagnostics['evolution_generation']}")
    print(f"Чекпоинт сохранён: {final_checkpoint}\n")
    
    return agent


def plot_seal_training(agent):
    """Визуализация процесса обучения SEAL"""
    fig, axes = plt.subplots(2, 2, figsize=(14, 10))
    fig.patch.set_facecolor('#0a0a0a')
    
    # 1. Rewards (protagonist vs adversary)
    ax1 = axes[0, 0]
    ax1.set_facecolor('#0a0a0a')
    if len(agent.training_rewards) > 0:
        ax1.plot(agent.training_rewards, color='#1E90FF', linewidth=2, label='Protagonist')
    if len(agent.adversarial_rewards) > 0:
        ax1.plot(agent.adversarial_rewards, color='#FF4500', linewidth=2, alpha=0.7, label='Adversary')
    ax1.set_title('Rewards: Protagonist vs Adversary', color='white', fontsize=12, fontweight='bold')
    ax1.set_xlabel('Episode', color='white')
    ax1.set_ylabel('Reward', color='white')
    ax1.tick_params(colors='white')
    ax1.grid(alpha=0.2, color='gray')
    ax1.legend(facecolor='#0a0a0a', edgecolor='white', labelcolor='white')
    for spine in ax1.spines.values():
        spine.set_color('white')
    
    # 2. Training Loss
    ax2 = axes[0, 1]
    ax2.set_facecolor('#0a0a0a')
    if len(agent.training_losses) > 0:
        ax2.plot(agent.training_losses, color='#32CD32', linewidth=2)
    ax2.set_title('Training Loss', color='white', fontsize=12, fontweight='bold')
    ax2.set_xlabel('Update Step', color='white')
    ax2.set_ylabel('Loss', color='white')
    ax2.tick_params(colors='white')
    ax2.grid(alpha=0.2, color='gray')
    for spine in ax2.spines.values():
        spine.set_color('white')
    
    # 3. Evolutionary Fitness
    ax3 = axes[1, 0]
    ax3.set_facecolor('#0a0a0a')
    if len(agent.best_fitness_history) > 0:
        ax3.plot(agent.best_fitness_history, color='#FFD700', linewidth=2, marker='o')
    ax3.set_title('Evolutionary Best Fitness', color='white', fontsize=12, fontweight='bold')
    ax3.set_xlabel('Generation', color='white')
    ax3.set_ylabel('Fitness', color='white')
    ax3.tick_params(colors='white')
    ax3.grid(alpha=0.2, color='gray')
    for spine in ax3.spines.values():
        spine.set_color('white')
    
    # 4. Curriculum Difficulty
    ax4 = axes[1, 1]
    ax4.set_facecolor('#0a0a0a')
    # Восстанавливаем историю difficulty (записываем через эпизоды)
    difficulty_history = [agent.difficulty_level] * len(agent.training_rewards)
    if len(difficulty_history) > 0:
        ax4.plot(difficulty_history, color='#FF69B4', linewidth=2)
    ax4.set_title('Curriculum Difficulty Level', color='white', fontsize=12, fontweight='bold')
    ax4.set_xlabel('Episode', color='white')
    ax4.set_ylabel('Difficulty', color='white')
    ax4.tick_params(colors='white')
    ax4.grid(alpha=0.2, color='gray')
    for spine in ax4.spines.values():
        spine.set_color('white')
    
    plt.tight_layout()
    
    filename = f"charts/seal_training_{datetime.now().strftime('%Y%m%d_%H%M%S')}.png"
    plt.savefig(filename, dpi=100, facecolor='#0a0a0a', edgecolor='none', bbox_inches='tight')
    print(f"\nГрафик обучения сохранён: {filename}")
    plt.close()


# ====================== ПАРСИНГ ======================
def parse_answer(text: str) -> dict:
    """Парсинг ответа модели с прогнозом цены"""
    prob = re.search(r"(?:УВЕРЕННОСТЬ|ВЕРОЯТНОСТЬ)[\s:]*(\d+)", text, re.I)
    direction = re.search(r"\b(UP|DOWN)\b", text, re.I)
    price_pred = re.search(r"ПРОГНОЗ ЦЕНЫ.*?(\d+\.\d+)", text, re.I)
    
    p = int(prob.group(1)) if prob else 50
    d = direction.group(1).upper() if direction else "DOWN"
    target_price = float(price_pred.group(1)) if price_pred else None
    
    return {"prob": p, "dir": d, "target_price": target_price}


# ====================== ВИЗУАЛИЗАЦИЯ ======================
def plot_results(balance_hist, equity_hist, slots):
    """График equity с фиксированной шириной 700px"""
    DPI = 100
    WIDTH_PX = 700
    HEIGHT_PX = 350
    
    fig = plt.figure(figsize=(WIDTH_PX / DPI, HEIGHT_PX / DPI), dpi=DPI)
    
    min_length = min(len(equity_hist), len(slots))
    dates = [s['datetime'] for s in slots[:min_length]]
    equity_to_plot = equity_hist[:min_length]
    
    plt.plot(dates, equity_to_plot, color='#1E90FF', linewidth=3.5, label='Equity')
    plt.title('Equity Curve', fontsize=16, fontweight='bold', color='white')
    plt.xlabel('Time', color='white')
    plt.ylabel('Balance ($)', color='white')
    
    ax = plt.gca()
    ax.set_facecolor('#0a0a0a')
    ax.spines['bottom'].set_color('white')
    ax.spines['top'].set_color('none')
    ax.spines['right'].set_color('none')
    ax.spines['left'].set_color('white')
    ax.tick_params(colors='white')
    plt.grid(alpha=0.2, color='gray')
    plt.xticks(rotation=45)
    
    plt.legend(facecolor='#0a0a0a', edgecolor='white', labelcolor='white')
    plt.tight_layout(pad=2.0)
    
    filename = f"charts/equity_{datetime.now().strftime('%Y%m%d_%H%M%S')}.png"
    plt.savefig(filename, dpi=DPI, facecolor='#0a0a0a', edgecolor='none', 
                bbox_inches='tight', pad_inches=0.1)
    print(f"\nГрафик сохранён: {filename} ({WIDTH_PX}×{HEIGHT_PX} px)")
    plt.close()


def calculate_max_drawdown(equity):
    """Расчёт максимальной просадки"""
    if len(equity) == 0:
        return 0
    peak = np.maximum.accumulate(equity)
    dd = (peak - equity) / (peak + 1e-8)
    return np.max(dd) * 100


# ====================== ФОРВАРДНЫЙ ТЕСТ ======================
def forward_test():
    """
    Форвардный тест на OUT-OF-SAMPLE данных
    """
    print("\n" + "=" * 80)
    print("ФОРВАРДНЫЙ ТЕСТ (OUT-OF-SAMPLE)")
    print("=" * 80)
    
    if not mt5 or not mt5.initialize():
        print("MT5 не подключен → используем синтетику")
        mock = True
    else:
        mock = False
        print("MT5 подключен")
    
    # Разделение данных: train/test
    # Train: последние BACKTEST_DAYS дней
    # Forward test: следующие FORWARD_TEST_DAYS дней
    
    end_train = datetime.now() - timedelta(days=FORWARD_TEST_DAYS)
    start_train = end_train - timedelta(days=BACKTEST_DAYS)
    
    end_forward = datetime.now()
    start_forward = end_train
    
    print(f"\nПериод обучения: {start_train.strftime('%Y-%m-%d')} - {end_train.strftime('%Y-%m-%d')}")
    print(f"Период форвардного теста: {start_forward.strftime('%Y-%m-%d')} - {end_forward.strftime('%Y-%m-%d')}\n")
    
    # Загрузка данных для форвардного теста
    data = {}
    
    for sym in SYMBOLS:
        if not mock:
            rates = mt5.copy_rates_range(sym, TIMEFRAME, start_forward, end_forward)
            if rates is None or len(rates) == 0:
                continue
            df = pd.DataFrame(rates)
            df["time"] = pd.to_datetime(df["time"], unit="s")
        else:
            dates = pd.date_range(start_forward, end_forward, freq="15min")
            close = 1.0800 + np.cumsum(np.random.randn(len(dates)) * 0.0002)
            df = pd.DataFrame({
                "time": dates,
                "open": close + np.random.randn(len(dates)) * 0.0001,
                "high": close + abs(np.random.randn(len(dates))) * 0.0003,
                "low": close - abs(np.random.randn(len(dates))) * 0.0003,
                "close": close,
                "tick_volume": np.random.randint(1000, 10000, len(dates)),
            })
        
        df.set_index("time", inplace=True)
        
        if len(df) > LOOKBACK + PREDICTION_HORIZON:
            data[sym] = df
            print(f"{sym}: {len(df)} баров")
    
    if not data:
        print("Нет данных для форвардного теста!")
        return
    
    # Инициализация
    balance = INITIAL_BALANCE
    trades = []
    balance_hist = [balance]
    equity_hist = [balance]
    slots = [{"datetime": start_forward}]
    
    SPREAD_PIPS = 2
    SWAP_LONG = -0.5
    SWAP_SHORT = -0.3
    
    print(f"\nСтарт форвардного теста...")
    print(f"Начальный баланс: ${balance:,.2f}\n")
    
    # Проверка доступности модели
    use_ai = False
    if ollama:
        try:
            ollama.list()
            use_ai = True
            print("Модель Ollama активна\n")
        except:
            print("Ollama недоступен, простая логика\n")
    
    # Основной цикл форвардного теста
    main_symbol = list(data.keys())[0]
    main_data = data[main_symbol]
    total_bars = len(main_data)
    
    analysis_points = list(range(LOOKBACK, total_bars - PREDICTION_HORIZON, PREDICTION_HORIZON))
    
    print(f"Точек анализа: {len(analysis_points)}\n")
    
    for point_idx, current_idx in enumerate(analysis_points):
        current_time = main_data.index[current_idx]
        
        for sym in SYMBOLS:
            if sym not in data:
                continue
            
            # КРИТИЧНО: используем только данные ДО current_idx
            historical_data = data[sym].iloc[:current_idx + 1].copy()
            
            if len(historical_data) < LOOKBACK:
                continue
            
            df_with_features = calculate_features(historical_data)
            if len(df_with_features) == 0:
                continue
            
            row = df_with_features.iloc[-1]
            
            if not mock:
                symbol_info = mt5.symbol_info(sym)
                if symbol_info is None:
                    continue
                point = symbol_info.point
                contract_size = symbol_info.trade_contract_size
            else:
                point = 0.0001
                contract_size = 100000
            
            prompt = f"""{sym} {current_time.strftime('%Y-%m-%d %H:%M')}
Текущая цена: {row['close']:.5f}
RSI: {row['RSI']:.1f}
MACD: {row['MACD']:.6f}
ATR: {row['ATR']:.5f}
Объёмы: {row['vol_ratio']:.2f}x
BB позиция: {row['BB_position']:.2f}
Stochastic K: {row['Stoch_K']:.1f}
Проанализируй и дай точный прогноз цены через 24 часа."""
            
            try:
                if use_ai:
                    resp = ollama.generate(model=MODEL_NAME, prompt=prompt, options={"temperature": 0.3})
                    result = parse_answer(resp["response"])
                else:
                    rsi_signal = 1 if row['RSI'] < 50 else -1
                    macd_signal = 1 if row['MACD'] > 0 else -1
                    combined = rsi_signal + macd_signal
                    result = {
                        "prob": min(95, max(65, 70 + abs(combined) * 10)),
                        "dir": "UP" if combined > 0 else "DOWN",
                        "target_price": None
                    }
                
                if result["prob"] < MIN_PROB:
                    continue
                
                entry_price = row['close'] + SPREAD_PIPS * point if result["dir"] == "UP" else row['close']
                
                exit_idx = current_idx + PREDICTION_HORIZON
                if exit_idx >= len(data[sym]):
                    continue
                
                exit_row = data[sym].iloc[exit_idx]
                exit_price = exit_row['close'] if result["dir"] == "UP" else exit_row['close'] + SPREAD_PIPS * point
                
                price_move_pips = (exit_price - entry_price) / point if result["dir"] == "UP" else (entry_price - exit_price) / point
                
                risk_amount = balance * RISK_PER_TRADE
                atr_pips = row['ATR'] / point
                stop_loss_pips = max(20, atr_pips * 2)
                lot_size = risk_amount / (stop_loss_pips * point * contract_size)
                lot_size = max(0.01, min(lot_size, 10.0))
                
                profit_pips = price_move_pips
                profit_usd = profit_pips * point * contract_size * lot_size
                swap_cost = SWAP_LONG if result["dir"] == "UP" else SWAP_SHORT
                swap_cost = swap_cost * (lot_size / 0.01)
                profit_usd -= swap_cost
                profit_usd -= SLIPPAGE * point * contract_size * lot_size
                
                balance += profit_usd
                
                actual_direction = "UP" if (exit_row['close'] > row['close']) else "DOWN"
                correct = (result["dir"] == actual_direction)
                
                trades.append({
                    "time": current_time,
                    "symbol": sym,
                    "direction": result["dir"],
                    "prob": result["prob"],
                    "entry_price": entry_price,
                    "exit_price": exit_price,
                    "lot_size": lot_size,
                    "profit_pips": profit_pips,
                    "profit_usd": profit_usd,
                    "balance": balance,
                    "correct": correct
                })
                
                status = "✓" if correct else "✗"
                print(f"{status} {current_time.strftime('%m-%d %H:%M')} | {sym} {result['dir']} {result['prob']}% | "
                      f"{entry_price:.5f}→{exit_price:.5f} | {profit_pips:+.1f}p ${profit_usd:+.2f} | ${balance:,.2f}")
            
            except Exception as e:
                log.error(f"Ошибка {sym}: {e}")
        
        balance_hist.append(balance)
        equity_hist.append(balance)
        slots.append({"datetime": current_time})
    
    # Результаты
    print(f"\n{'='*80}")
    print("РЕЗУЛЬТАТЫ ФОРВАРДНОГО ТЕСТА")
    print(f"{'='*80}")
    print(f"Период: {start_forward.strftime('%Y-%m-%d')} - {end_forward.strftime('%Y-%m-%d')}")
    print(f"Всего сделок: {len(trades)}")
    print(f"Начальный баланс: ${INITIAL_BALANCE:,.2f}")
    print(f"Конечный баланс: ${balance:,.2f}")
    print(f"Прибыль/убыток: ${balance - INITIAL_BALANCE:+,.2f} ({((balance/INITIAL_BALANCE - 1) * 100):+.2f}%)")
    
    if trades:
        wins = sum(1 for t in trades if t['profit_usd'] > 0)
        losses = len(trades) - wins
        win_rate = wins / len(trades) * 100
        
        print(f"\nСТАТИСТИКА:")
        print(f"Прибыльных: {wins} ({win_rate:.1f}%)")
        print(f"Убыточных: {losses} ({100 - win_rate:.1f}%)")
        
        if wins > 0:
            avg_win = np.mean([t['profit_usd'] for t in trades if t['profit_usd'] > 0])
            print(f"Средняя прибыль: ${avg_win:.2f}")
        
        if losses > 0:
            avg_loss = np.mean([t['profit_usd'] for t in trades if t['profit_usd'] < 0])
            print(f"Средний убыток: ${avg_loss:.2f}")
        
        if wins > 0 and losses > 0:
            profit_factor = abs(sum(t['profit_usd'] for t in trades if t['profit_usd'] > 0)) / abs(sum(t['profit_usd'] for t in trades if t['profit_usd'] < 0))
            print(f"Профит-фактор: {profit_factor:.2f}")
        
        max_dd = calculate_max_drawdown(np.array(equity_hist))
        print(f"Макс. просадка: {max_dd:.2f}%")
        
        correct_predictions = sum(1 for t in trades if t['correct'])
        prediction_accuracy = correct_predictions / len(trades) * 100
        print(f"Точность прогнозов: {prediction_accuracy:.1f}%")
        
        if len(equity_hist) > 1:
            plot_results(balance_hist, equity_hist, slots)
    
    if not mock:
        mt5.shutdown()
    
    print(f"\n{'='*80}\n")


# ====================== 3. БЭКТЕСТ ======================
def backtest():
    """Бэктест без утечки данных"""
    print("\n" + "=" * 80)
    print("3. БЭКТЕСТ (БЕЗ УТЕЧКИ ДАННЫХ)")
    print("=" * 80)
    
    if not mt5 or not mt5.initialize():
        print("MT5 не подключен → синтетика")
        mock = True
    else:
        mock = False
        print("MT5 подключен")
    
    end = datetime.now().replace(second=0, microsecond=0)
    start = end - timedelta(days=BACKTEST_DAYS)
    
    data = {}
    print(f"\nЗагрузка {start.strftime('%Y-%m-%d')} - {end.strftime('%Y-%m-%d')}...")
    
    for sym in SYMBOLS:
        if not mock:
            rates = mt5.copy_rates_range(sym, TIMEFRAME, start, end)
            if rates is None or len(rates) == 0:
                continue
            df = pd.DataFrame(rates)
            df["time"] = pd.to_datetime(df["time"], unit="s")
        else:
            dates = pd.date_range(start, end, freq="15min")
            close = 1.0800 + np.cumsum(np.random.randn(len(dates)) * 0.0002)
            df = pd.DataFrame({
                "time": dates,
                "open": close + np.random.randn(len(dates)) * 0.0001,
                "high": close + abs(np.random.randn(len(dates))) * 0.0003,
                "low": close - abs(np.random.randn(len(dates))) * 0.0003,
                "close": close,
                "tick_volume": np.random.randint(1000, 10000, len(dates)),
            })
        
        df.set_index("time", inplace=True)
        
        if len(df) > LOOKBACK + PREDICTION_HORIZON:
            data[sym] = df
            print(f"{sym}: {len(df)} баров")
    
    if not data:
        print("Нет данных!")
        return
    
    balance = INITIAL_BALANCE
    trades = []
    balance_hist = [balance]
    equity_hist = [balance]
    slots = [{"datetime": start}]
    
    SPREAD_PIPS = 2
    SWAP_LONG = -0.5
    SWAP_SHORT = -0.3
    
    print(f"\nНачало: ${balance:,.2f}\n")
    
    use_ai = False
    if ollama:
        try:
            ollama.list()
            use_ai = True
            print("Ollama активна\n")
        except:
            print("Ollama недоступна\n")
    
    main_symbol = list(data.keys())[0]
    main_data = data[main_symbol]
    total_bars = len(main_data)
    analysis_points = list(range(LOOKBACK, total_bars - PREDICTION_HORIZON, PREDICTION_HORIZON))
    
    print(f"Точек анализа: {len(analysis_points)}\n")
    
    for point_idx, current_idx in enumerate(analysis_points):
        current_time = main_data.index[current_idx]
        
        for sym in SYMBOLS:
            if sym not in data:
                continue
            
            historical_data = data[sym].iloc[:current_idx + 1].copy()
            
            if len(historical_data) < LOOKBACK:
                continue
            
            df_with_features = calculate_features(historical_data)
            if len(df_with_features) == 0:
                continue
            
            row = df_with_features.iloc[-1]
            
            if not mock:
                symbol_info = mt5.symbol_info(sym)
                if symbol_info is None:
                    continue
                point = symbol_info.point
                contract_size = symbol_info.trade_contract_size
            else:
                point = 0.0001
                contract_size = 100000
            
            prompt = f"""{sym} {current_time.strftime('%Y-%m-%d %H:%M')}
Текущая цена: {row['close']:.5f}
RSI: {row['RSI']:.1f}
MACD: {row['MACD']:.6f}
ATR: {row['ATR']:.5f}
Объёмы: {row['vol_ratio']:.2f}x
BB позиция: {row['BB_position']:.2f}
Stochastic K: {row['Stoch_K']:.1f}
Проанализируй и дай точный прогноз цены через 24 часа."""
            
            try:
                if use_ai:
                    resp = ollama.generate(model=MODEL_NAME, prompt=prompt, options={"temperature": 0.3})
                    result = parse_answer(resp["response"])
                else:
                    rsi_signal = 1 if row['RSI'] < 50 else -1
                    macd_signal = 1 if row['MACD'] > 0 else -1
                    combined = rsi_signal + macd_signal
                    result = {
                        "prob": min(95, max(65, 70 + abs(combined) * 10)),
                        "dir": "UP" if combined > 0 else "DOWN",
                        "target_price": None
                    }
                
                if result["prob"] < MIN_PROB:
                    continue
                
                entry_price = row['close'] + SPREAD_PIPS * point if result["dir"] == "UP" else row['close']
                exit_idx = current_idx + PREDICTION_HORIZON
                
                if exit_idx >= len(data[sym]):
                    continue
                
                exit_row = data[sym].iloc[exit_idx]
                exit_price = exit_row['close'] if result["dir"] == "UP" else exit_row['close'] + SPREAD_PIPS * point
                
                price_move_pips = (exit_price - entry_price) / point if result["dir"] == "UP" else (entry_price - exit_price) / point
                
                risk_amount = balance * RISK_PER_TRADE
                atr_pips = row['ATR'] / point
                stop_loss_pips = max(20, atr_pips * 2)
                lot_size = risk_amount / (stop_loss_pips * point * contract_size)
                lot_size = max(0.01, min(lot_size, 10.0))
                profit_pips = price_move_pips
                profit_usd = profit_pips * point * contract_size * lot_size
                swap_cost = SWAP_LONG if result["dir"] == "UP" else SWAP_SHORT
                swap_cost = swap_cost * (lot_size / 0.01)
                profit_usd -= swap_cost
                profit_usd -= SLIPPAGE * point * contract_size * lot_size
                
                balance += profit_usd
                
                actual_direction = "UP" if (exit_row['close'] > row['close']) else "DOWN"
                correct = (result["dir"] == actual_direction)
                
                trades.append({
                    "time": current_time,
                    "symbol": sym,
                    "direction": result["dir"],
                    "prob": result["prob"],
                    "entry_price": entry_price,
                    "exit_price": exit_price,
                    "lot_size": lot_size,
                    "profit_pips": profit_pips,
                    "profit_usd": profit_usd,
                    "balance": balance,
                    "correct": correct
                })
                
                status = "✓" if correct else "✗"
                print(f"{status} {current_time.strftime('%m-%d %H:%M')} | {sym} {result['dir']} {result['prob']}% | "
                      f"{profit_pips:+.1f}p ${profit_usd:+.2f} | ${balance:,.2f}")
            
            except Exception as e:
                log.error(f"Ошибка {sym}: {e}")
        
        balance_hist.append(balance)
        equity_hist.append(balance)
        slots.append({"datetime": current_time})
    
    print(f"\n{'='*80}")
    print("РЕЗУЛЬТАТЫ БЭКТЕСТА")
    print(f"{'='*80}")
    print(f"Сделок: {len(trades)}")
    print(f"Начало: ${INITIAL_BALANCE:,.2f}")
    print(f"Конец: ${balance:,.2f}")
    print(f"P&L: ${balance - INITIAL_BALANCE:+,.2f} ({((balance/INITIAL_BALANCE - 1) * 100):+.2f}%)")
    
    if trades:
        wins = sum(1 for t in trades if t['profit_usd'] > 0)
        win_rate = wins / len(trades) * 100
        
        print(f"\nПрибыльных: {wins} ({win_rate:.1f}%)")
        print(f"Убыточных: {len(trades) - wins}")
        
        if wins > 0:
            print(f"Средняя прибыль: ${np.mean([t['profit_usd'] for t in trades if t['profit_usd'] > 0]):.2f}")
        
        if len(trades) - wins > 0:
            print(f"Средний убыток: ${np.mean([t['profit_usd'] for t in trades if t['profit_usd'] < 0]):.2f}")
        
        max_dd = calculate_max_drawdown(np.array(equity_hist))
        print(f"Макс. просадка: {max_dd:.2f}%")
        
        if len(equity_hist) > 1:
            plot_results(balance_hist, equity_hist, slots)
    
    if not mock:
        mt5.shutdown()


# ====================== ФАЙНТЬЮН ======================
def finetune_with_ollama(dataset_path: str):
    """Файнтьюн модели через Ollama"""
    print("\n" + "=" * 80)
    print("ФАЙНТЬЮН ЧЕРЕЗ OLLAMA")
    print("=" * 80)
    
    try:
        subprocess.run(["ollama", "--version"], check=True, capture_output=True)
    except:
        print("Ollama не установлен!")
        return
    
    with open(dataset_path, 'r', encoding='utf-8') as f:
        training_data = [json.loads(line) for line in f]
    
    training_sample = training_data[:min(100, len(training_data))]
    
    modelfile_content = f"""FROM {BASE_MODEL}
PARAMETER temperature 0.55
PARAMETER top_p 0.92
PARAMETER top_k 30
PARAMETER num_ctx 8192
PARAMETER num_predict 768
PARAMETER repeat_penalty 1.1
SYSTEM \"\"\"
Ты — ShtencoAiTrader-3B-Ultra-Analyst v3 — элитный аналитик валютного рынка с прогнозами на 24 часа.
СТРОГИЕ ПРАВИЛА:
1. Только UP или DOWN — никакого FLAT, боковика, неуверенности
2. Уверенность всегда 65-98%
3. ОБЯЗАТЕЛЬНО давай прогноз цены через 24 часа в формате: X.XXXXX (±NN пунктов)
4. Детальный анализ каждого индикатора с учётом суточного таймфрейма
5. Конкретные рекомендации с целевой ценой
\"\"\"
"""
    
    for i, example in enumerate(training_sample[:500], 1):
        modelfile_content += f"""
MESSAGE user \"\"\"{example['prompt']}\"\"\"
MESSAGE assistant \"\"\"{example['response']}\"\"\"
"""
    
    modelfile_path = "Modelfile_finetune"
    with open(modelfile_path, 'w', encoding='utf-8') as f:
        f.write(modelfile_content)
    
    print(f"Modelfile создан ({len(training_sample)} примеров)")
    print(f"\nСоздание модели {MODEL_NAME}...\n")
    
    try:
        result = subprocess.run(
            ["ollama", "create", MODEL_NAME, "-f", modelfile_path],
            check=True,
            capture_output=True,
            text=True
        )
        print(result.stdout)
        print(f"\nМодель {MODEL_NAME} создана!")
        
        os.remove(modelfile_path)
        
        print(f"\nГОТОВО!")
    
    except subprocess.CalledProcessError as e:
        print(f"Ошибка: {e}")


# ====================== 2. ФАЙНТЬЮН РЕЖИМ ======================
def mode_finetune():
    """Режим файнтьюна"""
    print("\n" + "=" * 80)
    print("2. ФАЙНТЬЮН МОДЕЛИ + SEAL RL")
    print("=" * 80)
    print("\nВарианты:")
    print("A. Реальные данные MT5 + SEAL")
    print("B. Синтетика + SEAL")
    print("C. Существующий датасет + SEAL")
    print("D. Только датасет (без обучения)")
    
    choice = input("\nВыбор (A/B/C/D): ").strip().upper()
    
    if choice == "A":
        dataset = generate_real_dataset_from_mt5(FINETUNE_SAMPLES)
        dataset_path = save_dataset(dataset, "dataset/finetune_real_mt5.jsonl")
        
        # Ollama файнтьюн
        finetune_with_ollama(dataset_path)
        
        # SEAL RL обучение
        print("\nЗапуск SEAL RL обучения...")
        train_seal_agent(dataset_path, epochs=10)
    
    elif choice == "B":
        dataset = generate_synthetic_dataset(FINETUNE_SAMPLES)
        dataset_path = save_dataset(dataset)
        
        finetune_with_ollama(dataset_path)
        train_seal_agent(dataset_path, epochs=10)
    
    elif choice == "C":
        dataset_path = input("Путь к датасету: ").strip()
        if os.path.exists(dataset_path):
            finetune_with_ollama(dataset_path)
            train_seal_agent(dataset_path, epochs=10)
        else:
            print(f"Файл {dataset_path} не найден")
    
    elif choice == "D":
        dtype = input("1=MT5, 2=Синтетика: ").strip()
        num_samples = int(input("Кол-во примеров (1000): ").strip() or "1000")
        
        if dtype == "1":
            dataset = generate_real_dataset_from_mt5(num_samples)
            save_dataset(dataset, "dataset/finetune_real_mt5.jsonl")
        else:
            dataset = generate_synthetic_dataset(num_samples)
            save_dataset(dataset)
        
        print("\nДатасет готов!")


# ====================== 4. ЛАЙВ ======================
def live():
    """Живая торговля"""
    print("\n" + "=" * 80)
    print("4. ЖИВАЯ ТОРГОВЛЯ")
    print("=" * 80)
    
    if not mt5 or not mt5.initialize():
        print("MT5 не найден")
        return
    
    account_info = mt5.account_info()
    if account_info is None:
        print("Ошибка подключения к счёту")
        return
    
    print(f"Счёт: {account_info.login}")
    print(f"Баланс: ${account_info.balance:.2f}")
    print(f"Эквити: ${account_info.equity:.2f}")
    
    confirm = input("\nРЕАЛЬНАЯ ТОРГОВЛЯ! Продолжить? (YES): ").strip()
    if confirm != "YES":
        print("Отменено")
        return
    
    print("\nЗапуск... (Ctrl+C = стоп)\n")
    
    open_positions = {}
    last_analysis_time = None
    
    while True:
        try:
            now = datetime.now()
            positions = mt5.positions_get()
            
            # Закрытие через 24ч
            if positions:
                for pos in positions:
                    if pos.magic == MAGIC:
                        open_time = datetime.fromtimestamp(pos.time)
                        if (now - open_time).total_seconds() >= 86400:
                            request = {
                                "action": mt5.TRADE_ACTION_DEAL,
                                "symbol": pos.symbol,
                                "volume": pos.volume,
                                "type": mt5.ORDER_TYPE_SELL if pos.type == mt5.POSITION_TYPE_BUY else mt5.ORDER_TYPE_BUY,
                                "position": pos.ticket,
                                "price": mt5.symbol_info_tick(pos.symbol).bid if pos.type == mt5.POSITION_TYPE_BUY else mt5.symbol_info_tick(pos.symbol).ask,
                                "deviation": SLIPPAGE,
                                "magic": MAGIC,
                                "comment": "24h close",
                                "type_time": mt5.ORDER_TIME_GTC,
                                "type_filling": mt5.ORDER_FILLING_IOC,
                            }
                            result = mt5.order_send(request)
                            if result.retcode == mt5.TRADE_RETCODE_DONE:
                                print(f"Закрыта {pos.symbol} через 24ч | ${pos.profit:+.2f}")
                                if pos.ticket in open_positions:
                                    del open_positions[pos.ticket]
            
            # Новый анализ каждые 24ч
            if last_analysis_time is None or (now - last_analysis_time).total_seconds() >= 86400:
                last_analysis_time = now
                print(f"\n{'='*80}")
                print(f"АНАЛИЗ: {now.strftime('%Y-%m-%d %H:%M')}")
                print(f"{'='*80}\n")
                
                for sym in SYMBOLS:
                    has_position = any(p.symbol == sym and p.magic == MAGIC for p in (positions or []))
                    if has_position:
                        continue
                    
                    rates = mt5.copy_rates_from_pos(sym, TIMEFRAME, 0, LOOKBACK)
                    if rates is None or len(rates) == 0:
                        continue
                    
                    df = pd.DataFrame(rates)
                    df["time"] = pd.to_datetime(df["time"], unit="s")
                    df.set_index("time", inplace=True)
                    df = calculate_features(df)
                    
                    if len(df) == 0:
                        continue
                    
                    row = df.iloc[-1]
                    symbol_info = mt5.symbol_info(sym)
                    if symbol_info is None or not symbol_info.visible:
                        continue
                    
                    prompt = f"""{sym} {now.strftime('%Y-%m-%d %H:%M')}
Текущая цена: {row['close']:.5f}
RSI: {row['RSI']:.1f}
MACD: {row['MACD']:.6f}
ATR: {row['ATR']:.5f}
Объёмы: {row['vol_ratio']:.2f}x
BB позиция: {row['BB_position']:.2f}
Stochastic K: {row['Stoch_K']:.1f}
Проанализируй и дай точный прогноз цены через 24 часа."""
                    
                    resp = ollama.generate(model=MODEL_NAME, prompt=prompt, options={"temperature": 0.3})
                    result = parse_answer(resp["response"])
                    
                    print(f"{sym}: {result['dir']} ({result['prob']}%)")
                    
                    if result["prob"] < MIN_PROB:
                        continue
                    
                    order_type = mt5.ORDER_TYPE_BUY if result["dir"] == "UP" else mt5.ORDER_TYPE_SELL
                    tick = mt5.symbol_info_tick(sym)
                    if tick is None:
                        continue
                    price = tick.ask if result["dir"] == "UP" else tick.bid
                    
                    risk_amount = mt5.account_info().balance * RISK_PER_TRADE
                    point = symbol_info.point
                    atr_pips = row['ATR'] / point
                    stop_loss_pips = max(200, atr_pips * 2)
                    lot_size = risk_amount / (stop_loss_pips * point * symbol_info.trade_contract_size)
                    lot_step = symbol_info.volume_step
                    lot_size = round(lot_size / lot_step) * lot_step
                    lot_size = max(symbol_info.volume_min, min(lot_size, symbol_info.volume_max))
                    
                    sl = price - stop_loss_pips * point if result["dir"] == "UP" else price + stop_loss_pips * point
                    tp = price + stop_loss_pips * 3 * point if result["dir"] == "UP" else price - stop_loss_pips * 3 * point
                    
                    request = {
                        "action": mt5.TRADE_ACTION_DEAL,
                        "symbol": sym,
                        "volume": lot_size,
                        "type": order_type,
                        "price": price,
                        "sl": sl,
                        "tp": tp,
                        "deviation": SLIPPAGE,
                        "magic": MAGIC,
                        "comment": f"AI_{result['prob']}%",
                        "type_time": mt5.ORDER_TIME_GTC,
                        "type_filling": mt5.ORDER_FILLING_IOC,
                    }
                    
                    result_order = mt5.order_send(request)
                    if result_order.retcode == mt5.TRADE_RETCODE_DONE:
                        print(f" Открыто! Тикет: {result_order.order}\n")
                        open_positions[result_order.order] = {"symbol": sym, "open_time": now}
                
                print(f"Позиций: {len(open_positions)}\n")
            
            time.sleep(60)
        
        except KeyboardInterrupt:
            print("\nОстановка...")
            positions = mt5.positions_get(magic=MAGIC)
            if positions:
                for pos in positions:
                    request = {
                        "action": mt5.TRADE_ACTION_DEAL,
                        "symbol": pos.symbol,
                        "volume": pos.volume,
                        "type": mt5.ORDER_TYPE_SELL if pos.type == mt5.POSITION_TYPE_BUY else mt5.ORDER_TYPE_BUY,
                        "position": pos.ticket,
                        "price": mt5.symbol_info_tick(pos.symbol).bid if pos.type == mt5.POSITION_TYPE_BUY else mt5.symbol_info_tick(pos.symbol).ask,
                        "deviation": SLIPPAGE,
                        "magic": MAGIC,
                        "comment": "manual",
                        "type_time": mt5.ORDER_TIME_GTC,
                        "type_filling": mt5.ORDER_FILLING_IOC,
                    }
                    mt5.order_send(request)
            print("Стоп")
            break
        except Exception as e:
            log.error(f"Ошибка: {e}")
            time.sleep(60)
    
    mt5.shutdown()


# ====================== 5. ДАТАСЕТ ======================
def mode_dataset():
    """Генерация датасета"""
    print("\n" + "=" * 80)
    print("5. ГЕНЕРАЦИЯ ДАТАСЕТА")
    print("=" * 80)
    
    dtype = input("\n1=MT5, 2=Синтетика: ").strip()
    num_samples = int(input("Кол-во (1000): ").strip() or "1000")
    
    if dtype == "1":
        dataset = generate_real_dataset_from_mt5(num_samples)
        dataset_path = save_dataset(dataset, "dataset/finetune_real_mt5.jsonl")
    else:
        dataset = generate_synthetic_dataset(num_samples)
        dataset_path = save_dataset(dataset)
    
    print(f"\nДатасет: {dataset_path}")
    print(f"Примеров: {len(dataset)}")
    
    print("\n" + "=" * 80)
    print("ПРИМЕР:")
    print("=" * 80)
    example = dataset[0]
    print("\nПРОМПТ:")
    print(example['prompt'])
    print("\nОТВЕТ:")
    print(example['response'])


# ====================== 1. PUSH ======================
def mode_push():
    """Пуш базовой модели в Ollama"""
    print("\n" + "=" * 80)
    print("1. ПУШ БАЗОВОЙ МОДЕЛИ В OLLAMA")
    print("=" * 80)

    content = f"""FROM {BASE_MODEL}
PARAMETER temperature 0.55
PARAMETER top_p 0.92
PARAMETER top_k 30
PARAMETER num_ctx 8192
PARAMETER num_predict 768
SYSTEM \"\"\"
Ты — ShtencoAiTrader-3B-Ultra-Analyst v3 — лучший в мире аналитик валютного рынка.
Ты всегда даешь четкое направление: UP или DOWN. Слова FLAT, боковик, не уверен — полностью запрещены.
Ты ОБЯЗАТЕЛЬНО даёшь прогноз цены через 24 часа в формате: X.XXXXX (±NN пунктов)
Ты подробно разбираешь каждый индикатор (RSI, MACD, объемы, ATR, уровни, свечи и т.д.).
Формат ответа строго такой:
НАПРАВЛЕНИЕ: UP
УВЕРЕННОСТЬ: 87%
ПРОГНОЗ ЦЕНЫ ЧЕРЕЗ 24Ч: 1.08750 (+45 пунктов)
ПОЛНЫЙ АНАЛИЗ НА 24 ЧАСА:
- RSI: ...
- MACD: ...
- Объемы: ...
- ATR и волатильность: ...
- Уровни поддержки/сопротивления: ...
- Свечной паттерн: ...
ИТОГ: мощный бычий импульс с подтверждением по всем индикаторам, через 24ч цель 1.08750
Уверенность всегда 65–98%. Никаких сомнений.
\"\"\"
"""

    with open("Modelfile", "w", encoding="utf-8") as f:
        f.write(content)
    print("Modelfile создан")
    print("Скачивание базовой модели...")
    subprocess.run(["ollama", "pull", BASE_MODEL], check=True)

    print("Создание модели...")
    subprocess.run(["ollama", "create", MODEL_NAME, "-f", "Modelfile"], check=True)

    print("Пушим в реестр Ollama (5-20 минут)...")
    subprocess.run(["ollama", "push", MODEL_NAME], check=True)

    os.remove("Modelfile")
    print(f"\nГОТОВО! Модель доступна: https://ollama.com/{MODEL_NAME}")


# ====================== МЕНЮ ======================
def main():
    """Главное меню"""
    print("\n" + "=" * 80)
    print(" SHTENCO AI TRADER ULTRA 3B + SEAL RL")
    print(" Версия: 08.02.2026 (Balanced + SEAL MIT + Forward Test)")
    print("=" * 80)
    print("\nРЕЖИМЫ:")
    print("-" * 80)
    print("1 → Пуш базовой модели")
    print("2 → Файнтьюн (MT5/синтетика) + SEAL RL")
    print("3 → Бэктест на исторических данных")
    print("4 → Живая торговля MT5")
    print("5 → Генерация датасета")
    print("6 → ФОРВАРДНЫЙ ТЕСТ (out-of-sample)")
    print("7 → ТОЛЬКО SEAL RL обучение")
    print("-" * 80)
    
    choice = input("\nВыбор (1-7): ").strip()
    
    if choice == "1":
        mode_push()
    elif choice == "2":
        mode_finetune()
    elif choice == "3":
        backtest()
    elif choice == "4":
        live()
    elif choice == "5":
        mode_dataset()
    elif choice == "6":
        forward_test()
    elif choice == "7":
        dataset_path = input("Путь к датасету: ").strip()
        if os.path.exists(dataset_path):
            epochs = int(input("Кол-во эпох (10): ").strip() or "10")
            train_seal_agent(dataset_path, epochs=epochs)
        else:
            print(f"Файл {dataset_path} не найден")
    else:
        print("Неверный выбор")


if __name__ == "__main__":
    main()
 
