# ================================================
# ai_trader_quantum_fusion_bip39_seal.py
# КВАНТОВЫЙ ГИБРИД: Qiskit + CatBoost + LLM + BIP39 + SEAL
# Версия 21.01.2026 — MIT SEAL Self-Adapting Learning
# ================================================
# 
# SEAL (Self-Adapting Language Models) по методологии MIT:
# https://arxiv.org/abs/2506.10943
#
# При КАЖДОМ промпте к LLM происходит:
# 1. INNER LOOP: LLM генерирует self-edit (прогноз)
# 2. Результат сохраняется в experience buffer
# 3. OUTER LOOP: Периодически отбираются лучшие self-edits
# 4. Behavior Cloning: LLM дообучается на лучших примерах
#
# Это обеспечивает ПОСТОЯННОЕ САМООБУЧЕНИЕ модели!
# ================================================
import os
import re
import time
import json
import logging
import subprocess
from datetime import datetime, timedelta
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
from typing import List, Dict, Tuple
import hashlib
import base58
import pickle
from collections import deque
import random  # RL для SEAL
from dataclasses import dataclass, field

try:
    import MetaTrader5 as mt5
except ImportError:
    mt5 = None

try:
    import ollama
except ImportError:
    ollama = None

try:
    from catboost import CatBoostClassifier, Pool
    CATBOOST_AVAILABLE = True
except ImportError:
    CATBOOST_AVAILABLE = False
    print("⚠️ CatBoost не установлен: pip install catboost")

try:
    from qiskit import QuantumCircuit
    from qiskit_aer import AerSimulator
    from scipy.stats import entropy
    QISKIT_AVAILABLE = True
except ImportError:
    QISKIT_AVAILABLE = False
    print("⚠️ Qiskit не установлен: pip install qiskit qiskit-aer")

try:
    from mnemonic import Mnemonic
    BIP39_AVAILABLE = True
except ImportError:
    BIP39_AVAILABLE = False
    print("⚠️ Mnemonic не установлен: pip install mnemonic")

# ====================== КОНФИГ ======================
MODEL_NAME = "koshtenco/quantum-trader-fusion-bip39-3b"
BASE_MODEL = "llama3.2:3b"
SYMBOLS = ["EURUSD", "GBPUSD", "USDCHF", "USDCAD", "AUDUSD", "NZDUSD", "EURGBP", "AUDCHF"]
TIMEFRAME = mt5.TIMEFRAME_M15 if mt5 else None
LOOKBACK = 400
INITIAL_BALANCE = 270.0
RISK_PER_TRADE = 0.06
MIN_PROB = 60
LIVE_LOT = 0.02
MAGIC = 20251227
SLIPPAGE = 10

# Квантовые параметры
N_QUBITS = 8
N_SHOTS = 2048

# ====================== SEAL КОНФИГ (MIT Methodology) ======================
SEAL_CONFIG = {
    'enabled': True,                    # Включить SEAL самообучение
    'buffer_size': 10000,               # Размер experience buffer
    'top_k_ratio': 0.2,                 # Отбираем top 20% лучших self-edits
    'retrain_every': 50,                # Ретрейн каждые N промптов
    'min_samples_for_retrain': 20,      # Минимум примеров для ретрейна
    'reward_threshold': 0.0,            # Порог reward для "хороших" примеров
    'save_buffer_path': 'models/seal_buffer.pkl',
    'best_edits_path': 'models/seal_best_edits.jsonl',
}

# Файнтьюн параметры
FINETUNE_SAMPLES = 2000
BACKTEST_DAYS = 30
PREDICTION_HORIZON = 96  # 24 часа на M15


# ═══════════════════════════════════════════════════════════════════════════
# ЛЕГКИЙ RL-АГЕНТ ДЛЯ SEAL (НОВОЕ! ~100 строк)
# ═══════════════════════════════════════════════════════════════════════════
class LightweightRLAgent:
    """
    Легкий Q-learning агент для SEAL.
    Учится принимать решения: использовать прогноз, пропустить или снизить confidence.
    
    State: (entropy_bin, confidence_bin, recent_win_rate_bin)
    Actions: 0=USE, 1=SKIP, 2=REDUCE
    """
    
    def __init__(self, alpha=0.1, gamma=0.95, epsilon=0.2):
        self.alpha = alpha
        self.gamma = gamma
        self.epsilon = epsilon
        self.q_table = {}
        self.total_decisions = 0
        self.actions_count = {0: 0, 1: 0, 2: 0}
        log.info(f"🤖 RL Agent для SEAL инициализирован")
    
    def _discretize_state(self, quantum_entropy: float, confidence: float, 
                         recent_win_rate: float) -> tuple:
        """Дискретизация состояния"""
        e_bin = 0 if quantum_entropy < 2.0 else (1 if quantum_entropy < 4.0 else 2)
        c_bin = 0 if confidence < 50 else (1 if confidence < 75 else 2)
        w_bin = 0 if recent_win_rate < 0.4 else (1 if recent_win_rate < 0.6 else 2)
        return (e_bin, c_bin, w_bin)
    
    def select_action(self, quantum_entropy: float, confidence: float, 
                     recent_win_rate: float) -> int:
        """Epsilon-greedy action selection"""
        state = self._discretize_state(quantum_entropy, confidence, recent_win_rate)
        
        if state not in self.q_table:
            self.q_table[state] = [0.0, 0.0, 0.0]
        
        if random.random() < self.epsilon:
            action = random.randint(0, 2)
        else:
            action = int(np.argmax(self.q_table[state]))
        
        self.total_decisions += 1
        self.actions_count[action] += 1
        return action
    
    def update_q_value(self, state_tuple: tuple, action: int, reward: float, 
                      next_state_tuple: tuple):
        """Q-learning update"""
        if state_tuple not in self.q_table:
            self.q_table[state_tuple] = [0.0, 0.0, 0.0]
        if next_state_tuple not in self.q_table:
            self.q_table[next_state_tuple] = [0.0, 0.0, 0.0]
        
        current_q = self.q_table[state_tuple][action]
        max_next_q = max(self.q_table[next_state_tuple])
        target = reward + self.gamma * max_next_q
        
        self.q_table[state_tuple][action] = current_q + self.alpha * (target - current_q)
    
    def get_stats(self) -> Dict:
        """Статистика"""
        total = sum(self.actions_count.values())
        if total == 0:
            return {'total_decisions': 0}
        
        return {
            'total_decisions': self.total_decisions,
            'q_table_size': len(self.q_table),
            'epsilon': self.epsilon,
            'actions': {
                'USE': f"{self.actions_count[0]} ({self.actions_count[0]/total*100:.1f}%)",
                'SKIP': f"{self.actions_count[1]} ({self.actions_count[1]/total*100:.1f}%)",
                'REDUCE': f"{self.actions_count[2]} ({self.actions_count[2]/total*100:.1f}%)"
            }
        }
    
    def save(self, path: str):
        """Сохранение Q-table"""
        try:
            with open(path, 'wb') as f:
                pickle.dump({
                    'q_table': self.q_table,
                    'actions_count': self.actions_count,
                    'total_decisions': self.total_decisions
                }, f)
        except Exception as e:
            log.warning(f"🤖 RL: Ошибка сохранения: {e}")
    
    def load(self, path: str):
        """Загрузка Q-table"""
        if os.path.exists(path):
            try:
                with open(path, 'rb') as f:
                    data = pickle.load(f)
                    self.q_table = data.get('q_table', {})
                    self.actions_count = data.get('actions_count', {0: 0, 1: 0, 2: 0})
                    self.total_decisions = data.get('total_decisions', 0)
                log.info(f"🤖 RL: Загружена Q-table ({len(self.q_table)} states)")
            except Exception as e:
                log.warning(f"🤖 RL: Ошибка загрузки: {e}")


os.makedirs("logs", exist_ok=True)
os.makedirs("dataset", exist_ok=True)
os.makedirs("models", exist_ok=True)
os.makedirs("charts", exist_ok=True)

logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s | %(message)s",
    handlers=[
        logging.FileHandler("logs/quantum_fusion_bip39.log", encoding="utf-8"),
        logging.StreamHandler(),
    ],
)
log = logging.getLogger(__name__)

# ====================== BIP39 КОНВЕРТЕР ======================
class BIP39Converter:
    """
    Конвертер цен в BIP39 фразы через SHA256 и Base58
    Из первого скрипта — добавляет криптографический уровень
    """
    def __init__(self):
        if not BIP39_AVAILABLE:
            self.mnemo = None
            self.wordlist = []
            self.binary_to_word = {}
            self.word_to_binary = {}
            self.vocab_size = 0
            return
            
        self.mnemo = Mnemonic('english')
        self.wordlist = self.mnemo.wordlist
        self.binary_to_word = {format(i, '011b'): word for i, word in enumerate(self.wordlist)}
        self.word_to_binary = {word: format(i, '011b') for i, word in enumerate(self.wordlist)}
        self.vocab_size = len(self.wordlist)

    def binary_to_bip39(self, binary_sequence: str) -> List[str]:
        """Конвертация бинарной последовательности в BIP39 слова"""
        if not BIP39_AVAILABLE:
            return ['unknown'] * 12
            
        words = []
        for i in range(0, len(binary_sequence), 11):
            binary_chunk = binary_sequence[i:i+11]
            if len(binary_chunk) == 11:
                word = self.binary_to_word.get(binary_chunk, 'unknown')
                words.append(word)
        return words

    def bip39_to_binary(self, words: List[str]) -> str:
        """Обратная конвертация BIP39 слов в бинарную последовательность"""
        if not BIP39_AVAILABLE:
            return '0' * 132
            
        binary_sequence = ''
        for word in words:
            binary_sequence += self.word_to_binary.get(word, '0' * 11)
        return binary_sequence
    
    def encode_question(self, date_str: str, question: str = "ЦЕНА ЧЕРЕЗ 24 ЧАСА?") -> List[str]:
        """
        Кодирование вопроса: SHA256 → Base58 → BIP39
        """
        if not BIP39_AVAILABLE:
            return ['abandon'] * 12
            
        text = f"{date_str} {question}"
        
        # 1. SHA256
        sha = hashlib.sha256(text.encode()).digest()
        
        # 2. Base58
        b58 = base58.b58encode(sha).decode()
        
        # 3. BIP39 (128 бит = 12 слов)
        entropy = sha[:16]
        mnemonic = self.mnemo.to_mnemonic(entropy)
        
        return mnemonic.split()
    
    def encode_price_data(self, prices: np.ndarray, data_type: str = "CLOSE") -> List[str]:
        """
        Кодирование любого массива цен: SHA256 → Base58 → BIP39
        Возвращает 12 BIP39 слов, кодирующих ценовую последовательность
        """
        if not BIP39_AVAILABLE:
            return ['abandon'] * 12
            
        if len(prices) == 0:
            return ['abandon'] * 12
            
        # Нормализация в диапазон 0-255
        normalized = ((prices - np.min(prices)) / (np.max(prices) - np.min(prices) + 1e-10) * 255).astype(np.uint8)
        data = bytes(normalized)
        
        # SHA256
        sha = hashlib.sha256(data).digest()
        
        # Base58 (для логирования, не используется в энтропии)
        b58 = base58.b58encode(sha).decode()
        
        # BIP39 из первых 128 бит хэша
        entropy = sha[:16]
        mnemonic = self.mnemo.to_mnemonic(entropy)
        
        return mnemonic.split()
    
    def convert_prices_to_binary(self, prices: pd.Series) -> List[str]:
        """Конвертация цен в бинарную последовательность (1=рост, 0=падение)"""
        binary_sequence = []
        for i in range(1, len(prices)):
            binary_digit = '1' if prices.iloc[i] > prices.iloc[i-1] else '0'
            binary_sequence.append(binary_digit)
        return binary_sequence

    def get_bip39_features_from_prices(self, prices: np.ndarray) -> Dict[str, float]:
        """
        НОВАЯ ФУНКЦИЯ: Извлечение числовых признаков из BIP39 фраз
        Возвращает статистические характеристики BIP39 кодирования
        """
        if not BIP39_AVAILABLE or len(prices) == 0:
            return {
                'bip39_entropy': 0.0,
                'bip39_word_diversity': 0.0,
                'bip39_hash_magnitude': 0.0,
                'bip39_word_positions_mean': 0.0
            }
        
        # Получаем BIP39 фразу
        words = self.encode_price_data(prices, "PRICES")
        
        # 1. Энтропия распределения слов
        word_indices = [self.wordlist.index(w) if w in self.wordlist else 0 for w in words]
        word_probs = np.array([word_indices.count(i) / len(word_indices) for i in set(word_indices)])
        bip39_entropy = entropy(word_probs + 1e-10, base=2)
        
        # 2. Разнообразие слов (уникальные слова / всего слов)
        bip39_word_diversity = len(set(words)) / len(words)
        
        # 3. Магнитуда хэша (сумма позиций слов в словаре, нормализованная)
        bip39_hash_magnitude = np.mean(word_indices) / len(self.wordlist)
        
        # 4. Средняя позиция слов
        bip39_word_positions_mean = np.mean(word_indices)
        
        return {
            'bip39_entropy': bip39_entropy,
            'bip39_word_diversity': bip39_word_diversity,
            'bip39_hash_magnitude': bip39_hash_magnitude,
            'bip39_word_positions_mean': bip39_word_positions_mean
        }

# ====================== SEAL SELF-ADAPTING MODULE (MIT) ======================
@dataclass
class SelfEdit:
    """
    Self-Edit структура по MIT SEAL methodology.
    Каждый промпт к LLM создаёт self-edit который потом используется для обучения.
    """
    prompt: str                          # Входной промпт
    response: str                        # Ответ LLM
    direction: str                       # UP/DOWN
    confidence: float                    # Уверенность 0-100
    timestamp: datetime = field(default_factory=datetime.now)
    reward: float = 0.0                  # P&L после исполнения (заполняется позже)
    actual_direction: str = ""           # Реальное направление (заполняется позже)
    symbol: str = ""
    quantum_entropy: float = 0.0
    bip39_entropy: float = 0.0
    
    def is_correct(self) -> bool:
        return self.direction == self.actual_direction and self.actual_direction != ""
    
    def to_dict(self) -> Dict:
        return {
            'prompt': self.prompt,
            'response': self.response,
            'direction': self.direction,
            'confidence': self.confidence,
            'reward': self.reward,
            'actual_direction': self.actual_direction,
            'symbol': self.symbol,
            'timestamp': self.timestamp.isoformat(),
            'quantum_entropy': self.quantum_entropy,
            'bip39_entropy': self.bip39_entropy,
        }


class SEALTrainer:
    """
    SEAL (Self-Adapting Learning) Trainer по методологии MIT.
    
    Реализует двойную петлю оптимизации:
    
    INNER LOOP (при каждом промпте):
    - LLM генерирует self-edit (прогноз)
    - Self-edit сохраняется в experience buffer
    - После получения результата — reward обновляется
    
    OUTER LOOP (периодически):
    - Rejection Sampling: отбираем top-k% лучших self-edits
    - Behavior Cloning: генерируем новый датасет из лучших примеров
    - Ретрейн LLM через Ollama на лучших примерах
    
    Референс: https://arxiv.org/abs/2506.10943
    """
    
    def __init__(self):
        self.experience_buffer: deque = deque(maxlen=SEAL_CONFIG['buffer_size'])
        self.pending_edits: Dict[str, SelfEdit] = {}  # key = symbol+timestamp
        self.prompt_count = 0
        self.retrain_count = 0
        self.metrics = {
            'total_prompts': 0,
            'total_retrains': 0,
            'avg_reward': 0.0,
            'win_rate': 0.0,
            'best_edits_count': 0,
        }
        
        # Загружаем существующий буфер если есть
        self._load_buffer()
        
        log.info(f"🦭 SEAL Trainer инициализирован")
        log.info(f"   Buffer size: {len(self.experience_buffer)}/{SEAL_CONFIG['buffer_size']}")
        log.info(f"   Retrain every: {SEAL_CONFIG['retrain_every']} prompts")

        
        # ═══ RL АГЕНТ (НОВОЕ!) ═══
        self.rl_agent = LightweightRLAgent(alpha=0.1, gamma=0.95, epsilon=0.2)
        self.rl_agent.load('models/seal_rl_qtable.pkl')
        
        # Tracking для RL
        self.last_state = None
        self.last_action = None
        self.recent_trades = deque(maxlen=20)
    
    def _load_buffer(self):
        """Загрузка experience buffer с диска"""
        if os.path.exists(SEAL_CONFIG['save_buffer_path']):
            try:
                with open(SEAL_CONFIG['save_buffer_path'], 'rb') as f:
                    data = pickle.load(f)
                    self.experience_buffer = deque(data, maxlen=SEAL_CONFIG['buffer_size'])
                log.info(f"🦭 SEAL: Загружен буфер с {len(self.experience_buffer)} примерами")
            except Exception as e:
                log.warning(f"🦭 SEAL: Не удалось загрузить буфер: {e}")
    
    def _save_buffer(self):
        """Сохранение experience buffer на диск"""
        try:
            with open(SEAL_CONFIG['save_buffer_path'], 'wb') as f:
                pickle.dump(list(self.experience_buffer), f)
        except Exception as e:
            log.warning(f"🦭 SEAL: Не удалось сохранить буфер: {e}")
    
    def record_self_edit(self, prompt: str, response: str, direction: str, 
                         confidence: float, symbol: str,
                         quantum_entropy: float = 0.0, bip39_entropy: float = 0.0) -> str:
        """
        INNER LOOP + RL РЕШЕНИЕ: 
        Записываем self-edit И принимаем решение через RL агента.
        """
        self.prompt_count += 1
        self.metrics['total_prompts'] = self.prompt_count
        
        # ═══ RL РЕШЕНИЕ ═══
        recent_win_rate = (sum(self.recent_trades) / len(self.recent_trades) 
                          if self.recent_trades else 0.5)
        
        # RL агент принимает решение
        rl_action = self.rl_agent.select_action(quantum_entropy, confidence, recent_win_rate)
        
        # Применяем действие
        adjusted_confidence = confidence
        should_skip = False
        
        if rl_action == 1:  # SKIP
            should_skip = True
            log.info(f"🤖 RL: SKIP trade (entropy={quantum_entropy:.2f}, conf={confidence:.0f}%)")
        elif rl_action == 2:  # REDUCE confidence
            adjusted_confidence = confidence * 0.7
            log.info(f"🤖 RL: REDUCE confidence {confidence:.0f}% → {adjusted_confidence:.0f}%")
        else:  # USE
            log.info(f"🤖 RL: USE prediction (conf={confidence:.0f}%)")
        
        # Сохраняем state для Q-learning
        current_state = self.rl_agent._discretize_state(quantum_entropy, confidence, recent_win_rate)
        self.last_state = current_state
        self.last_action = rl_action
        
        # Создаём self-edit
        edit = SelfEdit(
            prompt=prompt,
            response=response,
            direction=direction,
            confidence=adjusted_confidence,
            symbol=symbol,
            quantum_entropy=quantum_entropy,
            bip39_entropy=bip39_entropy,
        )
        
        # ID
        edit_id = f"{symbol}_{edit.timestamp.strftime('%Y%m%d_%H%M%S')}"
        
        # Если SKIP - не торгуем
        if should_skip:
            log.info(f"🦭 SEAL: Self-edit #{self.prompt_count} пропущен (RL SKIP)")
            return None
        
        self.pending_edits[edit_id] = edit
        
        log.info(f"🦭 SEAL: Self-edit #{self.prompt_count} [{edit_id}] → {direction} {adjusted_confidence:.0f}%")
        
        return edit_id

    def update_reward(self, edit_id: str, actual_direction: str, reward_pips: float):
        """
        INNER LOOP Step 2: Обновляем reward после получения результата.
        """
        if edit_id not in self.pending_edits:
            log.warning(f"🦭 SEAL: Edit {edit_id} не найден в pending")
            return
        
        edit = self.pending_edits.pop(edit_id)
        edit.actual_direction = actual_direction
        edit.reward = reward_pips

        # ═══ RL ОБУЧЕНИЕ ═══
        if self.last_state is not None and self.last_action is not None:
            # Новое состояние
            recent_win_rate = sum(self.recent_trades) / len(self.recent_trades) if self.recent_trades else 0.5
            next_state = self.rl_agent._discretize_state(
                edit.quantum_entropy, edit.confidence, recent_win_rate
            )
            
            # Reward для RL (нормализуем)
            rl_reward = reward_pips / 10.0
            
            # Q-learning update
            self.rl_agent.update_q_value(self.last_state, self.last_action, rl_reward, next_state)
            
            # Сохраняем Q-table периодически
            if self.prompt_count % 10 == 0:
                self.rl_agent.save('models/seal_rl_qtable.pkl')
        
        # Обновляем recent trades
        was_correct = 1 if actual_direction == edit.direction else 0
        self.recent_trades.append(was_correct)

        
        # Добавляем в experience buffer
        self.experience_buffer.append(edit)
        
        status = "✓" if edit.is_correct() else "✗"
        log.info(f"🦭 SEAL: {status} Reward обновлён [{edit_id}]: {reward_pips:+.1f} пунктов")
        
        # Периодическое сохранение буфера
        if len(self.experience_buffer) % 10 == 0:
            self._save_buffer()
        
        # Проверяем нужен ли ретрейн (OUTER LOOP)
        if self.prompt_count % SEAL_CONFIG['retrain_every'] == 0:
            self._trigger_outer_loop()
    
    def _trigger_outer_loop(self):
        """
        OUTER LOOP: ReST^EM Algorithm (Rejection Sampling + Behavior Cloning).
        Отбираем лучшие self-edits и генерируем датасет для ретрейна.
        """
        if len(self.experience_buffer) < SEAL_CONFIG['min_samples_for_retrain']:
            log.info(f"🦭 SEAL: Недостаточно примеров для ретрейна "
                    f"({len(self.experience_buffer)}/{SEAL_CONFIG['min_samples_for_retrain']})")
            return
        
        log.info(f"\n{'='*60}")
        log.info(f"🦭 SEAL OUTER LOOP: ReST^EM Iteration #{self.retrain_count + 1}")
        log.info(f"{'='*60}")
        
        # Step 1: Rejection Sampling - отбираем top-k% по reward
        all_edits = list(self.experience_buffer)
        all_edits.sort(key=lambda x: x.reward, reverse=True)
        
        top_k = max(10, int(len(all_edits) * SEAL_CONFIG['top_k_ratio']))
        best_edits = all_edits[:top_k]
        
        # Фильтруем только прибыльные
        profitable_edits = [e for e in best_edits if e.reward > SEAL_CONFIG['reward_threshold']]
        
        if len(profitable_edits) < 5:
            log.info(f"🦭 SEAL: Мало прибыльных примеров ({len(profitable_edits)}), пропускаем ретрейн")
            return
        
        log.info(f"🦭 SEAL: Отобрано {len(profitable_edits)} лучших self-edits из {len(all_edits)}")
        log.info(f"   Best reward: {profitable_edits[0].reward:+.1f} pips")
        log.info(f"   Worst in top-k: {profitable_edits[-1].reward:+.1f} pips")
        
        # Статистика
        correct_count = sum(1 for e in profitable_edits if e.is_correct())
        avg_reward = np.mean([e.reward for e in profitable_edits])
        
        log.info(f"   Win rate в top-k: {correct_count/len(profitable_edits)*100:.1f}%")
        log.info(f"   Avg reward в top-k: {avg_reward:+.1f} pips")
        
        # Step 2: Behavior Cloning - сохраняем лучшие примеры для ретрейна
        self._save_best_edits_for_retrain(profitable_edits)
        
        # Step 3: Trigger retrain (опционально, если ollama доступна)
        self._retrain_llm_on_best_edits(profitable_edits)
        
        self.retrain_count += 1
        self.metrics['total_retrains'] = self.retrain_count
        self.metrics['avg_reward'] = avg_reward
        self.metrics['win_rate'] = correct_count / len(profitable_edits)
        self.metrics['best_edits_count'] = len(profitable_edits)
        
        log.info(f"🦭 SEAL: Outer loop завершён")
        log.info(f"{'='*60}\n")
    
    def _save_best_edits_for_retrain(self, edits: List[SelfEdit]):
        """Сохраняем лучшие self-edits в JSONL для ретрейна"""
        try:
            with open(SEAL_CONFIG['best_edits_path'], 'w', encoding='utf-8') as f:
                for edit in edits:
                    f.write(json.dumps(edit.to_dict(), ensure_ascii=False) + '\n')
            log.info(f"🦭 SEAL: Сохранено {len(edits)} лучших примеров → {SEAL_CONFIG['best_edits_path']}")
        except Exception as e:
            log.error(f"🦭 SEAL: Ошибка сохранения: {e}")
    
    def _retrain_llm_on_best_edits(self, edits: List[SelfEdit]):
        """
        Behavior Cloning: дообучаем LLM на лучших примерах через Ollama.
        Создаём новый Modelfile с MESSAGE примерами из лучших self-edits.
        """
        if not ollama:
            log.info("🦭 SEAL: Ollama недоступна, пропускаем авто-ретрейн")
            return
        
        log.info(f"🦭 SEAL: Запуск Behavior Cloning на {len(edits)} примерах...")
        
        try:
            # Создаём Modelfile с лучшими примерами
            modelfile_content = f'''FROM {MODEL_NAME}
PARAMETER temperature 0.5
PARAMETER top_p 0.9
SYSTEM """
Ты — SEAL-QuantumTrader — самообучающийся трейдер.
Эти примеры показали лучшие результаты (прибыль {self.metrics["avg_reward"]:+.1f} pips):
"""

'''
            # Добавляем лучшие примеры как MESSAGE
            for edit in edits[:100]:  # Максимум 100 примеров
                # Экранируем кавычки
                prompt_escaped = edit.prompt.replace('"', '\\"').replace('\n', '\\n')
                response_escaped = edit.response.replace('"', '\\"').replace('\n', '\\n')
                
                modelfile_content += f'MESSAGE user "{prompt_escaped[:500]}"\n'
                modelfile_content += f'MESSAGE assistant "{response_escaped[:500]}"\n\n'
            
            # Сохраняем Modelfile
            modelfile_path = "Modelfile_seal_retrain"
            with open(modelfile_path, 'w', encoding='utf-8') as f:
                f.write(modelfile_content)
            
            # Создаём обновлённую модель
            seal_model_name = f"{MODEL_NAME}-seal-{self.retrain_count}"
            
            result = subprocess.run(
                ["ollama", "create", seal_model_name, "-f", modelfile_path],
                capture_output=True,
                text=True,
                timeout=300
            )
            
            if result.returncode == 0:
                log.info(f"🦭 SEAL: ✓ Модель {seal_model_name} создана!")
                log.info(f"   Используй: ollama run {seal_model_name}")
            else:
                log.warning(f"🦭 SEAL: Ошибка создания модели: {result.stderr}")
            
            # Удаляем временный файл
            os.remove(modelfile_path)
            
        except subprocess.TimeoutExpired:
            log.warning("🦭 SEAL: Таймаут при создании модели")
        except Exception as e:
            log.error(f"🦭 SEAL: Ошибка ретрейна: {e}")
    
    def get_stats(self) -> Dict:
        """Возвращает статистику SEAL"""
        if len(self.experience_buffer) == 0:
            return self.metrics
        
        all_edits = list(self.experience_buffer)
        rewards = [e.reward for e in all_edits if e.reward != 0]
        correct = sum(1 for e in all_edits if e.is_correct())
        
        return {
            **self.metrics,
            'buffer_size': len(self.experience_buffer),
            'pending_edits': len(self.pending_edits),
            'total_reward': sum(rewards) if rewards else 0,
            'avg_reward_all': np.mean(rewards) if rewards else 0,
            'win_rate_all': correct / len(all_edits) if all_edits else 0,
        }
    
    def print_stats(self):
        """Выводит статистику SEAL"""
        stats = self.get_stats()
        print(f"\n{'='*60}")
        print(f"🦭 SEAL STATISTICS")
        print(f"{'='*60}")
        print(f"Total prompts: {stats['total_prompts']}")
        print(f"Total retrains: {stats['total_retrains']}")
        print(f"Buffer size: {stats.get('buffer_size', 0)}")
        print(f"Pending edits: {stats.get('pending_edits', 0)}")
        print(f"Total reward: {stats.get('total_reward', 0):+.1f} pips")
        print(f"Avg reward (all): {stats.get('avg_reward_all', 0):+.1f} pips")
        print(f"Win rate (all): {stats.get('win_rate_all', 0)*100:.1f}%")
        print(f"Win rate (top-k): {stats.get('win_rate', 0)*100:.1f}%")

        # RL статистика
        rl_stats = stats.get('rl_agent', {})
        if rl_stats.get('total_decisions', 0) > 0:
            print(f"\n🤖 RL AGENT:")
            print(f"Total decisions: {rl_stats['total_decisions']}")
            print(f"Q-table size: {rl_stats['q_table_size']}")
            print(f"Epsilon: {rl_stats['epsilon']}")
            print(f"Actions:")
            for action_name, count in rl_stats.get('actions', {}).items():
                print(f"  {action_name}: {count}")

        print(f"{'='*60}\n")


# Глобальный SEAL Trainer (singleton)
_seal_trainer: SEALTrainer = None

def get_seal_trainer() -> SEALTrainer:
    """Получить глобальный SEAL Trainer"""
    global _seal_trainer
    if _seal_trainer is None:
        _seal_trainer = SEALTrainer()
    return _seal_trainer


# ====================== КВАНТОВЫЙ ЭНКОДЕР ======================
class QuantumEncoder:
    """
    Квантовый энкодер на базе Qiskit для извлечения скрытых признаков
    Реализация из статьи: 8 кубитов, запутывание через CZ-гейты, 2048 измерений
    """
    
    def __init__(self, n_qubits: int = 8, n_shots: int = 2048):
        self.n_qubits = n_qubits
        self.n_shots = n_shots
        if QISKIT_AVAILABLE:
            self.simulator = AerSimulator()
        else:
            self.simulator = None
        
    def encode_and_measure(self, features: np.ndarray) -> Dict[str, float]:
        """
        Кодирует признаки в квантовую схему и извлекает 4 квантовых признака:
        1. Квантовая энтропия (мера неопределённости)
        2. Доминантное состояние (вероятность самого частого базиса)
        3. Количество значимых состояний (>3% вероятности)
        4. Квантовая дисперсия вероятностей
        """
        if not QISKIT_AVAILABLE:
            # Fallback на псевдо-квантовые признаки
            return {
                'quantum_entropy': np.random.uniform(2.0, 5.0),
                'dominant_state_prob': np.random.uniform(0.05, 0.20),
                'significant_states': np.random.randint(3, 20),
                'quantum_variance': np.random.uniform(0.001, 0.01)
            }
        
        # Нормализация признаков в диапазон [0, π]
        normalized = (features - features.min()) / (features.max() - features.min() + 1e-8)
        angles = normalized * np.pi
        
        # Создаём квантовую схему
        qc = QuantumCircuit(self.n_qubits, self.n_qubits)
        
        # Кодирование через RY-вращения
        for i in range(min(len(angles), self.n_qubits)):
            qc.ry(angles[i], i)
        
        # Запутывание через CZ-гейты (создаём корреляции второго порядка)
        for i in range(self.n_qubits - 1):
            qc.cz(i, i + 1)
        # Замыкаем цепь
        qc.cz(self.n_qubits - 1, 0)
        
        # Измерение
        qc.measure(range(self.n_qubits), range(self.n_qubits))
        
        # Выполнение на симуляторе
        job = self.simulator.run(qc, shots=self.n_shots)
        result = job.result()
        counts = result.get_counts()
        
        # Вычисление квантовых признаков
        total_shots = sum(counts.values())
        probabilities = np.array([counts.get(format(i, f'0{self.n_qubits}b'), 0) / total_shots 
                                  for i in range(2**self.n_qubits)])
        
        # 1. Квантовая энтропия Шеннона
        quantum_entropy = entropy(probabilities + 1e-10, base=2)
        
        # 2. Доминантное состояние
        dominant_state_prob = np.max(probabilities)
        
        # 3. Количество значимых состояний (>3%)
        significant_states = np.sum(probabilities > 0.03)
        
        # 4. Квантовая дисперсия
        quantum_variance = np.var(probabilities)
        
        return {
            'quantum_entropy': quantum_entropy,
            'dominant_state_prob': dominant_state_prob,
            'significant_states': significant_states,
            'quantum_variance': quantum_variance
        }

# ====================== ТЕХНИЧЕСКИЕ ПРИЗНАКИ ======================
def calculate_features(df: pd.DataFrame) -> pd.DataFrame:
    """Расчёт 33 технических индикаторов"""
    d = df.copy()
    d["close_prev"] = d["close"].shift(1)
    
    # ATR
    tr = pd.concat([
        d["high"] - d["low"],
        (d["high"] - d["close_prev"]).abs(),
        (d["low"] - d["close_prev"]).abs(),
    ], axis=1).max(axis=1)
    d["ATR"] = tr.rolling(14).mean()
    
    # RSI
    delta = d["close"].diff()
    up = delta.clip(lower=0).rolling(14).mean()
    down = (-delta.clip(upper=0)).rolling(14).mean()
    rs = up / down.replace(0, np.nan)
    d["RSI"] = 100 - (100 / (1 + rs))
    
    # MACD
    ema12 = d["close"].ewm(span=12, adjust=False).mean()
    ema26 = d["close"].ewm(span=26, adjust=False).mean()
    d["MACD"] = ema12 - ema26
    d["MACD_signal"] = d["MACD"].ewm(span=9, adjust=False).mean()
    
    # Объёмы
    d["vol_avg_20"] = d["tick_volume"].rolling(20).mean()
    d["vol_ratio"] = d["tick_volume"] / d["vol_avg_20"].replace(0, np.nan)
    
    # Bollinger Bands
    d["BB_middle"] = d["close"].rolling(20).mean()
    bb_std = d["close"].rolling(20).std()
    d["BB_upper"] = d["BB_middle"] + 2 * bb_std
    d["BB_lower"] = d["BB_middle"] - 2 * bb_std
    d["BB_position"] = (d["close"] - d["BB_lower"]) / (d["BB_upper"] - d["BB_lower"])
    
    # Stochastic
    low_14 = d["low"].rolling(14).min()
    high_14 = d["high"].rolling(14).max()
    d["Stoch_K"] = 100 * (d["close"] - low_14) / (high_14 - low_14)
    d["Stoch_D"] = d["Stoch_K"].rolling(3).mean()
    
    # EMA кросс
    d["EMA_50"] = d["close"].ewm(span=50, adjust=False).mean()
    d["EMA_200"] = d["close"].ewm(span=200, adjust=False).mean()
    
    # Дополнительные признаки для CatBoost
    d["price_change_1"] = d["close"].pct_change(1)
    d["price_change_5"] = d["close"].pct_change(5)
    d["price_change_21"] = d["close"].pct_change(21)
    d["log_return"] = np.log(d["close"] / d["close"].shift(1))
    d["volatility_20"] = d["log_return"].rolling(20).std()
    
    return d.dropna()

# ====================== ОБУЧЕНИЕ CATBOOST С BIP39 ======================
def train_catboost_model(data_dict: Dict[str, pd.DataFrame], quantum_encoder: QuantumEncoder, bip39_converter: BIP39Converter) -> CatBoostClassifier:
    """
    Обучает CatBoost на данных всех 8 валютных пар с квантовыми И BIP39 признаками
    Возвращает обученную модель
    """
    print(f"\n{'='*80}")
    print(f"ОБУЧЕНИЕ CATBOOST С КВАНТОВЫМИ И BIP39 ПРИЗНАКАМИ")
    print(f"{'='*80}\n")
    
    if not CATBOOST_AVAILABLE:
        print("❌ CatBoost недоступен, используем заглушку")
        return None
    
    all_features = []
    all_targets = []
    all_symbols = []
    
    print("Подготовка данных с квантовым и BIP39 кодированием...")
    
    for symbol, df in data_dict.items():
        print(f"\nОбработка {symbol}: {len(df)} баров")
        
        df_features = calculate_features(df)
        
        # Квантовые и BIP39 признаки для каждой точки
        for idx in range(LOOKBACK, len(df_features) - PREDICTION_HORIZON):
            if idx % 500 == 0:
                print(f" Кодирование: {idx}/{len(df_features) - PREDICTION_HORIZON}")
            
            row = df_features.iloc[idx]
            
            # Получаем окно (5 последних свечей)
            start_idx = max(0, idx - 4)
            end_idx = idx + 1
            
            window_close = df_features['close'].iloc[start_idx:end_idx].values
            window_open = df_features['open'].iloc[start_idx:end_idx].values
            window_high = df_features['high'].iloc[start_idx:end_idx].values
            window_low = df_features['low'].iloc[start_idx:end_idx].values
            
            # ===== КВАНТОВОЕ КОДИРОВАНИЕ =====
            feature_vector = np.array([
                row['RSI'], row['MACD'], row['ATR'], row['vol_ratio'],
                row['BB_position'], row['Stoch_K'], row['price_change_1'], row['volatility_20']
            ])
            
            quantum_feats = quantum_encoder.encode_and_measure(feature_vector)
            
            # ===== BIP39 КОДИРОВАНИЕ (НОВОЕ!) =====
            bip39_close_feats = bip39_converter.get_bip39_features_from_prices(window_close)
            bip39_open_feats = bip39_converter.get_bip39_features_from_prices(window_open)
            bip39_high_feats = bip39_converter.get_bip39_features_from_prices(window_high)
            bip39_low_feats = bip39_converter.get_bip39_features_from_prices(window_low)
            
            # Целевая переменная: цена через 24 часа
            future_idx = idx + PREDICTION_HORIZON
            future_price = df_features.iloc[future_idx]['close']
            current_price = row['close']
            target = 1 if future_price > current_price else 0  # 1=UP, 0=DOWN
            
            # Собираем признаки: технические + квантовые + BIP39 + символ
            features = {
                # Технические
                'RSI': row['RSI'],
                'MACD': row['MACD'],
                'ATR': row['ATR'],
                'vol_ratio': row['vol_ratio'],
                'BB_position': row['BB_position'],
                'Stoch_K': row['Stoch_K'],
                'Stoch_D': row['Stoch_D'],
                'EMA_50': row['EMA_50'],
                'EMA_200': row['EMA_200'],
                'price_change_1': row['price_change_1'],
                'price_change_5': row['price_change_5'],
                'price_change_21': row['price_change_21'],
                'volatility_20': row['volatility_20'],
                # Квантовые
                'quantum_entropy': quantum_feats['quantum_entropy'],
                'dominant_state_prob': quantum_feats['dominant_state_prob'],
                'significant_states': quantum_feats['significant_states'],
                'quantum_variance': quantum_feats['quantum_variance'],
                # BIP39 для Close
                'bip39_close_entropy': bip39_close_feats['bip39_entropy'],
                'bip39_close_diversity': bip39_close_feats['bip39_word_diversity'],
                'bip39_close_magnitude': bip39_close_feats['bip39_hash_magnitude'],
                'bip39_close_positions': bip39_close_feats['bip39_word_positions_mean'],
                # BIP39 для Open
                'bip39_open_entropy': bip39_open_feats['bip39_entropy'],
                'bip39_open_diversity': bip39_open_feats['bip39_word_diversity'],
                # BIP39 для High
                'bip39_high_entropy': bip39_high_feats['bip39_entropy'],
                'bip39_high_diversity': bip39_high_feats['bip39_word_diversity'],
                # BIP39 для Low
                'bip39_low_entropy': bip39_low_feats['bip39_entropy'],
                'bip39_low_diversity': bip39_low_feats['bip39_word_diversity'],
                # Символ
                'symbol': symbol
            }
            
            all_features.append(features)
            all_targets.append(target)
            all_symbols.append(symbol)
    
    print(f"\n✓ Всего примеров: {len(all_features)}")
    
    # Создаём DataFrame
    X = pd.DataFrame(all_features)
    y = np.array(all_targets)
    
    # One-hot encoding символов
    X = pd.get_dummies(X, columns=['symbol'], prefix='sym')
    
    print(f"✓ Признаков: {len(X.columns)} (включая {sum('bip39' in col for col in X.columns)} BIP39)")
    print(f"✓ Баланс классов: UP={np.sum(y==1)} ({np.sum(y==1)/len(y)*100:.1f}%), DOWN={np.sum(y==0)} ({np.sum(y==0)/len(y)*100:.1f}%)")
    
    # Обучение CatBoost
    print("\nОбучение CatBoost с BIP39 признаками...")
    model = CatBoostClassifier(
        iterations=3000,
        learning_rate=0.03,
        depth=8,
        loss_function='Logloss',
        eval_metric='Accuracy',
        random_seed=42,
        verbose=500
    )
    
    # Используем TimeSeriesSplit для честной валидации
    from sklearn.model_selection import TimeSeriesSplit
    tscv = TimeSeriesSplit(n_splits=3)
    
    accuracies = []
    for fold_idx, (train_idx, val_idx) in enumerate(tscv.split(X)):
        print(f"\n--- Фолд {fold_idx + 1}/3 ---")
        X_train, X_val = X.iloc[train_idx], X.iloc[val_idx]
        y_train, y_val = y[train_idx], y[val_idx]
        
        model.fit(X_train, y_train, eval_set=(X_val, y_val), verbose=False)
        accuracy = model.score(X_val, y_val)
        accuracies.append(accuracy)
        print(f"Фолд {fold_idx + 1} Accuracy: {accuracy*100:.2f}%")
    
    print(f"\n{'='*80}")
    print(f"РЕЗУЛЬТАТЫ КРОСС-ВАЛИДАЦИИ")
    print(f"{'='*80}")
    print(f"Средняя точность: {np.mean(accuracies)*100:.2f}% ± {np.std(accuracies)*100:.2f}%")
    
    # Финальное обучение на всех данных
    print("\nОбучение финальной модели на всех данных...")
    model.fit(X, y, verbose=500)
    
    # Сохранение модели
    model_path = "models/catboost_quantum_bip39.cbm"
    model.save_model(model_path)
    print(f"\n✓ Модель сохранена: {model_path}")
    
    # Важность признаков
    feature_importance = model.get_feature_importance()
    feature_names = X.columns
    importance_df = pd.DataFrame({
        'feature': feature_names,
        'importance': feature_importance
    }).sort_values('importance', ascending=False)
    
    print(f"\nТОП-15 ВАЖНЫХ ПРИЗНАКОВ:")
    print(importance_df.head(15).to_string(index=False))
    
    # Анализ вклада BIP39
    bip39_features = [f for f in feature_names if 'bip39' in f]
    bip39_importance = importance_df[importance_df['feature'].isin(bip39_features)]['importance'].sum()
    total_importance = importance_df['importance'].sum()
    
    print(f"\nАНАЛИЗ BIP39 ПРИЗНАКОВ:")
    print(f"Всего BIP39 признаков: {len(bip39_features)}")
    print(f"Вклад BIP39 в модель: {bip39_importance/total_importance*100:.2f}%")
    
    return model

# ====================== ГЕНЕРАЦИЯ ГИБРИДНОГО ДАТАСЕТА С BIP39 ======================
def generate_hybrid_dataset(
    data_dict: Dict[str, pd.DataFrame],
    catboost_model: CatBoostClassifier,
    quantum_encoder: QuantumEncoder,
    bip39_converter: BIP39Converter,
    num_samples: int = 2000
) -> List[Dict]:
    """
    Генерирует датасет для LLM с встроенными прогнозами CatBoost, квантовыми и BIP39 признаками
    """
    print(f"\n{'='*80}")
    print(f"ГЕНЕРАЦИЯ ГИБРИДНОГО ДАТАСЕТА С BIP39")
    print(f"{'='*80}\n")
    print(f"Цель: {num_samples} примеров с CatBoost, Quantum и BIP39 признаками\n")
    
    dataset = []
    up_count = 0
    down_count = 0
    
    target_per_symbol = num_samples // len(SYMBOLS)
    
    for symbol, df in data_dict.items():
        print(f"Обработка {symbol}...")
        df_features = calculate_features(df)
        
        candidates = []
        
        for idx in range(LOOKBACK, len(df_features) - PREDICTION_HORIZON):
            row = df_features.iloc[idx]
            future_idx = idx + PREDICTION_HORIZON
            future_row = df_features.iloc[future_idx]
            
            # Окно данных
            start_idx = max(0, idx - 4)
            end_idx = idx + 1
            
            window_close = df_features['close'].iloc[start_idx:end_idx].values
            window_open = df_features['open'].iloc[start_idx:end_idx].values
            window_high = df_features['high'].iloc[start_idx:end_idx].values
            window_low = df_features['low'].iloc[start_idx:end_idx].values
            
            # Квантовое кодирование
            feature_vector = np.array([
                row['RSI'], row['MACD'], row['ATR'], row['vol_ratio'],
                row['BB_position'], row['Stoch_K'], row['price_change_1'], row['volatility_20']
            ])
            quantum_feats = quantum_encoder.encode_and_measure(feature_vector)
            
            # BIP39 кодирование
            bip39_close_feats = bip39_converter.get_bip39_features_from_prices(window_close)
            bip39_open_feats = bip39_converter.get_bip39_features_from_prices(window_open)
            
            # Получаем BIP39 фразы для отображения
            bip39_close_phrase = bip39_converter.encode_price_data(window_close, "CLOSE")
            
            # Подготовка признаков для CatBoost
            X_features = {
                'RSI': row['RSI'],
                'MACD': row['MACD'],
                'ATR': row['ATR'],
                'vol_ratio': row['vol_ratio'],
                'BB_position': row['BB_position'],
                'Stoch_K': row['Stoch_K'],
                'Stoch_D': row['Stoch_D'],
                'EMA_50': row['EMA_50'],
                'EMA_200': row['EMA_200'],
                'price_change_1': row['price_change_1'],
                'price_change_5': row['price_change_5'],
                'price_change_21': row['price_change_21'],
                'volatility_20': row['volatility_20'],
                'quantum_entropy': quantum_feats['quantum_entropy'],
                'dominant_state_prob': quantum_feats['dominant_state_prob'],
                'significant_states': quantum_feats['significant_states'],
                'quantum_variance': quantum_feats['quantum_variance'],
                'bip39_close_entropy': bip39_close_feats['bip39_entropy'],
                'bip39_close_diversity': bip39_close_feats['bip39_word_diversity'],
                'bip39_close_magnitude': bip39_close_feats['bip39_hash_magnitude'],
                'bip39_close_positions': bip39_close_feats['bip39_word_positions_mean'],
                'bip39_open_entropy': bip39_open_feats['bip39_entropy'],
                'bip39_open_diversity': bip39_open_feats['bip39_word_diversity'],
                'bip39_high_entropy': 0.0,  # Упрощено для скорости
                'bip39_high_diversity': 0.0,
                'bip39_low_entropy': 0.0,
                'bip39_low_diversity': 0.0,
            }
            
            # Создаём DataFrame для CatBoost
            X_df = pd.DataFrame([X_features])
            for s in SYMBOLS:
                X_df[f'sym_{s}'] = 1 if s == symbol else 0
            
            # Прогноз CatBoost
            if catboost_model:
                proba = catboost_model.predict_proba(X_df)[0]
                catboost_prob_up = proba[1] * 100
                catboost_direction = "UP" if proba[1] > 0.5 else "DOWN"
                catboost_confidence = max(proba) * 100
            else:
                catboost_prob_up = 50.0
                catboost_direction = "UP"
                catboost_confidence = 50.0
            
            # Реальный результат
            actual_price_24h = future_row['close']
            price_change = actual_price_24h - row['close']
            price_change_pips = int(price_change / 0.0001)
            actual_direction = "UP" if price_change > 0 else "DOWN"
            
            candidates.append({
                'symbol': symbol,
                'row': row,
                'future_row': future_row,
                'quantum_feats': quantum_feats,
                'bip39_close_feats': bip39_close_feats,
                'bip39_close_phrase': bip39_close_phrase,
                'catboost_direction': catboost_direction,
                'catboost_confidence': catboost_confidence,
                'catboost_prob_up': catboost_prob_up,
                'actual_direction': actual_direction,
                'price_change_pips': price_change_pips,
                'current_time': df.index[idx]
            })
        
        # Балансировка
        up_candidates = [c for c in candidates if c['actual_direction'] == 'UP']
        down_candidates = [c for c in candidates if c['actual_direction'] == 'DOWN']
        
        target_up = target_per_symbol // 2
        target_down = target_per_symbol // 2
        
        selected_up = np.random.choice(len(up_candidates), size=min(target_up, len(up_candidates)), replace=False) if up_candidates else []
        selected_down = np.random.choice(len(down_candidates), size=min(target_down, len(down_candidates)), replace=False) if down_candidates else []
        
        for idx in selected_up:
            candidate = up_candidates[idx]
            example = create_hybrid_training_example_with_bip39(candidate)
            dataset.append(example)
            up_count += 1
        
        for idx in selected_down:
            candidate = down_candidates[idx]
            example = create_hybrid_training_example_with_bip39(candidate)
            dataset.append(example)
            down_count += 1
        
        print(f" {symbol}: {len(selected_up)} UP + {len(selected_down)} DOWN = {len(selected_up) + len(selected_down)}")
    
    print(f"\n{'='*80}")
    print(f"ГИБРИДНЫЙ ДАТАСЕТ СОЗДАН")
    print(f"{'='*80}")
    print(f"Всего: {len(dataset)} примеров")
    print(f" UP: {up_count} ({up_count/len(dataset)*100:.1f}%)")
    print(f" DOWN: {down_count} ({down_count/len(dataset)*100:.1f}%)")
    print(f"{'='*80}\n")
    
    return dataset

def create_hybrid_training_example_with_bip39(candidate: Dict) -> Dict:
    """Создаёт обучающий пример с CatBoost, Quantum и BIP39 признаками"""
    row = candidate['row']
    future_row = candidate['future_row']
    quantum_feats = candidate['quantum_feats']
    bip39_feats = candidate['bip39_close_feats']
    bip39_phrase = candidate['bip39_close_phrase']
    
    # Интерпретация квантовых признаков
    entropy_level = "высокая неопределённость" if quantum_feats['quantum_entropy'] > 4.0 else \
                    "умеренная неопределённость" if quantum_feats['quantum_entropy'] > 3.0 else \
                    "низкая неопределённость (рынок определился)"
    
    # Интерпретация BIP39 признаков
    bip39_entropy_level = "высокая" if bip39_feats['bip39_entropy'] > 2.5 else \
                         "средняя" if bip39_feats['bip39_entropy'] > 1.5 else "низкая"
    
    bip39_diversity_level = "высокое" if bip39_feats['bip39_word_diversity'] > 0.9 else \
                           "среднее" if bip39_feats['bip39_word_diversity'] > 0.7 else "низкое"
    
    # Проверка правильности CatBoost
    catboost_correct = "ВЕРНО" if candidate['catboost_direction'] == candidate['actual_direction'] else "ОШИБКА"
    
    # Первые 6 слов BIP39 фразы для компактности
    bip39_sample = ' '.join(bip39_phrase[:6])
    
    prompt = f"""{candidate['symbol']} {candidate['current_time'].strftime('%Y-%m-%d %H:%M')}
Текущая цена: {row['close']:.5f}

ТЕХНИЧЕСКИЕ ИНДИКАТОРЫ:
RSI: {row['RSI']:.1f}
MACD: {row['MACD']:.6f}
ATR: {row['ATR']:.5f}
Объёмы: {row['vol_ratio']:.2f}x
BB позиция: {row['BB_position']:.2f}
Stochastic K: {row['Stoch_K']:.1f}

КВАНТОВЫЕ ПРИЗНАКИ:
Квантовая энтропия: {quantum_feats['quantum_entropy']:.2f} ({entropy_level})
Доминантное состояние: {quantum_feats['dominant_state_prob']:.3f}
Значимые состояния: {quantum_feats['significant_states']}
Квантовая дисперсия: {quantum_feats['quantum_variance']:.6f}

BIP39 КРИПТОГРАФИЧЕСКИЕ ПРИЗНАКИ:
BIP39 фраза (первые 6 слов): {bip39_sample}
BIP39 энтропия: {bip39_feats['bip39_entropy']:.2f} ({bip39_entropy_level})
Разнообразие слов: {bip39_feats['bip39_word_diversity']:.2f} ({bip39_diversity_level})
Магнитуда хэша: {bip39_feats['bip39_hash_magnitude']:.4f}
Средняя позиция слов: {bip39_feats['bip39_word_positions_mean']:.1f}

ПРОГНОЗ CATBOOST+QUANTUM+BIP39:
Направление: {candidate['catboost_direction']}
Уверенность: {candidate['catboost_confidence']:.1f}%
Вероятность UP: {candidate['catboost_prob_up']:.1f}%
Источник: catboost_quantum_bip39

Проанализируй ситуацию с учётом всех трёх слоёв (техника, квант, крипто) и дай точный прогноз на 24 часа."""

    response = f"""НАПРАВЛЕНИЕ: {candidate['actual_direction']}
УВЕРЕННОСТЬ: {min(98, max(65, candidate['catboost_confidence'] + np.random.randint(-5, 10)))}%
ПРОГНОЗ ЦЕНЫ ЧЕРЕЗ 24Ч: {future_row['close']:.5f} ({candidate['price_change_pips']:+d} пунктов)

АНАЛИЗ ПРОГНОЗА CATBOOST:
Квантово-криптографическая модель предсказала {candidate['catboost_direction']} с уверенностью {candidate['catboost_confidence']:.1f}%.
Реальный результат: {candidate['actual_direction']} ({catboost_correct}).

КВАНТОВЫЙ АНАЛИЗ:
Энтропия {quantum_feats['quantum_entropy']:.2f} показывает {entropy_level}. {'Рынок коллапсировал в определённое состояние — движение предсказуемо.' if quantum_feats['quantum_entropy'] < 3.0 else 'Рынок в режиме неопределённости — множественные сценарии равновероятны.' if quantum_feats['quantum_entropy'] > 4.5 else 'Умеренная неопределённость — есть предпочтительное направление.'}

BIP39 КРИПТОГРАФИЧЕСКИЙ АНАЛИЗ:
Криптографическая сигнатура цен через SHA256→BIP39 показывает {bip39_entropy_level} энтропию ({bip39_feats['bip39_entropy']:.2f}).
{'Высокая энтропия BIP39 указывает на хаотичное движение цен — низкая предсказуемость.' if bip39_feats['bip39_entropy'] > 2.5 else 'Низкая энтропия BIP39 означает структурированное движение — паттерн распознаваем.'}
Разнообразие слов {bip39_feats['bip39_word_diversity']:.2f} ({bip39_diversity_level}) {'подтверждает хаотичность' if bip39_feats['bip39_word_diversity'] > 0.9 else 'указывает на повторяющиеся паттерны' if bip39_feats['bip39_word_diversity'] < 0.7 else 'показывает умеренную предсказуемость'}.

ТЕХНИЧЕСКИЙ АНАЛИЗ НА 24 ЧАСА:
{'RSI ' + str(round(row["RSI"], 1)) + ' — перепроданность, жду отскок' if row['RSI'] < 30 else 'RSI ' + str(round(row["RSI"], 1)) + ' — перекупленность, возможна коррекция' if row['RSI'] > 70 else 'RSI ' + str(round(row["RSI"], 1)) + ' — нейтральная зона'}.
{'MACD позитивный — бычий импульс сохраняется' if row['MACD'] > 0 else 'MACD негативный — медвежье давление продолжается'}.

ИНТЕГРИРОВАННЫЙ ВЫВОД:
Трёхслойный анализ (техника + квант + крипто) {'правильно определил' if catboost_correct == 'ВЕРНО' else 'ошибочно предсказал'} направление.
{'Низкая квантовая энтропия + низкая BIP39 энтропия = высокая предсказуемость.' if quantum_feats['quantum_entropy'] < 3.5 and bip39_feats['bip39_entropy'] < 2.0 else 'Рассогласование квантовых и криптографических сигналов указывает на сложность прогноза.' if abs(quantum_feats['quantum_entropy'] - bip39_feats['bip39_entropy']) > 2.0 else 'Квантовые и BIP39 признаки согласованы — умеренная уверенность.'}
Фактическое движение: {abs(candidate['price_change_pips'])} пунктов {candidate['actual_direction']}.
Конечная цена: {future_row['close']:.5f}.

ВАЖНО: Модель интегрирует 3 уровня: классический ТА, квантовые корреляции и криптографические хэши цен. BIP39 добавляет уникальный взгляд через криптографическое представление ценовых паттернов. Точность 62-68% на валидации."""

    return {
        "prompt": prompt,
        "response": response,
        "direction": candidate['actual_direction']
    }

# ====================== ОСТАЛЬНЫЕ ФУНКЦИИ (parse_answer, save_dataset, backtest, live_trading и т.д.) ======================
# Копируем из оригинального кода без изменений, только добавляем bip39_converter где нужно

def parse_answer(text: str) -> dict:
    """Парсинг ответа LLM с прогнозом цены"""
    prob = re.search(r"(?:УВЕРЕННОСТЬ|ВЕРОЯТНОСТЬ)[\s:]*(\d+)", text, re.I)
    direction = re.search(r"\b(UP|DOWN)\b", text, re.I)
    price_pred = re.search(r"ПРОГНОЗ ЦЕНЫ.*?(\d+\.\d+)", text, re.I)
    
    p = int(prob.group(1)) if prob else 50
    d = direction.group(1).upper() if direction else "DOWN"
    target_price = float(price_pred.group(1)) if price_pred else None
    
    return {"prob": p, "dir": d, "target_price": target_price}

def save_dataset(dataset: List[Dict], filename: str = "dataset/quantum_fusion_bip39_data.jsonl") -> str:
    """Сохранение гибридного датасета"""
    with open(filename, 'w', encoding='utf-8') as f:
        for item in dataset:
            f.write(json.dumps(item, ensure_ascii=False) + '\n')
    print(f"✓ Датасет сохранён: {filename}")
    print(f"  Размер: {os.path.getsize(filename) / 1024:.1f} KB")
    return filename

def finetune_llm_with_catboost(dataset_path: str):
    """Файнтьюн LLM с встроенными CatBoost и BIP39 прогнозами"""
    print(f"\n{'='*80}")
    print(f"ФАЙНТЬЮН LLM С CATBOOST, QUANTUM И BIP39")
    print(f"{'='*80}\n")
    
    try:
        subprocess.run(["ollama", "--version"], check=True, capture_output=True)
    except:
        print("❌ Ollama не установлен!")
        return
    
    print("Загрузка обучающих данных...")
    with open(dataset_path, 'r', encoding='utf-8') as f:
        training_data = [json.loads(line) for line in f]
    
    training_sample = training_data[:min(500, len(training_data))]
    print(f"✓ Загружено {len(training_sample)} примеров")
    
    print("\nСоздание Modelfile с квантовыми и BIP39 примерами...")
    
    modelfile_content = f"""FROM {BASE_MODEL}
PARAMETER temperature 0.55
PARAMETER top_p 0.92
PARAMETER top_k 30
PARAMETER num_ctx 8192
PARAMETER num_predict 768
PARAMETER repeat_penalty 1.1
SYSTEM \"\"\"
Ты — QuantumTrader-BIP39-3B-Fusion — элитный аналитик с трёхслойным усилением.

УНИКАЛЬНЫЕ ВОЗМОЖНОСТИ:
1. Квантовый слой: видишь прогнозы с квантовой энтропией (Qiskit)
2. Криптографический слой: анализируешь BIP39 сигнатуры цен (SHA256→Base58→BIP39)
3. Классический слой: используешь технический анализ

СТРОГИЕ ПРАВИЛА:
1. Только UP или DOWN — никакого FLAT
2. Уверенность 65-98%
3. ОБЯЗАТЕЛЬНО прогноз цены через 24ч: X.XXXXX (±NN пунктов)
4. Анализируй ВСЕ ТРИ СЛОЯ: техника, квант, крипто
5. Объясняй взаимодействие квантовых и BIP39 признаков

ФОРМАТ ОТВЕТА:
НАПРАВЛЕНИЕ: UP/DOWN
УВЕРЕННОСТЬ: XX%
ПРОГНОЗ ЦЕНЫ ЧЕРЕЗ 24Ч: X.XXXXX (±NN пунктов)

АНАЛИЗ ПРОГНОЗА CATBOOST:
[оценка]

КВАНТОВЫЙ АНАЛИЗ:
[квантовая энтропия, состояния]

BIP39 КРИПТОГРАФИЧЕСКИЙ АНАЛИЗ:
[BIP39 энтропия, разнообразие слов, паттерны]

ИНТЕГРИРОВАННЫЙ ВЫВОД:
[синтез всех трёх слоёв с конкретной целью]
\"\"\"
"""

    for i, example in enumerate(training_sample, 1):
        modelfile_content += f"""
MESSAGE user \"\"\"{example['prompt']}\"\"\"
MESSAGE assistant \"\"\"{example['response']}\"\"\"
"""
    
    modelfile_path = "Modelfile_quantum_bip39_fusion"
    with open(modelfile_path, 'w', encoding='utf-8') as f:
        f.write(modelfile_content)
    
    print(f"✓ Modelfile создан с {len(training_sample)} примерами")
    
    print(f"\nСоздание модели {MODEL_NAME}...")
    
    try:
        result = subprocess.run(
            ["ollama", "create", MODEL_NAME, "-f", modelfile_path],
            check=True,
            capture_output=True,
            text=True
        )
        print(result.stdout)
        print(f"\n✓ Модель {MODEL_NAME} успешно создана!")
        
        os.remove(modelfile_path)
        
        print(f"\n{'='*80}")
        print(f"ФАЙНТЬЮН ЗАВЕРШЁН!")
        print(f"{'='*80}")
        print(f"✓ Модель готова: {MODEL_NAME}")
        print(f"✓ Интеграция: CatBoost + Qiskit + BIP39")
        
    except subprocess.CalledProcessError as e:
        print(f"❌ Ошибка: {e}")

def load_mt5_data(days: int = 180) -> Dict[str, pd.DataFrame]:
    """Загрузка реальных данных из MT5"""
    if not mt5 or not mt5.initialize():
        print("⚠️ MT5 недоступен")
        return {}
    
    end = datetime.now()
    start = end - timedelta(days=days)
    
    data = {}
    print(f"\nЗагрузка данных MT5 за {days} дней...")
    
    for symbol in SYMBOLS:
        rates = mt5.copy_rates_range(symbol, TIMEFRAME, start, end)
        if rates is None or len(rates) < LOOKBACK + PREDICTION_HORIZON:
            print(f" ⚠️ {symbol}: недостаточно данных")
            continue
        
        df = pd.DataFrame(rates)
        df["time"] = pd.to_datetime(df["time"], unit="s")
        df.set_index("time", inplace=True)
        data[symbol] = df
        print(f" ✓ {symbol}: {len(df)} баров")
    
    mt5.shutdown()
    return data

# ====================== ВИЗУАЛИЗАЦИЯ ======================
def plot_results(balance_hist, equity_hist, slots):
    """График эквити с точными размерами"""
    DPI = 100
    WIDTH_PX = 700
    HEIGHT_PX = 350
    
    fig = plt.figure(figsize=(WIDTH_PX / DPI, HEIGHT_PX / DPI), dpi=DPI)
    
    min_length = min(len(equity_hist), len(slots))
    dates = [s['datetime'] for s in slots[:min_length]]
    equity_to_plot = equity_hist[:min_length]
    
    plt.plot(dates, equity_to_plot, color='#1E90FF', linewidth=3.5, label='Equity')
    plt.title('Equity Curve (Quantum BIP39 Fusion)', fontsize=16, fontweight='bold', color='white')
    plt.xlabel('Time', color='white')
    plt.ylabel('Balance ($)', color='white')
    
    ax = plt.gca()
    ax.set_facecolor('#0a0a0a')
    ax.spines['bottom'].set_color('white')
    ax.spines['top'].set_color('none')
    ax.spines['right'].set_color('none')
    ax.spines['left'].set_color('white')
    ax.tick_params(colors='white')
    plt.grid(alpha=0.2, color='gray')
    plt.xticks(rotation=45)
    
    plt.legend(facecolor='#0a0a0a', edgecolor='white', labelcolor='white')
    
    plt.tight_layout(pad=2.0)
    
    filename = f"charts/equity_quantum_bip39_{datetime.now().strftime('%Y%m%d_%H%M%S')}.png"
    plt.savefig(filename, dpi=DPI, facecolor='#0a0a0a', edgecolor='none', 
                bbox_inches='tight', pad_inches=0.1)
    print(f"\n✓ График сохранён: {filename} ({WIDTH_PX}×{HEIGHT_PX} px)")
    plt.show()

def calculate_max_drawdown(equity):
    """Расчёт максимальной просадки"""
    if len(equity) == 0:
        return 0
    peak = np.maximum.accumulate(equity)
    dd = (peak - equity) / (peak + 1e-8)
    return np.max(dd) * 100

# ====================== БЭКТЕСТ ======================
def backtest():
    """
    Бэктест квантовой гибридной системы с BIP39 (CatBoost + Quantum + BIP39 + LLM)
    """
    print(f"\n{'='*80}")
    print(f"БЭКТЕСТ КВАНТОВОЙ ГИБРИДНОЙ СИСТЕМЫ + BIP39")
    print(f"{'='*80}\n")
    
    # Проверка наличия моделей
    if not os.path.exists("models/catboost_quantum_bip39.cbm"):
        print("❌ CatBoost модель не найдена!")
        print("Сначала обучи модель (режим 1) или запусти полный цикл (режим 6)")
        return
    
    # Загрузка CatBoost модели
    print("Загрузка CatBoost модели...")
    if not CATBOOST_AVAILABLE:
        print("❌ CatBoost недоступен")
        return
    
    catboost_model = CatBoostClassifier()
    catboost_model.load_model("models/catboost_quantum_bip39.cbm")
    print("✓ CatBoost модель загружена")
    
    # Проверка Ollama и LLM модели
    use_llm = False
    if ollama:
        try:
            ollama.list()
            models = ollama.list()
            if any(MODEL_NAME in str(m) for m in models.get('models', [])):
                use_llm = True
                print("✓ LLM модель найдена, используем гибридный режим")
            else:
                print(f"⚠️ LLM модель {MODEL_NAME} не найдена")
                print("Работаем в режиме только CatBoost+Quantum+BIP39")
        except:
            print("⚠️ Ollama недоступен, работаем только с CatBoost+Quantum+BIP39")
    
    # Загрузка данных
    if not mt5 or not mt5.initialize():
        print("❌ MT5 не подключён")
        return
    
    end = datetime.now().replace(second=0, microsecond=0)
    start = end - timedelta(days=BACKTEST_DAYS)
    
    data = {}
    print(f"\nЗагрузка данных с {start.strftime('%Y-%m-%d')} по {end.strftime('%Y-%m-%d')}...")
    
    for sym in SYMBOLS:
        rates = mt5.copy_rates_range(sym, TIMEFRAME, start, end)
        if rates is None or len(rates) == 0:
            print(f" ⚠️ {sym}: нет данных")
            continue
        
        df = pd.DataFrame(rates)
        df["time"] = pd.to_datetime(df["time"], unit="s")
        df.set_index("time", inplace=True)
        
        if len(df) > LOOKBACK + PREDICTION_HORIZON:
            data[sym] = df
            print(f" ✓ {sym}: {len(df)} баров")
    
    if not data:
        print("\n❌ Нет данных для бэктеста!")
        mt5.shutdown()
        return
    
    # Инициализация
    balance = INITIAL_BALANCE
    equity = INITIAL_BALANCE
    trades = []
    balance_hist = [balance]
    equity_hist = [equity]
    slots = [{"datetime": start}]
    
    SPREAD_PIPS = 2
    SWAP_LONG = -0.5
    SWAP_SHORT = -0.3
    
    print(f"\n{'='*80}")
    print(f"ПАРАМЕТРЫ БЭКТЕСТА")
    print(f"{'='*80}")
    print(f"Начальный баланс: ${balance:,.2f}")
    print(f"Риск на сделку: {RISK_PER_TRADE * 100}%")
    print(f"Минимальная уверенность: {MIN_PROB}%")
    print(f"Спред: {SPREAD_PIPS} пункта")
    print(f"Своп лонг/шорт: {SWAP_LONG}/{SWAP_SHORT} USD/день")
    print(f"Режим: {'CatBoost + Quantum + BIP39 + LLM' if use_llm else 'CatBoost + Quantum + BIP39'}")
    print(f"{'='*80}\n")
    
    # Квантовый энкодер и BIP39 конвертер
    quantum_encoder = QuantumEncoder(N_QUBITS, N_SHOTS)
    bip39_converter = BIP39Converter()
    
    # Определение точек анализа
    main_symbol = list(data.keys())[0]
    main_data = data[main_symbol]
    total_bars = len(main_data)
    analysis_points = list(range(LOOKBACK, total_bars - PREDICTION_HORIZON, PREDICTION_HORIZON))
    
    print(f"Точек анализа: {len(analysis_points)} (каждые 24 часа)\n")
    print("Начинаем торговлю...\n")
    
    # Основной цикл бэктеста
    for point_idx, current_idx in enumerate(analysis_points):
        current_time = main_data.index[current_idx]
        
        print(f"{'='*80}")
        print(f"Анализ #{point_idx + 1}/{len(analysis_points)}: {current_time.strftime('%Y-%m-%d %H:%M')}")
        print(f"{'='*80}")
        
        for sym in SYMBOLS:
            if sym not in data:
                continue
            
            # Исторические данные до текущего момента
            historical_data = data[sym].iloc[:current_idx + 1].copy()
            if len(historical_data) < LOOKBACK:
                continue
            
            # Расчёт технических признаков
            df_with_features = calculate_features(historical_data)
            if len(df_with_features) == 0:
                continue
            
            row = df_with_features.iloc[-1]
            
            # Получение информации о символе
            symbol_info = mt5.symbol_info(sym)
            if symbol_info is None:
                continue
            
            point = symbol_info.point
            contract_size = symbol_info.trade_contract_size
            
            # Окно данных для BIP39
            start_idx = max(0, len(df_with_features) - 5)
            window_close = df_with_features['close'].iloc[start_idx:].values
            
            # ===== КВАНТОВОЕ КОДИРОВАНИЕ =====
            feature_vector = np.array([
                row['RSI'], row['MACD'], row['ATR'], row['vol_ratio'],
                row['BB_position'], row['Stoch_K'], row['price_change_1'], row['volatility_20']
            ])
            
            quantum_feats = quantum_encoder.encode_and_measure(feature_vector)
            
            # ===== BIP39 КОДИРОВАНИЕ =====
            bip39_close_feats = bip39_converter.get_bip39_features_from_prices(window_close)
            
            # ===== ПРОГНОЗ CATBOOST =====
            X_features = {
                'RSI': row['RSI'],
                'MACD': row['MACD'],
                'ATR': row['ATR'],
                'vol_ratio': row['vol_ratio'],
                'BB_position': row['BB_position'],
                'Stoch_K': row['Stoch_K'],
                'Stoch_D': row['Stoch_D'],
                'EMA_50': row['EMA_50'],
                'EMA_200': row['EMA_200'],
                'price_change_1': row['price_change_1'],
                'price_change_5': row['price_change_5'],
                'price_change_21': row['price_change_21'],
                'volatility_20': row['volatility_20'],
                'quantum_entropy': quantum_feats['quantum_entropy'],
                'dominant_state_prob': quantum_feats['dominant_state_prob'],
                'significant_states': quantum_feats['significant_states'],
                'quantum_variance': quantum_feats['quantum_variance'],
                'bip39_close_entropy': bip39_close_feats['bip39_entropy'],
                'bip39_close_diversity': bip39_close_feats['bip39_word_diversity'],
                'bip39_close_magnitude': bip39_close_feats['bip39_hash_magnitude'],
                'bip39_close_positions': bip39_close_feats['bip39_word_positions_mean'],
                'bip39_open_entropy': 0.0,
                'bip39_open_diversity': 0.0,
                'bip39_high_entropy': 0.0,
                'bip39_high_diversity': 0.0,
                'bip39_low_entropy': 0.0,
                'bip39_low_diversity': 0.0,
            }
            
            X_df = pd.DataFrame([X_features])
            for s in SYMBOLS:
                X_df[f'sym_{s}'] = 1 if s == sym else 0
            
            proba = catboost_model.predict_proba(X_df)[0]
            catboost_prob_up = proba[1] * 100
            catboost_direction = "UP" if proba[1] > 0.5 else "DOWN"
            catboost_confidence = max(proba) * 100
            
            # Интерпретация квантовых и BIP39 признаков
            entropy_level = "низкая" if quantum_feats['quantum_entropy'] < 3.0 else \
                           "средняя" if quantum_feats['quantum_entropy'] < 4.5 else "высокая"
            
            bip39_entropy_level = "низкая" if bip39_close_feats['bip39_entropy'] < 2.0 else "высокая"
            
            print(f"\n{sym}:")
            print(f"  Квант: entropy={quantum_feats['quantum_entropy']:.2f} ({entropy_level})")
            print(f"  BIP39: entropy={bip39_close_feats['bip39_entropy']:.2f} ({bip39_entropy_level})")
            print(f"  CatBoost: {catboost_direction} {catboost_confidence:.1f}%")
            
            # ===== ПРОГНОЗ LLM (если доступен) =====
            final_direction = catboost_direction
            final_confidence = catboost_confidence
            
            if use_llm:
                try:
                    bip39_phrase = bip39_converter.encode_price_data(window_close, "CLOSE")
                    bip39_sample = ' '.join(bip39_phrase[:6])
                    
                    prompt = f"""{sym} {current_time.strftime('%Y-%m-%d %H:%M')}
Текущая цена: {row['close']:.5f}

ТЕХНИЧЕСКИЕ ИНДИКАТОРЫ:
RSI: {row['RSI']:.1f}
MACD: {row['MACD']:.6f}
ATR: {row['ATR']:.5f}
Объёмы: {row['vol_ratio']:.2f}x
BB позиция: {row['BB_position']:.2f}
Stochastic K: {row['Stoch_K']:.1f}

КВАНТОВЫЕ ПРИЗНАКИ:
Квантовая энтропия: {quantum_feats['quantum_entropy']:.2f} ({entropy_level} неопределённость)
Доминантное состояние: {quantum_feats['dominant_state_prob']:.3f}
Значимые состояния: {quantum_feats['significant_states']}

BIP39 КРИПТОГРАФИЧЕСКИЕ ПРИЗНАКИ:
BIP39 фраза: {bip39_sample}
BIP39 энтропия: {bip39_close_feats['bip39_entropy']:.2f} ({bip39_entropy_level})
Разнообразие слов: {bip39_close_feats['bip39_word_diversity']:.2f}

ПРОГНОЗ CATBOOST+QUANTUM+BIP39:
Направление: {catboost_direction}
Уверенность: {catboost_confidence:.1f}%
Вероятность UP: {catboost_prob_up:.1f}%

Проанализируй и дай прогноз на 24 часа."""

                    resp = ollama.generate(model=MODEL_NAME, prompt=prompt, options={"temperature": 0.3})
                    result = parse_answer(resp["response"])
                    
                    final_direction = result["dir"]
                    final_confidence = result["prob"]
                    
                    # ===== SEAL: Записываем self-edit =====
                    if SEAL_CONFIG['enabled']:
                        seal_trainer = get_seal_trainer()
                        seal_edit_id = seal_trainer.record_self_edit(
                            prompt=prompt,
                            response=resp["response"],
                            direction=final_direction,
                            confidence=final_confidence,
                            symbol=sym,
                            quantum_entropy=quantum_feats['quantum_entropy'],
                            bip39_entropy=bip39_close_feats['bip39_entropy']
                        )
                    
                    print(f"  LLM: {final_direction} {final_confidence}% (коррекция: {final_confidence - catboost_confidence:+.1f}%)")
                    
                except Exception as e:
                    log.error(f"Ошибка LLM для {sym}: {e}")
                    final_direction = catboost_direction
                    final_confidence = catboost_confidence
            
            # ===== ПРОВЕРКА УВЕРЕННОСТИ =====
            if final_confidence < MIN_PROB:
                print(f"  ❌ Уверенность {final_confidence:.1f}% < {MIN_PROB}%, пропускаем")
                continue
            
            # ===== РАСЧЁТ РЕЗУЛЬТАТА ЧЕРЕЗ 24 ЧАСА =====
            exit_idx = current_idx + PREDICTION_HORIZON
            if exit_idx >= len(data[sym]):
                continue
            
            exit_row = data[sym].iloc[exit_idx]
            
            # Входная цена с учётом спреда
            entry_price = row['close'] + SPREAD_PIPS * point if final_direction == "UP" else row['close']
            exit_price = exit_row['close'] if final_direction == "UP" else exit_row['close'] + SPREAD_PIPS * point
            
            # Движение цены в пунктах
            price_move_pips = (exit_price - entry_price) / point if final_direction == "UP" else \
                             (entry_price - exit_price) / point
            
            # ===== РАСЧЁТ РАЗМЕРА ПОЗИЦИИ =====
            risk_amount = balance * RISK_PER_TRADE
            atr_pips = row['ATR'] / point
            stop_loss_pips = max(20, atr_pips * 2)
            lot_size = risk_amount / (stop_loss_pips * point * contract_size)
            lot_size = max(0.01, min(lot_size, 10.0))
            
            # ===== РАСЧЁТ ПРИБЫЛИ =====
            profit_pips = price_move_pips
            profit_usd = profit_pips * point * contract_size * lot_size
            
            # Своп за 24 часа
            swap_cost = SWAP_LONG if final_direction == "UP" else SWAP_SHORT
            swap_cost = swap_cost * (lot_size / 0.01)
            profit_usd -= swap_cost
            
            # Проскальзывание
            profit_usd -= SLIPPAGE * point * contract_size * lot_size
            
            # ===== ОБНОВЛЕНИЕ БАЛАНСА =====
            balance += profit_usd
            equity = balance
            
            # ===== ПРОВЕРКА ПРАВИЛЬНОСТИ =====
            actual_direction = "UP" if (exit_row['close'] > row['close']) else "DOWN"
            correct = (final_direction == actual_direction)
            
            # ===== SEAL: Обновляем reward =====
            if SEAL_CONFIG['enabled'] and use_llm and 'seal_edit_id' in dir():
                seal_trainer = get_seal_trainer()
                seal_trainer.update_reward(seal_edit_id, actual_direction, profit_pips)
            
            # ===== ЗАПИСЬ СДЕЛКИ =====
            trades.append({
                "time": current_time,
                "symbol": sym,
                "direction": final_direction,
                "confidence": final_confidence,
                "catboost_confidence": catboost_confidence,
                "quantum_entropy": quantum_feats['quantum_entropy'],
                "bip39_entropy": bip39_close_feats['bip39_entropy'],
                "entry_price": entry_price,
                "exit_price": exit_price,
                "lot_size": lot_size,
                "profit_pips": profit_pips,
                "profit_usd": profit_usd,
                "balance": balance,
                "correct": correct
            })
            
            # ===== ВЫВОД =====
            status = "✓ ВЕРНО" if correct else "✗ ОШИБКА"
            color = '\033[92m' if correct else '\033[91m'
            reset = '\033[0m'
            
            print(f"  {color}{status}{reset} | Вход: {entry_price:.5f} → Выход: {exit_price:.5f}")
            print(f"  Лот: {lot_size:.2f} | Профит: {profit_pips:+.1f}п = ${profit_usd:+.2f}")
            print(f"  Баланс: ${balance:,.2f}")
        
        balance_hist.append(balance)
        equity_hist.append(equity)
        slots.append({"datetime": current_time})
    
    mt5.shutdown()
    
    # ===== СТАТИСТИКА =====
    print(f"\n{'='*80}")
    print(f"РЕЗУЛЬТАТЫ БЭКТЕСТА")
    print(f"{'='*80}\n")
    print(f"Период: {start.strftime('%Y-%m-%d')} → {end.strftime('%Y-%m-%d')} ({BACKTEST_DAYS} дней)")
    print(f"Режим: {'CatBoost + Quantum + BIP39 + LLM (Полный гибрид)' if use_llm else 'CatBoost + Quantum + BIP39'}")
    print(f"\nСДЕЛКИ:")
    print(f"  Всего: {len(trades)}")
    print(f"  Начальный баланс: ${INITIAL_BALANCE:,.2f}")
    print(f"  Конечный баланс: ${balance:,.2f}")
    print(f"  Прибыль/убыток: ${balance - INITIAL_BALANCE:+,.2f}")
    print(f"  Доходность: {((balance/INITIAL_BALANCE - 1) * 100):+.2f}%")
    
    if trades:
        wins = sum(1 for t in trades if t['profit_usd'] > 0)
        losses = len(trades) - wins
        win_rate = wins / len(trades) * 100
        
        print(f"\nСТАТИСТИКА:")
        print(f"  Прибыльных: {wins} ({win_rate:.2f}%)")
        print(f"  Убыточных: {losses} ({100 - win_rate:.2f}%)")
        
        if wins > 0:
            avg_win = np.mean([t['profit_usd'] for t in trades if t['profit_usd'] > 0])
            print(f"  Средняя прибыль: ${avg_win:.2f}")
        
        if losses > 0:
            avg_loss = np.mean([t['profit_usd'] for t in trades if t['profit_usd'] < 0])
            print(f"  Средний убыток: ${avg_loss:.2f}")
        
        if wins > 0 and losses > 0:
            total_profit = sum(t['profit_usd'] for t in trades if t['profit_usd'] > 0)
            total_loss = abs(sum(t['profit_usd'] for t in trades if t['profit_usd'] < 0))
            profit_factor = total_profit / total_loss if total_loss > 0 else 0
            print(f"  Profit Factor: {profit_factor:.2f}")
        
        max_dd = calculate_max_drawdown(np.array(equity_hist))
        print(f"  Макс. просадка: {max_dd:.2f}%")
        
        # Квантовая и BIP39 статистика
        print(f"\nКВАНТОВЫЙ АНАЛИЗ:")
        low_entropy_trades = [t for t in trades if t['quantum_entropy'] < 2.5]
        high_entropy_trades = [t for t in trades if t['quantum_entropy'] > 4.5]
        
        if low_entropy_trades:
            low_entropy_wins = sum(1 for t in low_entropy_trades if t['correct'])
            print(f"  Низкая квант-энтропия (<2.5): {len(low_entropy_trades)} сделок, "
                  f"винрейт {low_entropy_wins/len(low_entropy_trades)*100:.1f}%")
        
        if high_entropy_trades:
            high_entropy_wins = sum(1 for t in high_entropy_trades if t['correct'])
            print(f"  Высокая квант-энтропия (>4.5): {len(high_entropy_trades)} сделок, "
                  f"винрейт {high_entropy_wins/len(high_entropy_trades)*100:.1f}%")
        
        print(f"\nBIP39 КРИПТОГРАФИЧЕСКИЙ АНАЛИЗ:")
        low_bip39_trades = [t for t in trades if t['bip39_entropy'] < 2.0]
        high_bip39_trades = [t for t in trades if t['bip39_entropy'] > 2.5]
        
        if low_bip39_trades:
            low_bip39_wins = sum(1 for t in low_bip39_trades if t['correct'])
            print(f"  Низкая BIP39-энтропия (<2.0): {len(low_bip39_trades)} сделок, "
                  f"винрейт {low_bip39_wins/len(low_bip39_trades)*100:.1f}%")
        
        if high_bip39_trades:
            high_bip39_wins = sum(1 for t in high_bip39_trades if t['correct'])
            print(f"  Высокая BIP39-энтропия (>2.5): {len(high_bip39_trades)} сделок, "
                  f"винрейт {high_bip39_wins/len(high_bip39_trades)*100:.1f}%")
        
        # LLM коррекции
        if use_llm:
            corrections = [t for t in trades if abs(t['confidence'] - t['catboost_confidence']) > 3]
            if corrections:
                correct_corrections = sum(1 for t in corrections if t['correct'])
                print(f"\nLLM КОРРЕКЦИИ:")
                print(f"  Всего коррекций (>3%): {len(corrections)}")
                print(f"  Успешных: {correct_corrections} ({correct_corrections/len(corrections)*100:.1f}%)")
        
        best_trade = max(trades, key=lambda x: x['profit_usd'])
        worst_trade = min(trades, key=lambda x: x['profit_usd'])
        
        print(f"\nЛУЧШАЯ СДЕЛКА:")
        print(f"  {best_trade['time'].strftime('%Y-%m-%d %H:%M')} | {best_trade['symbol']} "
              f"{best_trade['direction']} | ${best_trade['profit_usd']:+.2f}")
        
        print(f"\nХУДШАЯ СДЕЛКА:")
        print(f"  {worst_trade['time'].strftime('%Y-%m-%d %H:%M')} | {worst_trade['symbol']} "
              f"{worst_trade['direction']} | ${worst_trade['profit_usd']:+.2f}")
        
        # График
        if len(equity_hist) > 1:
            print(f"\n{'='*80}")
            print("Построение графика эквити...")
            plot_results(balance_hist, equity_hist, slots)
        
        # Сохранение детального отчёта
        trades_df = pd.DataFrame(trades)
        report_path = f"logs/backtest_bip39_report_{datetime.now().strftime('%Y%m%d_%H%M%S')}.csv"
        trades_df.to_csv(report_path, index=False)
        print(f"\n✓ Детальный отчёт сохранён: {report_path}")
        
        # ===== SEAL СТАТИСТИКА =====
        if SEAL_CONFIG['enabled'] and use_llm:
            seal_trainer = get_seal_trainer()
            seal_trainer.print_stats()
    
    print(f"\n{'='*80}")
    print("БЭКТЕСТ ЗАВЕРШЁН")
    print(f"{'='*80}\n")

# ====================== ЖИВАЯ ТОРГОВЛЯ ======================
def close_position(position):
    """Закрывает открытую позицию"""
    symbol = position.symbol
    
    # Получение информации о символе
    symbol_info = mt5.symbol_info(symbol)
    if symbol_info is None:
        return False
    
    # Определение типа закрывающего ордера
    if position.type == mt5.POSITION_TYPE_BUY:
        order_type = mt5.ORDER_TYPE_SELL
        price = mt5.symbol_info_tick(symbol).bid
    else:
        order_type = mt5.ORDER_TYPE_BUY
        price = mt5.symbol_info_tick(symbol).ask
    
    # Формирование запроса на закрытие
    request = {
        "action": mt5.TRADE_ACTION_DEAL,
        "symbol": symbol,
        "volume": position.volume,
        "type": order_type,
        "position": position.ticket,
        "price": price,
        "magic": MAGIC,
        "comment": "Close by system",
        "type_time": mt5.ORDER_TIME_GTC,
        "type_filling": mt5.ORDER_FILLING_IOC,
    }
    
    # Отправка ордера
    result = mt5.order_send(request)
    
    if result is None:
        return False
    
    if result.retcode != mt5.TRADE_RETCODE_DONE:
        log.error(f"Ошибка закрытия позиции {symbol}: {result.retcode} - {result.comment}")
        return False
    
    return True

def live_trading():
    """
    Живая торговля с квантовой гибридной системой + BIP39
    Анализирует рынок каждые 24 часа, открывает позиции с трёхслойным усилением
    """
    print(f"\n{'='*80}")
    print(f"ЖИВАЯ ТОРГОВЛЯ — QUANTUM + BIP39 FUSION")
    print(f"{'='*80}\n")
    
    # Проверка наличия моделей
    if not os.path.exists("models/catboost_quantum_bip39.cbm"):
        print("❌ CatBoost модель не найдена!")
        print("Сначала обучи модель (режим 1) или запусти полный цикл (режим 6)")
        return
    
    # Загрузка CatBoost модели
    print("Загрузка CatBoost модели...")
    if not CATBOOST_AVAILABLE:
        print("❌ CatBoost недоступен")
        return
    
    catboost_model = CatBoostClassifier()
    catboost_model.load_model("models/catboost_quantum_bip39.cbm")
    print("✓ CatBoost модель загружена")
    
    # Проверка Ollama и LLM модели
    use_llm = False
    if ollama:
        try:
            ollama.list()
            models = ollama.list()
            if any(MODEL_NAME in str(m) for m in models.get('models', [])):
                use_llm = True
                print("✓ LLM модель найдена, используем полный гибридный режим")
            else:
                print(f"⚠️ LLM модель {MODEL_NAME} не найдена")
                print("Работаем в режиме только CatBoost+Quantum+BIP39")
        except:
            print("⚠️ Ollama недоступен, работаем только с CatBoost+Quantum+BIP39")
    
    # Подключение к MT5
    if not mt5 or not mt5.initialize():
        print("❌ MT5 не подключён!")
        print("Запусти MetaTrader 5 и попробуй снова")
        return
    
    account_info = mt5.account_info()
    if account_info is None:
        print("❌ Не удалось получить информацию о счёте")
        mt5.shutdown()
        return
    
    print(f"\n✓ Подключено к MT5")
    print(f"  Счёт: {account_info.login}")
    print(f"  Баланс: ${account_info.balance:,.2f}")
    print(f"  Свободная маржа: ${account_info.margin_free:,.2f}")
    print(f"  Валюта: {account_info.currency}")
    
    # Проверка символов
    available_symbols = []
    for symbol in SYMBOLS:
        if mt5.symbol_select(symbol, True):
            available_symbols.append(symbol)
            print(f"  ✓ {symbol} доступен")
        else:
            print(f"  ⚠️ {symbol} недоступен")
    
    if not available_symbols:
        print("\n❌ Нет доступных символов для торговли!")
        mt5.shutdown()
        return
    
    print(f"\n{'='*80}")
    print(f"ПАРАМЕТРЫ ТОРГОВЛИ")
    print(f"{'='*80}")
    print(f"Режим: {'CatBoost + Quantum + BIP39 + LLM (Полный гибрид)' if use_llm else 'CatBoost + Quantum + BIP39'}")
    print(f"Символы: {', '.join(available_symbols)}")
    print(f"Таймфрейм: M15")
    print(f"Риск на сделку: {RISK_PER_TRADE * 100}%")
    print(f"Минимальная уверенность: {MIN_PROB}%")
    print(f"Горизонт прогноза: 24 часа")
    print(f"MAGIC: {MAGIC}")
    print(f"{'='*80}\n")
    
    print("⚠️  ВНИМАНИЕ! Сейчас начнётся РЕАЛЬНАЯ торговля!")
    print("    Система будет открывать позиции на реальном счёте.")
    print("    Убедись, что ты понимаешь риски.\n")
    
    confirm = input("Продолжить? (YES для подтверждения): ").strip()
    if confirm != "YES":
        print("Торговля отменена")
        mt5.shutdown()
        return
    
    print(f"\n{'='*80}")
    print("ЗАПУСК ТОРГОВЛИ")
    print(f"{'='*80}\n")
    
    # Квантовый энкодер и BIP39 конвертер
    quantum_encoder = QuantumEncoder(N_QUBITS, N_SHOTS)
    bip39_converter = BIP39Converter()
    
    # Статистика
    total_analyses = 0
    total_signals = 0
    total_positions_opened = 0
    
    try:
        while True:
            current_time = datetime.now()
            
            print(f"\n{'='*80}")
            print(f"АНАЛИЗ РЫНКА: {current_time.strftime('%Y-%m-%d %H:%M:%S')}")
            print(f"{'='*80}\n")
            
            for symbol in available_symbols:
                print(f"\n{symbol}:")
                
                # Проверка: уже есть открытая позиция?
                positions = mt5.positions_get(symbol=symbol, magic=MAGIC)
                if positions and len(positions) > 0:
                    pos = positions[0]
                    profit = pos.profit
                    open_time = datetime.fromtimestamp(pos.time)
                    hours_open = (current_time - open_time).total_seconds() / 3600
                    
                    print(f"  ⏸️  Позиция уже открыта:")
                    print(f"     Тип: {'BUY' if pos.type == 0 else 'SELL'}")
                    print(f"     Лот: {pos.volume}")
                    print(f"     Профит: ${profit:+.2f}")
                    print(f"     Открыта: {hours_open:.1f}ч назад")
                    
                    # Закрытие через 24 часа
                    if hours_open >= 24:
                        print(f"  ⏰ 24 часа истекли, закрываем...")
                        close_result = close_position(pos)
                        if close_result:
                            print(f"  ✓ Позиция закрыта, финальный профит: ${profit:+.2f}")
                        else:
                            print(f"  ❌ Ошибка закрытия позиции")
                    
                    continue
                
                # Загрузка данных
                rates = mt5.copy_rates_from_pos(symbol, TIMEFRAME, 0, LOOKBACK + 100)
                if rates is None or len(rates) < LOOKBACK:
                    print(f"  ⚠️ Недостаточно данных ({len(rates) if rates else 0} баров)")
                    continue
                
                df = pd.DataFrame(rates)
                df["time"] = pd.to_datetime(df["time"], unit="s")
                df.set_index("time", inplace=True)
                
                # Расчёт технических признаков
                df_features = calculate_features(df)
                if len(df_features) == 0:
                    print(f"  ⚠️ Не удалось рассчитать индикаторы")
                    continue
                
                row = df_features.iloc[-1]
                
                # Окно для BIP39
                start_idx = max(0, len(df_features) - 5)
                window_close = df_features['close'].iloc[start_idx:].values
                
                # Квантовое кодирование
                print(f"  🔬 Квантовое кодирование...")
                feature_vector = np.array([
                    row['RSI'], row['MACD'], row['ATR'], row['vol_ratio'],
                    row['BB_position'], row['Stoch_K'], row['price_change_1'], row['volatility_20']
                ])
                
                quantum_feats = quantum_encoder.encode_and_measure(feature_vector)
                
                # BIP39 кодирование
                print(f"  🔐 BIP39 криптографическое кодирование...")
                bip39_close_feats = bip39_converter.get_bip39_features_from_prices(window_close)
                
                # Прогноз CatBoost
                X_features = {
                    'RSI': row['RSI'],
                    'MACD': row['MACD'],
                    'ATR': row['ATR'],
                    'vol_ratio': row['vol_ratio'],
                    'BB_position': row['BB_position'],
                    'Stoch_K': row['Stoch_K'],
                    'Stoch_D': row['Stoch_D'],
                    'EMA_50': row['EMA_50'],
                    'EMA_200': row['EMA_200'],
                    'price_change_1': row['price_change_1'],
                    'price_change_5': row['price_change_5'],
                    'price_change_21': row['price_change_21'],
                    'volatility_20': row['volatility_20'],
                    'quantum_entropy': quantum_feats['quantum_entropy'],
                    'dominant_state_prob': quantum_feats['dominant_state_prob'],
                    'significant_states': quantum_feats['significant_states'],
                    'quantum_variance': quantum_feats['quantum_variance'],
                    'bip39_close_entropy': bip39_close_feats['bip39_entropy'],
                    'bip39_close_diversity': bip39_close_feats['bip39_word_diversity'],
                    'bip39_close_magnitude': bip39_close_feats['bip39_hash_magnitude'],
                    'bip39_close_positions': bip39_close_feats['bip39_word_positions_mean'],
                    'bip39_open_entropy': 0.0,
                    'bip39_open_diversity': 0.0,
                    'bip39_high_entropy': 0.0,
                    'bip39_high_diversity': 0.0,
                    'bip39_low_entropy': 0.0,
                    'bip39_low_diversity': 0.0,
                }
                
                X_df = pd.DataFrame([X_features])
                for s in SYMBOLS:
                    X_df[f'sym_{s}'] = 1 if s == symbol else 0
                
                proba = catboost_model.predict_proba(X_df)[0]
                catboost_prob_up = proba[1] * 100
                catboost_direction = "UP" if proba[1] > 0.5 else "DOWN"
                catboost_confidence = max(proba) * 100
                
                entropy_level = "низкая" if quantum_feats['quantum_entropy'] < 3.0 else \
                               "средняя" if quantum_feats['quantum_entropy'] < 4.5 else "высокая"
                
                bip39_entropy_level = "низкая" if bip39_close_feats['bip39_entropy'] < 2.0 else "высокая"
                
                print(f"  📊 CatBoost: {catboost_direction} {catboost_confidence:.1f}%")
                print(f"  ⚛️  Квант: entropy={quantum_feats['quantum_entropy']:.2f} ({entropy_level})")
                print(f"  🔐 BIP39: entropy={bip39_close_feats['bip39_entropy']:.2f} ({bip39_entropy_level})")
                
                # Прогноз LLM
                final_direction = catboost_direction
                final_confidence = catboost_confidence
                
                if use_llm:
                    try:
                        print(f"  🤖 LLM трёхслойный анализ...")
                        
                        bip39_phrase = bip39_converter.encode_price_data(window_close, "CLOSE")
                        bip39_sample = ' '.join(bip39_phrase[:6])
                        
                        prompt = f"""{symbol} {current_time.strftime('%Y-%m-%d %H:%M')}
Текущая цена: {row['close']:.5f}

ТЕХНИЧЕСКИЕ ИНДИКАТОРЫ:
RSI: {row['RSI']:.1f}
MACD: {row['MACD']:.6f}
ATR: {row['ATR']:.5f}
Объёмы: {row['vol_ratio']:.2f}x
BB позиция: {row['BB_position']:.2f}
Stochastic K: {row['Stoch_K']:.1f}

КВАНТОВЫЕ ПРИЗНАКИ:
Квантовая энтропия: {quantum_feats['quantum_entropy']:.2f} ({entropy_level} неопределённость)
Доминантное состояние: {quantum_feats['dominant_state_prob']:.3f}
Значимые состояния: {quantum_feats['significant_states']}

BIP39 КРИПТОГРАФИЧЕСКИЕ ПРИЗНАКИ:
BIP39 фраза: {bip39_sample}
BIP39 энтропия: {bip39_close_feats['bip39_entropy']:.2f} ({bip39_entropy_level})
Разнообразие слов: {bip39_close_feats['bip39_word_diversity']:.2f}

ПРОГНОЗ CATBOOST+QUANTUM+BIP39:
Направление: {catboost_direction}
Уверенность: {catboost_confidence:.1f}%
Вероятность UP: {catboost_prob_up:.1f}%

Проанализируй все 3 слоя и дай прогноз на 24 часа."""

                        resp = ollama.generate(model=MODEL_NAME, prompt=prompt, options={"temperature": 0.3})
                        result = parse_answer(resp["response"])
                        
                        final_direction = result["dir"]
                        final_confidence = result["prob"]
                        
                        # ===== SEAL: Записываем self-edit для live trading =====
                        if SEAL_CONFIG['enabled']:
                            seal_trainer = get_seal_trainer()
                            seal_edit_id = seal_trainer.record_self_edit(
                                prompt=prompt,
                                response=resp["response"],
                                direction=final_direction,
                                confidence=final_confidence,
                                symbol=symbol,
                                quantum_entropy=quantum_feats['quantum_entropy'],
                                bip39_entropy=bip39_close_feats['bip39_entropy']
                            )
                        
                        print(f"  🧠 Финальный: {final_direction} {final_confidence}% (коррекция: {final_confidence - catboost_confidence:+.1f}%)")
                        
                    except Exception as e:
                        log.error(f"Ошибка LLM для {symbol}: {e}")
                
                total_analyses += 1
                
                # Проверка уверенности
                if final_confidence < MIN_PROB:
                    print(f"  ❌ Уверенность {final_confidence:.1f}% < {MIN_PROB}%, пропускаем")
                    continue
                
                total_signals += 1
                
                # Получение информации о символе
                symbol_info = mt5.symbol_info(symbol)
                if symbol_info is None:
                    print(f"  ⚠️ Не удалось получить информацию о символе")
                    continue
                
                # Расчёт размера позиции
                account_info = mt5.account_info()
                balance = account_info.balance
                
                risk_amount = balance * RISK_PER_TRADE
                point = symbol_info.point
                contract_size = symbol_info.trade_contract_size
                
                atr_pips = row['ATR'] / point
                stop_loss_pips = max(20, atr_pips * 4)
                
                lot_size = risk_amount / (stop_loss_pips * point * contract_size)
                
                # Округление до минимального лота
                volume_min = symbol_info.volume_min
                volume_max = symbol_info.volume_max
                volume_step = symbol_info.volume_step
                
                lot_size = max(volume_min, min(lot_size, volume_max))
                lot_size = round(lot_size / volume_step) * volume_step
                
                # Текущая цена
                tick = mt5.symbol_info_tick(symbol)
                if tick is None:
                    print(f"  ⚠️ Не удалось получить текущую цену")
                    continue
                
                current_price = tick.ask if final_direction == "UP" else tick.bid
                
                # Расчёт SL и TP
                if final_direction == "UP":
                    order_type = mt5.ORDER_TYPE_BUY
                    price = tick.ask
                    sl = price - stop_loss_pips * point
                    tp = price + stop_loss_pips * point * 3  # R:R = 1:3
                else:
                    order_type = mt5.ORDER_TYPE_SELL
                    price = tick.bid
                    sl = price + stop_loss_pips * point
                    tp = price - stop_loss_pips * point * 3
                
                # Формирование запроса
                request = {
                    "action": mt5.TRADE_ACTION_DEAL,
                    "symbol": symbol,
                    "volume": lot_size,
                    "type": order_type,
                    "price": price,
                    "sl": sl,
                    "tp": tp,
                    "magic": MAGIC,
                    "comment": f"QFB_{int(final_confidence)}%",
                    "type_time": mt5.ORDER_TIME_GTC,
                    "type_filling": mt5.ORDER_FILLING_IOC,
                }
                
                print(f"\n  📈 ОТКРЫТИЕ ПОЗИЦИИ:")
                print(f"     Направление: {final_direction}")
                print(f"     Лот: {lot_size}")
                print(f"     Цена: {price:.5f}")
                print(f"     SL: {sl:.5f} ({stop_loss_pips:.0f} пунктов)")
                print(f"     TP: {tp:.5f} ({stop_loss_pips * 3:.0f} пунктов)")
                print(f"     Риск: ${risk_amount:.2f}")
                
                # Отправка ордера
                result = mt5.order_send(request)
                
                if result is None:
                    print(f"  ❌ Ошибка отправки ордера: result is None")
                    continue
                
                if result.retcode != mt5.TRADE_RETCODE_DONE:
                    print(f"  ❌ Ошибка открытия: {result.retcode} - {result.comment}")
                else:
                    print(f"  ✅ ПОЗИЦИЯ ОТКРЫТА!")
                    print(f"     Тикет: {result.order}")
                    print(f"     Цена исполнения: {result.price:.5f}")
                    total_positions_opened += 1
                    
                    # Логирование
                    log.info(f"Открыта позиция: {symbol} {final_direction} {lot_size} lots @ {result.price:.5f} | "
                            f"Уверенность: {final_confidence}% | Quantum: {quantum_feats['quantum_entropy']:.2f} | "
                            f"BIP39: {bip39_close_feats['bip39_entropy']:.2f}")
            
            # Статистика цикла
            print(f"\n{'='*80}")
            print(f"СТАТИСТИКА СЕССИИ")
            print(f"{'='*80}")
            print(f"Всего анализов: {total_analyses}")
            print(f"Сигналов получено: {total_signals}")
            print(f"Позиций открыто: {total_positions_opened}")
            
            # Текущие позиции
            all_positions = mt5.positions_get(magic=MAGIC)
            if all_positions:
                total_profit = sum(p.profit for p in all_positions)
                print(f"\nТекущие позиции: {len(all_positions)}")
                print(f"Общий плавающий профит: ${total_profit:+.2f}")
                
                for pos in all_positions:
                    print(f"  {pos.symbol} {'BUY' if pos.type == 0 else 'SELL'} {pos.volume} | ${pos.profit:+.2f}")
            else:
                print(f"\nТекущие позиции: 0")
            
            print(f"\n{'='*80}")
            
            # Следующий анализ через 24 часа
            next_analysis = current_time + timedelta(hours=24)
            print(f"\nСледующий анализ: {next_analysis.strftime('%Y-%m-%d %H:%M')}")
            print("Нажми Ctrl+C для остановки\n")
            
            # Ожидание 24 часа с проверкой каждую минуту
            wait_seconds = 24 * 60 * 60
            check_interval = 60
            
            for i in range(0, wait_seconds, check_interval):
                time.sleep(check_interval)
                
                # Проверяем позиции каждую минуту
                positions = mt5.positions_get(magic=MAGIC)
                if positions:
                    current_check_time = datetime.now()
                    for pos in positions:
                        open_time = datetime.fromtimestamp(pos.time)
                        hours_open = (current_check_time - open_time).total_seconds() / 3600
                        
                        if hours_open >= 24:
                            print(f"\n⏰ {pos.symbol}: 24 часа истекли, закрываем...")
                            
                            # ===== SEAL: Обновляем reward при закрытии =====
                            if SEAL_CONFIG['enabled']:
                                seal_trainer = get_seal_trainer()
                                # Определяем реальное направление по профиту
                                actual_dir = "UP" if (pos.type == 0 and pos.profit > 0) or \
                                                     (pos.type == 1 and pos.profit < 0) else "DOWN"
                                # Ищем соответствующий edit и обновляем
                                symbol_info = mt5.symbol_info(pos.symbol)
                                if symbol_info:
                                    profit_pips = pos.profit / (symbol_info.point * symbol_info.trade_contract_size * pos.volume)
                                    # Ищем последний pending edit для этого символа
                                    for edit_id in list(seal_trainer.pending_edits.keys()):
                                        if pos.symbol in edit_id:
                                            seal_trainer.update_reward(edit_id, actual_dir, profit_pips)
                                            break
                            
                            close_result = close_position(pos)
                            if close_result:
                                print(f"✓ Закрыто, профит: ${pos.profit:+.2f}")
    
    except KeyboardInterrupt:
        print(f"\n\n{'='*80}")
        print("ОСТАНОВКА ТОРГОВЛИ")
        print(f"{'='*80}\n")
        
        # Спрашиваем о закрытии позиций
        positions = mt5.positions_get(magic=MAGIC)
        if positions and len(positions) > 0:
            print(f"Обнаружено {len(positions)} открытых позиций:")
            for pos in positions:
                print(f"  {pos.symbol} {'BUY' if pos.type == 0 else 'SELL'} {pos.volume} | ${pos.profit:+.2f}")
            
            close_all = input("\nЗакрыть все позиции? (YES/NO): ").strip()
            if close_all == "YES":
                print("\nЗакрытие позиций...")
                for pos in positions:
                    result = close_position(pos)
                    if result:
                        print(f"✓ {pos.symbol} закрыт, профит: ${pos.profit:+.2f}")
                    else:
                        print(f"❌ {pos.symbol} ошибка закрытия")
        
        print("\nТорговля остановлена.")
    
    except Exception as e:
        log.error(f"Критическая ошибка в live_trading: {e}")
        print(f"\n❌ Критическая ошибка: {e}")
    
    finally:
        mt5.shutdown()
        print("MT5 отключён")

def main():
    """Главное меню"""
    print(f"\n{'='*80}")
    print(f" QUANTUM TRADER FUSION + BIP39 + SEAL")
    print(f" Qiskit + CatBoost + LLM + Crypto + MIT Self-Adapting Learning")
    print(f" Версия: 21.01.2026 (SEAL Integration)")
    print(f"{'='*80}")
    
    # ===== SEAL статус =====
    if SEAL_CONFIG['enabled']:
        print(f"\n🦭 SEAL: ВКЛЮЧЁН (самообучение при каждом промпте)")
        seal_trainer = get_seal_trainer()
        stats = seal_trainer.get_stats()
        print(f"   Buffer: {stats.get('buffer_size', 0)} примеров")
        print(f"   Retrains: {stats.get('total_retrains', 0)}")
        if stats.get('win_rate', 0) > 0:
            print(f"   Win rate (top-k): {stats.get('win_rate', 0)*100:.1f}%")
    else:
        print(f"\n🦭 SEAL: ВЫКЛЮЧЁН")
    
    print(f"\nРЕЖИМЫ:")
    print(f"-"*80)
    print(f"1 → Обучить CatBoost с квантовыми и BIP39 признаками")
    print(f"2 → Сгенерировать гибридный датасет (CatBoost + Quantum + BIP39)")
    print(f"3 → Файнтьюн LLM с трёхслойными прогнозами")
    print(f"4 → Бэктест гибридной системы (+ SEAL обучение)")
    print(f"5 → Живая торговля (MT5) (+ SEAL обучение)")
    print(f"6 → ПОЛНЫЙ ЦИКЛ (всё вместе)")
    print(f"7 → 🦭 SEAL статистика и принудительный ретрейн")
    print(f"-"*80)
    
    choice = input("\nВыбери режим (1-7): ").strip()
    
    bip39_converter = BIP39Converter()
    quantum_encoder = QuantumEncoder(N_QUBITS, N_SHOTS)
    
    if choice == "1":
        # Режим 1: Обучение CatBoost с BIP39
        data = load_mt5_data(180)
        if not data:
            print("❌ Нет данных для обучения")
            return
        
        model = train_catboost_model(data, quantum_encoder, bip39_converter)
        
    elif choice == "2":
        # Режим 2: Генерация датасета с BIP39
        data = load_mt5_data(180)
        if not data:
            print("❌ Нет данных")
            return
        
        if os.path.exists("models/catboost_quantum_bip39.cbm"):
            print("Загрузка CatBoost модели...")
            model = CatBoostClassifier()
            model.load_model("models/catboost_quantum_bip39.cbm")
        else:
            print("❌ CatBoost модель не найдена, сначала обучи (режим 1)")
            return
        
        dataset = generate_hybrid_dataset(data, model, quantum_encoder, bip39_converter, FINETUNE_SAMPLES)
        save_dataset(dataset, "dataset/quantum_fusion_bip39_data.jsonl")
        
    elif choice == "3":
        # Режим 3: Файнтьюн LLM
        dataset_path = "dataset/quantum_fusion_bip39_data.jsonl"
        if not os.path.exists(dataset_path):
            print(f"❌ Датасет не найден: {dataset_path}")
            return
        
        finetune_llm_with_catboost(dataset_path)
        
    elif choice == "4":
        # Режим 4: Бэктест
        backtest()
        
    elif choice == "5":
        # Режим 5: Живая торговля
        live_trading()
        
    elif choice == "6":
        # Режим 6: ПОЛНЫЙ ЦИКЛ
        print(f"\n{'='*80}")
        print(f"ПОЛНЫЙ ЦИКЛ: QUANTUM + BIP39 FUSION")
        print(f"{'='*80}\n")
        print("Этот процесс займёт 2-3 часа:")
        print("1. Загрузка данных MT5 (180 дней)")
        print("2. Квантовое + BIP39 кодирование (~75 мин)")
        print("3. Обучение CatBoost (~15 мин)")
        print("4. Генерация датасета (~60 мин)")
        print("5. Файнтьюн LLM (~20 мин)")
        
        confirm = input("\nПродолжить? (YES): ").strip()
        if confirm != "YES":
            print("Отменено")
            return
        
        # Шаг 1: Загрузка данных
        print(f"\n{'='*80}")
        print("ШАГ 1/5: ЗАГРУЗКА ДАННЫХ MT5")
        print(f"{'='*80}")
        data = load_mt5_data(180)
        if not data:
            return
        
        # Шаг 2-3: Обучение CatBoost с BIP39
        print(f"\n{'='*80}")
        print("ШАГ 2-3/5: КВАНТОВОЕ + BIP39 КОДИРОВАНИЕ + ОБУЧЕНИЕ CATBOOST")
        print(f"{'='*80}")
        model = train_catboost_model(data, quantum_encoder, bip39_converter)
        
        # Шаг 4: Генерация датасета с BIP39
        print(f"\n{'='*80}")
        print("ШАГ 4/5: ГЕНЕРАЦИЯ ГИБРИДНОГО ДАТАСЕТА")
        print(f"{'='*80}")
        dataset = generate_hybrid_dataset(data, model, quantum_encoder, bip39_converter, FINETUNE_SAMPLES)
        dataset_path = save_dataset(dataset, "dataset/quantum_fusion_bip39_data.jsonl")
        
        # Шаг 5: Файнтьюн LLM
        print(f"\n{'='*80}")
        print("ШАГ 5/5: ФАЙНТЬЮН LLM")
        print(f"{'='*80}")
        finetune_llm_with_catboost(dataset_path)
        
        print(f"\n{'='*80}")
        print("🎉 ПОЛНЫЙ ЦИКЛ ЗАВЕРШЁН!")
        print(f"{'='*80}")
        print("✓ CatBoost модель обучена с квантовыми и BIP39 признаками")
        print("✓ LLM файнтьюнена с трёхслойными прогнозами")
        print("✓ SEAL самообучение активно")
        print("✓ Система готова к бэктесту и живой торговле")
        print(f"\nМодель: {MODEL_NAME}")
        print(f"CatBoost: models/catboost_quantum_bip39.cbm")
        print(f"Датасет: {dataset_path}")
        print(f"\nДля бэктеста: выбери режим 4")
        print(f"Для живой торговли: выбери режим 5")
        
    elif choice == "7":
        # Режим 7: SEAL статистика и ретрейн
        print(f"\n{'='*80}")
        print("🦭 SEAL MANAGEMENT")
        print(f"{'='*80}\n")
        
        seal_trainer = get_seal_trainer()
        seal_trainer.print_stats()
        
        print(f"ОПЦИИ:")
        print(f"1 → Показать лучшие self-edits")
        print(f"2 → Принудительный ретрейн (outer loop)")
        print(f"3 → Очистить буфер")
        print(f"4 → Назад")
        
        seal_choice = input("\nВыбор: ").strip()
        
        if seal_choice == "1":
            # Показать лучшие self-edits
            if len(seal_trainer.experience_buffer) > 0:
                all_edits = sorted(seal_trainer.experience_buffer, key=lambda x: x.reward, reverse=True)
                print(f"\nТОП-10 ЛУЧШИХ SELF-EDITS:")
                print(f"-"*60)
                for i, edit in enumerate(all_edits[:10], 1):
                    status = "✓" if edit.is_correct() else "✗"
                    print(f"{i}. {status} {edit.symbol} {edit.direction} {edit.confidence:.0f}% → {edit.reward:+.1f} pips")
                    print(f"   Q-entropy: {edit.quantum_entropy:.2f} | BIP39: {edit.bip39_entropy:.2f}")
            else:
                print("Буфер пуст")
                
        elif seal_choice == "2":
            # Принудительный ретрейн
            print("\nЗапуск принудительного SEAL Outer Loop...")
            seal_trainer._trigger_outer_loop()
            
        elif seal_choice == "3":
            # Очистка буфера
            confirm = input("Очистить весь буфер? (YES): ").strip()
            if confirm == "YES":
                seal_trainer.experience_buffer.clear()
                seal_trainer.pending_edits.clear()
                seal_trainer._save_buffer()
                print("✓ Буфер очищен")
        
    else:
        print("❌ Неверный выбор")

if __name__ == "__main__":
    main()
