﻿// =====================================================================
//  Получение много-инструментальных синхронизированных по времени данных.
//  Файл: "TimeSyncManager.mqh"
// =====================================================================
#property copyright		"Dima S., 2026"
#property link				"dimascub@mail.ru"
#property version			"1.00"
#property description	"MultyTrade & MultyCurrency"
// ---------------------------------------------------------------------


// =====================================================================
//  Общие настройки:
// =====================================================================


// =====================================================================
//	Подключаемые библиотеки:
// =====================================================================
#include  <Object.mqh>
#include  <Generic\SortedMap.mqh>
//---------------------------------------------------------------------

// ---------------------------------------------------------------------
#define DESTROY_OBJECT(object)  { if(CheckPointer(object) == POINTER_DYNAMIC) delete(object); }
#define CLEAR_AND_DESTROY_OBJECT(object)  { if(CheckPointer(object) == POINTER_DYNAMIC) {object.Clear(); delete(object);} }
// ---------------------------------------------------------------------

// =====================================================================
//  Параметры бара:
// =====================================================================
enum ENUM_BAR_TYPE
{
  ENUM_BAR_LIVE,                                                      // реальный бар
  ENUM_BAR_INTERPOLATED,                                              // бар получен интерполяцией из предыдущего (предыдущих))
  ENUM_BAR_EMPTY                                                      // отсутствующий бар 
};
// =====================================================================
//  Контейнер данных:
// =====================================================================
class CSymbolData : public CObject
{
protected:
  string          symbol;                                             // название символа
  ENUM_TIMEFRAMES data_timeframe;                                     // ТФ ланных
  datetime        new_bar_time;                                       // время появления последнего нового бара

public:
  CSortedMap<datetime, double>          open;                         // цены Open
  CSortedMap<datetime, double>          high;                         // цены High
  CSortedMap<datetime, double>          low;                          // цены Low
  CSortedMap<datetime, double>          close;                        // цены Close
  CSortedMap<datetime, long>            tick_volume;                  // тиковый объем
  CSortedMap<datetime, long>            real_volume;                  // биржевой объем
  CSortedMap<datetime, int>             spread;                       // спред
  CSortedMap<datetime, datetime>        time;                         // время
  CSortedMap<datetime, ENUM_BAR_TYPE>   bar_type;                     // тип бара

public:
  // =====================================================================
  //  Конструктор:
  // =====================================================================
  CSymbolData(const string _symbol_name, const ENUM_TIMEFRAMES _data_timeframe)
  :
  symbol(_symbol_name),
  data_timeframe(_data_timeframe),
  new_bar_time(0)
  {
  }
  // =====================================================================
  //  Конструктор без параметров:
  // =====================================================================
  CSymbolData()
  :
  symbol(NULL),
  data_timeframe(PERIOD_CURRENT),
  new_bar_time(0)
  {
  }
  // =====================================================================
  //  Деструктор:
  // =====================================================================
  ~CSymbolData()
  {
    //  Очистим буферы:
    this.open.Clear();
    this.high.Clear();
    this.low.Clear();
    this.close.Clear();
    this.tick_volume.Clear();
    this.real_volume.Clear();
    this.spread.Clear();
    this.bar_type.Clear();
    this.time.Clear();
  }
  // =====================================================================
  //  Получить имя инструмента:
  // =====================================================================
  string    GetSymbolName()
  {
    return(this.symbol);
  }
  // =====================================================================
  //  Получить ТФ:
  // =====================================================================
  ENUM_TIMEFRAMES GetTF()
  {
    return(this.data_timeframe);
  }
  // =====================================================================
  //  Получить время появления последнего нового бара:
  // =====================================================================
  datetime  GetNewBarTime()
  {
    return(this.new_bar_time);
  }
  // =====================================================================
  //  Добавление баров данных:
  // ===================================================================== 
  int  AddFullData(const MqlRates& _rates[], const int _start_index, const int _count, const bool _keep_old_data = false)
  {
    int   size = MathMin(_count, (int)_rates.Size());
    if(size > 0)
    {
      //  Если НЕ НАДО сохранить старые данные, то очистим массивы контейнера:
      if(_keep_old_data != true && this.time.Count() > 0)
      {
        this.open.Clear();
        this.high.Clear();
        this.low.Clear();
        this.close.Clear();
        this.tick_volume.Clear();
        this.real_volume.Clear();
        this.spread.Clear();
        this.time.Clear();
        this.bar_type.Clear();
      }

      //  Учтем направление расположения данных во входном массиве:
      bool as_series = ArrayGetAsSeries(_rates);
      if(as_series == true)
      {
        ArraySetAsSeries(_rates, false); 
      }
      
      int start_index = _start_index;
      if(_start_index <= 0)
      {
        start_index = 0;
      }
      
      //  Цикл перекачки данных:
      datetime  dt ;
      int       index;
      for(int i = 0; i < size; i++)
      {
        index = start_index + i;
        dt = _rates[index].time;
        this.open.Add(dt, _rates[index].open);
        this.high.Add(dt, _rates[index].high);
        this.low.Add(dt, _rates[index].low);
        this.close.Add(dt, _rates[index].close);
        this.tick_volume.Add(dt, _rates[index].tick_volume);
        this.real_volume.Add(dt, _rates[index].real_volume);
        this.spread.Add(dt, _rates[index].spread);
        this.bar_type.Add(dt, ENUM_BAR_LIVE);
        this.time.Add(dt, dt);
      }

      //  Время появления последнего нового бара:
      this.new_bar_time = _rates[start_index + size - 1].time;

      //  Восстановим направление расположения данных во входном массиве:
      ArraySetAsSeries(_rates, as_series); 
      return(size);
    }

    return(0);
  }
  // =====================================================================
  //  Обновление и/или добавление (при необходимости) последних баров:
  // ===================================================================== 
  int  UpdateLastData(const MqlRates& _rates[], const int _start_index, const int _count)
  {
    int   size = MathMin(_count, (int)_rates.Size());
    if(size > 0)
    {
      //  Учтем направление расположения данных во входном массиве:
      bool as_series = ArrayGetAsSeries(_rates);
      if(as_series == false)
      {
        ArraySetAsSeries(_rates, true); 
      }
      datetime  dt;
      int       index;
      //  Цикл перекачки данных:
      for(int i = 0; i < size; i++)
      {
        index = _start_index + i;
        dt = _rates[index].time;
        //  Если бар с таким временем уже есть в списке, то просто ОБНОВИМ данные для него:
        if(this.time.ContainsKey(dt) == true)
        {
          this.open.TrySetValue(dt, _rates[index].open);
          this.high.TrySetValue(dt, _rates[index].high);
          this.low.TrySetValue(dt, _rates[index].low);
          this.close.TrySetValue(dt, _rates[index].close);
          this.tick_volume.TrySetValue(dt, _rates[index].tick_volume);
          this.real_volume.TrySetValue(dt, _rates[index].real_volume);
          this.spread.TrySetValue(dt, _rates[index].spread);
        }
        //  Бара с таким временем нет в списке, значит добавим НОВЫЙ БАР:
        else
        {
          this.open.Add(dt, _rates[index].open);
          this.high.Add(dt, _rates[index].high);
          this.low.Add(dt, _rates[index].low);
          this.close.Add(dt, _rates[index].close);
          this.tick_volume.Add(dt, _rates[index].tick_volume);
          this.real_volume.Add(dt, _rates[index].real_volume);
          this.spread.Add(dt, _rates[index].spread);
          this.bar_type.Add(dt, ENUM_BAR_LIVE);
          this.time.Add(dt, dt);

          //  Время появления последнего нового бара:
          this.new_bar_time = _rates[index].time;
        }
      }

      //  Восстановим направление расположения данных во входном массиве:
      ArraySetAsSeries(_rates, as_series); 
      return(size);
    }

    return(0);
  }  
  // =====================================================================
  //  Копирование только последних обновленных данных:
  // =====================================================================
  #define COPY_LAST_DATA(data_type, data_src, count, data_dst, dst_start) \
  {\
    if(data_src.Count() > 0) \                                          // если данные загружены
    {\
      bool as_series = ArrayGetAsSeries(data_dst); \
      if(as_series == true) \                                           // учтем направление расположения данных во входном массиве
      { ArraySetAsSeries(data_dst, false); } \
      datetime    dt[]; \
      data_type   val[]; \
      data_src.CopyTo(dt, val); \
      for(int i = 0; i < count; i++) \                                  // копируем заданное число данных
      {\
        data_dst[dst_start + i] = val[this.time.Count() - count + i]; \
      }\
      ArraySetAsSeries(data_dst, as_series); \                          // восстановим направление расположения данных во входном массиве
    }\
  }
  // =====================================================================
  //  Получить обновленные данные Time[]:
  // ---------------------------------------------------------------------
  //   _src_start_index - с какого индекса берем из источника
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastTime(const int _count, datetime& _dst[], const int _dst_start_index)
  {
    COPY_LAST_DATA(datetime, this.time, _count, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить обновленные данные Open[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastOpen(const int _count, double& _dst[], const int _dst_start_index)
  {
    COPY_LAST_DATA(double, this.open, _count, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить обновленные данные High[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastHigh(const int _count, double& _dst[], const int _dst_start_index)
  {
    COPY_LAST_DATA(double, this.high, _count, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить обновленные данные Low[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastLow(const int _count, double& _dst[], const int _dst_start_index)
  {
    COPY_LAST_DATA(double, this.low, _count, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить обновленные данные Close[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastClose(const int _count, double& _dst[], const int _dst_start_index)
  {
    COPY_LAST_DATA(double, this.close, _count, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить обновленные данные TickVolume[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastTickVolume(const int _count, long& _dst[], const int _dst_start_index)
  {
    COPY_LAST_DATA(long, this.tick_volume, _count, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить обновленные данные RealVolume[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastRealVolume(const int _count, long& _dst[], const int _dst_start_index)
  {
    COPY_LAST_DATA(long, this.real_volume, _count, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить обновленные данные Spread[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastSpread(const int _count, int& _dst[], const int _dst_start_index)
  {
    COPY_LAST_DATA(int, this.spread, _count, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Копирование полного массива данных:
  // =====================================================================
  #define COPY_FULL_DATA(data_type, data_src, data_dst, dst_start) \
  {\
    if(data_src.Count() > 0) \                                          // если данные загружены
    {\
      bool as_series = ArrayGetAsSeries(data_dst); \
      if(as_series == true) \                                           // учтем направление расположения данных во входном массиве
      { ArraySetAsSeries(data_dst, false); } \
      datetime    dt[]; \
      data_type   val[]; \
      int data_copied = data_src.CopyTo(dt, val); \
      for(int i = 0; i < data_src.Count(); i++) \                                  // копируем заданное число данных
      {\
        data_dst[dst_start + i] = val[i]; \                             // копируем столько данных, сколько реально есть
      }\
      ArraySetAsSeries(data_dst, as_series); \                           // восстановим направление расположения данных во входном массиве
      return(data_copied); \
    }\
    return(0);\
  }
  // =====================================================================
  //  Получить данные Time[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  // =====================================================================
  int  GetFullTime(datetime& _dst[], const int _dst_start_index)
  {
    COPY_FULL_DATA(datetime, this.time, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить данные Open[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  // =====================================================================
  int  GetFullOpen(double& _dst[], const int _dst_start_index)
  {
    COPY_FULL_DATA(double, this.open, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить данные High[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  // =====================================================================
  int  GetFullHigh(double& _dst[], const int _dst_start_index)
  {
    COPY_FULL_DATA(double, this.high, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить данные Low[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  // =====================================================================
  int  GetFullLow(double& _dst[], const int _dst_start_index)
  {
    COPY_FULL_DATA(double, this.low, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить данные Close[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  // =====================================================================
  int  GetFullClose(double& _dst[], const int _dst_start_index)
  {
    COPY_FULL_DATA(double, this.close, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить данные TickVolume[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  // =====================================================================
  int  GetFullTickVolume(long& _dst[], const int _dst_start_index)
  {
    COPY_FULL_DATA(long, this.tick_volume, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить данные RealVolume[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  // =====================================================================
  int  GetFullRealVolume(long& _dst[], const int _dst_start_index)
  {
    COPY_FULL_DATA(long, this.real_volume, _dst, _dst_start_index);
  }
  // =====================================================================
  //  Получить данные Spread[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  // =====================================================================
  int  GetFullSpread(int& _dst[], const int _dst_start_index)
  {
    COPY_FULL_DATA(int, this.spread, _dst, _dst_start_index);
  }
};
// =====================================================================
//  Состояние синхронизации баров:
// =====================================================================
enum ENUM_BARS_SYNC_STATE
{
  ENUM_BARS_SYNC_FULL,                                                // проведена полная синхронизация баров
  ENUM_BARS_SYNC_PARTIAL,                                             // проведена частичная синхронизация баров
  ENUM_BARS_SYNC_NOTHING                                              // синхронизация не проводилась
};
// =====================================================================
//  Тип синхронизации баров:
// =====================================================================
enum ENUM_SYNC_BARS_TYPE
{
  ENUM_SYNC_BARS_INTERPOLATED,                                        // пустой бар заменяем предыдущим
  ENUM_SYNC_BARS_EMPTY                                                // добавляем пустой бар
};
// =====================================================================
//  Выбор символа для синхронизации баров:
// =====================================================================
enum ENUM_SYNC_SYMBOL_TYPE
{
  ENUM_SYNC_SYMBOL_CUSTOM,                                            // синхронизируем по заданному
  ENUM_SYNC_SYMBOL_AUTO                                               // автоматическая синхронизация
};
// =====================================================================
//  Тип ожидания появления новых баров:
// =====================================================================
enum ENUM_NEW_BARS_TYPE
{
  ENUM_NEW_BARS_WAITING_ALL,                                          // ждем появления нового бара на всех инструментах
  ENUM_NEW_BARS_UPDATE_ALL                                            // при появлении нового бара берем предыдущий бар для остальных (если нет нового)
};
// =====================================================================
//  Управление синхронизацией данных ВР для портфеля инструментов:
// =====================================================================
class CTimeSyncManager
{
protected:
  ENUM_BARS_SYNC_STATE    sync_state;                                 // состояние синхронизации
  ENUM_SYNC_BARS_TYPE     sync_bars_type;                             // тип синхронизации баров
  ENUM_SYNC_SYMBOL_TYPE   sync_symbol_type;                           // выбор символа для синхронизации баров
  ENUM_NEW_BARS_TYPE      new_bars_type;                              // тип ожидания появления новых баров

protected:
  bool  is_syncronized_FLAG;                                          // true - все данные синхронизированы
  int   data_capacity;

protected:
  CSymbolData*            sync_symbol_data;                           // инструмент синхронизации
  CSymbolData*            symbols_data_Array[];                       // инструменты портфеля

public:
  typedef static string (*SymbolDataUpdate)(CSymbolData& _data);
  typedef void (*AllSymbolsDataUpdate)(CSymbolData& _data[]);
  typedef string (*SymbolNewBarUpdate)(CSymbolData& _data);
  typedef void (*AllSymbolsNewBarUpdate)(CSymbolData& _data[]);

public:
  // =====================================================================
  //  Конструктор:
  // =====================================================================
  CTimeSyncManager(const int _symbols_capacity = 28, const int _data_capacity = 60000)
  :
  sync_state(ENUM_BARS_SYNC_NOTHING),
  sync_bars_type(ENUM_SYNC_BARS_INTERPOLATED),
  sync_symbol_type(ENUM_SYNC_SYMBOL_CUSTOM),
  new_bars_type(ENUM_NEW_BARS_WAITING_ALL),
  is_syncronized_FLAG(false),
  data_capacity(_data_capacity),
  sync_symbol_data(NULL)
  {
  }
  // =====================================================================
  //  Деструктор:
  // =====================================================================
  ~CTimeSyncManager()
  {
    DESTROY_OBJECT(this.sync_symbol_data);
    for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++)
    {
      DESTROY_OBJECT(this.symbols_data_Array[i]);
    }
  }
  // =====================================================================
  void  OnCalculateUpdate(SymbolDataUpdate _data_updated, AllSymbolsDataUpdate _all_data_updated, SymbolNewBarUpdate _newbar_updated, AllSymbolsNewBarUpdate _all_newbar_updated)
  {
  }
  // =====================================================================
  void  OnTimerUpdate()
  {
  }
  // =====================================================================
  //  Получить имя символа синхронизации:
  // =====================================================================
  string  GetSyncSymbolName()
  {
    return(this.sync_symbol_data.GetSymbolName());
  }
  // =====================================================================
  //  Получить список имен добавленных ранее символов:
  // =====================================================================
  int  GetSymbolNames(string& _symbols[])
  {
    if(this.symbols_data_Array.Size() > 0)
    {
      ArrayResize(_symbols, this.symbols_data_Array.Size());
      for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++)
      {
        _symbols[i] = ((CSymbolData*)this.symbols_data_Array[i]).GetSymbolName();
      }
      return((int)this.symbols_data_Array.Size());
    }

    return(0);
  }
  // =====================================================================
  //  Добавление символа синхронизации с данными:
  // ===================================================================== 
  int  SyncSymbolAddFullData(const string _symbol, const ENUM_TIMEFRAMES _data_timeframe, const MqlRates& _rates[], const int _src_start, const int _count, const bool _keep_old_data = false)
  {
    this.is_syncronized_FLAG = false;

    if(this.sync_symbol_data == NULL)
    {
      this.sync_symbol_data = new CSymbolData(_symbol, _data_timeframe);
    }

    //  Добавим сами данные в список:
    return(this.sync_symbol_data.AddFullData(_rates, _src_start, _count, _keep_old_data));
  }
  // =====================================================================
  //  Добавление/обновление последних баров символа синхронизации:
  // ===================================================================== 
  int  SyncSymbolUpdateLastData(const MqlRates& _rates[], const int _src_start, const int _count)
  {
    this.is_syncronized_FLAG = false;

    return(this.sync_symbol_data.UpdateLastData(_rates, _src_start, _count));
  }
  // =====================================================================
  //  Добавление символа с данными в список:
  // ===================================================================== 
  int  SymbolAddFullData(const string _symbol, const ENUM_TIMEFRAMES _data_timeframe, const MqlRates& _rates[], const int _src_start, const int _count, const bool _keep_old_data = false)
  {
    this.is_syncronized_FLAG = false;

    //CSymbolData*  curr_symbol;
    if(this.symbols_data_Array.Size() > 0)
    {
      for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++)
      {
        //  Ищем символ среди уже добавленных:
        if(this.symbols_data_Array[i].GetSymbolName() == _symbol)
        {
          return(this.symbols_data_Array[i].AddFullData(_rates, _src_start, _count, _keep_old_data));
        }
      }
    }

    //  Если попали сюда, значит символ в списке не найден или список пустой:
    CSymbolData*  curr_symbol = new CSymbolData(_symbol, _data_timeframe);

    //  Добавим данные:
    int sz = curr_symbol.AddFullData(_rates, _src_start, _count, _keep_old_data);

    //  Добавим сам символ в список:
    ArrayResize(this.symbols_data_Array, this.symbols_data_Array.Size() + 1);
    this.symbols_data_Array[this.symbols_data_Array.Size() - 1] = curr_symbol;

    return(sz);
  }
  // =====================================================================
  //  Добавление/обновление последних баров символа:
  // ===================================================================== 
  void  SymbolUpdateLastData(const string _symbol, const MqlRates& _rates[], const int _src_start, const int _count) 
  {
    this.is_syncronized_FLAG = false;

    if(this.symbols_data_Array.Size() > 0)
    {
      //CSymbolData*  curr_symbol;
      for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++)
      {
        //  Ищем символ среди уже добавленных:
        if(this.symbols_data_Array[i].GetSymbolName() == _symbol)
        {
          this.symbols_data_Array[i].UpdateLastData(_rates, _src_start, _count);
          break;
        }
      }
    }
  }
  // =====================================================================
  #define GET_FULL_DATA(symbol, dst, dst_start_index, Func) \
  {\
    if(symbol == this.sync_symbol_data.GetSymbolName()) \               // если это СИМВОЛ СИНХРОНИЗАЦИИ
    { return(this.sync_symbol_data.Func(dst, dst_start_index)); } \
    if(this.symbols_data_Array.Size() > 0) \                            // если это НЕ СИМВОЛ СИНХРОНИЗАЦИИ
    {\
      for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++) \
      {\
        if(this.symbols_data_Array[i].GetSymbolName() == symbol) \      // ищем символ среди уже добавленных
        { return(this.symbols_data_Array[i].Func(dst, dst_start_index)); } \
      }\
    }\
    return(0); \
  }
  // =====================================================================
  //  Получить данные Open для заданного символа в виде массива:
  // =====================================================================
  int  GetFullOpen(const string _symbol, double& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:

    GET_FULL_DATA(_symbol, _dst, _dst_start_index, GetFullOpen);
  }
  // =====================================================================
  //  Получить данные High для заданного символа в виде массива:
  // =====================================================================
  int  GetFullHigh(const string _symbol, double& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    GET_FULL_DATA(_symbol, _dst, _dst_start_index, GetFullHigh);
  }
  // =====================================================================
  //  Получить данные Low для заданного символа в виде массива:
  // =====================================================================
  int  GetFullLow(const string _symbol, double& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    GET_FULL_DATA(_symbol, _dst, _dst_start_index, GetFullLow);
  }
  // =====================================================================
  //  Получить данные Close для заданного символа в виде массива:
  // =====================================================================
  int  GetFullClose(const string _symbol, double& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    GET_FULL_DATA(_symbol, _dst, _dst_start_index, GetFullClose);
  }
  // =====================================================================
  //  Получить данные Tick_Volume для заданного символа в виде массива:
  // =====================================================================
  int  GetFullTickVolume(const string _symbol, long& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    GET_FULL_DATA(_symbol, _dst, _dst_start_index, GetFullTickVolume);
  }
  // =====================================================================
  //  Получить данные Real_Volume для заданного символа в виде массива:
  // =====================================================================
  int  GetFullRealVolume(const string _symbol, long& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    GET_FULL_DATA(_symbol, _dst, _dst_start_index, GetFullRealVolume);
  }
  // =====================================================================
  //  Получить данные символа Spread в виде массива:
  // =====================================================================
  int  GetFullSpread(const string _symbol, int& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    GET_FULL_DATA(_symbol, _dst, _dst_start_index, GetFullSpread);
  }
  // =====================================================================
  //  Получить обновленные данные Time[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastTime(const string _symbol, const int _count, datetime& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    //  Если это СИМВОЛ СИНХРОНИЗАЦИИ:
    if(_symbol == this.sync_symbol_data.GetSymbolName())
    {
      this.sync_symbol_data.GetLastTime(_count, _dst, _dst_start_index);
    }

    //  Если это НЕ СИМВОЛ СИНХРОНИЗАЦИИ:
    if(this.symbols_data_Array.Size() > 0)
    {
      for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++)
      {
        //  Ищем символ среди уже добавленных:
        if(this.symbols_data_Array[i].GetSymbolName() == _symbol)
        {
          this.symbols_data_Array[i].GetLastTime(_count, _dst, _dst_start_index);
          break;
        }
      }
    }
  }
  // =====================================================================
  //  Получить обновленные данные Open[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastOpen(const string _symbol, const int _count, double& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    //  Если это СИМВОЛ СИНХРОНИЗАЦИИ:
    if(_symbol == this.sync_symbol_data.GetSymbolName())
    {
      this.sync_symbol_data.GetLastOpen(_count, _dst, _dst_start_index);
    }

    //  Если это НЕ СИМВОЛ СИНХРОНИЗАЦИИ:
    if(this.symbols_data_Array.Size() > 0)
    {
      for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++)
      {
        //  Ищем символ среди уже добавленных:
        if(this.symbols_data_Array[i].GetSymbolName() == _symbol)
        {
          this.symbols_data_Array[i].GetLastOpen(_count, _dst, _dst_start_index);
          break;
        }
      }
    }
  }
  // =====================================================================
  //  Получить обновленные данные High[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastHigh(const string _symbol, const int _count, double& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    //  Если это СИМВОЛ СИНХРОНИЗАЦИИ:
    if(_symbol == this.sync_symbol_data.GetSymbolName())
    {
      this.sync_symbol_data.GetLastHigh(_count, _dst, _dst_start_index);
    }

    //  Если это НЕ СИМВОЛ СИНХРОНИЗАЦИИ:
    if(this.symbols_data_Array.Size() > 0)
    {
      for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++)
      {
        //  Ищем символ среди уже добавленных:
        if(this.symbols_data_Array[i].GetSymbolName() == _symbol)
        {
          this.symbols_data_Array[i].GetLastHigh(_count, _dst, _dst_start_index);
          break;
        }
      }
    }
  }
  // =====================================================================
  //  Получить обновленные данные Low[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastLow(const string _symbol, const int _count, double& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    //  Если это СИМВОЛ СИНХРОНИЗАЦИИ:
    if(_symbol == this.sync_symbol_data.GetSymbolName())
    {
      this.sync_symbol_data.GetLastLow(_count, _dst, _dst_start_index);
    }

    //  Если это НЕ СИМВОЛ СИНХРОНИЗАЦИИ:
    if(this.symbols_data_Array.Size() > 0)
    {
      for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++)
      {
        //  Ищем символ среди уже добавленных:
        if(this.symbols_data_Array[i].GetSymbolName() == _symbol)
        {
          this.symbols_data_Array[i].GetLastLow(_count, _dst, _dst_start_index);
          break;
        }
      }
    }
  }
  // =====================================================================
  //  Получить обновленные данные Close[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastClose(const string _symbol, const int _count, double& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    //  Если это СИМВОЛ СИНХРОНИЗАЦИИ:
    if(_symbol == this.sync_symbol_data.GetSymbolName())
    {
      this.sync_symbol_data.GetLastClose(_count, _dst, _dst_start_index);
    }

    //  Если это НЕ СИМВОЛ СИНХРОНИЗАЦИИ:
    if(this.symbols_data_Array.Size() > 0)
    {
      for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++)
      {
        //  Ищем символ среди уже добавленных:
        if(this.symbols_data_Array[i].GetSymbolName() == _symbol)
        {
          this.symbols_data_Array[i].GetLastClose(_count, _dst, _dst_start_index);
          break;
        }
      }
    }
  }
  // =====================================================================
  //  Получить обновленные данные Spread[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastSpread(const string _symbol, const int _count, int& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    //  Если это СИМВОЛ СИНХРОНИЗАЦИИ:
    if(_symbol == this.sync_symbol_data.GetSymbolName())
    {
      this.sync_symbol_data.GetLastSpread(_count, _dst, _dst_start_index);
    }

    //  Если это НЕ СИМВОЛ СИНХРОНИЗАЦИИ:
    if(this.symbols_data_Array.Size() > 0)
    {
      for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++)
      {
        //  Ищем символ среди уже добавленных:
        if(this.symbols_data_Array[i].GetSymbolName() == _symbol)
        {
          this.symbols_data_Array[i].GetLastSpread(_count, _dst, _dst_start_index);
          break;
        }
      }
    }
  }
  // =====================================================================
  //  Получить обновленные данные TickVolume[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastTickVolume(const string _symbol, const int _count, long& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    //  Если это СИМВОЛ СИНХРОНИЗАЦИИ:
    if(_symbol == this.sync_symbol_data.GetSymbolName())
    {
      this.sync_symbol_data.GetLastTickVolume(_count, _dst, _dst_start_index);
    }

    //  Если это НЕ СИМВОЛ СИНХРОНИЗАЦИИ:
    if(this.symbols_data_Array.Size() > 0)
    {
      for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++)
      {
        //  Ищем символ среди уже добавленных:
        if(this.symbols_data_Array[i].GetSymbolName() == _symbol)
        {
          this.symbols_data_Array[i].GetLastTickVolume(_count, _dst, _dst_start_index);
          break;
        }
      }
    }
  }
  // =====================================================================
  //  Получить обновленные данные RealVolume[]:
  // ---------------------------------------------------------------------
  //   _dst_start_index - с какого индекса пишем в приемник
  //   _count           - сколько данных копировать
  // =====================================================================
  void  GetLastRealVolume(const string _symbol, const int _count, long& _dst[], const int _dst_start_index)
  {
    //  Если данные с такими параметрами уже запращивались:
    
    //  Если это СИМВОЛ СИНХРОНИЗАЦИИ:
    if(_symbol == this.sync_symbol_data.GetSymbolName())
    {
      this.sync_symbol_data.GetLastRealVolume(_count, _dst, _dst_start_index);
    }

    //  Если это НЕ СИМВОЛ СИНХРОНИЗАЦИИ:
    if(this.symbols_data_Array.Size() > 0)
    {
      for(int i = 0; i < (int)this.symbols_data_Array.Size(); i++)
      {
        //  Ищем символ среди уже добавленных:
        if(this.symbols_data_Array[i].GetSymbolName() == _symbol)
        {
          this.symbols_data_Array[i].GetLastRealVolume(_count, _dst, _dst_start_index);
          break;
        }
      }
    }
  }
  // =====================================================================
  //  Синхронизировать все добавленные данные:
  // =====================================================================
  bool  SyncAllData(const ENUM_SYNC_BARS_TYPE _sync_bars_type)
  {
    return(this.is_syncronized_FLAG = CustomSyncAllData(_sync_bars_type));
  }
  // ---------------------------------------------------------------------
  //  Синхронизация ПО ЗАДАННОМУ пользователем СИМВОЛУ:
  // ---------------------------------------------------------------------
  bool  CustomSyncAllData(const ENUM_SYNC_BARS_TYPE _sync_bars_type)
  {
    //  Если символ синхронизации не содержит данных или отсутствуют остальные инструменты, то на этом все:
    if(this.sync_symbol_data.time.Count() <= 0 || this.symbols_data_Array.Size() <= 0)
    {
      return(false);
    }

    double        prev_value_dbl = 0.0;
    int           prev_value_int = 0;
    int           sync_symbol_key_size;
    datetime      sync_symbol_time;
    int           to_delete_count = 0;

    datetime      prev_time = 0;
    bool          keep_prev_time = false;

    //  Массив моментов времени символа синхронизации:
    datetime  sync_symbol_key[];
    datetime  sync_symbol_val[];
    this.sync_symbol_data.time.CopyTo(sync_symbol_key, sync_symbol_val);
    sync_symbol_key_size = (int)sync_symbol_key.Size();

    //  Перебираем символы и синхронизируем каждый из них с СС:
    for(int k = 0; k < (int)this.symbols_data_Array.Size(); k++)
    {
      keep_prev_time = false;

      //  Двигаемся по времени в символе синхронизации (СС) и сравниваем с остальными символами: 
      for(int sync_symbol_key_index = 1; sync_symbol_key_index < sync_symbol_key_size; sync_symbol_key_index++)
      {
        sync_symbol_time = sync_symbol_key[sync_symbol_key_index];

        //  Если в синхронизируемом символе НЕТ бара с нужным временем, то вставим его:
        if(this.symbols_data_Array[k].time.ContainsKey(sync_symbol_time) != true)
        {
          //  Если задана интерполяция баров, то установим значения в пустых барах:
          if(_sync_bars_type == ENUM_SYNC_BARS_INTERPOLATED)
          {
            //  Получим значения предыдущего бара:
            if(keep_prev_time == false)
            {
              prev_time = sync_symbol_key[sync_symbol_key_index - 1];
              keep_prev_time = true;

              this.symbols_data_Array[k].close.TryGetValue(prev_time, prev_value_dbl);
              this.symbols_data_Array[k].spread.TryGetValue(prev_time, prev_value_int);
            }

            this.symbols_data_Array[k].open.Add(sync_symbol_time, prev_value_dbl);
            this.symbols_data_Array[k].high.Add(sync_symbol_time, prev_value_dbl);
            this.symbols_data_Array[k].low.Add(sync_symbol_time, prev_value_dbl);
            this.symbols_data_Array[k].close.Add(sync_symbol_time, prev_value_dbl);
            this.symbols_data_Array[k].tick_volume.Add(sync_symbol_time, 0);
            this.symbols_data_Array[k].real_volume.Add(sync_symbol_time, 0);
            this.symbols_data_Array[k].bar_type.Add(sync_symbol_time, ENUM_BAR_INTERPOLATED);
            this.symbols_data_Array[k].spread.Add(sync_symbol_time, prev_value_int);
            this.symbols_data_Array[k].time.Add(sync_symbol_time, sync_symbol_time);
          }
          else
          {
            this.symbols_data_Array[k].open.Add(sync_symbol_time, 0);
            this.symbols_data_Array[k].high.Add(sync_symbol_time, 0);
            this.symbols_data_Array[k].low.Add(sync_symbol_time, 0);
            this.symbols_data_Array[k].close.Add(sync_symbol_time, 0);
            this.symbols_data_Array[k].tick_volume.Add(sync_symbol_time, 0);
            this.symbols_data_Array[k].real_volume.Add(sync_symbol_time, 0);
            this.symbols_data_Array[k].bar_type.Add(sync_symbol_time, ENUM_BAR_EMPTY);
            this.symbols_data_Array[k].spread.Add(sync_symbol_time, 0);
            this.symbols_data_Array[k].time.Add(sync_symbol_time, sync_symbol_time);
          }

          to_delete_count++;
        }
        else
        {
          keep_prev_time = false;
        }
      }

      //  Удалим "лишние" элементы (если они есть) из начала массива:
      if(to_delete_count > 0)
      {
        datetime  _symbol_key[];
        datetime  _symbol_val[];
        this.symbols_data_Array[k].time.CopyTo(_symbol_key, _symbol_val);

        for(int i = to_delete_count - 1; i >= 0; i--)
        {
          this.symbols_data_Array[k].open.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].high.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].low.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].close.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].tick_volume.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].real_volume.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].bar_type.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].spread.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].time.Remove(_symbol_key[i]);
        }
      }

      //  Если задана интерполяция баров, то установим значения в пустых барах:
//      if(_sync_bars_type == ENUM_SYNC_BARS_INTERPOLATED)
//      {
//        datetime        _symbol_bt_key[];
//        ENUM_BAR_TYPE   _symbol_bt_val[];
//        this.symbols_data_Array[k].bar_type.CopyTo(_symbol_bt_key, _symbol_bt_val);
//
//        for(int i = 1; i < (int)_symbol_bt_key.Size(); i++)
//        {
//          if(_symbol_bt_val[i] == ENUM_BAR_EMPTY)
//          {
//            //  Получим значения предыдущего бара:
//            this.symbols_data_Array[k].close.TryGetValue(_symbol_bt_key[i - 1], prev_value_dbl);
//            this.symbols_data_Array[k].spread.TryGetValue(_symbol_bt_key[i - 1], prev_value_int);
//
//            this.symbols_data_Array[k].open.TrySetValue(_symbol_bt_key[i], prev_value_dbl);
//            this.symbols_data_Array[k].high.TrySetValue(_symbol_bt_key[i], prev_value_dbl);
//            this.symbols_data_Array[k].low.TrySetValue(_symbol_bt_key[i], prev_value_dbl);
//            this.symbols_data_Array[k].close.TrySetValue(_symbol_bt_key[i], prev_value_dbl);
//            this.symbols_data_Array[k].spread.TrySetValue(_symbol_bt_key[i], prev_value_int);
//            this.symbols_data_Array[k].bar_type.TrySetValue(_symbol_bt_key[i], ENUM_BAR_INTERPOLATED);
//          }
//        }
//      }
    }

    return(true);
  }
  // =====================================================================
  //  Синхронизировать данные для последних по времени баров:
  // =====================================================================
  bool  SyncLastBars(const int _count, const ENUM_SYNC_BARS_TYPE _sync_bars_type)
  {
    return(this.is_syncronized_FLAG = CustomSyncLastBars(_count, _sync_bars_type));
  }
  // ---------------------------------------------------------------------
  //  Синхронизация ПО ЗАДАННОМУ пользователем СИМВОЛУ для последних по времени баров:
  // ---------------------------------------------------------------------
  bool  CustomSyncLastBars(const int _count, const ENUM_SYNC_BARS_TYPE _sync_bars_type)
  {
    //  Если символ синхронизации не содержит данных или отсутствуют остальные инструменты, то на этом все:
    if(this.sync_symbol_data.time.Count() <= 0 || this.symbols_data_Array.Size() <= 0)
    {
      return(false);
    }

    double        prev_value_dbl;
    int           prev_value_int;
    int           sync_symbol_key_size;
    datetime      sync_symbol_time;
    int           to_delete_count = 0;

    //  Массив моментов времени символа синхронизации:
    datetime  sync_symbol_key[];
    datetime  sync_symbol_val[];
    this.sync_symbol_data.time.CopyTo(sync_symbol_key, sync_symbol_val);
    sync_symbol_key_size = (int)sync_symbol_key.Size();

    //  Перебираем символы и синхронизируем каждый из них с СС:
    for(int k = 0; k < (int)this.symbols_data_Array.Size(); k++)
    {
      //  Двигаемся по времени в символе синхронизации (СС) и сравниваем с остальными символами: 
      for(int sync_symbol_key_index = sync_symbol_key_size - _count; sync_symbol_key_index < sync_symbol_key_size; sync_symbol_key_index++)
      {
        sync_symbol_time = sync_symbol_key[sync_symbol_key_index];

        //  Если в синхронизируемом символе НЕТ бара с нужным временем, то вставим его:
        if(this.symbols_data_Array[k].time.ContainsKey(sync_symbol_time) != true)
        {
          this.symbols_data_Array[k].open.Add(sync_symbol_time, 0);
          this.symbols_data_Array[k].high.Add(sync_symbol_time, 0);
          this.symbols_data_Array[k].low.Add(sync_symbol_time, 0);
          this.symbols_data_Array[k].close.Add(sync_symbol_time, 0);
          this.symbols_data_Array[k].tick_volume.Add(sync_symbol_time, 0);
          this.symbols_data_Array[k].real_volume.Add(sync_symbol_time, 0);
          this.symbols_data_Array[k].bar_type.Add(sync_symbol_time, ENUM_BAR_EMPTY);
          this.symbols_data_Array[k].spread.Add(sync_symbol_time, 0);
          this.symbols_data_Array[k].time.Add(sync_symbol_time, sync_symbol_time);

          to_delete_count++;
        }
      }

      //  Удалим "лишние" элементы (если они есть) из начала массива:
      if(to_delete_count > 0)
      {
        datetime  _symbol_key[];
        datetime  _symbol_val[];
        this.symbols_data_Array[k].time.CopyTo(_symbol_key, _symbol_val);

        for(int i = to_delete_count - 1; i >= 0; i--)
        {
          this.symbols_data_Array[k].open.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].high.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].low.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].close.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].tick_volume.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].real_volume.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].bar_type.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].spread.Remove(_symbol_key[i]);
          this.symbols_data_Array[k].time.Remove(_symbol_key[i]);
        }
      }

      //  Если задана интерполяция баров, то установим значения в пустых барах:
      if(_sync_bars_type == ENUM_SYNC_BARS_INTERPOLATED)
      {
        datetime        _symbol_bt_key[];
        ENUM_BAR_TYPE   _symbol_bt_val[];
        this.symbols_data_Array[k].bar_type.CopyTo(_symbol_bt_key, _symbol_bt_val);

        for(int i = (int)_symbol_bt_key.Size() - _count + 1; i < (int)_symbol_bt_key.Size(); i++)
        {
          if(_symbol_bt_val[i] == ENUM_BAR_EMPTY)
          {
            //  Получим значения предыдущего бара:
            this.symbols_data_Array[k].close.TryGetValue(_symbol_bt_key[i - 1], prev_value_dbl);
            this.symbols_data_Array[k].spread.TryGetValue(_symbol_bt_key[i - 1], prev_value_int);

            this.symbols_data_Array[k].open.TrySetValue(_symbol_bt_key[i], prev_value_dbl);
            this.symbols_data_Array[k].high.TrySetValue(_symbol_bt_key[i], prev_value_dbl);
            this.symbols_data_Array[k].low.TrySetValue(_symbol_bt_key[i], prev_value_dbl);
            this.symbols_data_Array[k].close.TrySetValue(_symbol_bt_key[i], prev_value_dbl);
            this.symbols_data_Array[k].spread.TrySetValue(_symbol_bt_key[i], prev_value_int);
            this.symbols_data_Array[k].bar_type.TrySetValue(_symbol_bt_key[i], ENUM_BAR_INTERPOLATED);
          }
        }
      }
    }

    return(true);
  }
};
// ---------------------------------------------------------------------
