//+------------------------------------------------------------------+
//|                                                 ParafracEA.mq5   |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Wamek EA"
#property link      "Eawamek@gmail.com"
#property version   "1.00"
#property description "Trading system based on Parafrac/V2 indicator"

// Include the Trade class
#include <Trade/Trade.mqh>
CTrade trade;

//--- Indicator Type enum
enum IndicatorType
  {
   TypParafrac=0,
   TypParafracV2=1
  };

// Input parameters for trading
input IndicatorType UseIndicator = TypParafrac;  // Select Parafrac or V2 model
input double LotSize = 0.01;           // Lot size
input int StopLossPoints = 700;       // Stop loss in points
input double RiskRewardRatio = 2.0;   // Risk/Reward ratio
input double pstep = 0.02;
input double pMax = 0.2;
input int AtrPeriod = 7;              //ATRperiod for V2
input bool UseStrategy1 = true;       // Enable Strategy 1
input double threshold4Strat1 =1.5;    //threshold for strat 1
input bool UseStrategy2 = false;       // Enable Strategy 2
input bool UseStrategy3 = false;       // Enable Strategy 3


// Global variables
int MagicNumber =1483;
int indicatorHandle;
double upBuffer[];
double downBuffer[];
MqlRates priceData[];

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{

   if( !TerminalInfoInteger(TERMINAL_TRADE_ALLOWED) ){ 
        MessageBox("Enable AutoTrading"); 
        return (0); 
       }


   // Create indicator handle
   indicatorHandle = (UseIndicator==TypParafrac)?
    iCustom(_Symbol, _Period, "ParaFracOscillator", pstep, pMax ):
    iCustom(_Symbol, _Period, "ParaFracV2_Oscillator", pstep, pMax, AtrPeriod );
   
   
   if(indicatorHandle == INVALID_HANDLE)
   {
      Print("Error creating indicator handle");
      return(INIT_FAILED);
   }
   
  ArraySetAsSeries(upBuffer,true);
  ArraySetAsSeries(downBuffer,true); 
   // Set magic number for trades
   trade.SetDeviationInPoints(10);
   trade.SetExpertMagicNumber(MagicNumber);
   
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   // Release indicator handle
   if(indicatorHandle != INVALID_HANDLE)
      IndicatorRelease(indicatorHandle);
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
  //check for new bar
   static datetime lastBar;
   datetime currentBar = iTime(Symbol(), 0, 0);

   if(currentBar == lastBar) return;
    lastBar = currentBar;

  
   // Check trading time (2-22 UTC)
   MqlDateTime timeNow;
   TimeCurrent(timeNow);
   int currentHour = timeNow.hour;
   if(currentHour < 2 || currentHour >= 22) return;
  
   
   // Get indicator values
   if(CopyBuffer(indicatorHandle, 0, 1, 3, upBuffer) < 3)
      return;
   if(CopyBuffer(indicatorHandle, 1, 1, 3, downBuffer) < 3)
      return;
   
   // Get price data
   if(CopyRates(_Symbol, _Period, 0, 3, priceData) < 3)
      return;
   
   // Calculate current indicator values
   double currentValue = (upBuffer[0] != 0) ? upBuffer[0] : downBuffer[0];
   double prevValue = (upBuffer[1] != 0) ? upBuffer[1] : downBuffer[1];
   double prevValue2 = (upBuffer[2] != 0) ? upBuffer[2] : downBuffer[2];
   
      
   // Strategy 1: Zero line crossover with strong momentum
   if(UseStrategy1 && !Check4TradesOpen(POSITION_TYPE_BUY) )
   {
      // Buy condition: Cross up zero and any of first 2 histograms above threshold
      if( (prevValue <= 0 && currentValue > 0 || prevValue2 <= 0 && prevValue > 0) 
           && currentValue > threshold4Strat1  )
      {
         ExecuteBuy();
      }
   }
   
   if(UseStrategy1 && !Check4TradesOpen(POSITION_TYPE_SELL))
   {
      // Sell condition: Cross down zero and any of first 2 histograms below -threshold
      if( (prevValue >= 0 && currentValue < 0 || prevValue2 >= 0 && prevValue < 0 )  
           && currentValue < -threshold4Strat1 )
      {
         ExecuteSell();
      }
   }
   
   
   // Strategy 2: Momentum reversal
   if(UseStrategy2 && !Check4TradesOpen(POSITION_TYPE_BUY))
   {
      // Buy condition: 3 histograms increasing below zero
      if(currentValue < 0 && prevValue < 0 && downBuffer[2] < 0 &&
         currentValue > prevValue && prevValue > prevValue2)
      {
         ExecuteBuy();
      }
   } 
   
   if(UseStrategy2 && !Check4TradesOpen(POSITION_TYPE_SELL))
   {
      // Sell condition: 3 histograms decreasing above zero
      if(currentValue > 0 && prevValue > 0 && prevValue2 > 0 &&
         currentValue < prevValue && prevValue < prevValue2)
      {
         ExecuteSell();
      }
   }
   

   
   // Strategy 3: Momentum continuation with confirmation
   if(UseStrategy3 && !Check4TradesOpen(POSITION_TYPE_BUY))
   {
      // Buy condition: 3 histograms increasing above zero AND bearish candle
      if(currentValue > 0 && prevValue > 0 && prevValue2 > 0 &&
         currentValue > prevValue && prevValue > prevValue2 &&
         priceData[0].close < priceData[0].open)
      {
         ExecuteBuy();
      }
   }
   
   if(UseStrategy3 && !Check4TradesOpen(POSITION_TYPE_SELL))
   {
      // Sell condition: 3 histograms decreasing below zero AND bullish candle
      if(currentValue < 0 && prevValue < 0 && prevValue2 < 0 &&
         currentValue < prevValue && prevValue < prevValue2 &&
         priceData[0].close > priceData[0].open)
      {
         ExecuteSell();
      }
   }
}

//+------------------------------------------------------------------+
//| Execute buy order                                                |
//+------------------------------------------------------------------+
void ExecuteBuy()
{
   double ask = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   double sl = ask - StopLossPoints * _Point;
   double tp = ask + (StopLossPoints * RiskRewardRatio) * _Point;
   
   trade.Buy(LotSize, _Symbol, ask, sl, tp, "Parafrac EA Buy");
}

//+------------------------------------------------------------------+
//| Execute sell order                                               |
//+------------------------------------------------------------------+
void ExecuteSell()
{
   double bid = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   double sl = bid + StopLossPoints * _Point;
   double tp = bid - (StopLossPoints * RiskRewardRatio) * _Point;
   
   trade.Sell(LotSize, _Symbol, bid, sl, tp, "Parafrac EA Sell");
}
//+------------------------------------------------------------------+


//+------------------------------------------------------------------+
//| Check existing positions                                         |
//+------------------------------------------------------------------+
bool Check4TradesOpen(ENUM_POSITION_TYPE type)
  {
   for(int i=PositionsTotal()-1; i>=0; i--)
     {
      ulong ticket=PositionGetTicket(i);
      if(ticket && 
         PositionGetString(POSITION_SYMBOL)==_Symbol && 
         PositionGetInteger(POSITION_MAGIC)==MagicNumber &&
         PositionGetInteger(POSITION_TYPE)==type)
         return true;
     }
   return false;
  }
//+------------------------------------------------------------------+
