#property strict

#include "ILogHandler.mqh"
#include "LogLevels.mqh"

//+------------------------------------------------------------------+
//| Class: ConsoleLogHandler                                         |
//| Description: Implements ILogHandler to output log messages to    |
//|              the MetaTrader 5 Experts tab (console).             |
//+------------------------------------------------------------------+
class ConsoleLogHandler : public ILogHandler
  {
private:
   LogLevel          m_min_level;       // Minimum level to log
   string            m_format;          // Log message format string

   //--- Helper to format the log message
   string            FormatMessage(const datetime time, const LogLevel level, const string origin, const string message);
   //--- Helper to get string representation of LogLevel
   string            LogLevelToString(const LogLevel level);

public:
                     ConsoleLogHandler(const LogLevel min_level = LOG_LEVEL_INFO, const string format = "[{time}] {level}: {origin} - {message}");
                    ~ConsoleLogHandler();

   //--- ILogHandler implementation
   virtual bool      Setup(const string settings="") override;
   virtual void      Log(const datetime time, const LogLevel level, const string origin, const string message, const long expert_id=0) override;
   virtual void      Shutdown() override;

   //--- Setters
   void              SetMinLevel(const LogLevel level) { m_min_level = level; }
   void              SetFormat(const string format)    { m_format = format; }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
ConsoleLogHandler::ConsoleLogHandler(const LogLevel min_level = LOG_LEVEL_INFO, const string format = "[{time}] {level}: {origin} - {message}")
  {
   m_min_level = min_level;
   m_format = format;
   // No specific setup needed for console logging initially
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
ConsoleLogHandler::~ConsoleLogHandler()
  {
   // No specific cleanup needed
  }
//+------------------------------------------------------------------+
//| Setup                                                            |
//+------------------------------------------------------------------+
bool ConsoleLogHandler::Setup(const string settings="")
  {
   // Settings could be used to parse format or min_level, but we use constructor args for now
   // Example: Parse settings string if needed
   return true;
  }
//+------------------------------------------------------------------+
//| Log                                                              |
//+------------------------------------------------------------------+
void ConsoleLogHandler::Log(const datetime time, const LogLevel level, const string origin, const string message, const long expert_id=0)
  {
   // Check if the message level meets the minimum requirement
   if(level >= m_min_level && level < LOG_LEVEL_OFF)
     {
      // Format and print the message to the Experts tab
      Print(FormatMessage(time, level, origin, message));
     }
  }
//+------------------------------------------------------------------+
//| Shutdown                                                         |
//+------------------------------------------------------------------+
void ConsoleLogHandler::Shutdown()
  {
   // No specific shutdown actions needed for console logging
   PrintFormat("%s: ConsoleLogHandler shutdown.", __FUNCTION__);
  }
//+------------------------------------------------------------------+
//| FormatMessage                                                    |
//+------------------------------------------------------------------+
string ConsoleLogHandler::FormatMessage(const datetime time, const LogLevel level, const string origin, const string message)
  {
   string formatted_message = m_format;

   // Replace placeholders
   StringReplace(formatted_message, "{time}", TimeToString(time, TIME_DATE | TIME_SECONDS));
   StringReplace(formatted_message, "{level}", LogLevelToString(level));
   StringReplace(formatted_message, "{origin}", origin);
   StringReplace(formatted_message, "{message}", message);

   return formatted_message;
  }
//+------------------------------------------------------------------+
//| LogLevelToString                                                 |
//+------------------------------------------------------------------+
string ConsoleLogHandler::LogLevelToString(const LogLevel level)
  {
   switch(level)
     {
      case LOG_LEVEL_DEBUG: return "DEBUG";
      case LOG_LEVEL_INFO:  return "INFO";
      case LOG_LEVEL_WARN:  return "WARN";
      case LOG_LEVEL_ERROR: return "ERROR";
      case LOG_LEVEL_FATAL: return "FATAL";
      default:              return "UNKNOWN";
     }
  }
//+------------------------------------------------------------------+
