
#property strict

#include "LogLevels.mqh"
#include "ILogHandler.mqh"

//+------------------------------------------------------------------+
//| Class: CLogger                                                   |
//| Description: Singleton class for managing and dispatching log    |
//|              messages to registered handlers.                    |
//+------------------------------------------------------------------+
class CLogger
  {
private:
   static CLogger   *s_instance;
   ILogHandler*     m_handlers[];  
   LogLevel          m_global_min_level;
   long              m_expert_magic;
   string            m_expert_name;

   //--- Private constructor for Singleton
                     CLogger();
                    ~CLogger();

public:
   //--- Get the singleton instance
   static CLogger*   Instance();
   //--- Cleanup the singleton instance
   static void       Release();

   //--- Handler management
   bool              AddHandler(ILogHandler *handler);
   void              ClearHandlers();

   //--- Configuration
   void              SetGlobalMinLevel(const LogLevel level);
   void              SetExpertInfo(const long magic, const string name);

   //--- Logging methods
   void              Log(const LogLevel level, const string origin, const string message);
   void              Debug(const string origin, const string message);
   void              Info(const string origin, const string message);
   void              Warn(const string origin, const string message);
   void              Error(const string origin, const string message);
   void              Fatal(const string origin, const string message);
   
   //--- Formatted logging methods
   void              LogFormat(const LogLevel level, const string origin, const string formatted_message);
   void              DebugFormat(const string origin, const string formatted_message);
   void              InfoFormat(const string origin, const string formatted_message);
   void              WarnFormat(const string origin, const string formatted_message);
   void              ErrorFormat(const string origin, const string formatted_message);
   void              FatalFormat(const string origin, const string formatted_message);
  };
//+------------------------------------------------------------------+
//| Static instance initialization                                   |
//+------------------------------------------------------------------+
CLogger *CLogger::s_instance = NULL;
//+------------------------------------------------------------------+
//| Get Singleton Instance                                           |
//+------------------------------------------------------------------+
CLogger* CLogger::Instance()
  {
   if(s_instance == NULL)
     {
      s_instance = new CLogger();
     }
   return s_instance;
  }
//+------------------------------------------------------------------+
//| Release Singleton Instance                                       |
//+------------------------------------------------------------------+
void CLogger::Release()
  {
   if(s_instance != NULL)
     {
      delete s_instance;
      s_instance = NULL;
     }
  }
//+------------------------------------------------------------------+
//| Constructor (Private)                                            |
//+------------------------------------------------------------------+
CLogger::CLogger()
  {
   m_global_min_level = LOG_LEVEL_DEBUG;
   m_expert_magic = 0;
   m_expert_name = "";
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CLogger::~CLogger()
  {
   ClearHandlers();
  }
//+------------------------------------------------------------------+
//| AddHandler                                                       |
//+------------------------------------------------------------------+
bool CLogger::AddHandler(ILogHandler *handler)
  {
   if(CheckPointer(handler) == POINTER_INVALID)
     {
      Print("CLogger::AddHandler - Error: Invalid handler pointer.");
      return false;
     }
   int size = ArraySize(m_handlers);
   ArrayResize(m_handlers, size + 1);
   m_handlers[size] = handler;
   return true;
  }
//+------------------------------------------------------------------+
//| ClearHandlers                                                    |
//+------------------------------------------------------------------+
void CLogger::ClearHandlers()
  {
   for(int i = 0; i < ArraySize(m_handlers); i++)
     {
      ILogHandler *handler = m_handlers[i];
      if(CheckPointer(handler) != POINTER_INVALID)
        {
         handler.Shutdown();
         delete handler;
        }
     }
   ArrayResize(m_handlers, 0);
  }
//+------------------------------------------------------------------+
//| SetGlobalMinLevel                                                |
//+------------------------------------------------------------------+
void CLogger::SetGlobalMinLevel(const LogLevel level)
  {
   m_global_min_level = level;
  }
//+------------------------------------------------------------------+
//| SetExpertInfo                                                    |
//+------------------------------------------------------------------+
void CLogger::SetExpertInfo(const long magic, const string name)
  {
   m_expert_magic = magic;
   m_expert_name = name;
  }
//+------------------------------------------------------------------+
//| Log                                                              |
//+------------------------------------------------------------------+
void CLogger::Log(const LogLevel level, const string origin, const string message)
  {
   // Check global level first
   if(level < m_global_min_level || level >= LOG_LEVEL_OFF)
      return;

   datetime current_time = TimeCurrent();
   string effective_origin = origin;
   if(m_expert_name != "")
      effective_origin = m_expert_name + "::" + origin;
      
   // Dispatch to all registered handlers
   for(int i = 0; i < ArraySize(m_handlers); i++)
     {
      ILogHandler *handler = m_handlers[i];
      if(CheckPointer(handler) != POINTER_INVALID)
        {
         handler.Log(current_time, level, effective_origin, message, m_expert_magic);
        }
     }
  }
//+------------------------------------------------------------------+
//| Convenience Logging Methods                                      |
//+------------------------------------------------------------------+
void CLogger::Debug(const string origin, const string message) { Log(LOG_LEVEL_DEBUG, origin, message); }
void CLogger::Info(const string origin, const string message)  { Log(LOG_LEVEL_INFO, origin, message); }
void CLogger::Warn(const string origin, const string message)  { Log(LOG_LEVEL_WARN, origin, message); }
void CLogger::Error(const string origin, const string message) { Log(LOG_LEVEL_ERROR, origin, message); }
void CLogger::Fatal(const string origin, const string message) { Log(LOG_LEVEL_FATAL, origin, message); }

//+------------------------------------------------------------------+
//| LogFormat                                                        |
//+------------------------------------------------------------------+
void CLogger::LogFormat(const LogLevel level, const string origin, const string formatted_message)
  {
   // Check global level first
   if(level < m_global_min_level || level >= LOG_LEVEL_OFF)
      return;
   Log(level, origin, formatted_message);
  }
//+------------------------------------------------------------------+
//| Convenience Formatted Logging Methods                            |
//+------------------------------------------------------------------+
void CLogger::DebugFormat(const string origin, const string formatted_message) { LogFormat(LOG_LEVEL_DEBUG, origin, formatted_message); }
void CLogger::InfoFormat(const string origin, const string formatted_message)  { LogFormat(LOG_LEVEL_INFO, origin, formatted_message); }
void CLogger::WarnFormat(const string origin, const string formatted_message)  { LogFormat(LOG_LEVEL_WARN, origin, formatted_message); }
void CLogger::ErrorFormat(const string origin, const string formatted_message) { LogFormat(LOG_LEVEL_ERROR, origin, formatted_message); }
void CLogger::FatalFormat(const string origin, const string formatted_message) { LogFormat(LOG_LEVEL_FATAL, origin, formatted_message); }
//+------------------------------------------------------------------+
