//+------------------------------------------------------------------+
//|                                    MQL5 NEWS CALENDAR PART 7.mq5 |
//|                           Copyright 2025, Allan Munene Mutiiria. |
//|                                   https://t.me/Forex_Algo_Trader |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, Allan Munene Mutiiria."
#property link      "https://youtube.com/@ForexAlgo-Trader?"
#property version   "1.00"
#property strict

//---- Define resource file for economic calendar data
#resource "\\Files\\Database\\EconomicCalendar.csv" as string EconomicCalendarData

//---- Input parameter for start date of event filtering
input datetime StartDate = D'2025.03.01'; // Download Start Date
//---- Input parameter for end date of event filtering
input datetime EndDate = D'2025.03.21'; // Download End Date
//---- Input parameter to enable/disable time filtering
input bool ApplyTimeFilter = true;
//---- Input parameter for hours before event to consider
input int HoursBefore = 4;
//---- Input parameter for minutes before event to consider
input int MinutesBefore = 10;
//---- Input parameter for hours after event to consider
input int HoursAfter = 1;
//---- Input parameter for minutes after event to consider
input int MinutesAfter = 5;
//---- Input parameter to enable/disable currency filtering
input bool ApplyCurrencyFilter = true;
//---- Input parameter defining currencies to filter (comma-separated)
input string CurrencyFilter = "USD,EUR,GBP,JPY,AUD,NZD,CAD,CHF"; // All 8 major currencies
//---- Input parameter to enable/disable impact filtering
input bool ApplyImpactFilter = true;

//---- Enumeration for event importance filtering options
enum ENUM_IMPORTANCE {
   IMP_NONE = 0,                  // None
   IMP_LOW,                       // Low
   IMP_MEDIUM,                    // Medium
   IMP_HIGH,                      // High
   IMP_NONE_LOW,                  // None,Low
   IMP_NONE_MEDIUM,               // None,Medium
   IMP_NONE_HIGH,                 // None,High
   IMP_LOW_MEDIUM,                // Low,Medium
   IMP_LOW_HIGH,                  // Low,High
   IMP_MEDIUM_HIGH,               // Medium,High
   IMP_NONE_LOW_MEDIUM,           // None,Low,Medium
   IMP_NONE_LOW_HIGH,             // None,Low,High
   IMP_NONE_MEDIUM_HIGH,          // None,Medium,High
   IMP_LOW_MEDIUM_HIGH,           // Low,Medium,High
   IMP_ALL                        // None,Low,Medium,High (default)
};
//---- Input parameter for selecting importance filter
input ENUM_IMPORTANCE ImportanceFilter = IMP_ALL; // Impact Levels (Default to all)

//---- Structure to hold economic event data
struct EconomicEvent {
   string eventDate;      //---- Date of the event
   string eventTime;      //---- Time of the event
   string currency;       //---- Currency affected by the event
   string event;          //---- Event description
   string importance;     //---- Importance level of the event
   double actual;         //---- Actual value of the event
   double forecast;       //---- Forecasted value of the event
   double previous;       //---- Previous value of the event
};

//---- Array to store all economic events
EconomicEvent allEvents[];
//---- Array for currency filter values
string curr_filter[];
//---- Array for importance filter values
string imp_filter[];

//---- Function to check if a file exists
bool FileExists(string fileName) {
   //---- Open file in read mode to check existence
   int handle = FileOpen(fileName, FILE_READ | FILE_CSV);
   //---- Check if file opened successfully
   if (handle != INVALID_HANDLE) {
      //---- Close the file handle
      FileClose(handle);
      //---- Return true if file exists
      return true;
   }
   //---- Return false if file doesn't exist
   return false;
}

//---- Function to initialize currency and impact filters
void InitializeFilters() {
   //---- Currency Filter Section
   //---- Check if currency filter is enabled and has content
   if (ApplyCurrencyFilter && StringLen(CurrencyFilter) > 0) {
      //---- Split the currency filter string into array
      int count = StringSplit(CurrencyFilter, ',', curr_filter);
      //---- Loop through each currency filter entry
      for (int i = 0; i < ArraySize(curr_filter); i++) {
         //---- Temporary variable for trimming
         string temp = curr_filter[i];
         //---- Remove leading whitespace
         StringTrimLeft(temp);
         //---- Remove trailing whitespace
         StringTrimRight(temp);
         //---- Assign trimmed value back to array
         curr_filter[i] = temp;
         //---- Print currency filter for debugging
         Print("Currency filter [", i, "]: '", curr_filter[i], "'");
      }
   } else if (ApplyCurrencyFilter) {
      //---- Warn if currency filter is enabled but empty
      Print("Warning: CurrencyFilter is empty, no currency filtering applied");
      //---- Resize array to zero if no filter applied
      ArrayResize(curr_filter, 0);
   }

   //---- Impact Filter Section (using enum)
   //---- Check if impact filter is enabled
   if (ApplyImpactFilter) {
      //---- Switch based on selected importance filter
      switch (ImportanceFilter) {
         case IMP_NONE:
            //---- Resize array for single importance level
            ArrayResize(imp_filter, 1);
            //---- Set importance to "None"
            imp_filter[0] = "None";
            break;
         case IMP_LOW:
            //---- Resize array for single importance level
            ArrayResize(imp_filter, 1);
            //---- Set importance to "Low"
            imp_filter[0] = "Low";
            break;
         case IMP_MEDIUM:
            //---- Resize array for single importance level
            ArrayResize(imp_filter, 1);
            //---- Set importance to "Medium"
            imp_filter[0] = "Medium";
            break;
         case IMP_HIGH:
            //---- Resize array for single importance level
            ArrayResize(imp_filter, 1);
            //---- Set importance to "High"
            imp_filter[0] = "High";
            break;
         case IMP_NONE_LOW:
            //---- Resize array for two importance levels
            ArrayResize(imp_filter, 2);
            //---- Set first importance level
            imp_filter[0] = "None";
            //---- Set second importance level
            imp_filter[1] = "Low";
            break;
         case IMP_NONE_MEDIUM:
            //---- Resize array for two importance levels
            ArrayResize(imp_filter, 2);
            //---- Set first importance level
            imp_filter[0] = "None";
            //---- Set second importance level
            imp_filter[1] = "Medium";
            break;
         case IMP_NONE_HIGH:
            //---- Resize array for two importance levels
            ArrayResize(imp_filter, 2);
            //---- Set first importance level
            imp_filter[0] = "None";
            //---- Set second importance level
            imp_filter[1] = "High";
            break;
         case IMP_LOW_MEDIUM:
            //---- Resize array for two importance levels
            ArrayResize(imp_filter, 2);
            //---- Set first importance level
            imp_filter[0] = "Low";
            //---- Set second importance level
            imp_filter[1] = "Medium";
            break;
         case IMP_LOW_HIGH:
            //---- Resize array for two importance levels
            ArrayResize(imp_filter, 2);
            //---- Set first importance level
            imp_filter[0] = "Low";
            //---- Set second importance level
            imp_filter[1] = "High";
            break;
         case IMP_MEDIUM_HIGH:
            //---- Resize array for two importance levels
            ArrayResize(imp_filter, 2);
            //---- Set first importance level
            imp_filter[0] = "Medium";
            //---- Set second importance level
            imp_filter[1] = "High";
            break;
         case IMP_NONE_LOW_MEDIUM:
            //---- Resize array for three importance levels
            ArrayResize(imp_filter, 3);
            //---- Set first importance level
            imp_filter[0] = "None";
            //---- Set second importance level
            imp_filter[1] = "Low";
            //---- Set third importance level
            imp_filter[2] = "Medium";
            break;
         case IMP_NONE_LOW_HIGH:
            //---- Resize array for three importance levels
            ArrayResize(imp_filter, 3);
            //---- Set first importance level
            imp_filter[0] = "None";
            //---- Set second importance level
            imp_filter[1] = "Low";
            //---- Set third importance level
            imp_filter[2] = "High";
            break;
         case IMP_NONE_MEDIUM_HIGH:
            //---- Resize array for three importance levels
            ArrayResize(imp_filter, 3);
            //---- Set first importance level
            imp_filter[0] = "None";
            //---- Set second importance level
            imp_filter[1] = "Medium";
            //---- Set third importance level
            imp_filter[2] = "High";
            break;
         case IMP_LOW_MEDIUM_HIGH:
            //---- Resize array for three importance levels
            ArrayResize(imp_filter, 3);
            //---- Set first importance level
            imp_filter[0] = "Low";
            //---- Set second importance level
            imp_filter[1] = "Medium";
            //---- Set third importance level
            imp_filter[2] = "High";
            break;
         case IMP_ALL:
            //---- Resize array for all importance levels
            ArrayResize(imp_filter, 4);
            //---- Set first importance level
            imp_filter[0] = "None";
            //---- Set second importance level
            imp_filter[1] = "Low";
            //---- Set third importance level
            imp_filter[2] = "Medium";
            //---- Set fourth importance level
            imp_filter[3] = "High";
            break;
      }
      //---- Loop through impact filter array to print values
      for (int i = 0; i < ArraySize(imp_filter); i++) {
         //---- Print each impact filter value
         Print("Impact filter [", i, "]: '", imp_filter[i], "'");
      }
   } else {
      //---- Notify if impact filter is disabled
      Print("Impact filter disabled");
      //---- Resize impact filter array to zero
      ArrayResize(imp_filter, 0);
   }
}

//---- Function to write events to a CSV file
void WriteToCSV(string fileName, EconomicEvent &events[]) {
   //---- Open file for writing in CSV format
   int handle = FileOpen(fileName, FILE_WRITE | FILE_CSV, ',');
   //---- Check if file opening failed
   if (handle == INVALID_HANDLE) {
      //---- Print error message with last error code
      Print("Error creating file: ", GetLastError());
      //---- Exit function on failure
      return;
   }

   //---- Write CSV header row
   FileWrite(handle, "Date", "Time", "Currency", "Event", "Importance", "Actual", "Forecast", "Previous");
   //---- Loop through all events to write to file
   for (int i = 0; i < ArraySize(events); i++) {
      //---- Write event data to CSV file
      FileWrite(handle, events[i].eventDate, events[i].eventTime, events[i].currency, events[i].event,
                events[i].importance, DoubleToString(events[i].actual, 2), DoubleToString(events[i].forecast, 2),
                DoubleToString(events[i].previous, 2));
      //---- Print event details for debugging
      Print("Writing event ", i, ": ", events[i].eventDate, ", ", events[i].eventTime, ", ", events[i].currency, ", ",
            events[i].event, ", ", events[i].importance, ", ", DoubleToString(events[i].actual, 2), ", ",
            DoubleToString(events[i].forecast, 2), ", ", DoubleToString(events[i].previous, 2));
   }

   //---- Flush data to file
   FileFlush(handle);
   //---- Close the file handle
   FileClose(handle);
   //---- Print confirmation of data written
   Print("Data written to ", fileName, " with ", ArraySize(events), " events.");

   //---- Verify written file by reading it back
   int verifyHandle = FileOpen(fileName, FILE_READ | FILE_TXT);
   //---- Check if verification file opening succeeded
   if (verifyHandle != INVALID_HANDLE) {
      //---- Read entire file content
      string content = FileReadString(verifyHandle, (int)FileSize(verifyHandle));
      //---- Print file content for verification
      Print("File content after writing (size: ", FileSize(verifyHandle), " bytes):\n", content);
      //---- Close verification file handle
      FileClose(verifyHandle);
   }
}

//---- Function to load events from resource file
bool LoadEventsFromResource() {
   //---- Get data from resource
   string fileData = EconomicCalendarData;
   //---- Print raw resource content for debugging
   Print("Raw resource content (size: ", StringLen(fileData), " bytes):\n", fileData);

   //---- Array to hold lines from resource
   string lines[];
   //---- Split resource data into lines
   int lineCount = StringSplit(fileData, '\n', lines);
   //---- Check if resource has valid data
   if (lineCount <= 1) {
      //---- Print error if no data lines found
      Print("Error: No data lines found in resource! Raw data: ", fileData);
      //---- Return false on failure
      return false;
   }

   //---- Reset events array
   ArrayResize(allEvents, 0);
   //---- Index for event array
   int eventIndex = 0;

   //---- Loop through each line (skip header at i=0)
   for (int i = 1; i < lineCount; i++) {
      //---- Check for empty lines
      if (StringLen(lines[i]) == 0) {
         //---- Print message for skipped empty line
         Print("Skipping empty line ", i);
         //---- Skip to next iteration
         continue;
      }

      //---- Array to hold fields from each line
      string fields[];
      //---- Split line into fields
      int fieldCount = StringSplit(lines[i], ',', fields);
      //---- Print line details for debugging
      Print("Line ", i, ": ", lines[i], " (field count: ", fieldCount, ")");

      //---- Check if line has minimum required fields
      if (fieldCount < 8) {
         //---- Print error for malformed line
         Print("Malformed line ", i, ": ", lines[i], " (field count: ", fieldCount, ")");
         //---- Skip to next iteration
         continue;
      }

      //---- Extract date from field
      string dateStr = fields[0];
      //---- Extract time from field
      string timeStr = fields[1];
      //---- Extract currency from field
      string currency = fields[2];
      //---- Extract event description (handle commas in event name)
      string event = fields[3];
      //---- Combine multiple fields if event name contains commas
      for (int j = 4; j < fieldCount - 4; j++) {
         event += "," + fields[j];
      }
      //---- Extract importance from field
      string importance = fields[fieldCount - 4];
      //---- Extract actual value from field
      string actualStr = fields[fieldCount - 3];
      //---- Extract forecast value from field
      string forecastStr = fields[fieldCount - 2];
      //---- Extract previous value from field
      string previousStr = fields[fieldCount - 1];

      //---- Convert date and time to datetime format
      datetime eventDateTime = StringToTime(dateStr + " " + timeStr);
      //---- Check if datetime conversion failed
      if (eventDateTime == 0) {
         //---- Print error for invalid datetime
         Print("Error: Invalid datetime conversion for line ", i, ": ", dateStr, " ", timeStr);
         //---- Skip to next iteration
         continue;
      }

      //---- Resize events array for new event
      ArrayResize(allEvents, eventIndex + 1);
      //---- Assign event date
      allEvents[eventIndex].eventDate = dateStr;
      //---- Assign event time
      allEvents[eventIndex].eventTime = timeStr;
      //---- Assign event currency
      allEvents[eventIndex].currency = currency;
      //---- Assign event description
      allEvents[eventIndex].event = event;
      //---- Assign event importance
      allEvents[eventIndex].importance = importance;
      //---- Convert and assign actual value
      allEvents[eventIndex].actual = StringToDouble(actualStr);
      //---- Convert and assign forecast value
      allEvents[eventIndex].forecast = StringToDouble(forecastStr);
      //---- Convert and assign previous value
      allEvents[eventIndex].previous = StringToDouble(previousStr);
      //---- Print loaded event details
      Print("Loaded event ", eventIndex, ": ", dateStr, " ", timeStr, ", ", currency, ", ", event);
      //---- Increment event index
      eventIndex++;
   }

   //---- Print total events loaded
   Print("Loaded ", eventIndex, " events from resource into array.");
   //---- Return success if events were loaded
   return eventIndex > 0;
}

//---- Function to filter and print economic events
void FilterAndPrintEvents(datetime barTime) {
   //---- Get total number of events
   int totalEvents = ArraySize(allEvents);
   //---- Print total events considered
   Print("Total considered data size: ", totalEvents, " events");

   //---- Check if there are events to filter
   if (totalEvents == 0) {
      //---- Print message if no events loaded
      Print("No events loaded to filter.");
      //---- Exit function
      return;
   }

   //---- Array to store filtered events
   EconomicEvent filteredEvents[];
   //---- Counter for filtered events
   int filteredCount = 0;

   //---- Variables for time range
   datetime timeBefore, timeAfter;
   //---- Apply time filter if enabled
   if (ApplyTimeFilter) {
      //---- Structure for bar time
      MqlDateTime barStruct;
      //---- Convert bar time to structure
      TimeToStruct(barTime, barStruct);

      //---- Calculate time before event
      MqlDateTime timeBeforeStruct = barStruct;
      //---- Subtract hours before
      timeBeforeStruct.hour -= HoursBefore;
      //---- Subtract minutes before
      timeBeforeStruct.min -= MinutesBefore;
      //---- Adjust for negative minutes
      if (timeBeforeStruct.min < 0) {
         timeBeforeStruct.min += 60;
         timeBeforeStruct.hour -= 1;
      }
      //---- Adjust for negative hours
      if (timeBeforeStruct.hour < 0) {
         timeBeforeStruct.hour += 24;
         timeBeforeStruct.day -= 1;
      }
      //---- Convert structure to datetime
      timeBefore = StructToTime(timeBeforeStruct);

      //---- Calculate time after event
      MqlDateTime timeAfterStruct = barStruct;
      //---- Add hours after
      timeAfterStruct.hour += HoursAfter;
      //---- Add minutes after
      timeAfterStruct.min += MinutesAfter;
      //---- Adjust for minutes overflow
      if (timeAfterStruct.min >= 60) {
         timeAfterStruct.min -= 60;
         timeAfterStruct.hour += 1;
      }
      //---- Adjust for hours overflow
      if (timeAfterStruct.hour >= 24) {
         timeAfterStruct.hour -= 24;
         timeAfterStruct.day += 1;
      }
      //---- Convert structure to datetime
      timeAfter = StructToTime(timeAfterStruct);

      //---- Print time range for debugging
      Print("Bar time: ", TimeToString(barTime), ", Time range: ", TimeToString(timeBefore), " to ", TimeToString(timeAfter));
   } else {
      //---- Print message if no time filter applied
      Print("Bar time: ", TimeToString(barTime), ", No time filter applied, using StartDate to EndDate only.");
      //---- Set time range to date inputs
      timeBefore = StartDate;
      timeAfter = EndDate;
   }

   //---- Loop through all events for filtering
   for (int i = 0; i < totalEvents; i++) {
      //---- Convert event date and time to datetime
      datetime eventDateTime = StringToTime(allEvents[i].eventDate + " " + allEvents[i].eventTime);
      //---- Check if event is within date range
      bool inDateRange = (eventDateTime >= StartDate && eventDateTime <= EndDate);
      //---- Skip if not in date range
      if (!inDateRange) continue;

      //---- Time Filter Check
      //---- Check if event is within time range if filter applied
      bool timeMatch = !ApplyTimeFilter || (eventDateTime >= timeBefore && eventDateTime <= timeAfter);
      //---- Skip if time doesn't match
      if (!timeMatch) continue;
      //---- Print event details if time passes
      Print("Event ", i, ": Time passes (", allEvents[i].eventDate, " ", allEvents[i].eventTime, ") - ",
            "Currency: ", allEvents[i].currency, ", Event: ", allEvents[i].event, ", Importance: ", allEvents[i].importance,
            ", Actual: ", DoubleToString(allEvents[i].actual, 2), ", Forecast: ", DoubleToString(allEvents[i].forecast, 2),
            ", Previous: ", DoubleToString(allEvents[i].previous, 2));

      //---- Currency Filter Check
      //---- Default to match if filter disabled
      bool currencyMatch = !ApplyCurrencyFilter;
      //---- Apply currency filter if enabled
      if (ApplyCurrencyFilter && ArraySize(curr_filter) > 0) {
         //---- Initially set to no match
         currencyMatch = false;
         //---- Check each currency in filter
         for (int j = 0; j < ArraySize(curr_filter); j++) {
            //---- Check if event currency matches filter
            if (allEvents[i].currency == curr_filter[j]) {
               //---- Set match to true if found
               currencyMatch = true;
               //---- Exit loop on match
               break;
            }
         }
         //---- Skip if currency doesn't match
         if (!currencyMatch) continue;
      }
      //---- Print event details if currency passes
      Print("Event ", i, ": Currency passes (", allEvents[i].currency, ") - ",
            "Date: ", allEvents[i].eventDate, " ", allEvents[i].eventTime,
            ", Event: ", allEvents[i].event, ", Importance: ", allEvents[i].importance,
            ", Actual: ", DoubleToString(allEvents[i].actual, 2), ", Forecast: ", DoubleToString(allEvents[i].forecast, 2),
            ", Previous: ", DoubleToString(allEvents[i].previous, 2));

      //---- Impact Filter Check
      //---- Default to match if filter disabled
      bool impactMatch = !ApplyImpactFilter;
      //---- Apply impact filter if enabled
      if (ApplyImpactFilter && ArraySize(imp_filter) > 0) {
         //---- Initially set to no match
         impactMatch = false;
         //---- Check each importance in filter
         for (int k = 0; k < ArraySize(imp_filter); k++) {
            //---- Check if event importance matches filter
            if (allEvents[i].importance == imp_filter[k]) {
               //---- Set match to true if found
               impactMatch = true;
               //---- Exit loop on match
               break;
            }
         }
         //---- Skip if importance doesn't match
         if (!impactMatch) continue;
      }
      //---- Print event details if impact passes
      Print("Event ", i, ": Impact passes (", allEvents[i].importance, ") - ",
            "Date: ", allEvents[i].eventDate, " ", allEvents[i].eventTime,
            ", Currency: ", allEvents[i].currency, ", Event: ", allEvents[i].event,
            ", Actual: ", DoubleToString(allEvents[i].actual, 2), ", Forecast: ", DoubleToString(allEvents[i].forecast, 2),
            ", Previous: ", DoubleToString(allEvents[i].previous, 2));

      //---- Add event to filtered array
      ArrayResize(filteredEvents, filteredCount + 1);
      //---- Assign event to filtered array
      filteredEvents[filteredCount] = allEvents[i];
      //---- Increment filtered count
      filteredCount++;
   }

   //---- Print summary of filtered events
   Print("After ", (ApplyTimeFilter ? "time filter" : "date range filter"),
         ApplyCurrencyFilter ? " and currency filter" : "",
         ApplyImpactFilter ? " and impact filter" : "",
         ": ", filteredCount, " events remaining.");

   //---- Check if there are filtered events to print
   if (filteredCount > 0) {
      //---- Print header for filtered events
      Print("Filtered Events at Bar Time: ", TimeToString(barTime));
      //---- Print filtered events array
      ArrayPrint(filteredEvents, 2, " | ");
   } else {
      //---- Print message if no events found
      Print("No events found within the specified range.");
   }
}

//---- Initialization function
int OnInit() {
   //---- Define file path for CSV
   string fileName = "Database\\EconomicCalendar.csv";
   //---- Initialize filters
   InitializeFilters();

   //---- Check if not running in tester mode
   if (!MQLInfoInteger(MQL_TESTER)) {
      //---- Validate date range
      if (StartDate >= EndDate) {
         //---- Print error for invalid date range
         Print("Error: StartDate (", TimeToString(StartDate), ") must be earlier than EndDate (", TimeToString(EndDate), ")");
         //---- Return initialization failure
         return(INIT_PARAMETERS_INCORRECT);
      }

      //---- Array to hold calendar values
      MqlCalendarValue values[];
      //---- Fetch calendar data for date range
      if (!CalendarValueHistory(values, StartDate, EndDate)) {
         //---- Print error if calendar data fetch fails
         Print("Error fetching calendar data: ", GetLastError());
         //---- Return initialization failure
         return(INIT_FAILED);
      }

      //---- Array to hold economic events
      EconomicEvent events[];
      //---- Counter for events
      int eventCount = 0;

      //---- Loop through calendar values
      for (int i = 0; i < ArraySize(values); i++) {
         //---- Structure for event details
         MqlCalendarEvent eventDetails;
         //---- Fetch event details by ID
         if (!CalendarEventById(values[i].event_id, eventDetails)) continue;

         //---- Structure for country details
         MqlCalendarCountry countryDetails;
         //---- Fetch country details by ID
         if (!CalendarCountryById(eventDetails.country_id, countryDetails)) continue;

         //---- Structure for value details
         MqlCalendarValue value;
         //---- Fetch value details by ID
         if (!CalendarValueById(values[i].id, value)) continue;

         //---- Resize events array for new event
         ArrayResize(events, eventCount + 1);
         //---- Convert event time to string
         string dateTimeStr = TimeToString(values[i].time, TIME_DATE | TIME_MINUTES);
         //---- Extract date from datetime string
         events[eventCount].eventDate = StringSubstr(dateTimeStr, 0, 10);
         //---- Extract time from datetime string
         events[eventCount].eventTime = StringSubstr(dateTimeStr, 11, 5);
         //---- Assign currency from country details
         events[eventCount].currency = countryDetails.currency;
         //---- Assign event name
         events[eventCount].event = eventDetails.name;
         //---- Map importance level from enum to string
         events[eventCount].importance = (eventDetails.importance == 0) ? "None" :    // CALENDAR_IMPORTANCE_NONE
                                         (eventDetails.importance == 1) ? "Low" :     // CALENDAR_IMPORTANCE_LOW
                                         (eventDetails.importance == 2) ? "Medium" :  // CALENDAR_IMPORTANCE_MODERATE
                                         "High";                                      // CALENDAR_IMPORTANCE_HIGH
         //---- Assign actual value
         events[eventCount].actual = value.GetActualValue();
         //---- Assign forecast value
         events[eventCount].forecast = value.GetForecastValue();
         //---- Assign previous value
         events[eventCount].previous = value.GetPreviousValue();
         //---- Increment event count
         eventCount++;
      }

      //---- Check if file exists and print appropriate message
      if (!FileExists(fileName)) Print("Creating new file: ", fileName);
      else Print("Overwriting existing file: ", fileName);

      //---- Write events to CSV file
      WriteToCSV(fileName, events);
      //---- Print instructions for tester mode
      Print("Live mode: Data written. To use in tester, manually add ", fileName, " as a resource and recompile.");
   } else {
      //---- Check if resource data is empty in tester mode
      if (StringLen(EconomicCalendarData) == 0) {
         //---- Print error for empty resource
         Print("Error: Resource EconomicCalendarData is empty. Please run in live mode, add the file as a resource, and recompile.");
         //---- Return initialization failure
         return(INIT_FAILED);
      }
      //---- Print message for tester mode
      Print("Running in Strategy Tester, using embedded resource: Database\\EconomicCalendar.csv");

      //---- Load events from resource
      if (!LoadEventsFromResource()) {
         //---- Print error if loading fails
         Print("Failed to load events from resource.");
         //---- Return initialization failure
         return(INIT_FAILED);
      }
   }

   //---- Return successful initialization
   return(INIT_SUCCEEDED);
}

//---- Deinitialization function
void OnDeinit(const int reason) {
   //---- Print termination reason
   Print("EA terminated, reason: ", reason);
}

//---- Variable to track last bar time
datetime lastBarTime = 0;

//---- Tick event handler
void OnTick() {
   //---- Get current bar time
   datetime currentBarTime = iTime(_Symbol, _Period, 0);
   //---- Check if bar time has changed
   if (currentBarTime != lastBarTime) {
      //---- Update last bar time
      lastBarTime = currentBarTime;
      //---- Filter and print events for current bar
      FilterAndPrintEvents(currentBarTime);
   }
}