//+------------------------------------------------------------------+
//|                                               SignalFractals.mqh |
//|                             Copyright 2000-2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertSignal.mqh>
#include <Arrays\ArrayDouble.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals of Bill-Williams' Fractal Patterns.                |
//| Type=SignalAdvanced                                              |
//| Name=Fractals                                                    |
//| ShortName=Fractals                                               |
//| Class=CSignalFractals                                            |
//| Page=signal_fractals                                             |
//| Parameter=Pattern_0,int,50,Pattern 0                             |
//| Parameter=Pattern_1,int,50,Pattern 1                             |
//| Parameter=Pattern_2,int,50,Pattern 2                             |
//| Parameter=Pattern_3,int,50,Pattern 3                             |
//| Parameter=Pattern_4,int,50,Pattern 4                             |
//| Parameter=Pattern_5,int,50,Pattern 5                             |
//| Parameter=Pattern_6,int,50,Pattern 6                             |
//| Parameter=Pattern_7,int,50,Pattern 7                             |
//| Parameter=Pattern_8,int,50,Pattern 8                             |
//| Parameter=Pattern_9,int,50,Pattern 9                             |
//| Parameter=PatternsUsed,int,255,Patterns Used BitMap              |
//| Parameter=PeriodUsed,int,5,Used Period                           |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalFractals.                                           |
//| Purpose: Class of generator of trade signals based on            |
//|          Bill-Williams Fractal Patterns.                         |
//| Is derived from the CExpertSignal class.                         |
//+------------------------------------------------------------------+
class CSignalFractals : public CExpertSignal
{
protected:
   CiFractals        m_fractals;
   CiMA              m_ma;
   CiATR             m_atr;
   //--- adjusted parameters

   //--- "weights" of market models (0-100)
   int               m_pattern_0;      // model 0 "Consecutive Fractals"
   int               m_pattern_1;      // model 1 "Fractal Trend Breakout"
   int               m_pattern_2;      // model 2 "Inside Fractals"
   int               m_pattern_3;      // model 3 "Fractal Divergence"
   int               m_pattern_4;      // model 4 "Twin Opposing Fractals"
   int               m_pattern_5;      // model 5 "Fractal Confirmation of Swings"
   int               m_pattern_6;      // model 6 "Fractal Failure Swing"
   int               m_pattern_7;      // model 7 "Fractal Cluster at MA"
   int               m_pattern_8;      // model 8 "Fractal Gap Pattern"
   int               m_pattern_9;      // model 9 "Fractal Alignment with Fibo"

   int               m_periods;
   //
   //int               m_patterns_usage;   //

public:
   CSignalFractals(void);
   ~CSignalFractals(void);
   //--- methods of setting adjustable parameters
   //--- methods of adjusting "weights" of market models
   void              Pattern_0(int value)
   {  m_pattern_0 = value;
   }
   void              Pattern_1(int value)
   {  m_pattern_1 = value;
   }
   void              Pattern_2(int value)
   {  m_pattern_2 = value;
   }
   void              Pattern_3(int value)
   {  m_pattern_3 = value;
   }
   void              Pattern_4(int value)
   {  m_pattern_4 = value;
   }
   void              Pattern_5(int value)
   {  m_pattern_5 = value;
   }
   void              Pattern_6(int value)
   {  m_pattern_6 = value;
   }
   void              Pattern_7(int value)
   {  m_pattern_7 = value;
   }
   void              Pattern_8(int value)
   {  m_pattern_8 = value;
   }
   void              Pattern_9(int value)
   {  m_pattern_9 = value;
   }
   void              PatternsUsed(int value)
   {  m_patterns_usage = value;
      PatternsUsage(value);
   }
   void              PeriodUsed(int value)
   {  m_periods = value;
   }
   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the oscillator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int       LongCondition(void);
   virtual int       ShortCondition(void);

protected:
   //--- method of initialization of the oscillator
   bool              InitFractals(CIndicators *indicators);
   //--- methods of getting data
   double            FractalHigh(int ind)
   {  //
      m_high.Refresh(-1);
      if(m_high.MaxIndex(ind, 5) == ind)
      {  return(m_high.GetData(ind));
      }
      return(0.0);
   }
   double            FractalLow(int ind)
   {  //
      m_low.Refresh(-1);
      if(m_low.MinIndex(ind, 5) == ind)
      {  return(m_low.GetData(ind));
      }
      return(0.0);
   }
   //
   double            MA(int ind)
   {  //
      m_ma.Refresh(-1);
      return(m_ma.Main(ind));
   }
   bool Is_Fibo_Level(int ind)
   {  double _r=0.0;
      vector _h,_l;
      int _size = 3 * PeriodSeconds(PERIOD_MN1) / PeriodSeconds(m_period);
      _h.Init(_size);
      _l.Init(_size);
      if(_h.CopyRates(m_symbol.Name(),m_period,2,0,_size) && _l.CopyRates(m_symbol.Name(),m_period,4,0,_size))
      {  _r = _h.Max()-_l.Min();
         if(_l.Min()-ATR(ind) <= Close(ind) && Close(ind) <= _l.Min()+ATR(ind))
         {  return(true);
         }
         else if(_l.Min()+(0.236*_r)-ATR(ind) <= Close(ind) && Close(ind) <= _l.Min()+(0.236*_r)+ATR(ind))
         {  return(true);
         }
         else if(_l.Min()+(0.382*_r)-ATR(ind) <= Close(ind) && Close(ind) <= _l.Min()+(0.382*_r)+ATR(ind))
         {  return(true);
         }
         else if(_l.Min()+(0.5*_r)-ATR(ind) <= Close(ind) && Close(ind) <= _l.Min()+(0.5*_r)+ATR(ind))
         {  return(true);
         }
         else if(_l.Min()+(0.618*_r)-ATR(ind) <= Close(ind) && Close(ind) <= _l.Min()+(0.618*_r)+ATR(ind))
         {  return(true);
         }
         else if(_h.Max()-ATR(ind) <= Close(ind) && Close(ind) <= _h.Max()+ATR(ind))
         {  return(true);
         }
      }
      return(false);
   }
   double            ATR(int ind)
   {  //
      m_atr.Refresh(-1);
      return(m_atr.Main(ind));
   }
   double            Close(int ind)
   {  //
      m_close.Refresh(-1);
      return(m_close.GetData(ind));
   }
   double            High(int ind)
   {  //
      m_high.Refresh(-1);
      return(m_high.GetData(ind));
   }
   double            Low(int ind)
   {  //
      m_low.Refresh(-1);
      return(m_low.GetData(ind));
   }
   int               X()
   {  //
      return(StartIndex());
   }
   //--- methods to check for patterns
   bool              IsPattern_0(ENUM_POSITION_TYPE T);
   bool              IsPattern_1(ENUM_POSITION_TYPE T);
   bool              IsPattern_2(ENUM_POSITION_TYPE T);
   bool              IsPattern_3(ENUM_POSITION_TYPE T);
   bool              IsPattern_4(ENUM_POSITION_TYPE T);
   bool              IsPattern_5(ENUM_POSITION_TYPE T);
   bool              IsPattern_6(ENUM_POSITION_TYPE T);
   bool              IsPattern_7(ENUM_POSITION_TYPE T);
   bool              IsPattern_8(ENUM_POSITION_TYPE T);
   bool              IsPattern_9(ENUM_POSITION_TYPE T);
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CSignalFractals::CSignalFractals(void) : m_pattern_0(50),
   m_pattern_1(50),
   m_pattern_2(50),
   m_pattern_3(50),
   m_pattern_4(50),
   m_pattern_5(50),
   m_pattern_6(50),
   m_pattern_7(50),
   m_pattern_8(50),
   m_pattern_9(50)
   //m_patterns_usage(255)
{
//--- initialization of protected data
   m_used_series = USE_SERIES_OPEN + USE_SERIES_HIGH + USE_SERIES_LOW + USE_SERIES_CLOSE + USE_SERIES_TICK_VOLUME;
   PatternsUsage(m_patterns_usage);
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CSignalFractals::~CSignalFractals(void)
{
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalFractals::ValidationSettings(void)
{
//--- validation settings of additional filters
   if(!CExpertSignal::ValidationSettings())
      return(false);
//--- initial data checks
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalFractals::InitIndicators(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- create and initialize MA oscillator
   if(!InitFractals(indicators))
      return(false);
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Initialize MA indicators.                                        |
//+------------------------------------------------------------------+
bool CSignalFractals::InitFractals(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- add object to collection
   if(!indicators.Add(GetPointer(m_fractals)))
   {  printf(__FUNCTION__ + ": error adding object");
      return(false);
   }
//--- initialize object
   if(!m_fractals.Create(m_symbol.Name(), m_period))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_ma.Create(m_symbol.Name(), m_period, m_periods, 0, MODE_SMA, PRICE_CLOSE))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_atr.Create(m_symbol.Name(), m_period, m_periods))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalFractals::LongCondition(void)
{  int result  = 0, results = 0;
//--- if the model 0 is used and "Consecutive Fractals"
   if(((m_patterns_usage & 0x01) != 0) && IsPattern_0(POSITION_TYPE_BUY))
   {  result += m_pattern_0;
      results++;
   }
//--- if the model 1 is used and "Fractal Trend Breakout"
   if(((m_patterns_usage & 0x02) != 0) && IsPattern_1(POSITION_TYPE_BUY))
   {  result += m_pattern_1;
      results++;
   }
//--- if the model 2 is used and "Inside Fractals"
   if(((m_patterns_usage & 0x04) != 0) && IsPattern_2(POSITION_TYPE_BUY))
   {  result += m_pattern_2;
      results++;
   }
//--- if the model 3 is used and "Fractal Divergence"
   if(((m_patterns_usage & 0x08) != 0) && IsPattern_3(POSITION_TYPE_BUY))
   {  result += m_pattern_3;
      results++;
   }
//--- if the model 4 is used and "Twin Opposing Fractals"
   if(((m_patterns_usage & 0x10) != 0) && IsPattern_4(POSITION_TYPE_BUY))
   {  result += m_pattern_4;
      results++;
   }
//--- if the model 5 is used and "Fractal Confirmation of Swings"
   if(((m_patterns_usage & 0x20) != 0) && IsPattern_5(POSITION_TYPE_BUY))
   {  result += m_pattern_5;
      results++;
   }
//--- if the model 6 is used and "Fractal Failure Swing"
   if(((m_patterns_usage & 0x40) != 0) && IsPattern_6(POSITION_TYPE_BUY))
   {  result += m_pattern_6;
      results++;
   }
//--- if the model 7 is used and "Fractal Cluster at MA"
   if(((m_patterns_usage & 0x80) != 0) && IsPattern_7(POSITION_TYPE_BUY))
   {  result += m_pattern_7;
      results++;
   }
//--- if the model 8 is used and "Fractal Gap Pattern"
   if(((m_patterns_usage & 0x100) != 0) && IsPattern_8(POSITION_TYPE_BUY))
   {  result += m_pattern_8;
      results++;
   }
//--- if the model 9 is used and "Fractal Alignment with Fibo"
   if(((m_patterns_usage & 0x200) != 0) && IsPattern_9(POSITION_TYPE_BUY))
   {  result += m_pattern_9;
      results++;
   }
//--- return the result
//if(result > 0)printf(__FUNCSIG__+" result is: %i",result);
   if(results > 0)
   {  return(int(round(result / results)));
   }
   return(0);
}
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalFractals::ShortCondition(void)
{  int result  = 0, results = 0;
//--- if the model 0 is used and "Consecutive Fractals"
   if(((m_patterns_usage & 0x01) != 0) && IsPattern_0(POSITION_TYPE_SELL))
   {  result += m_pattern_0;
      results++;
   }
//--- if the model 1 is used and "Fractal Trend Breakout"
   if(((m_patterns_usage & 0x02) != 0) && IsPattern_1(POSITION_TYPE_SELL))
   {  result += m_pattern_1;
      results++;
   }
//--- if the model 2 is used and "Inside Fractals"
   if(((m_patterns_usage & 0x04) != 0) && IsPattern_2(POSITION_TYPE_SELL))
   {  result += m_pattern_2;
      results++;
   }
//--- if the model 3 is used and "Fractal Divergence"
   if(((m_patterns_usage & 0x08) != 0) && IsPattern_3(POSITION_TYPE_SELL))
   {  result += m_pattern_3;
      results++;
   }
//--- if the model 4 is used and "Twin Opposing Fractals"
   if(((m_patterns_usage & 0x10) != 0) && IsPattern_4(POSITION_TYPE_SELL))
   {  result += m_pattern_4;
      results++;
   }
//--- if the model 5 is used and "Fractal Confirmation of Swings"
   if(((m_patterns_usage & 0x20) != 0) && IsPattern_5(POSITION_TYPE_SELL))
   {  result += m_pattern_5;
      results++;
   }
//--- if the model 6 is used and "Fractal Failure Swing"
   if(((m_patterns_usage & 0x40) != 0) && IsPattern_6(POSITION_TYPE_SELL))
   {  result += m_pattern_6;
      results++;
   }
//--- if the model 7 is used and "Fractal Cluster at MA"
   if(((m_patterns_usage & 0x80) != 0) && IsPattern_7(POSITION_TYPE_SELL))
   {  result += m_pattern_7;
      results++;
   }
//--- if the model 8 is used and "Fractal Gap Pattern"
   if(((m_patterns_usage & 0x100) != 0) && IsPattern_8(POSITION_TYPE_SELL))
   {  result += m_pattern_8;
      results++;
   }
//--- if the model 9 is used and "Fractal Alignment with Fibo"
   if(((m_patterns_usage & 0x200) != 0) && IsPattern_9(POSITION_TYPE_SELL))
   {  result += m_pattern_9;
      results++;
   }
//--- return the result
//if(result > 0)printf(__FUNCSIG__+" result is: %i",result);
   if(results > 0)
   {  return(int(round(result / results)));
   }
   return(0);
}
//+------------------------------------------------------------------+
//| Check for Pattern 0.                                             |
//+------------------------------------------------------------------+
bool CSignalFractals::IsPattern_0(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && FractalLow(X() + 1) != 0.0 && FractalLow(X()) != 0.0)
   {  return(true);
   }
   else if(T == POSITION_TYPE_SELL && FractalHigh(X() + 1) != 0.0 && FractalHigh(X()) != 0.0)
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 1.                                             |
//+------------------------------------------------------------------+
bool CSignalFractals::IsPattern_1(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && FractalHigh(X() + 1) != 0.0 && Close(X()) > FractalHigh(X() + 1))
   {  return(true);
   }
   else if(T == POSITION_TYPE_SELL && FractalLow(X() + 1) != 0.0 && Close(X()) < FractalLow(X() + 1))
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 2.                                             |
//+------------------------------------------------------------------+
bool CSignalFractals::IsPattern_2(ENUM_POSITION_TYPE T)
{  CArrayDouble _buffer;
   if(T == POSITION_TYPE_BUY)
   {  for(int i = 0; i < m_periods; i++)
      {  if(FractalLow(X() + i) != 0.0)
         {  _buffer.Add(FractalLow(X() + i));
         }
      }
      if(_buffer[_buffer.Maximum(0, _buffer.Total())] - _buffer[_buffer.Minimum(0, _buffer.Total())] <= fabs(Close(X()) - Close(X() + 10)))
      {  return(true);
      }
   }
   else if(T == POSITION_TYPE_SELL)
   {  for(int i = 0; i < m_periods; i++)
      {  if(FractalHigh(X() + i) != 0.0)
         {  _buffer.Add(FractalHigh(X() + i));
         }
      }
      if(_buffer[_buffer.Maximum(0, _buffer.Total())] - _buffer[_buffer.Minimum(0, _buffer.Total())] <= fabs(Close(X()) - Close(X() + 10)))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 3.                                             |
//+------------------------------------------------------------------+
bool CSignalFractals::IsPattern_3(ENUM_POSITION_TYPE T)
{  CArrayDouble _buffer;
   if(T == POSITION_TYPE_BUY)
   {  for(int i = 0; i < m_periods; i++)
      {  if(FractalLow(X() + i) != 0.0)
         {  _buffer.Add(FractalLow(X() + i));
         }
         if(_buffer.Total() >= 2)
         {  break;
         }
      }
      if(_buffer[0] > _buffer[1] && Low(X()) < Low(X() + 1))
      {  return(true);
      }
   }
   else if(T == POSITION_TYPE_SELL)
   {  for(int i = 0; i < m_periods; i++)
      {  if(FractalHigh(X() + i) != 0.0)
         {  _buffer.Add(FractalHigh(X() + i));
         }
         if(_buffer.Total() >= 2)
         {  break;
         }
      }
      if(_buffer[0] < _buffer[1] && High(X()) > High(X() + 1))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 4.                                             |
//+------------------------------------------------------------------+
bool CSignalFractals::IsPattern_4(ENUM_POSITION_TYPE T)
{  bool _1 = (FractalHigh(X() + 1) != 0.0 && FractalLow(X() + 1) != 0.0);
   bool _2 = (FractalHigh(X() + 2) != 0.0 && FractalLow(X() + 2) != 0.0);
   if(_1 || _2)
   {  if(T == POSITION_TYPE_BUY)
      {  if((_1 && Close(X()) > FractalHigh(X() + 1)) || (_2 && Close(X()) > FractalHigh(X() + 2)))
         {  return(true);
         }
      }
      else if(T == POSITION_TYPE_SELL)
      {  if((_1 && Close(X()) < FractalLow(X() + 1)) || (_2 && Close(X()) < FractalLow(X() + 2)))
         {  return(true);
         }
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 5.                                             |
//+------------------------------------------------------------------+
bool CSignalFractals::IsPattern_5(ENUM_POSITION_TYPE T)
{  vector _std;
   _std.Init(5);
   _std.Fill(0.0);
   if(T == POSITION_TYPE_BUY && FractalLow(X() + m_periods) != 0.0)
   {  if(_std.CopyRates(m_symbol.Name(), m_period, 4, 0, m_periods) && _std.Std() <= fabs(Close(X()) - Close(X() + m_periods)))
      {  return(true);
      }
   }
   else if(T == POSITION_TYPE_SELL && FractalHigh(X() + m_periods))
   {  if(_std.CopyRates(m_symbol.Name(), m_period, 2, 0, m_periods) && _std.Std() <= fabs(Close(X()) - Close(X() + m_periods)))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 6.                                             |
//+------------------------------------------------------------------+
bool CSignalFractals::IsPattern_6(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && FractalLow(X() + 2) != 0.0)
   {  if(Low(X() + 1) <= FractalLow(X() + 2) && Close(X()) >= High(X() + 2))
      {  return(true);
      }
   }
   else if(T == POSITION_TYPE_SELL && FractalHigh(X() + 2) != 0.0)
   {  if(High(X() + 1) >= FractalHigh(X() + 2) && Close(X()) <= Low(X() + 2))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 7.                                             |
//+------------------------------------------------------------------+
bool CSignalFractals::IsPattern_7(ENUM_POSITION_TYPE T)
{  CArrayDouble _buffer;
   if(T == POSITION_TYPE_BUY)
   {  for(int i = 1; i < m_periods; i++)
      {  if(FractalLow(X() + i) != 0.0)
         {  _buffer.Add(fabs(FractalLow(X() + i) - MA(X() + i)));
         }
      }
      if(_buffer[_buffer.Maximum(0, _buffer.Total())] <= fabs(Close(X() + 1) - Close(X() + m_periods)) && Close(X() + 1) <= MA(X() + 1) && Close(X()) > MA(X()))
      {  return(true);
      }
   }
   else if(T == POSITION_TYPE_SELL)
   {  for(int i = 1; i < m_periods; i++)
      {  if(FractalHigh(X() + i) != 0.0)
         {  _buffer.Add(fabs(FractalHigh(X() + i) - MA(X() + i)));
         }
      }
      if(_buffer[_buffer.Maximum(0, _buffer.Total())] <= fabs(Close(X() + 1) - Close(X() + m_periods)) && Close(X() + 1) >= MA(X() + 1) && Close(X()) < MA(X()))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 8.                                             |
//+------------------------------------------------------------------+
bool CSignalFractals::IsPattern_8(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && FractalLow(X() + 2) != 0.0)
   {  if(Low(X() + 1) > High(X() + 2) && Close(X()) >= High(X() + 2))
      {  return(true);
      }
   }
   else if(T == POSITION_TYPE_SELL && FractalHigh(X() + 2) != 0.0)
   {  if(High(X() + 1) < Low(X() + 2) && Close(X()) <= Low(X() + 2))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 9.                                             |
//+------------------------------------------------------------------+
bool CSignalFractals::IsPattern_9(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && FractalLow(X()) != 0.0)
   {  if(Is_Fibo_Level(X()))
      {  return(true);
      }
   }
   else if(T == POSITION_TYPE_SELL && FractalHigh(X()) != 0.0)
   {  if(Is_Fibo_Level(X()))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
