//+------------------------------------------------------------------+
//|                                               Baseline Model.mq5 |
//|                                               Gamuchirai Ndawana |
//|                    https://www.mql5.com/en/users/gamuchiraindawa |
//+------------------------------------------------------------------+
#property copyright "Gamuchirai Ndawana"
#property link      "https://www.mql5.com/en/users/gamuchiraindawa"
#property version   "1.00"

//+------------------------------------------------------------------+
//| System Constants                                                 |
//+------------------------------------------------------------------+
#define SYMBOL_ONE   "XAGUSD"                                                       //--- Our primary   symbol, the price of Silver in USD
#define SYMBOL_TWO   "XAGEUR"                                                       //--- Our secondary symbol, the price of Silver in EUR
#define SYMBOL_THREE "EURUSD"                                                       //--- Our EURUSD exchange rate.
#define FETCH        24                                                             //--- How many bars of data should we fetch?
#define TF_1         PERIOD_H1                                                      //--- Our intended time frame
#define VOLUME       SymbolInfoDouble(SYMBOL_ONE,SYMBOL_VOLUME_MIN) * 10            //--- Our trading volume
#define XAGUSD_MA_PERIOD 8

//+------------------------------------------------------------------+
//| System resources                                                 |
//+------------------------------------------------------------------+
#resource "\\Files\\XAGUSD State Model.onnx" as  uchar xagusd_onnx_buffer[]
#resource "\\Files\\XAGEUR State Model.onnx" as  uchar xageur_onnx_buffer[]
#resource "\\Files\\EURUSD State Model.onnx" as  uchar eurusd_onnx_buffer[]

//+------------------------------------------------------------------+
//| Global variables                                                 |
//+------------------------------------------------------------------+
vector  eurusd,xagusd,xageur;
double  eurusd_growth,xagusd_growth,xageur_growth,bid,ask;
double  sl_width = 3e2 * _Point;
int     xagusd_f_ma_handler,xagusd_s_ma_handler;
double  xagusd_f[],xagusd_s[];
vectorf model_output = vectorf::Zeros(1);
long    onnx_model;
vectorf xageur_model_output = vectorf::Zeros(1);
long    xageur_onnx_model;
vectorf eurusd_model_output = vectorf::Zeros(1);
long    eurusd_onnx_model;
//+------------------------------------------------------------------+
//| Libraries                                                        |
//+------------------------------------------------------------------+
#include  <Trade\Trade.mqh>
CTrade    Trade;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
//--- Setup our technical indicators
   if(!setup())
     {
      return(INIT_FAILED);
     }
//---
   return(INIT_SUCCEEDED);
  }

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
//---
   OnnxRelease(onnx_model);
   OnnxRelease(xageur_onnx_model);
   OnnxRelease(eurusd_onnx_model);
   IndicatorRelease(xagusd_f_ma_handler);
   IndicatorRelease(xagusd_s_ma_handler);
   Print("System deinitialized");
  }

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
//--- New prices have been quoted
   new_quotes_received();
  }
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Custom functions                                                 |
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Updates system variables accordingly                             |
//+------------------------------------------------------------------+
void new_quotes_received(void)
  {
   static datetime time_stamp;
   datetime time = iTime(SYMBOL_ONE,TF_1,0);

   if(time_stamp != time)
     {
      time_stamp = time;
      update();
     }
  }

//+------------------------------------------------------------------+
//| Setup our technical indicators and select the symbols we need    |
//+------------------------------------------------------------------+
bool setup(void)
  {
//--- Select the symbols we need
   SymbolSelect(SYMBOL_ONE,true);
   SymbolSelect(SYMBOL_TWO,true);
   SymbolSelect(SYMBOL_THREE,true);

//--- Setup the moving averages
   xagusd_f_ma_handler = iMA(SYMBOL_ONE,TF_1,XAGUSD_MA_PERIOD,0,MODE_SMA,PRICE_OPEN);
   xagusd_s_ma_handler = iMA(SYMBOL_ONE,TF_1,XAGUSD_MA_PERIOD,0,MODE_SMA,PRICE_CLOSE);

   if((xagusd_f_ma_handler == INVALID_HANDLE) || (xagusd_s_ma_handler == INVALID_HANDLE))
     {
      Comment("Failed to load our technical indicators correctly. ", GetLastError());
      return(false);
     }

//--- Setup our statistical models
   onnx_model        = OnnxCreateFromBuffer(xagusd_onnx_buffer,ONNX_DEFAULT);
   xageur_onnx_model = OnnxCreateFromBuffer(xageur_onnx_buffer,ONNX_DEFAULT);
   eurusd_onnx_model = OnnxCreateFromBuffer(eurusd_onnx_buffer,ONNX_DEFAULT);

   if(onnx_model == INVALID_HANDLE)
     {
      Comment("Failed to create our XAGUSD ONNX model correctly. ",GetLastError());
      return(false);
     }

   if(xageur_onnx_model == INVALID_HANDLE)
     {
      Comment("Failed to create our XAGEUR ONNX model correctly. ",GetLastError());
      return(false);
     }

   if(eurusd_onnx_model == INVALID_HANDLE)
     {
      Comment("Failed to create our EURUSD ONNX model correctly. ",GetLastError());
      return(false);
     }

   ulong input_shape[] = {1,4};
   ulong output_shape[] = {1,1};

   if(!(OnnxSetInputShape(onnx_model,0,input_shape)))
     {
      Comment("Failed to specify XAGUSD model input shape. ",GetLastError());
      return(false);
     }


   if(!(OnnxSetInputShape(xageur_onnx_model,0,input_shape)))
     {
      Comment("Failed to specify XAGEUR model input shape. ",GetLastError());
      return(false);
     }


   if(!(OnnxSetInputShape(eurusd_onnx_model,0,input_shape)))
     {
      Comment("Failed to specify EURUSD model input shape. ",GetLastError());
      return(false);
     }

   if(!(OnnxSetOutputShape(onnx_model,0,output_shape)))
     {
      Comment("Failed to specify XAGUSD model output shape. ",GetLastError());
      return(false);
     }

   if(!(OnnxSetOutputShape(xageur_onnx_model,0,output_shape)))
     {
      Comment("Failed to specify XAGEUR model output shape. ",GetLastError());
      return(false);
     }

   if(!(OnnxSetOutputShape(eurusd_onnx_model,0,output_shape)))
     {
      Comment("Failed to specify EURUSD model output shape. ",GetLastError());
      return(false);
     }

   Print("System initialized succefully");

//--- If we have gotten this far, everything went fine.
   return(true);
  }

//+------------------------------------------------------------------+
//| Update our system setup                                          |
//+------------------------------------------------------------------+
void update(void)
  {
//--- Fetch updated prices
   xagusd.CopyRates(SYMBOL_ONE,TF_1,COPY_RATES_CLOSE,1,FETCH);
   xageur.CopyRates(SYMBOL_TWO,TF_1,COPY_RATES_CLOSE,1,FETCH);
   eurusd.CopyRates(SYMBOL_THREE,TF_1,COPY_RATES_CLOSE,1,FETCH);

//--- Calculate the growth in market prices
   eurusd_growth = eurusd[0] / eurusd[FETCH - 1];
   xageur_growth = xageur[0] / xageur[FETCH - 1];
   xagusd_growth = xagusd[0] / xagusd[FETCH - 1];

//--- Calculate the change in ratio of market prices
   CopyBuffer(xagusd_f_ma_handler,0,0,1,xagusd_f);
   CopyBuffer(xagusd_s_ma_handler,0,0,1,xagusd_s);

//--- Update system variables
   SymbolSelect(SYMBOL_ONE,true);
   bid = SymbolInfoDouble(SYMBOL_ONE,SYMBOL_BID);
   ask = SymbolInfoDouble(SYMBOL_ONE,SYMBOL_ASK);

//--- Check if we need to setup a new position
   if(PositionsTotal() == 0)
      find_setup();

//--- Check if we need to manage our positions
   if(PositionsTotal() > 0)
      manage_setup();

//--- Give feedback on the market growth
   Comment("EURUSD Growth: ",eurusd_growth,"\nXAGEUR Growth: ",xageur_growth,"\nXAGUSD Grwoth: ",xagusd_growth);
  }

//+------------------------------------------------------------------+
//| Fetch a prediction from our model                                |
//+------------------------------------------------------------------+
void model_predict(void)
  {
   vectorf model_inputs =  { (float) iOpen(SYMBOL_ONE,TF_1,1), (float) iHigh(SYMBOL_ONE,TF_1,1),(float) iLow(SYMBOL_ONE,TF_1,1),(float) iClose(SYMBOL_ONE,TF_1,1)};
   vectorf xageur_model_inputs =  { (float) iOpen(SYMBOL_TWO,TF_1,1),  (float) iHigh(SYMBOL_TWO,TF_1,1),  (float) iLow(SYMBOL_TWO,TF_1,1), (float) iClose(SYMBOL_TWO,TF_1,1)};
   vectorf eurusd_model_inputs =  { (float) iOpen(SYMBOL_THREE,TF_1,1), (float) iHigh(SYMBOL_THREE,TF_1,1),(float) iLow(SYMBOL_THREE,TF_1,1), (float) iClose(SYMBOL_THREE,TF_1,1)};
   OnnxRun(onnx_model,ONNX_DATA_TYPE_FLOAT,model_inputs,model_output);
   OnnxRun(onnx_model,ONNX_DATA_TYPE_FLOAT,xageur_model_inputs,xageur_model_output);
   OnnxRun(onnx_model,ONNX_DATA_TYPE_FLOAT,eurusd_model_inputs,eurusd_model_output);
  }

//+------------------------------------------------------------------+
//| Find setup                                                       |
//+------------------------------------------------------------------+
void find_setup(void)
  {
   model_predict();

//--- Check if the current market setup matches our expectations for selling
   if((eurusd_growth < 1) && (xageur_growth > 1) && (xagusd_growth <  1))
     {
      if(xagusd_s[0] < xagusd_f[0])
        {
         if((eurusd_model_output[0] < 0) && (xageur_model_output[0] > 0) && (model_output[0] < 0))
           {
            //--- If all our systems align, we have a high probability trade setup
            Trade.Sell(VOLUME * 2,SYMBOL_ONE,bid,(ask + sl_width),(ask - sl_width),"");
           }
         //--- Otherwise, we should trade conservatively
         Trade.Sell(VOLUME,SYMBOL_ONE,bid,(ask + sl_width),(ask - sl_width),"");
        }
     }

//--- Check if the current market setup matches our expectations for buying
   if((eurusd_growth > 1) && (xageur_growth < 1) && (xagusd_growth > 1))
     {
      if(xagusd_s[0] > xagusd_f[0])
        {
         if((eurusd_model_output[0] > 0) && (xageur_model_output[0] < 0) && (model_output[0] > 0))
           {
            Trade.Buy(VOLUME * 2,SYMBOL_ONE,ask,(bid - sl_width),(bid + sl_width),"");
           }

         Trade.Buy(VOLUME,SYMBOL_ONE,ask,(bid - sl_width),(bid + sl_width),"");
        }
     }
  }

//+------------------------------------------------------------------+
//| Manage setup                                                     |
//+------------------------------------------------------------------+
void manage_setup(void)
  {
//--- Select our open position
   if(PositionSelect(SYMBOL_ONE))
     {
      double current_sl = PositionGetDouble(POSITION_SL);
      double current_tp = PositionGetDouble(POSITION_TP);

      //--- Buy setup
      if(current_sl < current_tp)
        {
         if((bid - sl_width) > current_sl)
            Trade.PositionModify(SYMBOL_ONE,(bid - sl_width),(bid + sl_width));
        }

      //--- Sell setup
      if(current_sl > current_tp)
        {
         if((ask + sl_width) < current_sl)
            Trade.PositionModify(SYMBOL_ONE,(ask + sl_width),(ask - sl_width));
        }
     }
  }

//+------------------------------------------------------------------+
//| Undefine system constants                                        |
//+------------------------------------------------------------------+
#undef TF_1
#undef SYMBOL_ONE
#undef SYMBOL_TWO
#undef SYMBOL_THREE
#undef VOLUME
#undef FETCH
#undef XAGUSD_MA_PERIOD
//+------------------------------------------------------------------+
