//+--------------------------------------------------------------+
//| Simple Static Template MT5.mq5                               |
//| Copyright 2026, Centropolis Corp.                            |
//| https://www.mql5.com/en/users/w.hudson                       |
//+--------------------------------------------------------------+
#property copyright "Copyright 2026, Centropolis Corp."
#property link      "https://www.mql5.com/en/users/w.hudson"
#property version   "1.00"

//+--------------------------------------------------------------+
//| Include necessary libraries for trading operations           |
//+--------------------------------------------------------------+
#include <Trade\PositionInfo.mqh>
#include <Trade\Trade.mqh>

//+--------------------------------------------------------------+
//| Main robot constants                                         |
//+--------------------------------------------------------------+
#define EANICK "Simple Template"        // Robot name displayed in the graphical user interface
#define GLOBALVARPREFIX "HelpVar"       // Prefix for global variables, prevents conflicts when running multiple robot instances
#define EABARS 990                      // Maximum number of historical bars for analysis in the internal virtual chart
#define MINUTES_TRY 2                   // Maximum time in minutes for attempting to open/close positions from the start of a new bar

//+--------------------------------------------------------------+
//| Robot input parameters                                       |
//+--------------------------------------------------------------+
//+----------------- "Volumes" group -----------------------+
input group "Volumes";
sinput bool bInitLotControl=false;      // Automatic lot calculation
sinput double LotE=0.01;                // Fixed lot size
sinput double DepositE=1000.0;          // Deposit for lot calculation

//+----------------- "Waiting Out Losses Mode" group -----------------------+
input group "Waiting Out Losses Mode";
input bool bInitSitE=false;             // Loss linearization mode (sit out losses)
input int MinutesHoldE=144000;          // Position holding time in minutes

//+----------------- "Martingale" group -----------------------+
input group "Martingale";
input bool bInitMartinE=false;          // Martingale system
input int MaxMartinLotMultiplierStepsE=5; // Maximum Martingale steps

//+----------------- "Repurchase" group -----------------------+
input group "Repurchase";
input bool bInitRepurchaseE=false;      // Repurchase system
input double MinPercenPriceStepE=0.25;  // First price step for repurchase
input double NextStepMultiplierE=1.5;   // Repurchase step multiplier

//+----------------- "Other" group -----------------------+
input group "Other";
sinput int MagicE = 15464;              // Magic number
input int SLE=0;                        // Stop Loss in points
input int TPE=0;                        // Take Profit in points
input double ComissionPerLotE=0.0;      // Commission per lot
input const int SpreadEE=5000;          // Maximum spread in points
sinput string CommentI="Simple Template"; // Comment for orders

//+----------------- "Fighting The Obsolescence Of Settings" group -----------------------+
input group "Fighting The Obsolescence Of Settings";
sinput uint DaysToFuture=365;           // Number of days into the future

//+----------------- "Direction Variables" group -----------------------+
input group "Direction Variables";
input bool TBE=true;                    // Allow opening BUY positions
input bool TSE=true;                    // Allow opening SELL positions

//+----------------- "Strategy Variables" group -----------------------+
input group "Strategy Variables";
input int BarsChainForOpenE = 5;        // Concurrent bars for opening a position
input int BarsChainForCloseE = 5;       // Concurrent bars for closing a position

//+--------------------------------------------------------------+
//| Global variables                                             |
//+--------------------------------------------------------------+
int HistoryDaysLoadI=365;               // Number of history days to load

//+--------------------------------------------------------------+
//| Check if magic number belongs to our robot                   |
//| Parameters:                                                  |
//| magic1 - magic number to check                               |
//| Returns: true if magic belongs to our robot                  |
//+--------------------------------------------------------------+
bool bOurMagic(long magic1)
   {
   if ( (int)magic1 == BotO.MagicF ) return true;
   return false;                            // return false if magic does not belong to our robot
   }  
   
//+--------------------------------------------------------------+
//| Check possibility of additional position opening             |
//| Used for the Repurchase system                               |
//| Parameters:                                                  |
//| CurrentPrice - current price                                 |
//| bDirection - direction (true=BUY, false=SELL)                |
//| Returns: true if additional position can be opened           |
//+--------------------------------------------------------------+
bool AddtionalOpenPrice(double CurrentPrice,bool bDirection)
   {
   bool ord;                            // flag for successful position selection for analysis
   // first determine if there are open positions for the same instrument, if so determine the extreme open price
   double BorderPriceBUY = -1.0;       // exact (lowest) open price of BUY positions, -1.0 means no positions
   double BorderPriceSELL = -1.0;      // exact (highest) open price of SELL positions, -1.0 means no positions
   double PriceSymbolASK = SymbolInfoDouble(_Symbol,SYMBOL_ASK);  // current ASK price for buying
   double PriceSymbolBID = SymbolInfoDouble(_Symbol,SYMBOL_BID);  // current BID price for selling
   
   int PositionsBuy=0;                  // counter of open BUY positions for current symbol
   int PositionsSell=0;                 // counter of open SELL positions for current symbol
   
   // determine the extreme price to count from
   for ( int i=0; i<PositionsTotal(); i++ )
      {
      ulong ticket=PositionGetTicket(i);
      ord=PositionSelectByTicket(ticket); 
      long cMagic = PositionGetInteger(POSITION_MAGIC);          
      if ( ord && cMagic == MagicE )
         {
         string SymbolNotFiltered = PositionGetString(POSITION_SYMBOL);  
         if (_Symbol == SymbolNotFiltered)
            {
            if (PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
               {
               double PriceOpenPosition = PositionGetDouble(POSITION_PRICE_OPEN);
               if (BorderPriceBUY == -1.0 || PriceOpenPosition < BorderPriceBUY)
                  {
                  BorderPriceBUY = PriceOpenPosition;
                  PositionsBuy++;
                  }
               }                  
            else if (PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL)
               {
               double PriceOpenPosition = PositionGetDouble(POSITION_PRICE_OPEN);
               if (BorderPriceSELL == -1.0 || PriceOpenPosition > BorderPriceSELL)
                  {
                  BorderPriceSELL = PriceOpenPosition;
                  PositionsSell++;
                  }                  
               }               
            }
         }
      }
   
   // check for netting account mode
   if (AccountInfoInteger(ACCOUNT_MARGIN_MODE) == ACCOUNT_MARGIN_MODE_RETAIL_NETTING && (PositionsBuy > 0 || PositionsSell > 0)) return false;
   
   // after determining the extreme price, determine if a position can be opened right now
   if ( bDirection && (BorderPriceBUY == -1.0 || (BorderPriceBUY > PriceSymbolASK && PositionsBuy > 0 && PositionsSell == 0 && BorderPriceBUY-PriceSymbolASK >= MathPow(NextStepMultiplierE,double(PositionsBuy-1))*BorderPriceBUY*MinPercenPriceStepE/100.0 )) )
      {
      return true;                          // return true if condition for additional BUY opening is met
      }
   else if ( !bDirection && (BorderPriceSELL == -1.0 || (BorderPriceSELL < PriceSymbolBID && PositionsSell > 0 && PositionsBuy == 0 && PriceSymbolBID-BorderPriceSELL >= MathPow(NextStepMultiplierE,double(PositionsSell-1))*BorderPriceSELL*MinPercenPriceStepE/100.0 )) )
      {
      return true;                          // return true if condition for additional SELL opening is met
      }
   
   return false;                        // return false if conditions for additional opening are not met
   }

//+--------------------------------------------------------------+
//| Close positions to profit for repurchase system              |
//| Parameters:                                                  |
//| bDirection - direction (true=BUY, false=SELL)                |
//+--------------------------------------------------------------+
void CloseToProfit(bool bDirection)
   {
   bool ord;                            // flag for successful position selection for analysis
   ulong Tickets[];                     // dynamic array of position tickets for closing in profit
   int TicketsTotal=0;                  // total number of positions suitable for closing
   int TicketNumCurrent=0;              // current index when filling tickets array
   ulong ticket;                        // temporary variable to store current position ticket
   long cMagic=0;                       // magic number of current position for check
   
   double SummProfit=0.0;               // total profit of all positions in specified direction including swaps and commissions
   
   // calculating total profit by positions
   for ( int i=0; i<PositionsTotal(); i++ )
      {
      ticket=PositionGetTicket(i);
      ord=PositionSelectByTicket(ticket);
      if (ord && PositionGetString(POSITION_SYMBOL) == _Symbol && PositionGetInteger(POSITION_MAGIC) == MagicE )
         {
         if (bDirection && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
            {
            TicketsTotal=TicketsTotal+1;
            SummProfit+=PositionGetDouble(POSITION_PROFIT)+PositionGetDouble(POSITION_SWAP) - PositionGetDouble(POSITION_VOLUME)*ComissionPerLotE;
            }
         if (!bDirection && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL)
            {
            TicketsTotal=TicketsTotal+1;
            SummProfit+=PositionGetDouble(POSITION_PROFIT)+PositionGetDouble(POSITION_SWAP) - PositionGetDouble(POSITION_VOLUME)*ComissionPerLotE;
            } 
         }
      }
      
   if (SummProfit > 0.0)
      {
      ArrayResize(Tickets,TicketsTotal);
      for ( int i=0; i<PositionsTotal(); i++ )
         {
         ticket=PositionGetTicket(i);
         ord=PositionSelectByTicket(ticket);                  
         if (ord && PositionGetString(POSITION_SYMBOL) == _Symbol && PositionGetInteger(POSITION_MAGIC) == MagicE && TicketNumCurrent < TicketsTotal )
            {
            if ( (bDirection && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY) || (!bDirection && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL) )
               {
               Tickets[TicketNumCurrent]=ticket;
               TicketNumCurrent=TicketNumCurrent+1;
               }
            }
         }      
      }
   if (ArraySize(Tickets) > 0)
      {
      for ( int i=0; i<ArraySize(Tickets); i++ )
         {
         if (Tickets[i] != 0) BotO.m_trade.PositionClose(Tickets[i]);
         }
      }   
   }

//+--------------------------------------------------------------+
//| Max balance control                                          |
//+--------------------------------------------------------------+
double LastMaxBalance;              // maximum found balance for the last year to calculate drawdown
datetime LastTimeChecked;           // time of last max balance check for optimization
datetime LastTimeBorder;            // time of boundary deal, up to which history analysis is already performed

//+--------------------------------------------------------------+
//| Search for new balance highs to control drawdown             |
//| Analyzes deal history and determines maximum balance         |
//| for calculating permissible drawdown                         |
//+--------------------------------------------------------------+
void UpdateMaxBalance()
   {
   double TempSummDelta=0.0;         // accumulated profit delta to restore historical balance at the moment of each deal
   datetime LastTimeBorderTemp=LastTimeBorder; // temporary variable to store time of the newest processed deal in current analysis cycle
   
   // Load deal history for specified period
   HistorySelect(TimeCurrent()-HistoryDaysLoadI*86400,TimeCurrent());
   
   // Iterate through deal history from new to old
   for ( int i=HistoryDealsTotal()-1; i>=0; i-- ) 
      {
      ulong ticket=HistoryDealGetTicket(i);
      
      // Consider only closing deals of our robots
      if ( HistoryDealGetInteger(ticket,DEAL_ENTRY) == DEAL_ENTRY_OUT && bOurMagic(HistoryDealGetInteger(ticket,DEAL_MAGIC)) ) 
         {
         datetime DealTime = datetime(HistoryDealGetInteger(ticket,DEAL_TIME));
         
         // If we haven't considered this deal yet
         if (DealTime >= LastTimeChecked && DealTime >= LastTimeBorder )
            {
            // Subtract deal profit to restore balance at the moment before the deal
            TempSummDelta-=HistoryDealGetDouble(ticket,DEAL_PROFIT);
            double AB = AccountInfoDouble(ACCOUNT_BALANCE);  
            double ConstructedBalance = TempSummDelta + AB;
            
            // If greater balance extremum found
            if (ConstructedBalance > LastMaxBalance)
               {
               LastMaxBalance=ConstructedBalance;
               }
               
            // Protection against abnormal values - allow balance excess max by 20%
            if (AB > 0.0 && ConstructedBalance > AB && (ConstructedBalance - AB) / AB > 0.2 )
               {
               LastMaxBalance = AB * 1.2;  // Limit max excess to 20%
               break;
               }
            if (DealTime > LastTimeBorderTemp) LastTimeBorderTemp=DealTime;
            }
         else
            {
            break;  // Stop analysis, reached already processed deals
            } 
         }
      }
      
   // If max balance not defined, use current balance as starting one
   if (LastMaxBalance == 0) LastMaxBalance=AccountInfoDouble(ACCOUNT_BALANCE);
   
   // Update boundary time of processed deals
   LastTimeBorder=LastTimeBorderTemp;
   }

//+--------------------------------------------------------------+
//| MQL4 compatibility directives                                |
//| Define MQL4 constants to ensure code compatibility           |
//+--------------------------------------------------------------+

// Order and position working modes (MQL4 compatibility constants)
#define MODE_TRADES 0               // mode for working with active positions and orders
#define MODE_HISTORY 1              // mode for working with historical deal info

// Order/position selection methods (MQL4 compatibility constants)
#define SELECT_BY_POS 0             // order/position selection by index in list
#define SELECT_BY_TICKET 1          // order/position selection by unique ticket

// Operation types (MQL4 compatibility constants)
#define OP_BUY DEAL_TYPE_BUY        // buy operation (long position)
#define OP_SELL DEAL_TYPE_SELL      // sell operation (short position)

// Symbol information retrieval modes (MQL4 compatibility constants)
#define MODE_SPREAD 0               // current spread in points
#define MODE_STOPLEVEL 1            // minimum level for stop placement
#define MODE_MINLOT 2               // minimum lot size
#define MODE_MAXLOT 3               // maximum lot size
#define MODE_LOTSTEP 4              // lot size change step
#define MODE_TICKVALUE 5            // tick value in deposit currency
#define MODE_MARGINREQUIRED 6       // required margin for one lot
#define MODE_SWAPLONG 7             // swap size for long positions
#define MODE_SWAPSHORT 8            // swap size for short positions

//+------------------------------------------------------------------+

//+--------------------------------------------------------------+
//| Creation of the single chart object                          |
//| Initializes Chart object with arrays for storing             |
//| historical data (OHLC) of current symbol                     |
//+--------------------------------------------------------------+
void CreateChart()
   {
   Chart::TCN=EABARS;              // set number of bars for analysis
   ChartO = new Chart();           // create new chart object

   ChartO.lastcopied=0;            // reset copied bars counter
   
   // Set size for historical data storage arrays
   ArrayResize(ChartO.CloseI,Chart::TCN+2);    // close prices array
   ArrayResize(ChartO.OpenI,Chart::TCN+2);     // open prices array
   ArrayResize(ChartO.HighI,Chart::TCN+2);     // high prices array
   ArrayResize(ChartO.LowI,Chart::TCN+2);      // low prices array
   ArrayResize(ChartO.TimeI,Chart::TCN+2);     // bar times array

   ChartO.CurrentSymbol = _Symbol;             // set current symbol
   ChartO.Timeframe = PERIOD_CURRENT;          // set current period
   ChartO.GetTimeSeries();                     // load time series
   }

//+--------------------------------------------------------------+
//| Time series correction for tester                            |
//| Fixes issues with time series in MQL5 tester                 |
//| Determines trading sessions for different days of week       |
//+--------------------------------------------------------------+
void CorrectSeries()
   {
   // Minute equivalents of trading session start times for each day of week
   int MONDAY_MinuteEquivalentFrom;        // Monday trading session start time in minutes from 00:00
   int TUESDAY_MinuteEquivalentFrom;       // Tuesday trading session start time in minutes from 00:00
   int WEDNESDAY_MinuteEquivalentFrom;     // Wednesday trading session start time in minutes from 00:00
   int THURSDAY_MinuteEquivalentFrom;      // Thursday trading session start time in minutes from 00:00
   int FRIDAY_MinuteEquivalentFrom;        // Friday trading session start time in minutes from 00:00
   int SATURDAY_MinuteEquivalentFrom;      // Saturday trading session start time in minutes from 00:00
   int SUNDAY_MinuteEquivalentFrom;        // Sunday trading session start time in minutes from 00:00
   
   // Minute equivalents of trading session end times for each day of week
   int MONDAY_MinuteEquivalentTo;          // Monday trading session end time in minutes from 00:00
   int TUESDAY_MinuteEquivalentTo;         // Tuesday trading session end time in minutes from 00:00
   int WEDNESDAY_MinuteEquivalentTo;       // Wednesday trading session end time in minutes from 00:00
   int THURSDAY_MinuteEquivalentTo;        // Thursday trading session end time in minutes from 00:00
   int FRIDAY_MinuteEquivalentTo;          // Friday trading session end time in minutes from 00:00
   int SATURDAY_MinuteEquivalentTo;        // Saturday trading session end time in minutes from 00:00
   int SUNDAY_MinuteEquivalentTo;          // Sunday trading session end time in minutes from 00:00   
   
   // Variables for recording trading session start and end in datetime format
   datetime date_from_MONDAY;      // Monday trading session start time in datetime format
   datetime date_to_MONDAY;        // Monday trading session end time in datetime format
   datetime date_from_TUESDAY;     // Tuesday trading session start time in datetime format
   datetime date_to_TUESDAY;       // Tuesday trading session end time in datetime format
   datetime date_from_WEDNESDAY;   // Wednesday trading session start time in datetime format
   datetime date_to_WEDNESDAY;     // Wednesday trading session end time in datetime format
   datetime date_from_THURSDAY;    // Thursday trading session start time in datetime format
   datetime date_to_THURSDAY;      // Thursday trading session end time in datetime format
   datetime date_from_FRIDAY;      // Friday trading session start time in datetime format
   datetime date_to_FRIDAY;        // Friday trading session end time in datetime format
   datetime date_from_SATURDAY;    // Saturday trading session start time in datetime format
   datetime date_to_SATURDAY;      // Saturday trading session end time in datetime format
   datetime date_from_SUNDAY;      // Sunday trading session start time in datetime format
   datetime date_to_SUNDAY;        // Sunday trading session end time in datetime format
   
   // MqlDateTime structures for working with trading session time components
   MqlDateTime date_from_MONDAY0;     // Monday trading session start time structure
   MqlDateTime date_to_MONDAY0;       // Monday trading session end time structure
   MqlDateTime date_from_TUESDAY0;    // Tuesday trading session start time structure
   MqlDateTime date_to_TUESDAY0;      // Tuesday trading session end time structure
   MqlDateTime date_from_WEDNESDAY0;  // Wednesday trading session start time structure
   MqlDateTime date_to_WEDNESDAY0;    // Wednesday trading session end time structure
   MqlDateTime date_from_THURSDAY0;   // Thursday trading session start time structure
   MqlDateTime date_to_THURSDAY0;     // Thursday trading session end time structure
   MqlDateTime date_from_FRIDAY0;     // Friday trading session start time structure
   MqlDateTime date_to_FRIDAY0;       // Friday trading session end time structure
   MqlDateTime date_from_SATURDAY0;   // Saturday trading session start time structure
   MqlDateTime date_to_SATURDAY0;     // Saturday trading session end time structure
   MqlDateTime date_from_SUNDAY0;     // Sunday trading session start time structure
   MqlDateTime date_to_SUNDAY0;       // Sunday trading session end time structure
   
   // Get trading session information for each day of week
   bool bMONDAY = SymbolInfoSessionTrade(_Symbol,MONDAY,0,date_from_MONDAY,date_to_MONDAY);
   bool bTUESDAY = SymbolInfoSessionTrade(_Symbol,TUESDAY,0,date_from_TUESDAY,date_to_TUESDAY);
   bool bWEDNESDAY = SymbolInfoSessionTrade(_Symbol,WEDNESDAY,0,date_from_WEDNESDAY,date_to_WEDNESDAY);
   bool bTHURSDAY = SymbolInfoSessionTrade(_Symbol,THURSDAY,0,date_from_THURSDAY,date_to_THURSDAY);
   bool bFRIDAY = SymbolInfoSessionTrade(_Symbol,FRIDAY,0,date_from_FRIDAY,date_to_FRIDAY);
   bool bSATURDAY = SymbolInfoSessionTrade(_Symbol,SATURDAY,0,date_from_SATURDAY,date_to_SATURDAY);
   bool bSUNDAY = SymbolInfoSessionTrade(_Symbol,SUNDAY,0,date_from_SUNDAY,date_to_SUNDAY);
      
   // Convert datetime to MqlDateTime structures for convenient work with time components
   TimeToStruct(date_from_MONDAY,date_from_MONDAY0); 
   TimeToStruct(date_from_TUESDAY,date_from_TUESDAY0);
   TimeToStruct(date_from_WEDNESDAY,date_from_WEDNESDAY0);
   TimeToStruct(date_from_THURSDAY,date_from_THURSDAY0);
   TimeToStruct(date_from_FRIDAY,date_from_FRIDAY0);
   TimeToStruct(date_from_SATURDAY,date_from_SATURDAY0);
   TimeToStruct(date_from_SUNDAY,date_from_SUNDAY0);
   TimeToStruct(date_to_MONDAY,date_to_MONDAY0);
   TimeToStruct(date_to_TUESDAY,date_to_TUESDAY0);
   TimeToStruct(date_to_WEDNESDAY,date_to_WEDNESDAY0);
   TimeToStruct(date_to_THURSDAY,date_to_THURSDAY0);
   TimeToStruct(date_to_FRIDAY,date_to_FRIDAY0);
   TimeToStruct(date_to_SATURDAY,date_to_SATURDAY0);
   TimeToStruct(date_to_SUNDAY,date_to_SUNDAY0); 
   
   // Calculate minute equivalents of trading session start times
   MONDAY_MinuteEquivalentFrom = date_from_MONDAY0.hour*60 + date_from_MONDAY0.min;
   TUESDAY_MinuteEquivalentFrom = date_from_TUESDAY0.hour*60 + date_from_TUESDAY0.min;
   WEDNESDAY_MinuteEquivalentFrom = date_from_WEDNESDAY0.hour*60 + date_from_WEDNESDAY0.min;
   THURSDAY_MinuteEquivalentFrom = date_from_THURSDAY0.hour*60 + date_from_THURSDAY0.min;
   FRIDAY_MinuteEquivalentFrom = date_from_FRIDAY0.hour*60 + date_from_FRIDAY0.min;
   SATURDAY_MinuteEquivalentFrom = date_from_SATURDAY0.hour*60 + date_from_SATURDAY0.min;
   SUNDAY_MinuteEquivalentFrom = date_from_SUNDAY0.hour*60 + date_from_SUNDAY0.min;
   
   // Calculate minute equivalents of trading session end times
   MONDAY_MinuteEquivalentTo = date_to_MONDAY0.hour*60 + date_to_MONDAY0.min;
   TUESDAY_MinuteEquivalentTo = date_to_TUESDAY0.hour*60 + date_to_TUESDAY0.min;
   WEDNESDAY_MinuteEquivalentTo = date_to_WEDNESDAY0.hour*60 + date_to_WEDNESDAY0.min;
   THURSDAY_MinuteEquivalentTo = date_to_THURSDAY0.hour*60 + date_to_THURSDAY0.min;
   FRIDAY_MinuteEquivalentTo = date_to_FRIDAY0.hour*60 + date_to_FRIDAY0.min;
   SATURDAY_MinuteEquivalentTo = date_to_SATURDAY0.hour*60 + date_to_SATURDAY0.min;
   SUNDAY_MinuteEquivalentTo = date_to_SUNDAY0.hour*60 + date_to_SUNDAY0.min;      

   // Correction for cases 0:00 - 24:00 and when start time equals end time
   
   // Monday processing
   if (bMONDAY)
      {
      // Case of round-the-clock trading (0:00 - 0:00 next day)
      if (MONDAY_MinuteEquivalentFrom == 0 && MONDAY_MinuteEquivalentTo == 0 && date_from_MONDAY != date_to_MONDAY)
         {
         MONDAY_MinuteEquivalentTo=60*23+59;  // Set end to 23:59
         }
      // Case when start time equals end time (but different days)
      else if (MONDAY_MinuteEquivalentFrom == MONDAY_MinuteEquivalentTo && date_from_MONDAY != date_to_MONDAY)
         {
         if (MONDAY_MinuteEquivalentFrom < 60*23+59)
            {
            MONDAY_MinuteEquivalentTo = MONDAY_MinuteEquivalentFrom+1;  // Add 1 minute
            }
         else MONDAY_MinuteEquivalentTo = 0;  // Move to start of next day
         }
      // Case when start and end are on same day (no trading)
      else if (date_from_MONDAY == date_to_MONDAY)
         {
         MONDAY_MinuteEquivalentFrom = 0;
         MONDAY_MinuteEquivalentTo = 0;
         }
      }
   else
      {
      // If trading session not defined, set zero values
      MONDAY_MinuteEquivalentFrom = 0;
      MONDAY_MinuteEquivalentTo = 0;
      }
      
   // Tuesday processing
   if (bTUESDAY)
      {
      if (TUESDAY_MinuteEquivalentFrom == 0 && TUESDAY_MinuteEquivalentTo == 0 && date_from_TUESDAY != date_to_TUESDAY)
         {
         TUESDAY_MinuteEquivalentTo=60*23+59;
         }
      else if (TUESDAY_MinuteEquivalentFrom == TUESDAY_MinuteEquivalentTo && date_from_TUESDAY != date_to_TUESDAY)
         {
         if (TUESDAY_MinuteEquivalentFrom < 60*23+59)
            {
            TUESDAY_MinuteEquivalentTo = TUESDAY_MinuteEquivalentFrom+1;
            }
         else TUESDAY_MinuteEquivalentTo = 0;
         }
      else if (date_from_TUESDAY == date_to_TUESDAY)
         {
         TUESDAY_MinuteEquivalentFrom = 0;
         TUESDAY_MinuteEquivalentTo = 0;
         }                  
      }
   else
      {
      TUESDAY_MinuteEquivalentFrom = 0;
      TUESDAY_MinuteEquivalentTo = 0;
      }      
   if (bWEDNESDAY)
      {
      if (WEDNESDAY_MinuteEquivalentFrom == 0 && WEDNESDAY_MinuteEquivalentTo == 0 && date_from_WEDNESDAY != date_to_WEDNESDAY)
         {
         WEDNESDAY_MinuteEquivalentTo=60*23+59;
         }
      else if (WEDNESDAY_MinuteEquivalentFrom == WEDNESDAY_MinuteEquivalentTo && date_from_WEDNESDAY != date_to_WEDNESDAY)
         {
         if (WEDNESDAY_MinuteEquivalentFrom < 60*23+59)
            {
            WEDNESDAY_MinuteEquivalentTo = WEDNESDAY_MinuteEquivalentFrom+1;
            }
         else WEDNESDAY_MinuteEquivalentTo = 0;
         }
      else if (date_from_WEDNESDAY == date_to_WEDNESDAY)
         {
         WEDNESDAY_MinuteEquivalentFrom = 0;
         WEDNESDAY_MinuteEquivalentTo = 0;
         }                  
      }
   else
      {
      WEDNESDAY_MinuteEquivalentFrom = 0;
      WEDNESDAY_MinuteEquivalentTo = 0;
      }               
   if (bTHURSDAY)
      {
      if (THURSDAY_MinuteEquivalentFrom == 0 && THURSDAY_MinuteEquivalentTo == 0 && date_from_THURSDAY != date_to_THURSDAY)
         {
         THURSDAY_MinuteEquivalentTo=60*23+59;
         }
      else if (THURSDAY_MinuteEquivalentFrom == THURSDAY_MinuteEquivalentTo && date_from_THURSDAY != date_to_THURSDAY)
         {
         if (THURSDAY_MinuteEquivalentFrom < 60*23+59)
            {
            THURSDAY_MinuteEquivalentTo = THURSDAY_MinuteEquivalentFrom+1;
            }
         else THURSDAY_MinuteEquivalentTo = 0;
         }
      else if (date_from_THURSDAY == date_to_THURSDAY)
         {
         THURSDAY_MinuteEquivalentFrom = 0;
         THURSDAY_MinuteEquivalentTo = 0;
         }                  
      }
   else
      {
      THURSDAY_MinuteEquivalentFrom = 0;
      THURSDAY_MinuteEquivalentTo = 0;
      }      
   if (bFRIDAY)
      {
      if (FRIDAY_MinuteEquivalentFrom == 0 && FRIDAY_MinuteEquivalentTo == 0 && date_from_FRIDAY != date_to_FRIDAY)
         {
         FRIDAY_MinuteEquivalentTo=60*23+59;
         }
      else if (FRIDAY_MinuteEquivalentFrom == FRIDAY_MinuteEquivalentTo && date_from_FRIDAY != date_to_FRIDAY)
         {
         if (FRIDAY_MinuteEquivalentFrom < 60*23+59)
            {
            FRIDAY_MinuteEquivalentTo = FRIDAY_MinuteEquivalentFrom+1;
            }
         else FRIDAY_MinuteEquivalentTo = 0;
         }
      else if (date_from_FRIDAY == date_to_FRIDAY)
         {
         FRIDAY_MinuteEquivalentFrom = 0;
         FRIDAY_MinuteEquivalentTo = 0;
         }                  
      }
   else
      {
      FRIDAY_MinuteEquivalentFrom = 0;
      FRIDAY_MinuteEquivalentTo = 0;
      }      
   if (bSATURDAY)
      {
      if (SATURDAY_MinuteEquivalentFrom == 0 && SATURDAY_MinuteEquivalentTo == 0 && date_from_SATURDAY != date_to_SATURDAY)
         {
         SATURDAY_MinuteEquivalentTo=60*23+59;
         }
      else if (SATURDAY_MinuteEquivalentFrom == SATURDAY_MinuteEquivalentTo && date_from_SATURDAY != date_to_SATURDAY)
         {
         if (SATURDAY_MinuteEquivalentFrom < 60*23+59)
            {
            SATURDAY_MinuteEquivalentTo = SATURDAY_MinuteEquivalentFrom+1;
            }
         else SATURDAY_MinuteEquivalentTo = 0;
         }
      else if (date_from_SATURDAY == date_to_SATURDAY)
         {
         SATURDAY_MinuteEquivalentFrom = 0;
         SATURDAY_MinuteEquivalentTo = 0;
         }                  
      }
   else
      {
      SATURDAY_MinuteEquivalentFrom = 0;
      SATURDAY_MinuteEquivalentTo = 0;
      }      
   if (bSUNDAY)
      {
      if (SUNDAY_MinuteEquivalentFrom == 0 && SUNDAY_MinuteEquivalentTo == 0 && date_from_SUNDAY != date_to_SUNDAY)
         {
         SUNDAY_MinuteEquivalentTo=60*23+59;
         }
      else if (SUNDAY_MinuteEquivalentFrom == SUNDAY_MinuteEquivalentTo && date_from_SUNDAY != date_to_SUNDAY)
         {
         if (SUNDAY_MinuteEquivalentFrom < 60*23+59)
            {
            SUNDAY_MinuteEquivalentTo = SUNDAY_MinuteEquivalentFrom+1;
            }
         else SUNDAY_MinuteEquivalentTo = 0;
         }
      else if (date_from_SUNDAY == date_to_SUNDAY)
         {
         SUNDAY_MinuteEquivalentFrom = 0;
         SUNDAY_MinuteEquivalentTo = 0;
         }                  
      }     
   else
      {
      SUNDAY_MinuteEquivalentFrom = 0;
      SUNDAY_MinuteEquivalentTo = 0;
      }
      
   //
   ChartO.MONDAY_MinuteEquivalentFrom = MONDAY_MinuteEquivalentFrom;
   ChartO.MONDAY_MinuteEquivalentTo = MONDAY_MinuteEquivalentTo;
   ChartO.TUESDAY_MinuteEquivalentFrom = TUESDAY_MinuteEquivalentFrom;
   ChartO.TUESDAY_MinuteEquivalentTo = TUESDAY_MinuteEquivalentTo;
   ChartO.WEDNESDAY_MinuteEquivalentFrom = WEDNESDAY_MinuteEquivalentFrom;
   ChartO.WEDNESDAY_MinuteEquivalentTo = WEDNESDAY_MinuteEquivalentTo;
   ChartO.THURSDAY_MinuteEquivalentFrom = THURSDAY_MinuteEquivalentFrom;
   ChartO.THURSDAY_MinuteEquivalentTo = THURSDAY_MinuteEquivalentTo;
   ChartO.FRIDAY_MinuteEquivalentFrom = FRIDAY_MinuteEquivalentFrom;
   ChartO.FRIDAY_MinuteEquivalentTo = FRIDAY_MinuteEquivalentTo;
   ChartO.SATURDAY_MinuteEquivalentFrom = SATURDAY_MinuteEquivalentFrom;
   ChartO.SATURDAY_MinuteEquivalentTo = SATURDAY_MinuteEquivalentTo;
   ChartO.SUNDAY_MinuteEquivalentFrom = SUNDAY_MinuteEquivalentFrom;
   ChartO.SUNDAY_MinuteEquivalentTo = SUNDAY_MinuteEquivalentTo;    
   }

//+------------------------------------------------------------------+

//+--------------------------------------------------------------+
//| Create virtual robot instance                                |
//| Initializes the single BotInstance object to manage          |
//| the expert's trading logic                                   |
//+--------------------------------------------------------------+
void CreateInstance()
   {
   BotO = new BotInstance();           // Create a new trading robot instance
   }

//+--------------------------------------------------------------+
//| Virtual robot tick processing                                |
//| Main trading logic function, called on every tick            |
//| Detects new bars and triggers trading algorithms             |
//+--------------------------------------------------------------+
void BotTick()
   {
   // Check data readiness and trading session
   if ( ChartO.lastcopied >= Chart::TCN+1 && ChartO.ChartPoint > 0.0 && BotO.bInTradeSession() )
      {
      BotO.bNewBarV=BotO.bNewBar();    // Determine if a new bar has appeared
      } 
   else BotO.bNewBarV=false;           // If data is not ready, there is no new bar

   datetime tc=TimeCurrent();            // Current server time to check conditions for trading logic execution
   
   // Check conditions for trading logic execution
   // Either a new bar (START mode), or 20 seconds have passed (LATE mode)
   if ( (BotO.NewBarType == "START" && BotO.bNewBarV) || (BotO.NewBarType == "LATE" && tc - BotO.PreviousLateTick >= 20))
      {
      BotO.PreviousLateTick=tc;        // Update last tick time
      
      // Recalculate trading parameters only if time has changed
      if (BotO.PreviousTimeCalc != ChartO.TimeI[1])
         {
         BotO.CalculateForTrade();     // Calculate trading signals and parameters
         BotO.PreviousTimeCalc=ChartO.TimeI[1];  // Store calculation time
         } 
         
      BotO.bOpened=BotO.Opened();    // check for open positions     
                
      if (!BotO.bOpened) BotO.InstanceTick(true);  // first processing for closing positions
      BotO.InstanceTick(false);      // second processing for opening positions
      }
   }

//+--------------------------------------------------------------+
//| Class for working with chart data                            |
//| Manages time series, trading sessions, and ticks             |
//| Provides access to historical OHLC data                      |
//+--------------------------------------------------------------+
class Chart
   {
   public:
   // Historical data arrays
   datetime TimeI[];                   // bar time array
   double CloseI[];                    // close prices array
   double OpenI[];                     // open prices array
   double HighI[];                     // high prices array
   double LowI[];                      // low prices array
   
   // Basic chart parameters
   string BasicName;                   // basic name from substring
   string BasicSymbol;                 // basic symbol from substring
   string BasicPeriod;                 // basic period
   double ChartPoint;                  // current chart point size
   double ChartAsk;                    // chart Ask price
   double ChartBid;                    // chart Bid price
  
   datetime tTimeI[];                  // auxiliary array to monitor new bar appearance
   
   static int TCN;                     // total number of bars for analysis
   
   // Symbol and period parameters
   string CurrentSymbol;               // adjusted symbol
   ENUM_TIMEFRAMES Timeframe;          // chart timeframe
   int copied;                         // amount of copied data
   int lastcopied;                     // last amount of copied data
   datetime LastCloseTime;             // last bar time
   MqlTick LastTick;                   // last tick
   
   // Minute equivalents of trading sessions for each day of the week
   int MONDAY_MinuteEquivalentFrom;    // Monday trading session start time (in minutes)
   int TUESDAY_MinuteEquivalentFrom;   // Tuesday trading session start time (in minutes)
   int WEDNESDAY_MinuteEquivalentFrom; // Wednesday trading session start time (in minutes)
   int THURSDAY_MinuteEquivalentFrom;  // Thursday trading session start time (in minutes)
   int FRIDAY_MinuteEquivalentFrom;    // Friday trading session start time (in minutes)
   int SATURDAY_MinuteEquivalentFrom;  // Saturday trading session start time (in minutes)
   int SUNDAY_MinuteEquivalentFrom;    // Sunday trading session start time (in minutes)
   
   int MONDAY_MinuteEquivalentTo;      // Monday trading session end time (in minutes)
   int TUESDAY_MinuteEquivalentTo;     // Tuesday trading session end time (in minutes)
   int WEDNESDAY_MinuteEquivalentTo;   // Wednesday trading session end time (in minutes)
   int THURSDAY_MinuteEquivalentTo;    // Thursday trading session end time (in minutes)
   int FRIDAY_MinuteEquivalentTo;      // Friday trading session end time (in minutes)
   int SATURDAY_MinuteEquivalentTo;    // Saturday trading session end time (in minutes)
   int SUNDAY_MinuteEquivalentTo;      // Sunday trading session end time (in minutes)
   
   //+--------------------------------------------------------------+
   //| Get time series and setup trading sessions                   |
   //| Loads historical data and determines trading session times   |
   //+--------------------------------------------------------------+
   void GetTimeSeries()
      {
      // Variables for recording trading session start and end times
      datetime date_from_reserve=0;      // reserve session start time
      datetime date_to_reserve=0;        // reserve session end time
      
      datetime date_from_MONDAY;         // Monday session start time
      datetime date_to_MONDAY;           // Monday session end time
      datetime date_from_TUESDAY;        // Tuesday session start time
      datetime date_to_TUESDAY;          // Tuesday session end time
      datetime date_from_WEDNESDAY;      // Wednesday session start time
      datetime date_to_WEDNESDAY;        // Wednesday session end time
      datetime date_from_THURSDAY;       // Thursday session start time
      datetime date_to_THURSDAY;         // Thursday session end time
      datetime date_from_FRIDAY;         // Friday session start time
      datetime date_to_FRIDAY;           // Friday session end time
      datetime date_from_SATURDAY;       // Saturday session start time
      datetime date_to_SATURDAY;         // Saturday session end time
      datetime date_from_SUNDAY;         // Sunday session start time
      datetime date_to_SUNDAY;           // Sunday session end time
      
      // MqlDateTime structures for working with time components
      MqlDateTime date_from_MONDAY0;     // Monday session start time structure
      MqlDateTime date_to_MONDAY0;       // Monday session end time structure
      MqlDateTime date_from_TUESDAY0;    // Tuesday session start time structure
      MqlDateTime date_to_TUESDAY0;      // Tuesday session end time structure
      MqlDateTime date_from_WEDNESDAY0;  // Wednesday session start time structure
      MqlDateTime date_to_WEDNESDAY0;    // Wednesday session end time structure
      MqlDateTime date_from_THURSDAY0;   // Thursday session start time structure
      MqlDateTime date_to_THURSDAY0;     // Thursday session end time structure
      MqlDateTime date_from_FRIDAY0;     // Friday session start time structure
      MqlDateTime date_to_FRIDAY0;       // Friday session end time structure
      MqlDateTime date_from_SATURDAY0;   // Saturday session start time structure
      MqlDateTime date_to_SATURDAY0;     // Saturday session end time structure
      MqlDateTime date_from_SUNDAY0;     // Sunday session start time structure
      MqlDateTime date_to_SUNDAY0;       // Sunday session end time structure
      
      // Get trading session information for each day of the week
      bool bMONDAY = SymbolInfoSessionTrade(CurrentSymbol,MONDAY,0,date_from_MONDAY,date_to_MONDAY);
      bool bTUESDAY = SymbolInfoSessionTrade(CurrentSymbol,TUESDAY,0,date_from_TUESDAY,date_to_TUESDAY);
      bool bWEDNESDAY = SymbolInfoSessionTrade(CurrentSymbol,WEDNESDAY,0,date_from_WEDNESDAY,date_to_WEDNESDAY);
      bool bTHURSDAY = SymbolInfoSessionTrade(CurrentSymbol,THURSDAY,0,date_from_THURSDAY,date_to_THURSDAY);
      bool bFRIDAY = SymbolInfoSessionTrade(CurrentSymbol,FRIDAY,0,date_from_FRIDAY,date_to_FRIDAY);
      bool bSATURDAY = SymbolInfoSessionTrade(CurrentSymbol,SATURDAY,0,date_from_SATURDAY,date_to_SATURDAY);
      bool bSUNDAY = SymbolInfoSessionTrade(CurrentSymbol,SUNDAY,0,date_from_SUNDAY,date_to_SUNDAY);
         
      TimeToStruct(date_from_MONDAY,date_from_MONDAY0); 
      TimeToStruct(date_from_TUESDAY,date_from_TUESDAY0);
      TimeToStruct(date_from_WEDNESDAY,date_from_WEDNESDAY0);
      TimeToStruct(date_from_THURSDAY,date_from_THURSDAY0);
      TimeToStruct(date_from_FRIDAY,date_from_FRIDAY0);
      TimeToStruct(date_from_SATURDAY,date_from_SATURDAY0);
      TimeToStruct(date_from_SUNDAY,date_from_SUNDAY0);
      TimeToStruct(date_to_MONDAY,date_to_MONDAY0);
      TimeToStruct(date_to_TUESDAY,date_to_TUESDAY0);
      TimeToStruct(date_to_WEDNESDAY,date_to_WEDNESDAY0);
      TimeToStruct(date_to_THURSDAY,date_to_THURSDAY0);
      TimeToStruct(date_to_FRIDAY,date_to_FRIDAY0);
      TimeToStruct(date_to_SATURDAY,date_to_SATURDAY0);
      TimeToStruct(date_to_SUNDAY,date_to_SUNDAY0);   
       
      MONDAY_MinuteEquivalentFrom = date_from_MONDAY0.hour*60 + date_from_MONDAY0.min;
      TUESDAY_MinuteEquivalentFrom = date_from_TUESDAY0.hour*60 + date_from_TUESDAY0.min;
      WEDNESDAY_MinuteEquivalentFrom = date_from_WEDNESDAY0.hour*60 + date_from_WEDNESDAY0.min;
      THURSDAY_MinuteEquivalentFrom = date_from_THURSDAY0.hour*60 + date_from_THURSDAY0.min;
      FRIDAY_MinuteEquivalentFrom = date_from_FRIDAY0.hour*60 + date_from_FRIDAY0.min;
      SATURDAY_MinuteEquivalentFrom = date_from_SATURDAY0.hour*60 + date_from_SATURDAY0.min;
      SUNDAY_MinuteEquivalentFrom = date_from_SUNDAY0.hour*60 + date_from_SUNDAY0.min;
      
      MONDAY_MinuteEquivalentTo = date_to_MONDAY0.hour*60 + date_to_MONDAY0.min;
      TUESDAY_MinuteEquivalentTo = date_to_TUESDAY0.hour*60 + date_to_TUESDAY0.min;
      WEDNESDAY_MinuteEquivalentTo = date_to_WEDNESDAY0.hour*60 + date_to_WEDNESDAY0.min;
      THURSDAY_MinuteEquivalentTo = date_to_THURSDAY0.hour*60 + date_to_THURSDAY0.min;
      FRIDAY_MinuteEquivalentTo = date_to_FRIDAY0.hour*60 + date_to_FRIDAY0.min;
      SATURDAY_MinuteEquivalentTo = date_to_SATURDAY0.hour*60 + date_to_SATURDAY0.min;
      SUNDAY_MinuteEquivalentTo = date_to_SUNDAY0.hour*60 + date_to_SUNDAY0.min;
      
      // Stub for 0:00 - 24:00 and START = END cases
      if (bMONDAY)
         {
         if (MONDAY_MinuteEquivalentFrom == 0 && MONDAY_MinuteEquivalentTo == 0 && date_from_MONDAY != date_to_MONDAY)
            {
            MONDAY_MinuteEquivalentTo=60*23+59;
            }
         else if (MONDAY_MinuteEquivalentFrom == MONDAY_MinuteEquivalentTo && date_from_MONDAY != date_to_MONDAY)
            {
            if (MONDAY_MinuteEquivalentFrom < 60*23+59)
               {
               MONDAY_MinuteEquivalentTo = MONDAY_MinuteEquivalentFrom+1;
               }
            else MONDAY_MinuteEquivalentTo = 0;
            }
         else if (date_from_MONDAY == date_to_MONDAY)
            {
            MONDAY_MinuteEquivalentFrom = 0;
            MONDAY_MinuteEquivalentTo = 0;
            }
         }
      else
         {
         MONDAY_MinuteEquivalentFrom = 0;
         MONDAY_MinuteEquivalentTo = 0;
         }
      if (bTUESDAY)
         {
         if (TUESDAY_MinuteEquivalentFrom == 0 && TUESDAY_MinuteEquivalentTo == 0 && date_from_TUESDAY != date_to_TUESDAY)
            {
            TUESDAY_MinuteEquivalentTo=60*23+59;
            }
         else if (TUESDAY_MinuteEquivalentFrom == TUESDAY_MinuteEquivalentTo && date_from_TUESDAY != date_to_TUESDAY)
            {
            if (TUESDAY_MinuteEquivalentFrom < 60*23+59)
               {
               TUESDAY_MinuteEquivalentTo = TUESDAY_MinuteEquivalentFrom+1;
               }
            else TUESDAY_MinuteEquivalentTo = 0;
            }
         else if (date_from_TUESDAY == date_to_TUESDAY)
            {
            TUESDAY_MinuteEquivalentFrom = 0;
            TUESDAY_MinuteEquivalentTo = 0;
            }                  
         }
      else
         {
         TUESDAY_MinuteEquivalentFrom = 0;
         TUESDAY_MinuteEquivalentTo = 0;
         }      
      if (bWEDNESDAY)
         {
         if (WEDNESDAY_MinuteEquivalentFrom == 0 && WEDNESDAY_MinuteEquivalentTo == 0 && date_from_WEDNESDAY != date_to_WEDNESDAY)
            {
            WEDNESDAY_MinuteEquivalentTo=60*23+59;
            }
         else if (WEDNESDAY_MinuteEquivalentFrom == WEDNESDAY_MinuteEquivalentTo && date_from_WEDNESDAY != date_to_WEDNESDAY)
            {
            if (WEDNESDAY_MinuteEquivalentFrom < 60*23+59)
               {
               WEDNESDAY_MinuteEquivalentTo = WEDNESDAY_MinuteEquivalentFrom+1;
               }
            else WEDNESDAY_MinuteEquivalentTo = 0;
            }
         else if (date_from_WEDNESDAY == date_to_WEDNESDAY)
            {
            WEDNESDAY_MinuteEquivalentFrom = 0;
            WEDNESDAY_MinuteEquivalentTo = 0;
            }                  
         }
      else
         {
         WEDNESDAY_MinuteEquivalentFrom = 0;
         WEDNESDAY_MinuteEquivalentTo = 0;
         }               
      if (bTHURSDAY)
         {
         if (THURSDAY_MinuteEquivalentFrom == 0 && THURSDAY_MinuteEquivalentTo == 0 && date_from_THURSDAY != date_to_THURSDAY)
            {
            THURSDAY_MinuteEquivalentTo=60*23+59;
            }
         else if (THURSDAY_MinuteEquivalentFrom == THURSDAY_MinuteEquivalentTo && date_from_THURSDAY != date_to_THURSDAY)
            {
            if (THURSDAY_MinuteEquivalentFrom < 60*23+59)
               {
               THURSDAY_MinuteEquivalentTo = THURSDAY_MinuteEquivalentFrom+1;
               }
            else THURSDAY_MinuteEquivalentTo = 0;
            }
         else if (date_from_THURSDAY == date_to_THURSDAY)
            {
            THURSDAY_MinuteEquivalentFrom = 0;
            THURSDAY_MinuteEquivalentTo = 0;
            }                  
         }
      else
         {
         THURSDAY_MinuteEquivalentFrom = 0;
         THURSDAY_MinuteEquivalentTo = 0;
         }      
      if (bFRIDAY)
         {
         if (FRIDAY_MinuteEquivalentFrom == 0 && FRIDAY_MinuteEquivalentTo == 0 && date_from_FRIDAY != date_to_FRIDAY)
            {
            FRIDAY_MinuteEquivalentTo=60*23+59;
            }
         else if (FRIDAY_MinuteEquivalentFrom == FRIDAY_MinuteEquivalentTo && date_from_FRIDAY != date_to_FRIDAY)
            {
            if (FRIDAY_MinuteEquivalentFrom < 60*23+59)
               {
               FRIDAY_MinuteEquivalentTo = FRIDAY_MinuteEquivalentFrom+1;
               }
            else FRIDAY_MinuteEquivalentTo = 0;
            }
         else if (date_from_FRIDAY == date_to_FRIDAY)
            {
            FRIDAY_MinuteEquivalentFrom = 0;
            FRIDAY_MinuteEquivalentTo = 0;
            }                  
         }
      else
         {
         FRIDAY_MinuteEquivalentFrom = 0;
         FRIDAY_MinuteEquivalentTo = 0;
         }      
      if (bSATURDAY)
         {
         if (SATURDAY_MinuteEquivalentFrom == 0 && SATURDAY_MinuteEquivalentTo == 0 && date_from_SATURDAY != date_to_SATURDAY)
            {
            SATURDAY_MinuteEquivalentTo=60*23+59;
            }
         else if (SATURDAY_MinuteEquivalentFrom == SATURDAY_MinuteEquivalentTo && date_from_SATURDAY != date_to_SATURDAY)
            {
            if (SATURDAY_MinuteEquivalentFrom < 60*23+59)
               {
                SATURDAY_MinuteEquivalentTo = SATURDAY_MinuteEquivalentFrom+1;
                }
             else SATURDAY_MinuteEquivalentTo = 0;
             }
          else if (date_from_SATURDAY == date_to_SATURDAY)
             {
             SATURDAY_MinuteEquivalentFrom = 0;
             SATURDAY_MinuteEquivalentTo = 0;
             }                  
          }
       else
          {
          SATURDAY_MinuteEquivalentFrom = 0;
          SATURDAY_MinuteEquivalentTo = 0;
          }      
       if (bSUNDAY)
          {
          if (SUNDAY_MinuteEquivalentFrom == 0 && SUNDAY_MinuteEquivalentTo == 0 && date_from_SUNDAY != date_to_SUNDAY)
             {
             SUNDAY_MinuteEquivalentTo=60*23+59;
             }
          else if (SUNDAY_MinuteEquivalentFrom == SUNDAY_MinuteEquivalentTo && date_from_SUNDAY != date_to_SUNDAY)
             {
             if (SUNDAY_MinuteEquivalentFrom < 60*23+59)
                {
                SUNDAY_MinuteEquivalentTo = SUNDAY_MinuteEquivalentFrom+1;
                }
             else SUNDAY_MinuteEquivalentTo = 0;
             }
          else if (date_from_SUNDAY == date_to_SUNDAY)
             {
             SUNDAY_MinuteEquivalentFrom = 0;
             SUNDAY_MinuteEquivalentTo = 0;
             }                  
          }     
       else
          {
          SUNDAY_MinuteEquivalentFrom = 0;
          SUNDAY_MinuteEquivalentTo = 0;
          }
       }
 
    //+--------------------------------------------------------------+
    //| Chart tick processing                                        |
    //| Updates historical data and current quotes                   |
    //| Copies new bars when they appear                             |
    //+--------------------------------------------------------------+
    void ChartTick()
       {
       // Get the last tick for the current symbol
       SymbolInfoTick(CurrentSymbol,LastTick);
       
       // Check for new bar appearance
       ArraySetAsSeries(tTimeI,false);
       copied=CopyTime(CurrentSymbol,Timeframe,0,2,tTimeI);  // Copy time of the last 2 bars
       ArraySetAsSeries(tTimeI,true);
       
       // If a new bar appeared, update all historical data
       if ( copied == 2 && tTimeI[1] > LastCloseTime )
          {
          // Set arrays as ordinary (not as time series)
          ArraySetAsSeries(CloseI,false);                        
          ArraySetAsSeries(OpenI,false);                           
          ArraySetAsSeries(HighI,false);                        
          ArraySetAsSeries(LowI,false);                              
          ArraySetAsSeries(TimeI,false);
          
          // Copy OHLC historical data
          lastcopied=CopyClose(CurrentSymbol,Timeframe,0,Chart::TCN+2,CloseI);  // Close prices
          lastcopied=CopyOpen(CurrentSymbol,Timeframe,0,Chart::TCN+2,OpenI);    // Open prices
          lastcopied=CopyHigh(CurrentSymbol,Timeframe,0,Chart::TCN+2,HighI);    // High prices
          lastcopied=CopyLow(CurrentSymbol,Timeframe,0,Chart::TCN+2,LowI);      // Low prices
          lastcopied=CopyTime(CurrentSymbol,Timeframe,0,Chart::TCN+2,TimeI);    // Bar times
          
          // Set arrays as time series (index 0 = last bar)
          ArraySetAsSeries(CloseI,true);
          ArraySetAsSeries(OpenI,true);
          ArraySetAsSeries(HighI,true);                        
          ArraySetAsSeries(LowI,true);
          ArraySetAsSeries(TimeI,true);
          
          LastCloseTime=tTimeI[1];  // Store last bar time
          }
          
       // Update current quotes
       ChartBid=LastTick.bid;  // Bid price
       ChartAsk=LastTick.ask;  // Ask price
       ChartPoint=SymbolInfoDouble(CurrentSymbol,SYMBOL_POINT);
       }
    };
 int Chart::TCN = 0;                   // static variable for number of bars for analysis
 Chart *ChartO;                        // pointer to chart object for working with historical data
 
 //+--------------------------------------------------------------+
 //| Trading robot instance class                                 |
 //| Contains all trading logic, parameters and robot state       |
 //| Manages position opening/closing and signal calculation      |
 //+--------------------------------------------------------------+
 class BotInstance
   {
   public:
   //+--------------------------------------------------------------+
   //| Individual values for a specific robot                       |
   //+--------------------------------------------------------------+
   bool bInitBuy;                      // initial buys flag
   bool bInitSell;                     // initial sells flag
   
   // Additional variables for trading strategy (counter-trend)
   int BarsChainForOpen;               // number of consecutive bars in one direction to open a counter-trend position
   int BarsChainForClose;              // number of consecutive bars in one direction to close a counter-trend position
   
   //+--------------------------------------------------------------+
   //| General values for a specific robot                          |
   //+--------------------------------------------------------------+
   bool bInitSpreadControl;            // spread control flag
   int SpreadE;                        // maximum spread for opening
   int SpreadCloseE;                   // maximum spread for closing
   int MagicF;                         // robot magic number
   string CurrentSymbol;               // trading symbol
   
   datetime DatetimeStart;             // start date from settings file
   
   // Trading signals
   int TradeDirection;                 // trade direction signal (0-none, 1-buy, 2-sell)
   int TradeDirectionClose;            // direction signal for closing positions
   
   bool bOpened;                       // flag indicating presence of open positions
   
   // Objects for working with trading
   CPositionInfo  m_position;          // object for working with positions
   CTrade         m_trade;             // object for performing trading operations
   
   bool bNewBarV;                      // new bar appearance flag
   
   //+--------------------------------------------------------------+
   //| BotInstance class constructor                                |
   //| Initializes all robot parameters with default values         |
   //+--------------------------------------------------------------+
   BotInstance()
      {
      bOpened = false;                 // initially there are no positions
      
      SpreadE=SpreadEE;                // set maximum spread
      MagicF=MagicE;                   // set magic number
      RestartParams();                 // initialize other parameters
      }
      
   //+--------------------------------------------------------------+
   //| Main function for robot trading logic                        |
   //+--------------------------------------------------------------+
   void CalculateForTrade()
      {
      // The proper tone for signal designation (price movement direction prediction)
      
      // For opening new positions:
      // In other parts of the code:
      //   BotO.TradeDirection - 1 - this is BUY
      //   BotO.TradeDirection - 2 - this is SELL
      //   BotO.TradeDirection - 0 - no signal
      // Locally:
      //   TradeDirection - 1 - this is BUY
      //   TradeDirection - 2 - this is SELL
      //   TradeDirection - 0 - no signal
      
      // For closing existing positions:
      // In other parts of the code:
      //   BotO.TradeDirectionClose - 1 - this is BUY
      //   BotO.TradeDirectionClose - 2 - this is SELL
      //   BotO.TradeDirectionClose - 0 - no signal
      // Locally:
      //   TradeDirectionClose - 1 - this is BUY
      //   TradeDirectionClose - 2 - this is SELL
      //   TradeDirectionClose - 0 - no signal
      
      // ChartO - access to chart data where the virtual robot is attached
      
      // Example of calculating these values based on the strategy used
      // Counter-trend strategy: searching for a sequence of bars in one direction,
      // then opening a position in the opposite direction (expecting a bounce/retracement)
      
      // First, determine the direction of the most recent closed bar (index [1])
      int firstTradeDirection = 0;      
      if ( ChartO.OpenI[1] > ChartO.CloseI[1] ) firstTradeDirection = 1; // bearish bar (close price below open)
      if ( ChartO.CloseI[1] > ChartO.OpenI[1] ) firstTradeDirection = 2; // bullish bar (close price above open)
      
      // Based on the direction of this most recent bar, search for previous bars
      // that indicate the same direction and determine if there is an opening
      // or closing signal and in what direction
      // BarsChainForOpen - number of consecutive bars for an opening signal
      // BarsChainForClose - number of consecutive bars for a closing signal
      
      bool Aborted=false; // bar chain search interruption flag
      if (firstTradeDirection == 1)
         {
         // Signal definition for opening a long position (counter-trend to bearish trend)
         for ( int i=2; i<=BarsChainForOpen; i++ )
            {
            // Check that all previous bars are also bearish
            if (ChartO.OpenI[i] <= ChartO.CloseI[i])
               {
               Aborted=true; // found a bullish bar - interrupt search
               }
            }
         if ( !Aborted ) TradeDirection = 1; // open long
         else TradeDirection = 0;
         
         Aborted=false;
         // Determine signal for closing short positions
         for ( int i=2; i<=BarsChainForClose; i++ )
            {
            // Check that all previous bars are also bearish
            if (ChartO.OpenI[i] <= ChartO.CloseI[i])
               {
               Aborted=true; // found a bullish bar - interrupt search
               break;
               }            
            }
         if ( !Aborted ) TradeDirectionClose = 1; // close short
         else TradeDirectionClose = 0;            
         }
      else if (firstTradeDirection == 2)
         {
         // Signal definition for opening a short position (counter-trend to bullish trend)
         for ( int i=2; i<=BarsChainForOpen; i++ )
            {
            // Check that all previous bars are also bullish
            if (ChartO.CloseI[i] <= ChartO.OpenI[i])
               {
               Aborted=true; // found a bearish bar - interrupt search
               break;
               }
            }
         if ( !Aborted ) TradeDirection = 2; // open short
         else TradeDirection = 0;
         
         Aborted=false;
         // Determine signal for closing long positions
         for ( int i=2; i<=BarsChainForClose; i++ )
            {
            // Check that all previous bars are also bullish
            if (ChartO.CloseI[i] <= ChartO.OpenI[i])
               {
               Aborted=true; // found a bearish bar - interrupt search
               break;
               }            
            }
         if ( !Aborted ) TradeDirectionClose = 2; // close long
         else TradeDirectionClose = 0;            
         }
      else
         {
         TradeDirection = 0;        // no signal for opening new positions
         TradeDirectionClose = 0;   // no signal for closing existing positions
         }                   
      }

   //+--------------------------------------------------------------+
   //| Method for processing position closing logic                 |
   //| Determines which positions to close based on signal          |
   //| TradeDirectionClose: 1-close sells, 2-close buys             |
   //+--------------------------------------------------------------+
   void TradeClose()
      {      
      //CloseBuyF(); //standard BUY position closing logic
      //CloseSellF(); //standard SELL position closing logic
      
      // Close positions depending on the closing signal
      if (TradeDirectionClose==1) CloseSellF();        // Close sells
      else if (TradeDirectionClose==2) CloseBuyF();    // Close buys
      }
      
   //+--------------------------------------------------------------+
   //| Method for processing position opening logic                 |
   //| Opens new positions based on trading signal                  |
   //| TradeDirection: 1-buy, 2-sell                                |
   //+--------------------------------------------------------------+
   void Trade()
      {
      //SellF();
      //BuyF();
      
      // Open positions based on signal and time limit
      if (TradeDirection==1 && Days() <= DaysToFuture) BuyF();      // Open buy
      else if (TradeDirection==2 && Days() <= DaysToFuture) SellF(); // Open sell
      }

   //+--------------------------------------------------------------+
   //| Restart robot parameters                                     |
   //| Initializes all robot parameters with values from settings   |
   //| Called when a new robot instance is created                  |
   //+--------------------------------------------------------------+
   void RestartParams()
      {
      DatetimeStart=__DATETIME__;        // Set start date
      
      // Initialize main trading parameters
      bInitSpreadControl=true;           // Enable spread control
      SpreadCloseE = SpreadEE;           // Set maximum spread for closing
      MagicF=MagicE;                     // Set magic number
      CurrentSymbol=ChartO.CurrentSymbol; // Set current symbol
     bInitSell = TSE;                   // Permission for sells
     
     // Initialize trading strategy parameters
     BarsChainForOpen = BarsChainForOpenE;   // Number of bars for opening a position
     BarsChainForClose = BarsChainForCloseE; // Number of bars for closing a position
     
     this.m_trade.SetExpertMagicNumber(MagicF);
     }

   //+-------------------------------------------------------------------+
   //| Check for open positions on current candle                        |
   //| Determines if any positions were opened in the current bar        |
   //| Returns: true if there are positions opened on the current candle |
   //+-------------------------------------------------------------------+
   bool Opened()
      {
      ulong ticket;
      bool ord;
      
      // Iterate through all open positions
      for ( int i=0; i<PositionsTotal(); i++ )
         {
         ticket=PositionGetTicket(i);
         ord=PositionSelectByTicket(ticket);      
         
         // Check if the position belongs to our robot and symbol
         if ( ord && PositionGetInteger(POSITION_MAGIC) == MagicF && PositionGetString(POSITION_SYMBOL) == CurrentSymbol )
            {
            // If the position was opened on the current candle or later
            if ( PositionGetInteger(POSITION_TIME) >= ChartO.TimeI[0] ) return true;
            }
         }
      return false;  // No positions on the current candle
      }

   //+--------------------------------------------------------------+
   //| Check if within allowed trading window                       |
   //| Determines if trading is allowed at the current time         |
   //| Considers trading sessions and time settings                 |
   //+--------------------------------------------------------------+
   bool bInTradeSession()
      {
         MqlDateTime ME;                  // current bar's time structure (year, month, day, hour, minute, day of week)
         TimeToStruct(ChartO.TimeI[0],ME);
         int MinuteEqui = ME.hour*60+ME.min; // minute equivalent of current time (minutes from the start of the day)

         if (ME.day_of_week == MONDAY)
         {
             if (ChartO.MONDAY_MinuteEquivalentFrom < ChartO.MONDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.MONDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.MONDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.MONDAY_MinuteEquivalentFrom > ChartO.MONDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.MONDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.MONDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         else if (ME.day_of_week == TUESDAY)
         {
             if (ChartO.TUESDAY_MinuteEquivalentFrom < ChartO.TUESDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.TUESDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.TUESDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.TUESDAY_MinuteEquivalentFrom > ChartO.TUESDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.TUESDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.TUESDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         else if (ME.day_of_week == WEDNESDAY)
         {
             if (ChartO.WEDNESDAY_MinuteEquivalentFrom < ChartO.WEDNESDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.WEDNESDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.WEDNESDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.WEDNESDAY_MinuteEquivalentFrom > ChartO.WEDNESDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.WEDNESDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.WEDNESDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         else if (ME.day_of_week == THURSDAY)
         {
             if (ChartO.THURSDAY_MinuteEquivalentFrom < ChartO.THURSDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.THURSDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.THURSDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.THURSDAY_MinuteEquivalentFrom > ChartO.THURSDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.THURSDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.THURSDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         else if (ME.day_of_week == FRIDAY)
         {
             if (ChartO.FRIDAY_MinuteEquivalentFrom < ChartO.FRIDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.FRIDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.FRIDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.FRIDAY_MinuteEquivalentFrom > ChartO.FRIDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.FRIDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.FRIDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         else if (ME.day_of_week == SATURDAY)
         {
             if (ChartO.SATURDAY_MinuteEquivalentFrom < ChartO.SATURDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.SATURDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.SATURDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.SATURDAY_MinuteEquivalentFrom > ChartO.SATURDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.SATURDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.SATURDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         else if (ME.day_of_week == SUNDAY)
         {
             if (ChartO.SUNDAY_MinuteEquivalentFrom < ChartO.SUNDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.SUNDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.SUNDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.SUNDAY_MinuteEquivalentFrom > ChartO.SUNDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.SUNDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.SUNDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         else
         {
             return false;
         }
         return false;
      }

   //+------------------------------------------------------------------+
   //| Internal robot tick                                              |
   //| Controls sequence of trading operations                          |
   //| Parameters:                                                      |
   //|   bFirst - first processing flag (true=close, false=open)        |
   //+------------------------------------------------------------------+
   void InstanceTick(bool bFirst)
      {
      if ((bTester && bNewBarV) || !bTester)                         // if new bar appeared
         {
         if ( bFirst )                      // first processing - close positions
            {
            TradeClose();                   // execute position closing logic
            } 
         if ( !bFirst )      // second processing - open positions
            {
            Trade();                        // execute position opening logic        
            }
         }
      }
      
   public:
   datetime Time0;                    // current bar time to determine new bar
   string NewBarType;                 // new bar processing type ("START" or "LATE")
   datetime PreviousLateTick;         // last tick time during late processing
   datetime PreviousTimeCalc;          // last trading signals calculation time

   //+----------------------------------------------------------------+
   //| Detection of a new bar on the chart                            |
   //| Function checks for new bar appearance and determines the type |
   //| of processing (START - at bar start, LATE - delayed)           |
   //| Returns: true if a new bar appeared, false - otherwise         |
   //| Sets: NewBarType ("START" or "LATE")                           |
   //+----------------------------------------------------------------+
   bool bNewBar()
      {
      // Check for new bar appearance with processing within the first 20 seconds
      if (ChartO.LastTick.time > 0 && ChartO.TimeI[1] > 0 && ChartO.TimeI[1] != Time0 && ChartO.ChartPoint != 0.0 
      && ChartO.LastTick.time - ChartO.TimeI[0] <= 20 && ChartO.LastTick.time - ChartO.TimeI[0] >= 0 )
         {
         NewBarType = "START";           // Processing at the start of the bar
         Time0=ChartO.TimeI[1];          // Store current bar time
         return true;         
         }
      // Check for new bar appearance with late processing (after 20 seconds)
      else if ( ChartO.LastTick.time > 0 && ChartO.TimeI[1] > 0 && ChartO.TimeI[1] != Time0 && ChartO.ChartPoint != 0.0 )
         {
         NewBarType = "LATE";            // Delayed bar processing
         Time0=ChartO.TimeI[1];          // Store current bar time
         return true;
         }
      else return false;                 // No new bar detected
      }

   //+-----------------------------Auxiliary functions-------------------------------------+
   
   //+--------------------------------------------------------------+
   //| Hour correction                                              |
   //| Function checks if the hour value is correct and returns     |
   //| a valid value in the range from 1 to 23                      |
   //| Parameters: hour0 - initial hour value                       |
   //| Returns: corrected hour value or 0 on error                  |
   //+--------------------------------------------------------------+
   int HourCorrect(int hour0)
   {
   int rez=0;                             // hour correction result (0 on error, otherwise valid hour value)
   if ( hour0 < 24 && hour0 > 0 )
      {
      rez=hour0;
      }
   return rez;
   }
      
   //+--------------------------------------------------------------+
   //| Minute correction                                            |
   //| Function checks if the minute value is correct and returns   |
   //| a valid value in the range from 1 to 59                      |
   //| Parameters: minute0 - initial minute value                   |
   //| Returns: corrected minute value or 0 on error                |
   //+--------------------------------------------------------------+
   int MinuteCorrect(int minute0)
      {
      int rez=0;                           // minute correction result (0 on error, otherwise valid minute value)
      if ( minute0 < 60 && minute0 > 0 )
         {
         rez=minute0;
         }
      return rez;      
      }   
   //+--------------------------------------------------------------+
   //| Calculation of total robot positions                         |
   //| Counts all positions with our magic number                   |
   //| Returns: number of open positions (int)                      |
   //+--------------------------------------------------------------+
   int OrdersG()
      {
      ulong ticket;                       // current position ticket for processing
      bool ord;                           // flag for successful position selection for analysis
      int OrdersG=0;                      // counter for robot's open positions
      for ( int i=0; i<PositionsTotal(); i++ )
         {
         ticket=PositionGetTicket(i);
         ord=PositionSelectByTicket(ticket);
         if ( ord && PositionGetInteger(POSITION_MAGIC) == MagicF && PositionGetString(POSITION_SYMBOL) == CurrentSymbol )
            {
            OrdersG++;
            }
         }
      return OrdersG;
      }   

   //+-----------------------------------------------------------------+
   //| Check for positions of a specific direction                     |
   //| Parameters: direction - direction (true=SELL, false=BUY)        |
   //| Returns: true if there are positions of the specified direction |
   //+-----------------------------------------------------------------+
   bool OrdersG(bool direction)
      {
      ulong ticket;
      bool ord;
      for ( int i=0; i<PositionsTotal(); i++ )
         {
         ticket=PositionGetTicket(i);
         ord=PositionSelectByTicket(ticket);      
         if ( ord && PositionGetInteger(POSITION_MAGIC) == MagicF && PositionGetString(POSITION_SYMBOL) == CurrentSymbol && ((!direction && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY) || (direction && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL)) )
            {
            return false;
            }
         }
      return true;
      }
   
   //+--------------------------------------------------------------+
   //| Count SELL positions for the current symbol                  |
   //| Function iterates through all open positions and counts SELL |
   //| positions with our magic number                              |
   //| Used to control the number of open short positions           |
   //| Returns: number of SELL positions (int)                      |
   //+--------------------------------------------------------------+
   int OrdersS()
      {
      ulong ticket;                       // current position ticket for processing
      bool ord;                           // flag for successful position selection for analysis
      int OrdersG=0;                      // counter for robot's open positions
      for ( int i=0; i<PositionsTotal(); i++ )
         {
         ticket=PositionGetTicket(i);
         ord=PositionSelectByTicket(ticket);
         bool B1=bOurMagic(PositionGetInteger(POSITION_MAGIC)); // flag indicating the position belongs to our robot by magic number
         if ( ord && B1 && PositionGetString(POSITION_SYMBOL) == CurrentSymbol )
            {
            OrdersG++;
            }
         }
      return OrdersG;
      }   

   //+--------------------------------------------------------------+
   //| Structure for storing stop-loss and take-profit levels       |
   //| Used to pass calculated stop levels in points                |
   //| between trading system functions                             |
   //+--------------------------------------------------------------+
   struct StopsPoints
      {
      int StopLossPoints;          // stop-loss level in points from position open price
      int TakeProfitPoints;        // take-profit level in points from position open price
      };
   
   //+-----------------------------Trading functions block-------------------------------------+
   
   //+--------------------------------------------------------------+
   //| Function for opening buy positions (BUY)                     |
   //| Checks conditions and opens a long position                  |
   //| Considers lot, stop-loss, take-profit and re-buy settings    |
   //| Controls opening timing and account mode (hedging/netting)   |
   //+--------------------------------------------------------------+
   void BuyF()
      {
      double DtA;                         // time difference between the current moment and the last operation to control position opening frequency
      double SLTemp0=MathAbs(SLE);         // absolute value of stop-loss in points
      double TPTemp0=MathAbs(TPE);        // absolute value of take-profit in points 
   
      DtA=double(TimeCurrent())-GlobalVariableGet(GLOBALVARPREFIX+IntegerToString(MagicF));
      if ( ChartO.LastTick.bid > 0.0 && ChartO.OpenI[0] > 0.0 && ((NewBarType == "LATE" && ChartO.OpenI[0] >= ChartO.LastTick.bid) || NewBarType == "START") && (DtA > 0 || DtA < 0) && ((AccountInfoInteger(ACCOUNT_MARGIN_MODE) == ACCOUNT_MARGIN_MODE_RETAIL_HEDGING) || (AccountInfoInteger(ACCOUNT_MARGIN_MODE) == ACCOUNT_MARGIN_MODE_RETAIL_NETTING && OrdersS()==0)) )
         {
         if (bInitBuy && (DtA > 0 || DtA < 0) && ( (!bInitRepurchaseE && OrdersG() == 0) || (bInitRepurchaseE && OrdersG(true) && AddtionalOpenPrice(ChartO.LastTick.ask,true))) )
            {
            CheckForOpen(MagicF,OP_BUY,ChartO.ChartBid,ChartO.ChartAsk,ChartO.CloseI[0],int(SLTemp0),int(TPTemp0),LotE,ChartO.CurrentSymbol,0,CommentI,bInitLotControl,bInitSpreadControl,SpreadE);
            }            
         }
      }
      
   //+--------------------------------------------------------------+
   //| Function for opening sell positions (SELL)                   |
   //| Checks conditions and opens a short position                 |
   //| Considers lot, stop-loss, take-profit and re-buy settings    |
   //| Controls opening timing and account mode (hedging/netting)   |
   //+--------------------------------------------------------------+
   void SellF()
      {
      double DtA;                         // time difference between the current moment and the last operation to control position opening frequency
      double SLTemp0=MathAbs(SLE);         // absolute value of stop-loss in points
      double TPTemp0=MathAbs(TPE);         // absolute value of take-profit in points
   
      DtA=double(TimeCurrent())-GlobalVariableGet(GLOBALVARPREFIX+IntegerToString(MagicF));
      if ( ChartO.LastTick.bid > 0.0 && ChartO.OpenI[0] > 0.0 && ((NewBarType == "LATE" && ChartO.LastTick.bid >= ChartO.OpenI[0]) || NewBarType == "START") && (DtA > 0 || DtA < 0) && ((AccountInfoInteger(ACCOUNT_MARGIN_MODE) == ACCOUNT_MARGIN_MODE_RETAIL_HEDGING) || (AccountInfoInteger(ACCOUNT_MARGIN_MODE) == ACCOUNT_MARGIN_MODE_RETAIL_NETTING && OrdersS()==0)) )
         {
         if (bInitSell && (DtA > 0 || DtA < 0) && ( (!bInitRepurchaseE && OrdersG() == 0) || (bInitRepurchaseE  && OrdersG(false) && AddtionalOpenPrice(ChartO.LastTick.bid,false))) )
            {
            CheckForOpen(MagicF,OP_SELL,ChartO.ChartBid,ChartO.ChartAsk,ChartO.CloseI[0],int(SLTemp0),int(TPTemp0),LotE,ChartO.CurrentSymbol,0,CommentI,bInitLotControl,bInitSpreadControl,SpreadE);
            }            
         }
      }
      
   public:   
   //+----------------------------------------------------------------+
   //| Calculation of number of days since trading start              |
   //| Function calculates the number of days elapsed since the start |
   //| of the robot's work until the current time                     |
   //| Returns: number of days as a fractional number (double)        |
   //+----------------------------------------------------------------+
   double Days()
      {
      if ( double(ChartO.TimeI[0] - DatetimeStart) > 0 ) return double(ChartO.TimeI[0] - DatetimeStart)/86400.0;
      else return 0.0;     
      }
   private:

   //+--------------------------------------------------------------+
   //| Check for sufficient funds to open a position                |
   //| Function checks if there is enough free margin to open a     |
   //| position of the specified volume for the given symbol        |
   //| Parameters: symb - symbol, lots - volume, type - order type  |
   //| Returns: true if funds are sufficient, false - otherwise     |
   //+--------------------------------------------------------------+
   bool CheckMoneyForTrade(string symb,double lots,ENUM_ORDER_TYPE type)
      {
       //--- get open price
       MqlTick mqltick;                    // symbol's last tick structure to get current prices
       SymbolInfoTick(symb,mqltick);
       double price=mqltick.ask;          // open price for buy (ASK)
       if(type==ORDER_TYPE_SELL)
       price=mqltick.bid;                  // open price for sell (BID)
       //--- values of required and free margin
       double margin;                      // required margin to open a position of the specified volume
       double free_margin=AccountInfoDouble(ACCOUNT_MARGIN_FREE); // free margin on account
       //--- call checking function
       if(!OrderCalcMargin(type,symb,lots,price,margin))
         {
          //--- something went wrong, report and return false
          return(false);
         }
       //--- if there are not enough funds for the operation
       if(margin>free_margin)
         {
         //--- report the error and return false
         Print("Not enough money for ",EnumToString(type)," ",lots," ",symb," Error code=",GetLastError());
         return(false);
         }
      //--- check passed successfully
      return(true);
      }

   //+--------------------------------------------------------------+
   //| Position opening function with all conditions check          |
   //| Performs position opening with lot, spread, and risk control |
   //+--------------------------------------------------------------+
   void CheckForOpen(int Magic0,int OrdType,double PriceBid,double PriceAsk,double PriceClose0,int SL0,int TP0,double Lot0,string Symbol0,datetime Expiration0,string Comment0,bool bLotControl0,bool bSpreadControl0,int Spread0)
      {
      double LotTemp=Lot0;                // temporary lot size for calculations (may be changed by martingale system or auto-lot)
      double SpreadLocal=MarketInfo(CurrentSymbol,MODE_SPREAD); // current symbol spread in points to check opening validity
      double LotAntiError=MarketInfo(CurrentSymbol,MODE_MINLOT); // minimum lot size of the symbol for position size adjustment
   
      if ( (SpreadLocal <= Spread0 && bSpreadControl0 == true ) || ( bSpreadControl0 == false )   )                    
         {   
         if ( bLotControl0 == false )
            {
            LotTemp=Lot0;
            }
         else if ( bLotControl0 == true )
            {
            UpdateMaxBalance();//first update the maximum balance information
            LotTemp=Lot0*(LastMaxBalance/DepositE);
            }
         if (bInitMartinE)
            {
            LotTemp=CalcMartinLot(LotTemp);
            }
            
         LotAntiError=GetLotAniError(LotTemp);
         if ( OrdType == OP_SELL && LotAntiError > 0.0 && CheckMoneyForTrade(ChartO.CurrentSymbol,LotAntiError,ORDER_TYPE_SELL) )
            {
            bool rez = m_trade.Sell(LotAntiError,Symbol0,ChartO.ChartBid,SL0 != 0 ? ChartO.ChartBid+CorrectLevels(SL0)*ChartO.ChartPoint: 0,TP0 != 0 ? ChartO.ChartBid-CorrectLevels(TP0)*ChartO.ChartPoint: 0,Comment0); // result of short position opening operation
            GlobalVariableSet(GLOBALVARPREFIX+IntegerToString(MagicF),double(TimeCurrent()) );
            }
         if ( OrdType == OP_BUY && LotAntiError > 0.0 && CheckMoneyForTrade(ChartO.CurrentSymbol,LotAntiError,ORDER_TYPE_BUY) )
            {
            bool rez = m_trade.Buy(LotAntiError,Symbol0,ChartO.ChartAsk,SL0 != 0 ? ChartO.ChartAsk-CorrectLevels(SL0)*ChartO.ChartPoint: 0,TP0 != 0 ? ChartO.ChartAsk+CorrectLevels(TP0)*ChartO.ChartPoint: 0, Comment0); // result of buy position opening operation
            GlobalVariableSet(GLOBALVARPREFIX+IntegerToString(MagicF),double(TimeCurrent()) );
            }
         }
      }

   //calculate martingale lot enhancement relative to the base one   
   datetime LastMaxBalanceTime;           // last maximum balance time to limit deal history analysis
   datetime OptimizationBorderTime;       // boundary time for analysis optimization (avoiding redundant analysis of the same deals)
   //+--------------------------------------------------------------+
   //| Martingale system lot size calculation                       |
   //| Function analyzes deal history and calculates increased      |
   //| lot size to compensate for previous losses                   |
   //| Parameters: BasicLot - base lot size                         |
   //| Returns: adjusted lot size (double)                          |
   //+--------------------------------------------------------------+
   double CalcMartinLot(double BasicLot) 
      { 
      //martingale lot calculation
      //MaxMartinLotMultiplierStepsE
      bool bFirst=false;                  // flag for the first deal in the analyzed period
      bool bSecond=false;                 // flag for finding max balance in the analyzed period
      double TempSummProfit=0.0;          // cumulative profit sum for maximum balance calculation
      double LastMaxProfit=0.0;           // maximum found profit over the analyzed period
      datetime FirstDate=0;               // date of the first deal in the analyzed period
      HistorySelect(TimeCurrent()-HistoryDaysLoadI*86400,TimeCurrent());
      for ( int i=HistoryDealsTotal()-1; i>=0; i-- )//searching for the last maximum
         {
         ulong ticket=HistoryDealGetTicket(i);
         if ( HistoryDealGetInteger(ticket,DEAL_MAGIC) == MagicE && HistoryDealGetString(ticket,DEAL_SYMBOL) == ChartO.CurrentSymbol && HistoryDealGetInteger(ticket,DEAL_ENTRY) == DEAL_ENTRY_OUT ) 
            {
            if ( HistoryDealGetInteger(ticket,DEAL_TIME) >= OptimizationBorderTime)
               {
               double TempProfit=-(HistoryDealGetDouble(ticket,DEAL_PROFIT) + HistoryDealGetDouble(ticket,DEAL_COMMISSION) + HistoryDealGetDouble(ticket,DEAL_SWAP)); // current deal's profit considering commission and swap (inverted for loss calculation)
               TempSummProfit+=TempProfit;//
               if (!bFirst)
                  {
                  FirstDate=datetime(HistoryDealGetInteger(ticket,DEAL_TIME)+1);
                  bFirst=true;
                  }
               if (TempSummProfit > LastMaxProfit)
                  {
                  LastMaxProfit = TempSummProfit;
                  LastMaxBalanceTime=datetime(HistoryDealGetInteger(ticket,DEAL_TIME));
                  bSecond=true;
                  }
               }
            else break;
            }
         }
      if (!bSecond) LastMaxBalanceTime=FirstDate;
      if (LastMaxBalanceTime > OptimizationBorderTime) OptimizationBorderTime=LastMaxBalanceTime;   
     
      double TempSummLots=0.0;            // sum lot of all losing positions to calculate average loss per lot
      int summs=0;                        // counter for found losing positions (limited by MaxMartinLotMultiplierStepsE)
      TempSummProfit=0.0;                 // cumulative loss sum to calculate average loss per lot
      double MiddleProfit=0.0;           // average loss per one lot to calculate martingale multiplier
      // Now it remains to calculate the sum of lots from negative positions
     
      for ( int i=HistoryDealsTotal()-1; i>=0; i-- )//calculate average loss size over all trades per 1 lot
      {
      ulong ticket=HistoryDealGetTicket(i);
      if ( HistoryDealGetInteger(ticket,DEAL_MAGIC) == MagicE && HistoryDealGetString(ticket,DEAL_SYMBOL) == ChartO.CurrentSymbol && HistoryDealGetInteger(ticket,DEAL_ENTRY) == DEAL_ENTRY_OUT )
         {
         if (HistoryDealGetInteger(ticket,DEAL_TIME) < LastMaxBalanceTime) break;//interruption if we've gone into an already calculated part of the history
         if (summs >= MaxMartinLotMultiplierStepsE) break;//interruption if we exceed the allowed number of losses
        
         double TempProfit=HistoryDealGetDouble(ticket,DEAL_PROFIT) + HistoryDealGetDouble(ticket,DEAL_COMMISSION) + HistoryDealGetDouble(ticket,DEAL_SWAP); // current deal profit with commission and swap
         if (TempProfit < 0 )
            {
            double TempVolume=HistoryDealGetDouble(ticket,DEAL_VOLUME);//determine current deal volume
            TempSummProfit+=MathAbs(TempProfit/TempVolume);
            summs++;
            }
         }
      }   
     
      // finish calculating average loss per 1 lot, if there was more than 1 position
      if (summs>1)
         {
         MiddleProfit = TempSummProfit/summs;
         }
      //
     
      summs=0;//reset counter
      // now using this average loss size we will add weights to each loss lot
      for ( int i=HistoryDealsTotal()-1; i>=0; i-- )//calculate sum of negative positions lots (considering max quantity limit)
         {
         ulong ticket=HistoryDealGetTicket(i);
         if ( HistoryDealGetInteger(ticket,DEAL_MAGIC) == MagicE && HistoryDealGetString(ticket,DEAL_SYMBOL) == ChartO.CurrentSymbol && HistoryDealGetInteger(ticket,DEAL_ENTRY) == DEAL_ENTRY_OUT )
            {
            if (HistoryDealGetInteger(ticket,DEAL_TIME) < LastMaxBalanceTime) break;//interruption if we've gone into an already calculated part of history
            if (summs >= MaxMartinLotMultiplierStepsE) break;//interruption if we exceed allowed number of losses
           
            double TempProfit=HistoryDealGetDouble(ticket,DEAL_PROFIT) + HistoryDealGetDouble(ticket,DEAL_COMMISSION) + HistoryDealGetDouble(ticket,DEAL_SWAP); // current deal profit with commission and swap
            if (TempProfit < 0 )
               {
               double TempVolume=HistoryDealGetDouble(ticket,DEAL_VOLUME); // current deal volume in lots
               if (MiddleProfit > 0.0)
                  {             
                  double Multiplier = MathAbs(TempProfit/TempVolume)/MiddleProfit; // weight multiplier for current losing deal relative to average loss per lot
                  TempSummLots += TempVolume * Multiplier;
                  summs++;
                  } 
               else
                  {
                  TempSummLots += TempVolume;
                  summs++;                  
                  }
               }
            }
         }

     // decision making     
     return BasicLot + TempSummLots;  
     }
   
   //+--------------------------------------------------------------+
   //| Get day of week for current time                             |
   //| Function returns day of week for current server time         |
   //| Returns: day of week (ENUM_DAY_OF_WEEK)                      |
   //+--------------------------------------------------------------+
   ENUM_DAY_OF_WEEK DayOfWeek()
      {
      MqlDateTime TimeInfo;
      TimeToStruct(TimeCurrent(),TimeInfo);
      ENUM_DAY_OF_WEEK rez=ENUM_DAY_OF_WEEK(TimeInfo.day_of_week);
      
      return rez;
      }
      
   //+--------------------------------------------------------------+
   //| Get day of week for specified time                           |
   //| Function returns day of week for the passed time             |
   //| Parameters: Time00 - time for analysis                       |
   //| Returns: day of week (ENUM_DAY_OF_WEEK)                      |
   //+--------------------------------------------------------------+
   ENUM_DAY_OF_WEEK TimeDayOfWeek(datetime Time00)
      {
      MqlDateTime TimeInfo;
      TimeToStruct(Time00,TimeInfo);
      ENUM_DAY_OF_WEEK rez=ENUM_DAY_OF_WEEK(TimeInfo.day_of_week);
      
      return rez;
      }
      
   //+--------------------------------------------------------------+
   //| Get hour from specified time                                 |
   //| Function extracts hour from the passed time                  |
   //| Parameters: Time00 - time for analysis                       |
   //| Returns: hour in 0-23 format (int)                           |
   //+--------------------------------------------------------------+
   int TimeHour(datetime Time00)
      {
      MqlDateTime TimeInfo;               // time structure with components to extract the hour
      TimeToStruct(Time00,TimeInfo);
      int rez=int(TimeInfo.hour);         // extracted hour from the passed time
      
      return rez;
      }
      
   //+--------------------------------------------------------------+
   //| Get minutes from specified time                              |
   //| Function extracts minutes from the passed time               |
   //| Parameters: Time00 - time for analysis                       |
   //| Returns: minutes in 0-59 format (int)                        |
   //+--------------------------------------------------------------+
   int TimeMinute(datetime Time00)
      {
      MqlDateTime TimeInfo;               // time structure with components to extract the minutes
      TimeToStruct(Time00,TimeInfo);
      int rez=int(TimeInfo.min);          // extracted minutes from the passed time
      
      return rez;
      }   
   
   //+--------------------------------------------------------------+
   //| Get current hour                                             |
   //| Function returns the hour of the current server time         |
   //| Returns: hour in 0-23 format (int)                           |
   //+--------------------------------------------------------------+
   int Hour()
      {
      int rez;                            // current server time hour
      MqlDateTime C1Info;                 // current time structure with components
      TimeToStruct(TimeCurrent(),C1Info);
      rez=C1Info.hour;
      return rez;
      }
   
   //+--------------------------------------------------------------+
   //| Get current minutes                                          |
   //| Function returns the minutes of the current server time      |
   //| Returns: minutes in 0-59 format (int)                        |
   //+--------------------------------------------------------------+
   int Minute()
      {
      int rez;                            // current server time minutes
      MqlDateTime C1Info;                 // current time structure with components
      TimeToStruct(TimeCurrent(),C1Info);
      rez=C1Info.min;
      return rez;
      }   
   
   //+-------------------------------------------------------------------+
   //| Get account free margin                                           |
   //| Function returns the amount of free margin on the trading account |
   //| Returns: free margin in deposit currency (double)                 |
   //+-------------------------------------------------------------------+
   double AccountFreeMargin()
      {
      double rez;                         // free margin on the trading account
      rez=AccountInfoDouble(ACCOUNT_MARGIN_FREE);   
      return rez;
      }   
      
   //+--------------------------------------------------------------+
   //| Get account balance                                          |
   //| Function returns the current balance of the trading account  |
   //| Returns: balance in deposit currency (double)                |
   //+--------------------------------------------------------------+
   double AccountBalance()
      {
      double rez;                         // current balance of the trading account
      rez=AccountInfoDouble(ACCOUNT_BALANCE);
      return rez;
      }      
      
   //+--------------------------------------------------------------+
   //| Get symbol market information (MQL4 compatibility)           |
   //| Function emulates MQL4 MarketInfo function to get            |
   //| various parameters of the trading instrument                 |
   //| Parameters: Symbol0 - symbol, mode0 - request mode           |
   //| Returns: requested value (double)                            |
   //+--------------------------------------------------------------+
   double MarketInfo(string Symbol0,int mode0)
   {
      double rez=0;                       // result of getting symbol information (depends on mode0)
      if ( mode0 == MODE_SPREAD )
         {
         rez=double(SymbolInfoInteger(Symbol0,SYMBOL_SPREAD));            
         }
      if ( mode0 == MODE_STOPLEVEL )
         {
         rez=double(SymbolInfoInteger(Symbol0,SYMBOL_TRADE_STOPS_LEVEL));                  
         }
      if ( mode0 == MODE_MINLOT )
         {
         rez=SymbolInfoDouble(Symbol0,SYMBOL_VOLUME_MIN);      
         }
      if ( mode0 == MODE_MAXLOT )
         {
         rez=SymbolInfoDouble(Symbol0,SYMBOL_VOLUME_MAX);            
         }
      if ( mode0 == MODE_LOTSTEP )
         {
         rez=SymbolInfoDouble(Symbol0,SYMBOL_VOLUME_STEP);                  
         }                                                                          
      if ( mode0 == MODE_TICKVALUE )
         {
         rez=SymbolInfoDouble(Symbol0,SYMBOL_TRADE_TICK_VALUE);                        
         }
      if ( mode0 == MODE_MARGINREQUIRED )
         {
         bool b1=OrderCalcMargin(ORDER_TYPE_BUY,Symbol0,1.0,ChartO.CloseI[0],rez);
         }
      if ( mode0 == MODE_SWAPLONG )
         {
         rez=SymbolInfoDouble(Symbol0,SYMBOL_SWAP_LONG);                        
         }                                                  
      if ( mode0 == MODE_SWAPSHORT )
         {
         rez=SymbolInfoDouble(Symbol0,SYMBOL_SWAP_SHORT);                        
         }                                                                                                                                                                                                                                                                                      
      return rez;
   }   
   
   //+--------------------------------------------------------------+
   //| Lot size correction with error checking                      |
   //| Function corrects lot size according to broker requirements  |
   //| and checks for free margin sufficiency                       |
   //| Parameters: InputLot - initial lot size                      |
   //| Returns: corrected lot or -1.0 on insufficient funds         |
   //+--------------------------------------------------------------+
   double GetLotAniError(double InputLot)
      {
      double Free = AccountFreeMargin();  // free margin on account to check funds sufficiency
      double margin = MarketInfo(CurrentSymbol,MODE_MARGINREQUIRED); // required margin for one lot
      double minLot = MarketInfo(CurrentSymbol,MODE_MINLOT); // minimum symbol lot size
      double Max_Lot = MarketInfo(CurrentSymbol,MODE_MAXLOT); // maximum symbol lot size
      double Step = MarketInfo(CurrentSymbol,MODE_LOTSTEP); // symbol's lot change step
      double Lot13;                       // corrected lot size considering step and constraints
      int LotCorrection;                  // number of steps for lot size rounding
      LotCorrection=int(MathFloor(InputLot/Step));
      Lot13=LotCorrection*Step;   
      if(Lot13<=minLot) Lot13 = minLot;
      if(Lot13>=Max_Lot) Lot13 = Max_Lot;
      
      if( Lot13*margin>=Free ) Lot13=-1.0;
   
      return Lot13;
      }
   
   
   //+-----------------------------------------------------------------+
   //| Time interval check                                             |
   //| Function checks if the current time is within a specified       |
   //| time interval (server or local time)                            |
   //| Parameters: HourStart0, MinuteStart0 - interval start           |
   //| HourEnd0, MinuteEnd0 - interval end                             |
   //| bInitPCTime0 - use local time (true)                            |
   //| Returns: true if the time is in the interval, false - otherwise |
   //+-----------------------------------------------------------------+
   bool bTimeInterval(int HourStart0,int MinuteStart0,int HourEnd0,int MinuteEnd0,bool bInitPCTime0)
      {
      bool rez=false;                     // time interval check result
      
      if ( (Hour() >= HourStart0 && Hour() <= HourEnd0 && Minute() >= MinuteStart0 && Minute() <= MinuteEnd0 && bInitPCTime0==false) || ( TimeHour(TimeLocal()) >= HourStart0 && TimeHour(TimeLocal()) <= HourEnd0 && TimeMinute(TimeLocal()) >= MinuteStart0 && TimeMinute(TimeLocal()) <= MinuteEnd0 && bInitPCTime0==true) )
         {
         rez=true;
         }
         
      return rez;
      } 
   
   //+--------------------------------------------------------------+
   //| Stop-loss and take-profit levels correction                  |
   //| Function corrects SL/TP levels considering the minimum       |
   //| distance from the price (STOPLEVEL) and the current spread   |
   //| Parameters: level0 - initial level in points                 |
   //| Returns: corrected level (int)                               |
   //+--------------------------------------------------------------+
   int CorrectLevels(int level0)
      {
      int rez;                            // corrected stop-loss or take-profit level in points
      int ZeroLevel0=int(MathAbs(MarketInfo(CurrentSymbol,MODE_STOPLEVEL))+MathAbs(MarketInfo(CurrentSymbol,MODE_SPREAD))+1); // minimum allowed level considering stop level and spread
   
      if ( MathAbs(level0) > ZeroLevel0 )
         {
         rez=int(MathAbs(level0));
         }
      else
         {
         rez=ZeroLevel0;
         }   
      return rez;
      }

   //+--------------------------------------------------------------+
   //| Position holding time check function                         |
   //| Returns: true if holding time has expired                    |
   //+--------------------------------------------------------------+
   bool bTimeIsOut(int TimeStart)
      {
      if ( double(int(TimeCurrent()) - TimeStart)/60.0 > double(MinutesHoldE)  ) return true;
      else return false;
      }

   //+--------------------------------------------------------------+
   //| Function for closing sell positions (SELL)                   |
   //| Closes short positions according to strategy conditions      |
   //| Considers spread settings and holding time                   |
   //+--------------------------------------------------------------+
   void CloseSellF()
      {
      ulong ticket;                       // current position ticket for processing
      bool ord;                           // flag for successful position selection for analysis
      ulong Tickets[];                    // array of valid position tickets for closing
      int TicketsTotal=0;                  // total number of found positions for closing
      if ( ChartO.LastTick.bid > 0.0 && ChartO.OpenI[0] > 0.0 && ((NewBarType == "LATE" && ChartO.LastTick.bid >= ChartO.OpenI[0]) || NewBarType == "START") && (!bInitSpreadControl || (bInitSpreadControl && MarketInfo(ChartO.CurrentSymbol,MODE_SPREAD) <= SpreadCloseE)) )
         {
         if (!bInitRepurchaseE || bInitSitE)
            {
            for ( int i=0; i<PositionsTotal(); i++ )
               {
               ticket=PositionGetTicket(i);
               ord=PositionSelectByTicket(ticket);
               if ( ord && PositionGetInteger(POSITION_MAGIC) == MagicF && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL 
               && PositionGetString(POSITION_SYMBOL) == ChartO.CurrentSymbol && ( !bInitSitE || ( bInitSitE && ( (PositionGetDouble(POSITION_PROFIT) + PositionGetDouble(POSITION_SWAP) - PositionGetDouble(POSITION_VOLUME)*ComissionPerLotE) > 0.0 || bTimeIsOut(int(PositionGetInteger(POSITION_TIME))) ) ) ) )
                  {
                  if (ticket != 0)
                     {
                     TicketsTotal++;
                     } 
                  }
               }
            
            if (TicketsTotal > 0)
               {
               ArrayResize(Tickets,TicketsTotal);
               TicketsTotal=0;
               for ( int i=0; i<PositionsTotal(); i++ )
                  {
                  ticket=PositionGetTicket(i);
                  ord=PositionSelectByTicket(ticket);
                  if ( ord && PositionGetInteger(POSITION_MAGIC) == MagicF && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL 
                  && PositionGetString(POSITION_SYMBOL) == ChartO.CurrentSymbol && ( !bInitSitE || ( bInitSitE && ( (PositionGetDouble(POSITION_PROFIT) + PositionGetDouble(POSITION_SWAP) - PositionGetDouble(POSITION_VOLUME)*ComissionPerLotE) > 0.0 || bTimeIsOut(int(PositionGetInteger(POSITION_TIME))) ) ) ) )
                     {
                     if (ticket != 0)
                        {
                        Tickets[TicketsTotal]=ticket;
                        TicketsTotal++;
                        } 
                     }
                  }         
               
               for ( int i=0; i<TicketsTotal; i++ )
                  {
                  bool rez=m_trade.PositionClose(Tickets[i]); // result of position closing operation
                  }          
               }            
            }
         else
            {
            CloseToProfit(false);
            }
         }
      }
      
   //+--------------------------------------------------------------+
   //| Function for closing buy positions (BUY)                     |
   //| Closes long positions according to strategy conditions       |
   //| Considers spread settings and holding time                   |
   //+--------------------------------------------------------------+
   void CloseBuyF()
      {
      ulong ticket;                       // current position ticket for processing
      bool ord;                           // flag for successful position selection for analysis
      ulong Tickets[];                    // array of valid position tickets for closing
      int TicketsTotal=0;                  // total number of found positions for closing
      if ( ChartO.LastTick.bid > 0.0 && ChartO.OpenI[0] > 0.0 && ((NewBarType == "LATE" && ChartO.LastTick.bid >= ChartO.OpenI[0]) || NewBarType == "START") && (!bInitSpreadControl || (bInitSpreadControl && MarketInfo(ChartO.CurrentSymbol,MODE_SPREAD) <= SpreadCloseE)) )
         {         
         if (!bInitRepurchaseE || bInitSitE)
            {
            for ( int i=0; i<PositionsTotal(); i++ )
               {
               ticket=PositionGetTicket(i);
               ord=PositionSelectByTicket(ticket);
               if ( ord && PositionGetInteger(POSITION_MAGIC) == MagicF && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY 
               && PositionGetString(POSITION_SYMBOL) == ChartO.CurrentSymbol && ( !bInitSitE || ( bInitSitE && ( (PositionGetDouble(POSITION_PROFIT) + PositionGetDouble(POSITION_SWAP) - PositionGetDouble(POSITION_VOLUME)*ComissionPerLotE) > 0.0 || bTimeIsOut(int(PositionGetInteger(POSITION_TIME))) ) ) ) )
                  {
                  if (ticket != 0)
                     {
                     TicketsTotal++;
                     } 
                  }
               }
            
            if (TicketsTotal > 0)
               {
               ArrayResize(Tickets,TicketsTotal);
               TicketsTotal=0;
               for ( int i=0; i<PositionsTotal(); i++ )
                  {
                  ticket=PositionGetTicket(i);
                  ord=PositionSelectByTicket(ticket);
                  if ( ord && PositionGetInteger(POSITION_MAGIC) == MagicF && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY 
                  && PositionGetString(POSITION_SYMBOL) == ChartO.CurrentSymbol && ( !bInitSitE || ( bInitSitE && ( (PositionGetDouble(POSITION_PROFIT) + PositionGetDouble(POSITION_SWAP) - PositionGetDouble(POSITION_VOLUME)*ComissionPerLotE) > 0.0 || bTimeIsOut(int(PositionGetInteger(POSITION_TIME))) ) ) ) )
                     {
                     if (ticket != 0)
                        {
                        Tickets[TicketsTotal]=ticket;
                        TicketsTotal++;
                        } 
                     }
                  }         
               
               for ( int i=0; i<TicketsTotal; i++ )
                  {
                  bool rez=m_trade.PositionClose(Tickets[i]); // result of position closing operation
                  }          
               }            
            }
         else
            {
            CloseToProfit(true);
            }
         }
      } 
   };

BotInstance *BotO;                    // pointer to the trading robot object for managing trading logic

//+--------------------------------------------------------------+
//| Expert initialization function                               |
//| Called once when the expert is launched                      |
//| Sets up the timer and determines working mode (tester/real)  |
//+--------------------------------------------------------------+
bool bFirstTimer = false;           // first timer call flag
bool bFirstTick = false;            // first tick flag
bool bTimerCreated=false;           // timer successfully created flag

int OnInit()
  {
   // Determine if we are working in the tester
   bTester=MQLInfoInteger(MQL_TESTER);
   
   // Try to create a timer (up to 5 attempts)
   for (int i = 0; i < 5; i++)
      {
      if (!bTester)
         {
         bTimerCreated = EventSetTimer(1);  // Create a timer with a 1-second interval
         }
      else bTimerCreated=false;             // Timer is not needed in the tester
      
      // If the timer was not created, try to fix the situation
      if (!bTimerCreated)
         {
         EventKillTimer();                  // Remove any possible old timer
         Sleep(100);                        // A short pause before retelling
         }
      else break;                           // Timer created successfully, exit loop
      }
   return(INIT_SUCCEEDED);
  }

//+--------------------------------------------------------------+
//| Expert deinitialization function                             |
//| Called when the expert is finished                           |
//| Clears resources and stops the timer                         |
//+--------------------------------------------------------------+
void OnDeinit(const int reason)
  {
     bTimerCreated=false;               // Reset timer flag
     bFirstTimer = false;               // Reset first timer flag
     bFirstTick = false;                // Reset first tick flag
     DeInit();                          // Call custom deinitialization
     EventKillTimer();                  // Stop timer
  }

//+--------------------------------------------------------------+
//| Timer event handling function                                |
//| Called every second (if the timer is created)                |
//| Used for main real-time logic                                |
//+--------------------------------------------------------------+
void OnTimer()
   {
   if (bTimerCreated)
      {
      if (!bFirstTimer)
         {
         Init();                        // Initial initialization during first timer call
         bFirstTimer=true;
         } 
      else Simulated();                 // Main expert logic
      }
   }

//+--------------------------------------------------------------+
//| Tick handling function                                       |
//| Called when new quotes arrive                                |
//| In the tester, it is used instead of the timer               |
//+--------------------------------------------------------------+
void OnTick()
   {
   if (!bTimerCreated)                  // If timer is not created (tester mode)
      {
      if (!bFirstTick)
         {
         Init();                        // Initial initialization during first tick
         bFirstTick=true;
         }       
      else Simulated();
      }
   }

//+------------------------------------------------------------------+

datetime LastAdapt=0;                  // last re-adaptation time

//+--------------------------------------------------------------+
//| Tick or timer simulation                                     |
//| Main processing function called by timer or tick             |
//| Coordinates work of chart, robot, and interface              |
//+--------------------------------------------------------------+
void Simulated()
   {
   ChartO.ChartTick();                 // Generate tick on virtual chart
   BotTick();                          // Generate tick on virtual bot
   UpdateStatus();                     // Update interface status
   }

int Cor=0;                             // corrections counter
bool bTester = false;                  // tester mode flag

//+--------------------------------------------------------------+
//| System initialization                                        |
//| Creates all necessary objects and interface                  |
//| Called once when the expert is launched                      |
//| Sets up the timer and determines working mode (tester/real)  |
//+--------------------------------------------------------------+
void Init()
   {
   DeleteSimpleInterface();            // Remove previous interface
   
   CreateChart();                      // Create single chart
   CreateInstance();                   // Create single robot  
      
   CreateSimpleInterface();//create interface 
   UpdateStatus();//update interface status
   }   


void DeInit()
   {
   DeleteSimpleInterface();            // Remove interface  
   //clear dynamic memory
   delete ChartO;
   delete BotO;
   }
  
  
//+--------------------------------------------------------------+
//| Timeframe conversion to correction coefficient               |
//| Function converts timeframe period to a coefficient for      |
//| adjustment of average price movement calculation and lot     |
//| balancing depending on the time period                       |
//| Parameters: tf - timeframe for calculation                   |
//| Returns: correction coefficient (double)                     |
//+--------------------------------------------------------------+
double PeriodK(ENUM_TIMEFRAMES tf)
   {
   double ktemp;                            // temporary variable for storing the coefficient
   
   // Determine coefficient depending on timeframe
   switch(tf)
      {
      case  PERIOD_H1:                      // 1-hour timeframe
          ktemp = MathSqrt(1.0/60.0);       // coefficient for H1
          break;
      case  PERIOD_H4:                      // 4-hour timeframe
          ktemp = MathSqrt(1.0/240.0);      // coefficient for H4
          break;
      case PERIOD_M1:                       // 1-minute timeframe
          ktemp = 1.0;                      // base coefficient for M1
          break;
      case PERIOD_M5:                       // 5-minute timeframe
          ktemp = MathSqrt(1.0/5.0);        // coefficient for M5
          break;
      case PERIOD_M15:                      // 15-minute timeframe
          ktemp = MathSqrt(1.0/15.0);       // coefficient for M15
          break;
      case PERIOD_M30:                      // 30-minute timeframe
          ktemp = MathSqrt(1.0/30.0);       // coefficient for M30
          break;
      default: ktemp = 0;                   // for undefined timeframes, coefficient is zero
      }
   return ktemp;
   }

//+-----------------------------Interface Section-------------------------------------+
bool bInterface=true;                 // flag for enabling graphical user interface (GUI)

//+--------------------------------------------------------------+
//| Functions for creating graphical interface elements          |
//+--------------------------------------------------------------+

//+--------------------------------------------------------------+
//| Create a rectangular label for the interface                 |
//| Parameters: coordinates, dimensions, colors, styles          |
//| Returns: true on successful creation                         |
//+--------------------------------------------------------------+
bool RectLabelCreate(
const long chart_ID=0, // Chart ID 
const string name="RectLabel", // Label name 
const int sub_window=0, // Subwindow number 
const int x=0, // X coordinate 
const int y=0, // Y coordinate 
const int width=50, // width 
const int height=18, // height 
const color back_clr=C'236,233,216', // background color 
const ENUM_BORDER_TYPE border=BORDER_SUNKEN, // border type 
const ENUM_BASE_CORNER corner=CORNER_LEFT_UPPER, // chart corner for anchoring 
const color clr=clrRed, // flat border color (Flat) 
const ENUM_LINE_STYLE style=STYLE_SOLID, // flat border style 
const int line_width=1, // flat border thickness 
const bool back=false, // in the background 
bool selection=false, // select for moving 
const bool hidden=true, // hidden in the objects list 
const long z_order=0) // mouse click priority 
{ 
//--- reset error value 
ResetLastError(); 
//--- create rectangular label 
if(!ObjectCreate(chart_ID,name,OBJ_RECTANGLE_LABEL,sub_window,0,0)) 
   { 
   Print(__FUNCTION__, ": failed to create rectangular label! Error code = ",GetLastError()); 
   return(false); 
   } 
//--- set label coordinates 
ObjectSetInteger(chart_ID,name,OBJPROP_XDISTANCE,x); 
ObjectSetInteger(chart_ID,name,OBJPROP_YDISTANCE,y); 
//--- set label sizes 
ObjectSetInteger(chart_ID,name,OBJPROP_XSIZE,width); 
ObjectSetInteger(chart_ID,name,OBJPROP_YSIZE,height); 
//--- set background color 
ObjectSetInteger(chart_ID,name,OBJPROP_BGCOLOR,back_clr); 
//--- set border type 
ObjectSetInteger(chart_ID,name,OBJPROP_BORDER_TYPE,border); 
//--- set the chart corner relative to which coordinates will be determined 
ObjectSetInteger(chart_ID,name,OBJPROP_CORNER,corner); 
//--- set flat border color (in Flat mode) 
ObjectSetInteger(chart_ID,name,OBJPROP_COLOR,clr); 
//--- set flat border line style 
ObjectSetInteger(chart_ID,name,OBJPROP_STYLE,style); 
//--- set flat border thickness 
ObjectSetInteger(chart_ID,name,OBJPROP_WIDTH,line_width); 
//--- display on the foreground (false) or background (true) 
ObjectSetInteger(chart_ID,name,OBJPROP_BACK,back); 
//--- enable (true) or disable (false) the label movement mode by mouse 
ObjectSetInteger(chart_ID,name,OBJPROP_SELECTABLE,selection); 
ObjectSetInteger(chart_ID,name,OBJPROP_SELECTED,selection); 
//--- hide (true) or display (false) the GUI object name in the objects list 
ObjectSetInteger(chart_ID,name,OBJPROP_HIDDEN,hidden); 
//--- set priority for receiving a mouse click event on the chart 
ObjectSetInteger(chart_ID,name,OBJPROP_ZORDER,z_order); 
//--- successful execution 
return(true); 
}

//+--------------------------------------------------------------+
//| Create a text label for the interface                        |
//| Parameters: coordinates, text, font, color                   |
//| Returns: true on successful creation                         |
//+--------------------------------------------------------------+
bool LabelCreate(const long              chart_ID=0,               // Chart ID
                 const string            name="Label",             // Label name
                 const int               sub_window=0,             // Subwindow number
                 const int               x=0,                      // X coordinate
                 const int               y=0,                      // Y coordinate
                 const ENUM_BASE_CORNER  corner=CORNER_LEFT_UPPER, // Chart corner for anchoring
                 const string            text="Label",             // text
                 const string            font="Arial",             // font
                 const int               font_size=10,             // font size
                 const color             clr=clrRed,               // color
                 const double            angle=0.0,                // text angle
                 const ENUM_ANCHOR_POINT anchor=ANCHOR_LEFT_UPPER, // anchor type
                 const bool              back=false,               // in the background
                 const bool              selection=false,          // select for moving
                 const bool              hidden=true,              // hidden in the objects list
                 const long              z_order=0)                // mouse click priority
  {
//--- reset error value
   ResetLastError();
//--- create text label
   if(!ObjectCreate(chart_ID,name,OBJ_LABEL,sub_window,0,0))
     {
      Print(__FUNCTION__,
            ": failed to create text label! Error code = ",GetLastError());
      return(false);
     }
//--- set label coordinates
   ObjectSetInteger(chart_ID,name,OBJPROP_XDISTANCE,x);
   ObjectSetInteger(chart_ID,name,OBJPROP_YDISTANCE,y);
//--- set the chart corner relative to which coordinates will be determined
   ObjectSetInteger(chart_ID,name,OBJPROP_CORNER,corner);
//--- set text
   ObjectSetString(chart_ID,name,OBJPROP_TEXT,text);
//--- set text font
   ObjectSetString(chart_ID,name,OBJPROP_FONT,font);
//--- set font size
   ObjectSetInteger(chart_ID,name,OBJPROP_FONTSIZE,font_size);
//--- set text angle
   ObjectSetDouble(chart_ID,name,OBJPROP_ANGLE,angle);
//--- set anchor type
   ObjectSetInteger(chart_ID,name,OBJPROP_ANCHOR,anchor);
//--- set color
   ObjectSetInteger(chart_ID,name,OBJPROP_COLOR,clr);
//--- display on the foreground (false) or background (true)
   ObjectSetInteger(chart_ID,name,OBJPROP_BACK,back);
//--- enable (true) or disable (false) the label movement mode by mouse
   ObjectSetInteger(chart_ID,name,OBJPROP_SELECTABLE,selection);
   ObjectSetInteger(chart_ID,name,OBJPROP_SELECTED,selection);
//--- hide (true) or display (false) the GUI object name in the objects list
   ObjectSetInteger(chart_ID,name,OBJPROP_HIDDEN,hidden);
//--- set priority for receiving a mouse click event on the chart
   ObjectSetInteger(chart_ID,name,OBJPROP_ZORDER,z_order);
//--- successful execution
   return(true);
  }

//+-----------------------------End of interface creation functions----------------------------------+

//+--------------------------------------------------------------+
//| User interface object names array                            |
//| Contains unique names of all graphical interface elements    |
//| Used for creating and managing elements on the chart         |
//+--------------------------------------------------------------+
string OwnObjectNames[] = {
"AlgoWealthX-BorderPanel",      // graphical interface border frame (RectLabel)
"AlgoWealthX-MainPanel",        // main interface panel (RectLabel)
"AlgoWealthX-EAName",           // text label with expert name (Label)
"AlgoWealthX-Balance",          // text label with account balance (Label)
"AlgoWealthX-Equity",           // text label with account equity (Label)
"AlgoWealthX-Leverage",         // text label with account leverage (Label)
"AlgoWealthX-Broker",           // text label with broker name (Label)
"AlgoWealthX-DaysRemaining",    // text label with remaining work days (Label)
"AlgoWealthX-StartDate",        // text label with trading start date (Label)
"AlgoWealthX-EndDate",          // text label with trading end date (Label)
"AlgoWealthX-Magic",            // text label with expert magic number (Label)
};

//+--------------------------------------------------------------+
//| Calculation of days since trading start                      |
//| Used to limit the robot's working time                       |
//| Returns: number of days as a fractional number               |
//+--------------------------------------------------------------+
double Days()
   {
   datetime ClosestTime=0;                // closest start time (maximum value between 0 and DatetimeStart is used)
   if (BotO.DatetimeStart > ClosestTime) ClosestTime = BotO.DatetimeStart;
         
   if ( double(ChartO.TimeI[0] - ClosestTime) > 0 ) return double(ChartO.TimeI[0] - ClosestTime)/86400.0;
   else return 0.0;     
   }

datetime LastRead;                    // last settings read date for interface update frequency optimization

//+--------------------------------------------------------------+
//| Interface state update                                       |
//| Displays current information about expert operation          |
//+--------------------------------------------------------------+
//| Updates information labels on the chart                      |
//| Displays current trading account state and robot settings    |
//| Shows: balance, equity, leverage, broker, remaining days     |
//+--------------------------------------------------------------+
void UpdateStatus()
   {
   string TempText="EA name : ";        // temporary string variable for forming interface information label text
   TempText+=EANICK;//name
   ObjectSetString(0,OwnObjectNames[2],OBJPROP_TEXT,TempText);
   TempText="Balance : ";
   TempText+=DoubleToString(NormalizeDouble(AccountInfoDouble(ACCOUNT_BALANCE),2),2);
   ObjectSetString(0,OwnObjectNames[3],OBJPROP_TEXT,TempText);
   TempText="Equity : ";
   TempText+=DoubleToString(NormalizeDouble(AccountInfoDouble(ACCOUNT_EQUITY),2),2);
   ObjectSetString(0,OwnObjectNames[4],OBJPROP_TEXT,TempText);
   TempText="Leverage : 1/";
   TempText+=DoubleToString(AccountInfoInteger(ACCOUNT_LEVERAGE),0);
   ObjectSetString(0,OwnObjectNames[5],OBJPROP_TEXT,TempText);
   TempText="Broker : ";
   TempText+=AccountInfoString(ACCOUNT_COMPANY);
   ObjectSetString(0,OwnObjectNames[6],OBJPROP_TEXT,TempText);
   // Update information on working days
   TempText="Days remaining : ";
   if ( double(DaysToFuture)-Days() > 0 ) TempText+=DoubleToString(NormalizeDouble(double(DaysToFuture)-Days(),2),2);
   else TempText+="0, trading disabled!";
   ObjectSetString(0,OwnObjectNames[7],OBJPROP_TEXT,TempText);
   TempText="Optimization finished : ";
   MqlDateTime StartDate;                 // robot compilation time structure with components to display start date
   TimeToStruct(__DATETIME__,StartDate);
   TempText+=IntegerToString(StartDate.day)+"."+IntegerToString(StartDate.mon)+"."+IntegerToString(StartDate.year);
   ObjectSetString(0,OwnObjectNames[8],OBJPROP_TEXT,TempText);
   TempText="Will trade until : ";
   MqlDateTime EndDate;                   // robot termination time structure with components to display end trading date
   TimeToStruct(__DATETIME__+datetime(DaysToFuture*86400),EndDate);
   TempText+=IntegerToString(EndDate.day)+"."+IntegerToString(EndDate.mon)+"."+IntegerToString(EndDate.year);
   ObjectSetString(0,OwnObjectNames[9],OBJPROP_TEXT,TempText);
   // Update magic number information
   TempText="Magic : ";
   TempText+=IntegerToString(MagicE);
   ObjectSetString(0,OwnObjectNames[10],OBJPROP_TEXT,TempText);
   // Finish interface update
   }

//+--------------------------------------------------------------+
//| Calculate corrected font size for different DPI              |
//| Function adapts font size to screen resolution to            |
//| ensure interface readability on different monitors           |
//| Parameters: BasicFont - base font size                       |
//| Returns: corrected font size (from 6 to 72)                  |
//+--------------------------------------------------------------+
int CalcFontCorrected(int BasicFont)
   {
   int EtalonDPI=100;                     // reference DPI value (100%) for calculating font correction coefficient
   double DPI = TerminalInfoInteger(TERMINAL_SCREEN_DPI); // current terminal DPI value to adapt font size
   int FirstCorrection = int(MathFloor(double(EtalonDPI)/DPI * BasicFont)); // pre-corrected font size based on reference and current DPI ratio
   if (FirstCorrection < 6)//less than minimum font
      {
         return 6;
      }
   else if (FirstCorrection > 72)//greater than maximum font
      {
         return 72;
      }
   else
      {
      return FirstCorrection;
      }      
   }
   
//+----------------------------------------------------------------+
//| Create simple informational interface                          |
//| Function creates a panel with info on trading account state,   |
//| including balance, equity, leverage, broker, and working dates |
//| Places text labels with adaptive font size                     |
//+----------------------------------------------------------------+
void CreateSimpleInterface()
   {
   ENUM_BASE_CORNER corner=CORNER_LEFT_UPPER; // anchor all interface elements to the top-left corner of the chart
   int x=5;                                    // interface X-offset from the chart corner
   int y=15;                                   // interface Y-offset from the chart corner
   int Width = 280;                            // main interface panel width in pixels
   int Height = 195;                           // main interface panel height in pixels
   int Border = 5;                             // border size (offset) between frame and main panel
   
   RectLabelCreate(0,OwnObjectNames[0],0,x,y,Width+2*Border,Height+2*Border,clrLightCyan,BORDER_RAISED,corner,clrOrchid,STYLE_SOLID,3,false,false,true,0);//frame
   RectLabelCreate(0,OwnObjectNames[1],0,x+Border,y+Border,Width,Height,clrDarkSlateBlue,BORDER_RAISED,corner,clrOrchid,STYLE_SOLID,3,false,false,true,0);//main panel
   //text elements
   LabelCreate(0,OwnObjectNames[2],0,x+Border+2,y+8+Border,corner,"EA Name : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0);//
   LabelCreate(0,OwnObjectNames[3],0,x+Border+2,y+17+Border+20*1,corner,"Balance : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT);//
   LabelCreate(0,OwnObjectNames[4],0,x+Border+2,y+17+Border+20*2,corner,"Equity : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT);//
   LabelCreate(0,OwnObjectNames[5],0,x+Border+2,y+17+Border+20*3,corner,"Leverage : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT);//
   LabelCreate(0,OwnObjectNames[6],0,x+Border+2,y+17+Border+20*4,corner,"Broker : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT);//
   LabelCreate(0,OwnObjectNames[7],0,x+Border+2,y+17+Border+20*5,corner,"DaysRemaining : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT);//           
   LabelCreate(0,OwnObjectNames[8],0,x+Border+2,y+17+Border+20*6,corner,"StartDate : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT);//
   LabelCreate(0,OwnObjectNames[9],0,x+Border+2,y+17+Border+20*7,corner,"EndDate : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT);//
   LabelCreate(0,OwnObjectNames[10],0,x+Border+2,y+17+Border+20*8,corner,"Magic : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT);//
   }

//+--------------------------------------------------------------+
//| Remove all interface elements                                |
//| Function clears the chart from all created graphical objects |
//| with the "AlgoWealthX" prefix for correct termination        |
//+--------------------------------------------------------------+
void DeleteSimpleInterface()
   {
   ObjectsDeleteAll(0,"AlgoWealthX");
   }
